/*The example code involves the following conditions.
	CCLK = 1000 MHz, DCLK = 533 MHz, SYSCLK = 500  MHz, SCLK0 = 125 MHz
	- MDMA0 channel 8, required throughput =  500 MHz * 4 <= 2000 MB/s.
	- MDMA1 channel 18, required throughput = 500 MHz * 4 <= 2000 MB/s.
	- MDMA2 channel 39, required throughput = 500 MHz * 4 <= 2000 MB/s.
	- MDMA3 channel 43, required throughput = 500 MHz * 8 <= 4000 MB/s.
	- Throughout requirements depend upon the corresponding DMAx_BWLCNT register values. If not programmed, the MDMA channels request for the bandwidth with full throttle.

The code can be run under different conditions by commenting/un-commenting the corresponding conditions/optimization techniques in the "Multiple_DMAs.h" file.
*/
#include <sys/platform.h>
#include "adi_initialize.h"
#include "sru.h"
#include <services/int/adi_int.h>

#include "Multiple_DMAs.h"

char __argv_string[] = "";

volatile clock_t clock_start_mdma0=0;
volatile clock_t clock_stop_mdma0=0;
volatile clock_t cycles_mdma0=0;
volatile clock_t cycles_mdma0_pre;
float throughput_mdma0=0;

volatile clock_t clock_start_mdma1=0;
volatile clock_t clock_stop_mdma1=0;
volatile clock_t cycles_mdma1=0;
volatile clock_t cycles_mdma1_pre;
float throughput_mdma1=0;

volatile clock_t clock_start_mdma2=0;
volatile clock_t clock_stop_mdma2=0;
volatile clock_t cycles_mdma2=0;
volatile clock_t cycles_mdma2_pre;
float throughput_mdma2=0;

volatile clock_t clock_start_mdma3;
volatile clock_t clock_stop_mdma3=0;
volatile clock_t cycles_mdma3=0;
volatile clock_t cycles_mdma3_pre;
float throughput_mdma3=0;

float throughput_total=0;

#pragma align 32
unsigned char ucIntDest2Buff[MAX_SIZE];

#pragma align 32
unsigned char ucIntDest1Buff[MAX_SIZE];

#pragma align 32
unsigned char ucIntDestBuff[MAX_SIZE];

#pragma align 32
unsigned char ucIntSrc1Buff[MAX_SIZE];

void mdma0_Interrupt_Handler(uint32_t iid, void* handlerArg);
void mdma1_Interrupt_Handler(uint32_t iid, void* handlerArg);
void mdma2_Interrupt_Handler(uint32_t iid, void* handlerArg);
void mdma3_Interrupt_Handler(uint32_t iid, void* handlerArg);


int DMA_Count_mdma0=0;
int DMA_Count_mdma1=0;
int DMA_Count_mdma2=0;
int DMA_Count_mdma3=0;

int main(int argc, char *argv[])
{
	unsigned int uiBytes=0; int i;

	adi_initComponents();
	

	//Work unit size = 4 pages = 8K words
	uiBytes=2048*4;

	*pREG_SPU0_SECUREP110=0x3;
	*pREG_SPU0_SECUREP111=0x3;
	*pREG_SPU0_SECUREP112=0x3;
	*pREG_SPU0_SECUREP113=0x3;
	*pREG_SPU0_SECUREP101=0x3;
	*pREG_SPU0_SECUREP102=0x3;

	//Enables mdma1
	#ifdef ENABLE_MDMA0_DMA_2000MBPS
		adi_int_InstallHandler(INTR_MDMA0_DST, mdma0_Interrupt_Handler, NULL, true);
		MDMA_CONFIG(8, SOURCE_DMA_CFG_MDMA0 , MDMA0_DMA_DMC_START_ADDRESS, SOURCE_MSIZE_MDMA0, uiBytes/SOURCE_MSIZE_MDMA0,9,DEST_DMA_CFG_MDMA0, (uint32_t)ucIntDest2Buff|0x28000000, DEST_MSIZE_MDMA0, uiBytes/DEST_MSIZE_MDMA0);

		#ifdef LIMIT_MDMA0_BW
			*pREG_DMA8_BWLCNT=SOURCE_MSIZE_MDMA0*SYSCLK_SPEED/LIMIT_MDMA0_BW;
		#endif

	#endif

	//Enables mdma1
	#ifdef ENABLE_MDMA1_DMA_2000MBPS
		adi_int_InstallHandler(INTR_MDMA1_DST, mdma1_Interrupt_Handler, NULL, true);
		MDMA_CONFIG(18, SOURCE_DMA_CFG_MDMA1 , MDMA1_DMA_DMC_START_ADDRESS, SOURCE_MSIZE_MDMA1, uiBytes/SOURCE_MSIZE_MDMA1,19,DEST_DMA_CFG_MDMA1, (uint32_t)ucIntDest2Buff|0x28000000, DEST_MSIZE_MDMA2, uiBytes/DEST_MSIZE_MDMA2);

		#ifdef LIMIT_MDMA1_BW
			*pREG_DMA18_BWLCNT=SOURCE_MSIZE_MDMA1*SYSCLK_SPEED/LIMIT_MDMA1_BW;//2048;
		#endif

	#endif

	//Enables mdma2
	#ifdef ENABLE_MDMA2_DMA_2000MBPS
		adi_int_InstallHandler(INTR_MDMA2_DST, mdma2_Interrupt_Handler, NULL, true);
		MDMA_CONFIG(39, SOURCE_DMA_CFG_MDMA2 , MDMA2_DMA_DMC_START_ADDRESS, SOURCE_MSIZE_MDMA2, uiBytes/SOURCE_MSIZE_MDMA2,40,DEST_DMA_CFG_MDMA2, (uint32_t)ucIntDestBuff|0x28000000, DEST_MSIZE_MDMA2, uiBytes/DEST_MSIZE_MDMA2);

		#ifdef LIMIT_MDMA2_BW
			*pREG_DMA39_BWLCNT=SOURCE_MSIZE_MDMA2*SYSCLK_SPEED/LIMIT_MDMA2_BW;//2048;
		#endif
	#endif

	//Enables mdma3
	#ifdef ENABLE_MDMA3_DMA_4000MBPS
		adi_int_InstallHandler(INTR_MDMA3_DST, mdma3_Interrupt_Handler, NULL, true);
		MDMA_CONFIG(43, SOURCE_DMA_CFG_MDMA3 , MDMA3_DMA_DMC_START_ADDRESS, SOURCE_MSIZE_MDMA3, uiBytes/SOURCE_MSIZE_MDMA3,44,DEST_DMA_CFG_MDMA3, (uint32_t)ucIntDest1Buff|0x28000000, DEST_MSIZE_MDMA3, uiBytes/DEST_MSIZE_MDMA3);

		#ifdef LIMIT_MDMA3_BW
			*pREG_DMA43_BWLCNT=SOURCE_MSIZE_MDMA3*SYSCLK_SPEED/LIMIT_MDMA3_BW;//2048;
		#endif

	#endif

#define WAIT_COUNT 500

	while(DMA_Count_mdma0<WAIT_COUNT||DMA_Count_mdma1<WAIT_COUNT||DMA_Count_mdma2<WAIT_COUNT||DMA_Count_mdma3<WAIT_COUNT);

	//Disable the peripherals and MDMAs
	DISABLE_MDMAs(8,9);
	DISABLE_MDMAs(18,19);
	DISABLE_MDMAs(39,40);
	DISABLE_MDMAs(43,44);


	#ifdef ENABLE_MDMA0_DMA_2000MBPS
		throughput_mdma0=(float)uiBytes*CCLK_SPEED/cycles_mdma0;
		printf("Throughput mdma0=%f MB/s\n\n", throughput_mdma0);
	#endif

	#ifdef ENABLE_MDMA1_DMA_2000MBPS
		throughput_mdma1=(float)uiBytes*CCLK_SPEED/cycles_mdma1;
		printf("Throughput mdma1=%f MB/s\n\n", throughput_mdma1);
	#endif

	#ifdef ENABLE_MDMA2_DMA_2000MBPS
		throughput_mdma2=(float)uiBytes*CCLK_SPEED/cycles_mdma2;
		printf("Throughput mdma2=%f MB/s\n\n", throughput_mdma2);
	#endif

	#ifdef ENABLE_MDMA3_DMA_4000MBPS
		throughput_mdma3=(float)uiBytes*CCLK_SPEED/cycles_mdma3;
		printf("Throughput mdma3=%f MB/s\n\n", throughput_mdma3);
	#endif

	throughput_total = throughput_mdma0 + throughput_mdma1+ throughput_mdma2 + throughput_mdma3;

	printf("Throughput total=%f MB/s\n\n", throughput_total);

	while(1)
	{
			asm("nop;"); /* wait forever */
	}

}

void mdma0_Interrupt_Handler(uint32_t iid, void* handlerArg)
{
	clock_start_mdma0=clock_stop_mdma0;
	clock_stop_mdma0 = clock();
	cycles_mdma0_pre=cycles_mdma0;
	cycles_mdma0=clock_stop_mdma0-clock_start_mdma0;
	if(DMA_Count_mdma0>1)
	{
		cycles_mdma0=(DMA_Count_mdma0*cycles_mdma0_pre+cycles_mdma0)/(DMA_Count_mdma0+1);
	}

	*pREG_DMA9_STAT |= ENUM_DMA_STAT_IRQDONE;			 //Clear the W1C interrupt status bit

	DMA_Count_mdma0++;
}

void mdma1_Interrupt_Handler(uint32_t iid, void* handlerArg)
{
	clock_start_mdma1=clock_stop_mdma1;
	clock_stop_mdma1 = clock();
	cycles_mdma1_pre=cycles_mdma1;
	cycles_mdma1=clock_stop_mdma1-clock_start_mdma1;
	if(DMA_Count_mdma1>1)
	{
		cycles_mdma1=(DMA_Count_mdma1*cycles_mdma1_pre+cycles_mdma1)/(DMA_Count_mdma1+1);
	}

	*pREG_DMA19_STAT |= ENUM_DMA_STAT_IRQDONE;			 //Clear the W1C interrupt status bit

	DMA_Count_mdma1++;
}

void mdma2_Interrupt_Handler(uint32_t iid, void* handlerArg)
{
	clock_start_mdma2=clock_stop_mdma2;
	clock_stop_mdma2 = clock();
	cycles_mdma2_pre=cycles_mdma2;
	cycles_mdma2=clock_stop_mdma2-clock_start_mdma2;
	if(DMA_Count_mdma2>1)
	{
		cycles_mdma2=(DMA_Count_mdma2*cycles_mdma2_pre+cycles_mdma2)/(DMA_Count_mdma2+1);
	}

	*pREG_DMA40_STAT |= ENUM_DMA_STAT_IRQDONE;			 //Clear the W1C interrupt status bit

	DMA_Count_mdma2++;
}

void mdma3_Interrupt_Handler(uint32_t iid, void* handlerArg)
{
	clock_start_mdma3=clock_stop_mdma3;
	clock_stop_mdma3= clock();
	cycles_mdma3_pre=cycles_mdma3;
	cycles_mdma3=clock_stop_mdma3-clock_start_mdma3;
	if(DMA_Count_mdma3>1)
	{
		cycles_mdma3=(DMA_Count_mdma3*cycles_mdma3_pre+cycles_mdma3)/(DMA_Count_mdma3+1);
	}

	*pREG_DMA44_STAT |= ENUM_DMA_STAT_IRQDONE;			 //Clear the W1C interrupt status bit

	DMA_Count_mdma3++;
}
