/*This code shows how to make use of the DMC SCB priority feature to prioritize DMC transfers
 * for a specific master(e.g. MDMA).
 */

#include "main.h"
#include <sys/platform.h>
#include <sys/adi_core.h>
#include "time.h"
#include "stdio.h"
#include "adi_initialize.h"
#include <services/int/adi_int.h>

#pragma section("seg_l1_block1")
uint32_t ucIntBuff1[16384];

#pragma section("seg_l1_block2")
uint32_t ucIntBuff2[16384];

volatile clock_t clock_start;
volatile clock_t clock_stop_mdma0;
volatile clock_t cycles_mdma0;
float throughput_mdma0;

volatile clock_t clock_stop_mdma1;
volatile clock_t cycles_mdma1;
float throughput_mdma1;

int DMA_Done_MDMA0=0;
int DMA_Done_MDMA1=0;

void MDMA0_Interrupt_Handler(uint32_t iid, void* handlerArg);
void MDMA1_Interrupt_Handler(uint32_t iid, void* handlerArg);

int main(void)
{
	/* Initialize managed drivers and/or services */
    adi_initComponents();

	int failure=0; int i; float fThroughput;

	//DMA work unit size = 2048 bytes
	unsigned int uiBytes=2048;
	unsigned int uiBytes1=2048;

	*pREG_SPU0_SECUREP110=0x3;
	*pREG_SPU0_SECUREP111=0x3;
	*pREG_SPU0_SECUREP112=0x3;
	*pREG_SPU0_SECUREP113=0x3;


	//*pREG_DMC0_PRIO=0x86;			//DMA 8 high priority
	//*pREG_DMC0_PRIO=0x0504;		//DMA 18 high priority
	*pREG_DMC0_PRIOMSK=0xFFFF;

	//Initialize and map the interrupts here
	adi_int_InstallHandler(INTR_MDMA0_DST, MDMA0_Interrupt_Handler, NULL, true);
	adi_int_InstallHandler(INTR_MDMA1_DST, MDMA1_Interrupt_Handler, NULL, true);

	*pREG_DMA8_BWMCNT=0x7FFFFFFF;
	*pREG_DMA18_BWMCNT=0x7FFFFFFF;

	//Configure MDMA0 (8-9) to read from DMC
	MDMA_CONFIG(8, SOURCE_DMA_CFG, (void *)0x80000000, SOURCE_MSIZE, uiBytes/SOURCE_MSIZE,9,DEST_DMA_CFG, (uint32_t)ucIntBuff1|0x28000000, DEST_MSIZE, uiBytes/DEST_MSIZE);


	//Configure MDMA2 (18-19) to read from DMC
	MDMA_CONFIG(18, SOURCE_DMA_CFG, (void *)0x81000000, SOURCE_MSIZE, uiBytes1/SOURCE_MSIZE,19,DEST_DMA_CFG, (uint32_t)ucIntBuff2|0x28000000, DEST_MSIZE, uiBytes1/DEST_MSIZE);


	//Enable the DMAs here
	*pREG_DMA8_CFG |= 0x1;
	*pREG_DMA9_CFG |= 0x1;
	*pREG_DMA18_CFG |= 0x1;
	*pREG_DMA19_CFG |= 0x1;

	//Start the cycle count
	clock_start = 0x7FFFFFFF;

	//Wait for the DMA done
	while(DMA_Done_MDMA0==0||DMA_Done_MDMA1==0);

	//Stop the cycle counts
	cycles_mdma0=clock_start-clock_stop_mdma0;
	cycles_mdma1=clock_start-clock_stop_mdma1;

	//Disable the DMAs
	DISABLE_DMAs(8,9);
	DISABLE_DMAs(18,19);

	//Calculate the throughput values
	throughput_mdma0=(float)(uiBytes*MDMA_SPEED/cycles_mdma0);
	throughput_mdma1=(float)(uiBytes*MDMA_SPEED/cycles_mdma1);
	printf("\n\n No of bytes=%d, No of cycles for MDMA0 =%d\n\n ",uiBytes,cycles_mdma0);
	printf("\n\n No of bytes=%d, No of cycles for MDMA1 =%d\n\n",uiBytes,cycles_mdma1);
	printf("Throughput MDMA0=%f MB/s, Throughput MDMA1=%f MB/s\n\n",throughput_mdma0, throughput_mdma1);
	return 0;

}

void MDMA0_Interrupt_Handler(uint32_t iid, void* handlerArg)
{
	//Stop the cycle count for MDMA0
	clock_stop_mdma0 =  *pREG_DMA8_BWMCNT_CUR;

	*pREG_DMA9_STAT |= ENUM_DMA_STAT_IRQDONE;			 //Clear the W1C interrupt status bit

	DMA_Done_MDMA0 = 1;
}

void MDMA1_Interrupt_Handler(uint32_t iid, void* handlerArg)
{
	//Stop the cycle count for MDMA1
	clock_stop_mdma1 = *pREG_DMA18_BWMCNT_CUR;

	*pREG_DMA19_STAT |= ENUM_DMA_STAT_IRQDONE;			 //Clear the W1C interrupt status bit

	DMA_Done_MDMA1 = 1;
}
