/*************************************************************************
Program Name    : CodeUpdate
Hardware        : ADuC7020(rev.I)
Revision        : 0.1
Date            : Dec 09, 2005
Description     : This program emulates Serial-EEPROM(ATMEL AT24C series) 
                    
                   Clock rate: 180KHz max
                   SCL: P1.0
                   SDA: P1.1
                   I2C Slave Address: 0xA0 and 0xA2
                   Table size: 256 Bytes per slave ID

				  It also demonstrates the use of I2C to clear address 14
				  in order to update code

				  To erase address 14 the following addresses should be filled:
				    A0h[0xFB] = 0x12
				    A0h[0xFC] = 0x34
				    A0h[0xFD] = 0xA5
				    A0h[0xFE] = 0xA5
				    A0h[0xFF] = 0x01

**************************************************************************/

#include <ADuC7020.h>
#include "common.h"

BYTE A0h[256], A2h[256];
BYTE i2c_first = TRUE;
void flash_clear(void);
void protect_page(unsigned int);
void unprotect_page(unsigned int);
int status, ERROR;

int main(void){

    int i;
	ERROR = 0;

    // GPIO configuration
    GP0CON = 0x10000000;                // P0.7(ECLK)
    GP1CON = 0x00000022;                // P1.0(SCL), P1.1(SDA)
    GP4DAT = 0x04000000;                // P4.2(GPO)

    // I2C configuration
    I2C0ID0 = 0xA0;                     // primary I2C address A0h
    I2C0ID1 = 0xA2;                     // secondary I2C address A2h
    I2C0CFG = 0x4201;                   // enable I2C slave, Stop IRQ, disable TX FIFO requst IRQ


    // Initialize table    
    for(i=0;i<256;i++){
        A0h[i]=i;
        A2h[i]=(0xFF-i);
    }

    A0h[255] = 0;

    i2c_first = TRUE;

    FIQEN = 0x00000200;                  // enable I2C0SIRQ

    GP4DAT |= 0x00040000;

    while(1){
	
	        // check if A0h[255] is 0x01 then erase address 0x80014 for i2c loader
        if(A0h[255]==0x01){
			flash_clear();
		}
        if(A0h[255]==0x02){		// flash protection 
			protect_page(0xFFFFFFFE);	 // protect pages 0-3
		}
        if(A0h[255]==0x03){		// flash protection 
			unprotect_page(0xFFFFFFFF);	 // unprotect pages 0-3
		}
	}

    return (0);
}


/********************************************************************/
/*                                                                  */
/*      Flash/EE erase routine - Address 0x80014 for i2c loader     */
/*                                                                  */
/********************************************************************/

void flash_clear(void) __ram{
    WORD cnt = 3000;
    FEEMOD |= 0x08;
    FEEADR = 0x0000;          		 // set address
    FEECON = 0x05;         		     // page erase command
    while((cnt>1) && (!(FEESTA & 0x01))) {cnt--;}
    A0h[255] = 0;
	RSTSTA = 0x2;
    return;
}

void protect_page(unsigned int addr){
    WORD cnt = 3000;
	FEEPRO = addr;
	FEEADR = 0x1234;			 		// Key
	FEEDAT = 0xA5A5;					// Key
	FEEMOD = 0x48;
	FEECON = 0x0C;
    while((cnt>1) && (!(FEESTA & 0x01))) {cnt--;}
    A0h[255] = 0;
	return;								// return 
} 

void unprotect_page(unsigned int addr){
    WORD cnt = 3000;
	FEEPRO = addr;
	FEEADR = (A0h[251] << 8) + A0h[252];			 		// Key
	FEEDAT = (A0h[253] << 8) + A0h[254];					// Key
	FEEMOD = 0x48;
	FEECON = 0x0C;
    while((cnt>1) && (!(FEESTA & 0x01))) {cnt--;}
    A0h[255] = 0;
	return;								// return 
} 


