#include <Arduino.h>
#include <stdint.h>

// Constants
#define ID_STRING "USBSPI,PIC,01,01,DC,DC590,----------------------\n"
#define EVAL_ID_STRING "LTC5597,Cls,D5596,01,01,DC,DC3128A,-------------\n"

#define BUFFER_SIZE 64
#define BUFFER_LAST BUFFER_SIZE - 1

#define MEASURE_DELAY 5
#define NUM_AVERAGES 20

#define VOLT_PIN 0
#define ENABLE_PIN 7

#define SCALE 4.9

// Error codes
#define E_OK 0
#define E_NUM_NO_NEWLINE 1
#define E_BAD_COMMAND 2

// Globals

char buff[BUFFER_SIZE] = "Initial State";
char sm_buff[3] = "##";

// Helper functions

char num_to_char(int num) {
  return char(num + '0');
}

const char* char_to_str(char c) {
  sm_buff[0] = c;
  sm_buff[1] = '\0';
  return sm_buff;
}

// Error function:
// print !# where # is error code, also save full error to buff so user can ask for it.
void error(int error_code, const char* extra) {
  char c = num_to_char(error_code);
  buff[0] = c;
  if (extra != nullptr) {
    strcpy(buff + 1, ": ");
    strcat(buff, extra);
  }

  sm_buff[0] = '!';
  sm_buff[1] = c;
  Serial.println(sm_buff);
}

// Measurement functions

float average_read() {
  float val = 0.0;
  analogRead(VOLT_PIN); // throw out possibly state value
  for (int i = 0; i < NUM_AVERAGES; ++i) {
    val += analogRead(VOLT_PIN);
    delay(MEASURE_DELAY);
  }
  return val / NUM_AVERAGES;
}

double scale_to_mv(double value) {
  return value * SCALE;
}

// Serial functions

char serial_get_char() {
    while (Serial.available() <= 0) { /* wait */ }
    return Serial.read();
}

bool serial_read_line() {
  for (int i = 0; i < BUFFER_LAST; ++i) {
    char c = serial_get_char();
    if (c == '\n') {
      buff[i] = '\0';
      return true;
    }
    buff[i] = c;
  }
  buff[BUFFER_LAST] = '0';
  return false;
}

// User Commands

void id_command() {
  Serial.print(F(ID_STRING));
  Serial.print('\0');
  Serial.flush();
}

void eval_id_command() {
  Serial.print(F(EVAL_ID_STRING));
  Serial.print('\0');
  Serial.flush();
}

void read_counts_command() {
  float val = average_read();
  Serial.println(val);
}

void read_mv_command() {
  float val = scale_to_mv(average_read());
  Serial.println(val);
}

void enable_command(bool enable) {
  digitalWrite(ENABLE_PIN, enable ? HIGH : LOW);
  Serial.println(enable ? "en" : "ds");
}

// Command dispatchers
void handle_commands() {
  char command = serial_get_char();
  switch (command) {
    case 'i':
      id_command();
      return;
    case 'I':
      eval_id_command();
      return;
    case 'r':
      read_counts_command();
      return;
    case 'v':
      read_mv_command();
      return;
    case 'e':
      enable_command(true);
      return;
    case 'E':
      enable_command(false);
      return;
    case 'm':
      Serial.println(buff);
      return;
    default:
      error(E_BAD_COMMAND, char_to_str(command));
  }
}

void setup() {
  pinMode(ENABLE_PIN, OUTPUT);
  digitalWrite(ENABLE_PIN, HIGH);
  Serial.begin(115200);
}

void loop() {
  handle_commands();
}

