"""
This program uses I2C control only, to read/write to the memory chip on the LTC5100 Eval board, DC499A.

IMPORTANT NOTE: AS-IS, THIS PROGRAM DOES NOT WRITE TO THE LTC5100 DIRECTLY.
IT ONLY WRITES AND READS THE 24LC00 MEMORY CHIP!

On power-up, the LTC5100 looks for a memory chip. If it finds one, the LTC5100 reads the register values
from memory, and loads them into the LTC5100 automatically. See LTC5100 data sheet for full details. 

AFTER USING THIS PROGRAM TO MAKE MEMORY CHIP CHANGES OR EDITS, BE SURE TO POWER-CYCLE THE LTC5100 BOARD.
YOU WILL BE AUTOMATICALLY PROMPTED TO DO SO. THIS CAUSES THE REGISTER VALUES TO BE READ INTO THE LTC5100.

Change history:
03/13/2019: New file based on 6414-xx Py.
06/25/2019: Deleted some inaccessable and unused code fragments.
	    This program designed to use the Linear Technology / Analog Devices DC590B 
	    USB MCU board. The board is readily obtainable. Be sure to install QuikEval on your PC or Mac. 
	    QuikEval will automatically install the required USB drivers that are necessary. 
	    One of those drivers will be a serial COM port to the DC590. 
	    Windows Device Manager will tell you the COM port number of the virtual COM port. 
			
	    To run this Python code, you need to install the USPP package, noted below.
	    To do so, simply make sure the uspp.* and SerialPort_win.* files are present in the same directory as this file.
	    
	    You will need to tell this Py code (below) the COM port # that the OS chose for the virtual serial port.
			
	    Hard-wire the LTC5100 SCL, SDA, and GND to the DC590B QuikEval pins with the same name.
			
	    Connect about 1k pullup resistor on the QuikEval connector, from pin 10 to pin 9.
	    This is essential for the DC590B firmware.
	    New users: search for 'the pullup resistor' below. Read the note. 
			
10/21/2019: Minor comments updates.

10/30/2019: More comments updates.
            Fix the 'd' command bug.
            Fixed the bug where the 'i' command caused memory read problems.
            Post new version to the LTC5100 product page.

	    If any questions, contact your Analog Devices Field Sales Engineer, or post your question on EngineerZone at analog.com.
"""

import time
from time import sleep
# import sys
# import msvcrt
# import string
from uspp import *  # USPP-Universal-Serial-Port-Python-Library
                    # Serial comm with DC590B

def Byte2String(int_i):
    """ Converts integers to hex string format as needed by the DC590: """
    String1= hex(int_i)
    String1= String1.upper()
    if int_i < 16:
        # pad the leading zero
        return('0' + String1[2])
    else:
        return(String1[2] + String1[3])
    
def WriteTwoBytes(B1, B2, Verbose):
    """ This is the core code for register writes:
        Writes I2C address + 2 bytes to DUT in <600 us.
        DC590B has I2C_clk ~11 us (92 kHz). """
    if B1 < 0:
        print "ERROR -- DATA RANGE IN WriteTwoBytes() -- nothing written."
        return
    if B1 > 0xff:
        print "ERROR -- DATA RANGE IN WriteTwoBytes() -- nothing written."
        return
    if B2 < 0:
        print "ERROR -- DATA RANGE IN WriteTwoBytes() -- nothing written."
        return
    if B2 > 0xff:
        print "ERROR -- DATA RANGE IN WriteTwoBytes() -- nothing written."
        return
    
    writeString= "sS" + Byte2String(MEM_I2C_Addr) + "S" + Byte2String(B1) + 'S' + Byte2String(B2) + 'p'
    sleep(0.05)
    
    h.write(writeString)
    """ DC590 f/w wait here depends on baud rate
        At 38400 baud, takes about 1ms per Addr+2-byte I2C write, with Verbose off. """
    if Verbose == 1:
        print "Verbose DC590 writeString: ", writeString

def WriteMemByte(MemAddr, MemData, Verbose):
    """ Write the 24LC00 bytes, one at a time. """
    if MemAddr < 0:
        print "ERROR -- ADDR RANGE IN WriteMemByte() -- no data written to IC."
        return
    if MemAddr > 15:
        print "ERROR -- ADDR RANGE IN WriteMemByte() -- no data written to IC."
        return
    if MemData < 0:
        print "ERROR -- DATA RANGE IN WriteMemByte() -- no data written to IC."
        return
    if MemData > 0xff:
        print "ERROR -- DATA RANGE IN WriteMemByte() -- no data written to IC."
        return
        
    WriteTwoBytes(MemAddr, MemData, Verbose)

def toggleBit(int_type, offset):
    mask = 1 << offset
    return(int_type ^ mask)

def ReadMemReg(addr, Verbose): 
    """ addr is int. 0 to 15 for 24LC00.
        Verbose = int 1 for debugging. """
        
    if addr < 0:
        print "ERROR -- ADDRESS RANGE IN ReadMemReg() -- nothing written."
        return
    if addr > 15:
        print "ERROR -- ADDRESS RANGE IN ReadMemReg() -- nothing written."
        return

    SecondByte= addr
    
    writeString= "sS" + Byte2String(MEM_I2C_Addr) + "S" + Byte2String(SecondByte) + "sS" + Byte2String(MEM_I2C_Addr + 1) + "Rp"
    h.write(writeString)

    if Verbose == 1:
        print 'Verbose DC590 writeString: %s.' % (writeString)

    """ Loop until we get the 2 hex char. to read from DC590B:
        Without .inWaiting, just sleep 20ms and do the read. """
    ReadBytes= h.inWaiting()
    while ReadBytes != 2:
        ReadBytes= h.inWaiting()
        
    readString= h.read(2)
    readString= str('0X' + readString)    
    data= int(readString,16)
    
    if Verbose == 1:
        print 'Verbose DC590 writeString: %s.  Readback is 0x%s, or  %s decimal.' % (writeString, '{0:02X}'.format(data), data)
    return(data)

def LoadMemRegSet(RegisterSet, Verbose):
        K= 0
        while K < len(RegisterSet):
            WriteMemByte(K, RegisterSet[K], Verbose)
            K= K + 1

def ReadDC590_info(): 
    print "DC590 Info string:"
    h.write('i')
    sleep(0.5)   # Wait for the complete response
    ReadBytes= h.inWaiting()
    readString= h.read(ReadBytes)
    print '%s bytes returned:' % (ReadBytes)
    print readString
    print ""

def SetDC590_I2C_mode():
        h.write('MI')   # Go to I2C mode
        print 'We just wrote MI to the DC590B.'
        sleep(0.5)
        h.write('spspsp')   # Go to I2C mode
        print 'We just wrote start-stop-start-stop-start-stop to the DC590B.'
        # heads up: for this to work, the DC590 NEEDS the pullup resistor on the DC590 Aux SDA line!!!!!!!!!
        # failure to do so will have you scratching your head for hours.
        sleep(0.5)

##############################################################################################################
""" Future Features List :

SOFTWARE/FIRMWARE FEATURE LIST:
* Have a way to test I2C comm with DUT.
  Eliminate the hangup problem if DUT not jumpered properly.
  Use DC590B 'S' command, look for 'N' or 'X' response.

HARDWARE TESTING AND CHANGES TO BE DONE:

"""

""" Global Variables Declaration """
MemRegName= ['   SYS_CONFIG', '     IB_LIMIT', 'FLT_CONFIG(L)', 'FLT_CONFIG(H)', '        IB(L)', '        IB(H)', '       IB_TC1', '       IB_TC2', '        IM(L)', '        IM(H)', '       IM_TC1', '       IM_TC2', '     T_NOM(L)', '     T_NOM(H)', '    LOOP_GAIN', '      PEAKING']

#            0     1     2     3     4     5     6     7     8     9    10    11    12    13    14    15
RegFav1= [0x12, 0x7f, 0xa0, 0x04, 0xff, 0x0c, 0x00, 0x00, 0xff, 0x0c, 0x00, 0x00, 0x50, 0x00, 0x84, 0x10]
# For input pwr of 0dBm CW, 30MHz,
# these values should give about -1 dBm output power on SpecAn. Don't forget the DC Block on SA.
# LTC5100 DC current consumption should show about 65mA from the +5V.

#            0     1     2     3     4     5     6     7     8     9    10    11    12    13    14    15
RegOldies= [0x18, 0x00, 0xe0, 0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x54, 0x02, 0x84, 0x10]
# For input pwr of 0dBm CW, 30MHz,
# WARNING - changing registers 4 and 8 will change the gain!!!!!

#I2C_Addr= 0x46    # 7-bit I2C address. Uses top 7 bits. Make sure LSB=0.
LTC_I2C_Addr= 0x0A
MEM_I2C_Addr= 0xAE

Verbose= 1

print 'Python ',sys.version
print 'This program dedicated to LTC5100 test. Requires a DC590B.'
print "Opening serial port comm to DC590B..."
print 'Uses the FTDI Windows USB to COM port driver, '
print 'automatically installed with the LTC QuikEval package.'
print 'You will need to make sure the Python COM port matches the COM port'
print 'that Windows assigned to your USB Serial Port for the DC590B.'
# Open serial port using USPP (uspp.py and SerialPort_win.py):
#h=SerialPort("COM8", 500, 38400)
h=SerialPort("COM9", 500, 38400)
SetDC590_I2C_mode()

print 'DUT PCB: DC499A.'
print ""

command = '_'
while command !='q':
    print ""
    print "================== LTC5100 I2C TEST MENU ===================="
    print " 1  load default  RF register values into Memory chip"
    print " 2  load old RF register values into Memory chip (test only)"
    print " d  Document the register values, all on 1 line."
    print " i  read DC590B info strings (for test only)"
    print " q  Quit "
    print " r  Read one memory register"
    print " R  Read all memory registers"
    print " w  Write a memory register"
    
    command = raw_input("Enter Command: ")

    if command == '1':  #  RF register values for testing:
        LoadMemRegSet(RegFav1, 0)
        print "Register values for RF test written to the Memory IC."
        print "Power-cycle the DC499A +5V to transfer all values into the LTC5100."
        
    if command == '2':  #  RF register values for testing:
        LoadMemRegSet(RegOldies, 0)
        print "Oldies register values for RF test written to the Memory IC."
        print "Power-cycle the DC499A +5V to transfer all values into the LTC5100."
        
    if command == 'd':  # Document the register values (print on one line)
        Kount= 0
        print 'Reg[0..15]= [',
        while Kount <= 15:
            AmpB=ReadMemReg(Kount, 0)
            print '0x%s' % ('{0:02X}'.format(AmpB)),
            Kount= Kount + 1
        print "]"
        print ""
        print "These values are read from the memory chip."
        print "Power-cycle the DC499A +5V to transfer all values into the LTC5100."
        
    if command == 'i':
        ReadDC590_info()
           
    if command == 'q':
        pass
    
    if command == 'r':
        addr=int(raw_input(" Enter byte number, 0 through 15 decimal: "))
        Readback=ReadMemReg(addr, Verbose)
        print 'The contents of memory chip, byte %s is 0x%s, or %s dec.' % (addr, '{0:02X}'.format(Readback), Readback)
        print ""
        print "Power-cycle the DC499A +5V to transfer all values into the LTC5100."


    if command == 'R':
        Kount= 0
        print "Memory chip contents are as follows:"
        print 'Reg #       RegName    Value'
        while Kount <= 15:
            Readback=ReadMemReg(Kount, 0)
            print ' %s     %s 0x%s' % ('{0:02}'.format(Kount), MemRegName[Kount], '{0:02X}'.format(Readback))
            Kount= Kount + 1
        print ""
        print "Power-cycle the DC499A +5V to transfer all values into the LTC5100."

    if command == 'w':  #Write memory register
        addr=int(raw_input(" Enter byte number, 0 through 15: "))
        data=raw_input(" Enter data, 2 hex digits: ")
        data= data.upper()
        data= int(data,16)
        WriteMemByte(addr, data, Verbose)

        
print ""
print "Finis..."
