/*
LTC2607
16-bit, Dual Rail-to-Rail DAC

I2C DATA FORMAT (MSB First):

       Byte #1                                    Byte #2                       Byte #3                             Byte #4
                                                                         MSB                                 LSB
START  SA6 SA5 SA4 SA3 SA2 SA1 SA0 W SACK  C3 C2 C1 C0 A3 A2 A1 A0 SACK  D15 D14 D13 D12 D11 D10 D9 D8 SACK  D7 D6 D5 D4 D3  D2  D1  D0  SACK  STOP

Sx   : Address Select Bit
Cx   : Command Bit
Ax   : DAC Address
Dx   : Data Bits

REVISION HISTORY
$Revision: 1382 $
$Date: 2013-03-15 15:41:08 -0700 (Fri, 15 Mar 2013) $

LICENSE
Permission to freely use, copy, modify, and distribute this software for any
purpose with or without fee is hereby granted, provided that the below
copyright notice and this permission notice appear in all copies:

THIS SOFTWARE IS PROVIDED "AS IS" AND LTC DISCLAIMS ALL WARRANTIES
INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS. IN NO
EVENT SHALL LTC BE LIABLE FOR ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL
DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM ANY USE OF SAME, INCLUDING
ANY LOSS OF USE OR DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE
OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
PERFORMANCE OF THIS SOFTWARE.

Copyright 2013 Nuvation Research Corporation
Copyright 2013 Linear Technology Corp. (LTC)
*/

#include <stdint.h>
#include <fcntl.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <linux/i2c-dev.h>
#include "LTC2607.h"

static char *deviceName = NULL;

// Set the device name for the I2C device
void LTC2607_set_device_name(char *device)
{
  if (deviceName != NULL) {
    free( deviceName );
  }

  deviceName = strdup(device);
}

// Query the device name for the I2C device
char *LTC2607_get_device_name()
{
  if (deviceName != NULL) {
    return deviceName;
  }

  return "/dev/i2c-0";
}

// Write the dac_command byte and 16-bit dac_code to the LTC2607.
// Returns the status of the I2C Address write. 0=successful, 1=unsuccessful.
int8_t LTC2607_write(uint8_t i2c_address, uint8_t dac_command, uint8_t dac_address, uint16_t dac_code)
{
  int fd;
  int ret;
  char *device;
  uint8_t command_byte;
  uint8_t buffer[3];

  command_byte = dac_command | dac_address; // Build the DAC command byte

  // Open the I2C device
  device = LTC2607_get_device_name();
  fd = open(device, O_RDWR);
  if (fd < 0)
  {
    return (1);
  }

  // Select the desired address
  ret = ioctl(fd, I2C_SLAVE, i2c_address);
  if (ret < 0)
  {
    close(fd);
    return (1);
  }

  // Build the I2C command
  buffer[0] = command_byte;
  buffer[1] = (dac_code >> 8) & 0xFF;
  buffer[2] = dac_code & 0xFF;

  // Write the command to the I2C bus
  ret = write(fd, buffer, 3);
  if (ret < 3)
  {
    close(fd);
    return (1);
  }

  // Close the device
  close(fd);

  return (0);
}

// Calculate a LTC2607 DAC code given the desired output voltage
// Returns the DAC code
uint16_t LTC2607_code(float dac_voltage, float LTC2607_lsb, float LTC2607_offset)
{
  uint16_t dac_code;
  float float_code;
  float_code = (dac_voltage - LTC2607_offset) / LTC2607_lsb;                                      // Calculate the DAC code
  float_code = (float_code > (floor(float_code) + 0.5)) ? ceil(float_code) : floor(float_code);   // Round
  if (float_code >= 65535.0)                                                                      // Limits the DAC code to 16 bits
    float_code = 65535.0;
  dac_code = (uint16_t) (float_code);                                                             // Convert to unsigned integer
  return (dac_code);
}

// Calculate the LTC2607 offset and LSB voltage given two measured voltages and their corresponding codes
void LTC2607_calibrate(uint16_t dac_code1, uint16_t dac_code2, float voltage1, float voltage2, float *LTC2607_lsb, float *LTC2607_offset)
{
  *LTC2607_lsb = (voltage2 - voltage1) / ((float) (dac_code2 - dac_code1));           // Calculate the LSB
  *LTC2607_offset = voltage1 - ((*LTC2607_lsb) * dac_code1);                          // Calculate the offset
}
