/*
GPIO

Functions to control the Cyclone HPS GPIO. Requires the the appropriate
driver modules are loaded:
  % cd /lib/modules/3.7.0/kernel/drivers/gpio
  % insmod gpio-generic.ko
  % insmod gpio-dw.ko

REVISION HISTORY
 $Revision: $
 $Date: 2013-04-04 $

LICENSE
Permission to freely use, copy, modify, and distribute this software for any
purpose with or without fee is hereby granted, provided that the below
copyright notice and this permission notice appear in all copies:

THIS SOFTWARE IS PROVIDED "AS IS" AND LTC DISCLAIMS ALL WARRANTIES
INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS. IN NO
EVENT SHALL LTC BE LIABLE FOR ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL
DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM ANY USE OF SAME, INCLUDING
ANY LOSS OF USE OR DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE
OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
PERFORMANCE OF THIS SOFTWARE.

Copyright 2013 Nuvation Research Corporation
*/

#include <stdio.h>
#include <stdint.h>
#include <stdlib.h>
#include <unistd.h>
#include "gpio.h"

static char *gpioPath = "/sys/class/gpio";

// Export a GPIO.
// id - the GPIO id number
// Returns a status. 0=successful, 1=unsuccessful.
static int gpio_export(id)
{
  FILE *fp;
  char filename[128];

  snprintf( filename, sizeof(filename), "%s/export", gpioPath);
  filename[sizeof(filename)-1] = 0;

  fp = fopen(filename, "w");
  if(NULL == fp)
  {
    fprintf( stderr, "Unable to open the GPIO export control file.\n");
    return 1;
  }

  fprintf( fp, "%d\n", id);

  fclose(fp);

  return 0;
}

// Initialize the GPIO.
// This function will export the specified GPIO if required and configure
// it for output.
// id - the GPIO id number
// Returns a status. 0=successful, 1=unsuccessful.
int gpio_init(int id)
{
  int ret;
  FILE *fp;
  char filename[128];

  // Create the filename for the GPIO direction control file.
  snprintf( filename, sizeof(filename), "%s/gpio%d/direction", gpioPath, id);
  filename[sizeof(filename)-1] = 0;

  // Try to open it.
  fp = fopen(filename, "r+");
  if(NULL == fp)
  {
    // Unable to open the direction control file, so assume that the GPIO has not
    // been exported. So, try exporting it.
    ret = gpio_export( id);
    if( ret )
    {
      return ret;
    }

    // The GPIO control files should be there now, so try again.
    fp = fopen(filename, "r+");
    if(NULL == fp)
    {
      fprintf(stderr, "Unable to open GPIO control files.\n");
      return 1;
    }
  }

  // Set the direction to be output.
  rewind(fp);
  fprintf(fp, "out\n");
  fclose(fp);

  return 0;
}

// Sets the state of a GPIO
// id - the GPIO id number
// high - if true, the GPIO is set high. It is set low otherwise.
// Returns a status. 0=successful, 1=unsuccessful.
int gpio_set_value(int id, int high)
{
  FILE *fp;
  char filename[128];

  // Create the filename for the GPIO value control file.
  snprintf( filename, sizeof(filename), "%s/gpio%d/value", gpioPath, id);
  filename[sizeof(filename)-1] = 0;

  // Try to open it.
  fp = fopen(filename, "r+");
  if(NULL == fp)
  {
    fprintf(stderr, "Unable to open GPIO control file.\n");
    return 1;
  }

  // Set the output value.
  rewind(fp);
  fprintf(fp, "%d\n", high ? 1 : 0);

  // Close the file
  fclose(fp);

  return 0;
}
