/*
DC934
LTC2607: 16-bit, Dual Rail-to-Rail DAC.

NOTES
  Setup:
   Set the terminal baud rate to 115200 and select the newline terminator.
   A precision voltmeter (to monitor null box voltage).
   No external power supply is required.

USER INPUT DATA FORMAT:
 decimal : 1024
 hex     : 0x400
 octal   : 02000  (leading 0)
 binary  : B10000000000
 float   : 1024.0

REVISION HISTORY
$Revision: 1362 $
$Date: 2013-03-09 20:44:22 -0800 (Sat, 09 Mar 2013) $

LICENSE
Permission to freely use, copy, modify, and distribute this software for any
purpose with or without fee is hereby granted, provided that the below
copyright notice and this permission notice appear in all copies:

THIS SOFTWARE IS PROVIDED "AS IS" AND LTC DISCLAIMS ALL WARRANTIES
INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS. IN NO
EVENT SHALL LTC BE LIABLE FOR ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL
DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM ANY USE OF SAME, INCLUDING
ANY LOSS OF USE OR DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE
OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
PERFORMANCE OF THIS SOFTWARE.

Copyright 2013 Nuvation Research Corporation
Copyright 2013 Linear Technology Corp. (LTC)
*/
#include <stdio.h>
#include <stdint.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include "UserInterface.h"
#include "LTC2607.h"
#include "LTC2422.h"
#include "GPIO.h"

// Define if we need to manually switch between I2C and SPI, if not GPIO is used
//#define MANUAL_SWITCHING

// If not manual, define if we use GPIO to switch between I2C and SPI
#define GPIO_SWITCHING

// Function Prototypes
void print_title();
void print_prompt(int16_t selected_dac);
int16_t prompt_voltage_or_code();
enum {PROMPT_VOLTAGE = 0, PROMPT_CODE = 1};
uint16_t get_voltage(float LTC2607_lsb, float LTC2607_offset);
uint16_t get_code();

int8_t menu_1_select_dac(int16_t *selected_dac);
int8_t menu_2_write_to_input_register(int16_t selected_dac);
int8_t menu_3_write_and_update_dac(int16_t selected_dac);
int8_t menu_4_update_power_up_dac(int16_t selected_dac);
int8_t menu_5_power_down_dac(int16_t selected_dac);
int8_t menu_6_read_adc();
int8_t menu_7_sweep();
int8_t menu_8_calibrate_all();

// Demo Board Name
char demo_name[] = "DC934";
const uint8_t address_map[3] = {LTC2607_DAC_A, LTC2607_DAC_B, LTC2607_ALL_DACS};  // Map entered option 0..2 to DAC address

// Global Variables
float LTC2607_lsb[2] = {7.6295109E-5, 7.6295109E-5};  // The LTC2607 least significant bit value with 5V full-scale
float LTC2422_lsb = 4.7683761E-6;                     // The LTC2422 least significant bit value with 5V full-scale
float LTC2607_offset[2] = {0.0, 0.0};                 // The LTC2422 offset variable

int8_t demo_board_connected;                       // Set to 1 if the demo board is connected.

// Global Constants
const uint16_t LTC2422_TIMEOUT= 1000;  //Set 1 second LTC2422 SPI timeout

const uint16_t CAL_LOW_DAC_CODE = 0x00FF;                               // Calibration code for low_side
const uint16_t CAL_HIGH_DAC_CODE = 0xFF00;                            // Calibration code for high_side

#ifdef MANUAL_SWITCHING
void pause_for_connection(int i2c)
{
  char buffer[256];
  printf( "Connect for %s and press ENTER when complete.",
          i2c ? "I2C" : "SPI");
  fflush(stdout);
  gets(buffer);
}

#define CONNECT_I2C()   pause_for_connection(1)
#define CONNECT_SPI()   pause_for_connection(0)
#else
#ifdef GPIO_SWITCHING
#define GPIO_ALTERA     213
#define GPIO_ARROW      227

static int gpioId = GPIO_ALTERA;

#define CONNECT_I2C()   {gpio_set_value(gpioId, 1); delay(100);}
#define CONNECT_SPI()   {gpio_set_value(gpioId, 0); delay(100);}
#else
#define CONNECT_I2C()
#define CONNECT_SPI()
#endif
#endif


void delay(unsigned int ms)
{
  usleep(ms*1000);
}

void setup()
// Setup the Program
{
  print_title();                          // Displays the title

#ifdef GPIO_SWITCHING
    // Determine which dev board we are running on.
    int ack;

    // Init the GPIOs for both boards
    gpio_init(GPIO_ALTERA);
    gpio_init(GPIO_ARROW);

    // Try Altera first
    // Configure Altera for I2C and Arrow for SPI
    gpioId = GPIO_ALTERA;
    gpio_set_value(GPIO_ALTERA, 1);
    gpio_set_value(GPIO_ARROW, 0);
    delay(100);

    // Try writing to the I2C bus
    ack = LTC2607_write(LTC2607_I2C_ADDRESS, LTC2607_WRITE_COMMAND, LTC2607_DAC_A, 0);
    if(ack)
    {
      // The write failed, so we don't appear to be the Altera board.
      // Swap the settings and try for Arrow.
      gpioId = GPIO_ARROW;
      gpio_set_value(GPIO_ALTERA, 0);
      gpio_set_value(GPIO_ARROW, 1);
      delay(100);

      // Need to change the device driver to use.
      LTC2607_set_device_name( "/dev/i2c-1");

      // Try writing to the I2C bus
      ack = LTC2607_write(LTC2607_I2C_ADDRESS, LTC2607_WRITE_COMMAND, LTC2607_DAC_A, 0);
      if(ack)
      {
        // The write failed, so we don't appear to be the Arrow board either...
        fprintf( stderr, "Unable to detect DC934A board.\n");
        demo_board_connected = 0;
      }
      else
      {
        demo_board_connected = 1;
      }
    }
    else
    {
      demo_board_connected = 1;
    }
#else
  demo_board_connected = 1;               // Assume that the board is connected
#endif

  if (demo_board_connected)               // Prints the prompt if the correct demo board is connected
  {
    print_prompt(0);                      // Prints prompt and indicates that "A" is selected.  Be sure that "A" is default in start of loop().
  } else {
    exit (EXIT_FAILURE);
  }
}

// Select DAC to update
int8_t menu_1_select_dac(int16_t *selected_dac)
{
  printf("Select DAC to be updated (0=A, 1=B, 2=All) ");
  fflush(stdout);

  *selected_dac = read_int();
  if (*selected_dac > 2) *selected_dac=2;  // If user enters and invalid option, default to ALL.

  return(0);  // Always returns success, consider adding a fail code later.
}

// Write to input register only
int8_t menu_2_write_to_input_register(int16_t selected_dac)
{
  uint16_t dac_code;
  int8_t ack = 0;

  if (prompt_voltage_or_code() == PROMPT_VOLTAGE)
    dac_code = get_voltage(LTC2607_lsb[selected_dac], LTC2607_offset[selected_dac]);
  else
    dac_code = get_code();

  CONNECT_I2C();

  ack = LTC2607_write(LTC2607_I2C_ADDRESS, LTC2607_WRITE_COMMAND, address_map[selected_dac], dac_code);
  return(ack);    // Write a code to the LTC2607}
}

int8_t menu_3_write_and_update_dac(int16_t selected_dac)
{
  int8_t ack;
  uint16_t dac_code;
  int16_t voltage_or_code;
  voltage_or_code = prompt_voltage_or_code();
  if (voltage_or_code == PROMPT_VOLTAGE)
    dac_code = get_voltage(LTC2607_lsb[selected_dac], LTC2607_offset[selected_dac]);
  else
    dac_code = get_code();

  CONNECT_I2C();

  ack = LTC2607_write(LTC2607_I2C_ADDRESS, LTC2607_WRITE_UPDATE_COMMAND, address_map[selected_dac], dac_code);    // Write a code to the LTC2607
  return(ack);
}

// Update/Power up DAC
int8_t menu_4_update_power_up_dac(int16_t selected_dac)
{
  int8_t ack;

  CONNECT_I2C();

  ack = LTC2607_write(LTC2607_I2C_ADDRESS, LTC2607_UPDATE_COMMAND, address_map[selected_dac], 0);   // Update only, data is ignored...
  return(ack);
}

// Power down DAC
int8_t menu_5_power_down_dac(int16_t selected_dac)
{
  int8_t ack;

  CONNECT_I2C();

  ack = LTC2607_write(LTC2607_I2C_ADDRESS, LTC2607_POWER_DOWN_COMMAND, address_map[selected_dac], 0);     // Write a code to the LTC2607
  return(ack);
}

int8_t menu_6_read_adc()
{
  float adc_voltage;
  int32_t adc_code;
  uint8_t adc_channel;
  int32_t  adc_code_array[2];           // Array for ADC data. Useful because you don't know which channel until the LTC2422 tells you.
  int8_t return_code;

  CONNECT_SPI();

  // Read ADC
  LTC2422_read(&adc_channel, &adc_code, LTC2422_TIMEOUT);   // Throw out the stale data
  delay(LTC2422_CONVERSION_TIME);

  return_code = LTC2422_read(&adc_channel, &adc_code, LTC2422_TIMEOUT);   // Get current data for both channels
  adc_code_array[adc_channel] = adc_code;                                 // Note that channels may return in any order,
  delay(LTC2422_CONVERSION_TIME);

  return_code = LTC2422_read(&adc_channel, &adc_code, LTC2422_TIMEOUT);   // that is, adc_channel will toggle each reading
  adc_code_array[adc_channel] = adc_code;

  // The DC934A board connects VOUTA to CH1
  adc_voltage = LTC2422_voltage(adc_code_array[1], LTC2422_lsb);
  printf("     ADC A : %6.4f\n", adc_voltage);

  // The DC934A board connects VOUTB to CH0
  adc_voltage = LTC2422_voltage(adc_code_array[0], LTC2422_lsb);
  printf("     ADC B : %6.4f\n", adc_voltage);

  return(return_code);
}

// Sweep
int8_t menu_7_sweep()
{
  int8_t ack = 0;
  printf("Enter the desired number of Sample points: ");
  fflush(stdout);

  uint16_t sample;
  sample = read_int();
  if (sample > 65535)
    sample = 65535;

  uint16_t sample_code;
  sample_code = 65535/sample;

  int32_t adc_code;
  uint8_t adc_channel;

  puts(" Code, DAC A, DAC B");

  float adc_voltage;  
  uint16_t dac_code;
  uint16_t i;
  for (i = 0; i <= sample; i++)
  {
    CONNECT_I2C();

    dac_code = (sample_code*i);
    if (dac_code > 65535)
      dac_code = 65535;
    ack |= LTC2607_write(LTC2607_I2C_ADDRESS, LTC2607_WRITE_UPDATE_COMMAND, LTC2607_ALL_DACS, dac_code);

    CONNECT_SPI();

    // Throw out old data. Need to do a read to start a new conversion.
    LTC2422_read(&adc_channel, &adc_code, LTC2422_TIMEOUT);
    delay(LTC2422_CONVERSION_TIME);

    if (adc_channel != 0)
    {
      // The DC934A board connects VOUTA to CH1
      // IF we just read channel 1 (DAC A), a conversion on channel B has just started.
      // We want the FIRST reading in the table to be DAC A, so flush this conversion,
      // starting another conversion on DAC A.
      LTC2422_read(&adc_channel, &adc_code, LTC2422_TIMEOUT);
      delay(LTC2422_CONVERSION_TIME);
    }

    LTC2422_read(&adc_channel, &adc_code, LTC2422_TIMEOUT);
    delay(LTC2422_CONVERSION_TIME);

    printf("%5d,", dac_code);

    adc_voltage = LTC2422_voltage(adc_code, LTC2422_lsb);
    printf("%6.4f,", adc_voltage);

    LTC2422_read(&adc_channel, &adc_code, LTC2422_TIMEOUT);
    adc_voltage = LTC2422_voltage(adc_code, LTC2422_lsb);
    printf("%6.4f\n", adc_voltage);

    if(adc_channel != 0)
    {
      puts("Out of sync!!"); // This only happens if something bad occurs
      return(1);
    }
  }

  puts("\nCopy and save data points to a .csv file");
  puts("and open in Excel to plot points.");
  return(ack);
}

int8_t menu_8_calibrate_all()
{
  int8_t ack = 0;
  float voltage1[2];                                  // Calibration voltage 1
  float voltage2[2];                                  // Calibration voltage 2

  // Calibrate All
  printf("Measure and Enter Reference Voltage for The ADC: ");
  fflush(stdout);

  float ref_voltage;
  ref_voltage = read_float();
  LTC2422_cal_voltage(ref_voltage, &LTC2422_lsb);

  CONNECT_I2C();

  ack |= LTC2607_write(LTC2607_I2C_ADDRESS, LTC2607_WRITE_UPDATE_COMMAND, LTC2607_DAC_A, CAL_LOW_DAC_CODE);
  ack |= LTC2607_write(LTC2607_I2C_ADDRESS, LTC2607_WRITE_UPDATE_COMMAND, LTC2607_DAC_B, CAL_LOW_DAC_CODE);

  printf("\nCalibrating DACs ... ");
  fflush(stdout);

  delay(2000);

  CONNECT_SPI();

  int32_t adc_code;
  uint8_t adc_channel;
  LTC2422_read(&adc_channel, &adc_code, LTC2422_TIMEOUT);  // Throw away last reading
  delay(LTC2422_CONVERSION_TIME);

  LTC2422_read(&adc_channel, &adc_code, LTC2422_TIMEOUT);
  delay(LTC2422_CONVERSION_TIME);
  if (adc_channel == 0)
    voltage1[1] = LTC2422_voltage(adc_code, LTC2422_lsb);
  if (adc_channel == 1)
    voltage1[0] = LTC2422_voltage(adc_code, LTC2422_lsb);

  LTC2422_read(&adc_channel, &adc_code, LTC2422_TIMEOUT);
  if (adc_channel == 0)
    voltage1[1] = LTC2422_voltage(adc_code, LTC2422_lsb);
  if (adc_channel == 1)
    voltage1[0] = LTC2422_voltage(adc_code, LTC2422_lsb);

  CONNECT_I2C();

  ack |= LTC2607_write(LTC2607_I2C_ADDRESS, LTC2607_WRITE_UPDATE_COMMAND, LTC2607_DAC_A, CAL_HIGH_DAC_CODE);   // Set DAC to Full Scale
  ack |= LTC2607_write(LTC2607_I2C_ADDRESS, LTC2607_WRITE_UPDATE_COMMAND, LTC2607_DAC_B, CAL_HIGH_DAC_CODE);   // Set DAC to Full Scale

  delay(2000);

  CONNECT_SPI();

  LTC2422_read(&adc_channel, &adc_code, LTC2422_TIMEOUT);  // Throw away last reading
  delay(LTC2422_CONVERSION_TIME);

  LTC2422_read(&adc_channel, &adc_code, LTC2422_TIMEOUT);
  delay(LTC2422_CONVERSION_TIME);
  if (adc_channel == 0)
    voltage2[1] = LTC2422_voltage(adc_code, LTC2422_lsb);
  if (adc_channel == 1)
    voltage2[0] = LTC2422_voltage(adc_code, LTC2422_lsb);

  LTC2422_read(&adc_channel, &adc_code, LTC2422_TIMEOUT);
  if (adc_channel == 0)
    voltage2[1] = LTC2422_voltage(adc_code, LTC2422_lsb);
  if (adc_channel == 1)
    voltage2[0] = LTC2422_voltage(adc_code, LTC2422_lsb);

  LTC2607_calibrate(CAL_LOW_DAC_CODE, CAL_HIGH_DAC_CODE, voltage1[0], voltage2[0], &LTC2607_lsb[0], &LTC2607_offset[0]);
  LTC2607_calibrate(CAL_LOW_DAC_CODE, CAL_HIGH_DAC_CODE, voltage1[1], voltage2[1], &LTC2607_lsb[1], &LTC2607_offset[1]);

  puts("Calibration Complete");
  printf("lsb DAC A: %6.4f V\n   offset: %6.4f V\n", LTC2607_lsb[0], LTC2607_offset[0]);
  printf("lsb DAC B: %6.4f V\n   offset: %6.4f V\n", LTC2607_lsb[1], LTC2607_offset[1]);

  return(ack);
}

int loop()
{
  int8_t i2c_ack = 0;
  int8_t spi_error_code = 0;
  static int16_t selected_dac = 0;    // The selected DAC to be updated (0=A, 1=B, 2=All).  Initialized to "A".

  // The main control loop
  if (demo_board_connected)  // Do nothing if the demo board is not connected
  {
      int16_t user_command;  // User input command

      user_command = read_int();  // Read the user command
      switch (user_command)
      {
        case 1:
          menu_1_select_dac(&selected_dac);  // Select a DAC to update
          break;

        case 2:
          i2c_ack |= menu_2_write_to_input_register(selected_dac);  // Write to input register only
          break;

        case 3:
          i2c_ack |= menu_3_write_and_update_dac(selected_dac);
          break;

        case 4:
          i2c_ack |= menu_4_update_power_up_dac(selected_dac);  // Update/Power up DAC
          break;

        case 5:
          i2c_ack |= menu_5_power_down_dac(selected_dac);  // Power down DAC
          break;

        case 6:
          spi_error_code = menu_6_read_adc();
          break;

        case 7:
          i2c_ack |= menu_7_sweep();
          break;

        case 8:
          i2c_ack |= menu_8_calibrate_all();
          break;

        case 9: // Quit
          return 1;
          break;

        default:
          puts("Incorrect Option");
          break;
      }

      if (i2c_ack != 0)
        puts("Error: No Acknowledge. Check I2C Address.");
      if (spi_error_code != 0)
        puts("Error: SPI communication error.");

      puts("\n*************************");
      print_prompt(selected_dac);
  }

  return (0);
}

int16_t prompt_voltage_or_code()
{
  int16_t user_input;
  printf("Type 1 to enter voltage, 2 to enter code: ");
  fflush(stdout);
  user_input = read_int();

  if (user_input != 2)
    return(PROMPT_VOLTAGE);
  else
    return(PROMPT_CODE);
}

uint16_t get_voltage(float LTC2607_lsb, float LTC2607_offset)
{
  float dac_voltage;

  printf("Enter Desired DAC output voltage: ");
  fflush(stdout);
  dac_voltage = read_float();

  return(LTC2607_code(dac_voltage, LTC2607_lsb, LTC2607_offset));
}

uint16_t get_code()
{
  uint16_t returncode;
  puts("Enter Desired DAC Code");
  printf("(Format 32768, 0x8000, 0100000, or B1000000000000000): ");
  fflush(stdout);
  returncode = (uint16_t) read_int();

  return(returncode);
}

void print_title()
// Print the title block
{
  puts("");
  puts("*****************************************************************");
  puts("* DC934A Demonstration Program                                  *");
  puts("*                                                               *");
  puts("* This program demonstrates how to send data to the LTC2607 I2C *");
  puts("* 2 channel DAC. This board also has an LTC2422 SPI ADC for     *");
  puts("* readback. This board demonstrates ADCs, DACs, I2C, SPI, and   *");
  puts("* multi-channel concepts.                                       *");
  puts("*                                                               *");
  puts("*****************************************************************");
}

void print_prompt(int16_t selected_dac)
// Prompt the user for an input command
{
  puts("\nCommand Summary:");

  puts("  1-Select DAC");
  puts("  2-Write to input register (no update)");
  puts("  3-Write and update DAC");
  puts("  4-Update/Power up DAC");
  puts("  5-Power Down DAC");
  puts("  6-Read ADC");
  puts("  7-Sweep");
  puts("  8-Calibrate ALL");
  puts("  9-Quit");
  printf("\n  Selected DAC: ");
  if (selected_dac != 2)
    putchar((char) (selected_dac + 0x41));
  else
    printf("All");
  printf("\n\nEnter a command: ");
}

int main()
{
  // Set up the board and program.
  setup();

  // Read and process user input.
  while( loop() ==  0 );

  return EXIT_SUCCESS;
}
