/*!*****************************************************************************
*
* LTC3351: Super Capacitor Backup Controller with HotSwap
*
* @verbatim
LTC3351 supercap charger with hotswap
* @endverbatim
*
* http://www.linear.com/product/LTC3351
*
* http://www.linear.com/product/LTC3351#demoboards
*
* REVISION HISTORY
* $Revision: 164 $
* $Date: 2018-04-02 18:29:52 -0400 (Mon, 02 Apr 2018) $
*
* Copyright (c) 2018 Analog Devices, Inc.
*
* All rights reserved.
*
* Redistribution and use in source and binary forms, with or without
* modification,
* are permitted provided that the following conditions are met:
*  - Redistributions of source code must retain the above copyright
*    notice, this list of conditions and the following disclaimer.
*  - Redistributions in binary form must reproduce the above copyright
*    notice, this list of conditions and the following disclaimer in
*    the documentation and/or other materials provided with the
*    distribution.
*  - Neither the name of Analog Devices, Inc. nor the names of its
*    contributors may be used to endorse or promote products derived
*    from this software without specific prior written permission.
*  - The use of this software may or may not infringe the patent rights
*    of one or more patent holders.  This license does not release you
*    from the requirement that you obtain separate licenses from these
*    patent holders to use this software.
*  - Use of the software either in source or binary form, must be run
*    on or directly connected to an Analog Devices Inc. component.
*
* THIS SOFTWARE IS PROVIDED BY ANALOG DEVICES "AS IS" AND ANY EXPRESS OR
* IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, NON-INFRINGEMENT,
* MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
* IN NO EVENT SHALL ANALOG DEVICES BE LIABLE FOR ANY DIRECT, INDIRECT,
* INCIDENTAL,SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
* LIMITED TO, INTELLECTUAL PROPERTY RIGHTS, PROCUREMENT OF SUBSTITUTE GOODS
* OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
* HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
* LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
* OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
* DAMAGE.
*
* The Linear Technology Linduino is not affiliated with the official Arduino
* team. However, the Linduino is only possible because of the Arduino team's
* commitment to the open-source community. Please, visit http://www.arduino.cc
* and http://store.arduino.cc, and consider a purchase that will help fund
* their ongoing work.
*
* Generated on: 2018-04-02
*
******************************************************************************/


/*! @file
 *  @ingroup LTC3351
 *  @brief Register aligned modifier of power-on defaults
 *
 *  This is an example program to demonstrate use of the API by
 *  writing each writable register in the LTC3351 back to its
 *  power-on default. The file can be changed easily to modify the LTC3351
 *  default behavior. This register aligned configurator is somewhat more economical
 *  of program memory and execution time than the similar bit-field
 *  aligned configurator at the expense of readability.
 */

#include "LTC3351.h"

void configure_LTC3351_reg(LTC3351_chip_cfg_t *chip)
{
  //ctl_reg: Control Register: Several independent control bits are grouped into this register.
  //  ctl_disable_balancer: Disables the balancer.
  //    LTC3351_CTL_DISABLE_BALANCER_DEFAULT: 0
  //  ctl_force_itst_on: This bit forces the ITST current on. This can be used to discharge the capacitor stack or manually measure capacitance. Note that this only enables the test current, it does not disable the charger. Set ctl_force_charger_off to disable the charger.
  //    LTC3351_CTL_FORCE_ITST_ON_DEFAULT: 0
  //  ctl_force_charger_off: This bit disables the charger.
  //    LTC3351_CTL_FORCE_CHARGER_OFF_DEFAULT: 0
  //  ctl_force_boost_off: This bit disables the boost.
  //    LTC3351_CTL_FORCE_BOOST_OFF_DEFAULT: 0
  //  ctl_hotswap_disable: Disables the HotSwap controller. The gate of the hotswap FET is forced low, disconnecting VIN and VOUT and forcing the switcher into backup mode. This can be used to simulate a power failure for testing.
  //    LTC3351_CTL_HOTSWAP_DISABLE_DEFAULT: 0
  //  ctl_disable_shunt: Disables the shunt feature.
  //    LTC3351_CTL_DISABLE_SHUNT_DEFAULT: 0
  //  ctl_cap_scale: Increases capacitor measurement resolution 100 times, this is used when measuring smaller capacitors.
  //    LTC3351_CTL_CAP_SCALE_ENUM_LARGE_CAP: 0
  //    LTC3351_CTL_CAP_SCALE_ENUM_SMALL_CAP: 1
  //    LTC3351_CTL_CAP_SCALE_DEFAULT: 0
  //  ctl_stop_cap_esr_meas: Stops an active capacitance/ESR measurement; this bit clears itself once a measurement cycle has been stopped.
  //    LTC3351_CTL_STOP_CAP_ESR_MEAS_ENUM_STOP_MEASUREMENT: 1
  //    LTC3351_CTL_STOP_CAP_ESR_MEAS_DEFAULT: 0
  //  ctl_gpi_buffer_en: A one in this bit location enables the input buffer on the GPI pin. With a zero in this location the GPI pin is measured without the buffer.
  //    LTC3351_CTL_GPI_BUFFER_EN_DEFAULT: 0
  //  ctl_start_cap_esr_meas: Begin a capacitance and ESR measurement when possible; this bit clears itself once a measurement cycle begins or becomes pending.
  //    LTC3351_CTL_START_CAP_ESR_MEAS_ENUM_START_MEASUREMENT: 1
  //    LTC3351_CTL_START_CAP_ESR_MEAS_DEFAULT: 0
  LTC3351_write_register(chip, LTC3351_CTL_REG,
                         (0u << LTC3351_CTL_DISABLE_BALANCER_OFFSET) | //SIZE: 1, MAX: 1, DEFAULT: 0
                         (0u << LTC3351_CTL_FORCE_ITST_ON_OFFSET) | //SIZE: 1, MAX: 1, DEFAULT: 0
                         (0u << LTC3351_CTL_FORCE_CHARGER_OFF_OFFSET) | //SIZE: 1, MAX: 1, DEFAULT: 0
                         (0u << LTC3351_CTL_FORCE_BOOST_OFF_OFFSET) | //SIZE: 1, MAX: 1, DEFAULT: 0
                         (0u << LTC3351_CTL_HOTSWAP_DISABLE_OFFSET) | //SIZE: 1, MAX: 1, DEFAULT: 0
                         (0u << LTC3351_CTL_DISABLE_SHUNT_OFFSET) | //SIZE: 1, MAX: 1, DEFAULT: 0
                         (LTC3351_CTL_CAP_SCALE_ENUM_LARGE_CAP << LTC3351_CTL_CAP_SCALE_OFFSET) | //SIZE: 1, MAX: 1, DEFAULT: 0
                         (0u << LTC3351_CTL_STOP_CAP_ESR_MEAS_OFFSET) | //SIZE: 1, MAX: 1, DEFAULT: 0
                         (0u << LTC3351_CTL_GPI_BUFFER_EN_OFFSET) | //SIZE: 1, MAX: 1, DEFAULT: 0
                         (0u << LTC3351_CTL_START_CAP_ESR_MEAS_OFFSET)); //SIZE: 1, MAX: 1, DEFAULT: 0

  //alarm_mask_reg: Mask Alarms Register: Writing a one to any bit in this register enables a rising edge of its respective bit in alarm_reg to trigger an SMBALERT.
  //  mask_alarm_esr_hi: ESR high alarm mask
  //    LTC3351_MASK_ALARM_ESR_HI_DEFAULT: 0
  //  mask_alarm_cap_lo: Capacitance low alarm mask
  //    LTC3351_MASK_ALARM_CAP_LO_DEFAULT: 0
  //  mask_alarm_cap_ov: Capacitor Over Voltage alarm mask
  //    LTC3351_MASK_ALARM_CAP_OV_DEFAULT: 0
  //  mask_alarm_cap_uv: Capacitor Under Voltage alarm mask
  //    LTC3351_MASK_ALARM_CAP_UV_DEFAULT: 0
  //  mask_alarm_iin_oc: Input overcurrent alarm mask
  //    LTC3351_MASK_ALARM_IIN_OC_DEFAULT: 0
  //  mask_alarm_ichg_uc: Charge undercurrent alarm mask
  //    LTC3351_MASK_ALARM_ICHG_UC_DEFAULT: 0
  //  mask_alarm_dtemp_hot: Die temperature hot alarm mask
  //    LTC3351_MASK_ALARM_DTEMP_HOT_DEFAULT: 0
  //  mask_alarm_dtemp_cold: Die temperature cold alarm mask
  //    LTC3351_MASK_ALARM_DTEMP_COLD_DEFAULT: 0
  //  mask_alarm_vout_ov: VOUT Over Voltage alarm mask
  //    LTC3351_MASK_ALARM_VOUT_OV_DEFAULT: 0
  //  mask_alarm_vout_uv: VOUT Under Voltage alarm mask
  //    LTC3351_MASK_ALARM_VOUT_UV_DEFAULT: 0
  //  mask_alarm_vcap_ov: VCAP Over Voltage alarm mask
  //    LTC3351_MASK_ALARM_VCAP_OV_DEFAULT: 0
  //  mask_alarm_vcap_uv: VCAP Under Voltage alarm mask
  //    LTC3351_MASK_ALARM_VCAP_UV_DEFAULT: 0
  //  mask_alarm_vin_ov: VIN Over Voltage alarm mask
  //    LTC3351_MASK_ALARM_VIN_OV_DEFAULT: 0
  //  mask_alarm_vin_uv: VIN Under Voltage alarm mask
  //    LTC3351_MASK_ALARM_VIN_UV_DEFAULT: 0
  //  mask_alarm_gpi_ov: GPI Over Voltage alarm mask
  //    LTC3351_MASK_ALARM_GPI_OV_DEFAULT: 0
  //  mask_alarm_gpi_uv: GPI Under Voltage alarm mask
  //    LTC3351_MASK_ALARM_GPI_UV_DEFAULT: 0
  LTC3351_write_register(chip, LTC3351_ALARM_MASK_REG,
                         (0u << LTC3351_MASK_ALARM_ESR_HI_OFFSET) | //SIZE: 1, MAX: 1, DEFAULT: 0
                         (0u << LTC3351_MASK_ALARM_CAP_LO_OFFSET) | //SIZE: 1, MAX: 1, DEFAULT: 0
                         (0u << LTC3351_MASK_ALARM_CAP_OV_OFFSET) | //SIZE: 1, MAX: 1, DEFAULT: 0
                         (0u << LTC3351_MASK_ALARM_CAP_UV_OFFSET) | //SIZE: 1, MAX: 1, DEFAULT: 0
                         (0u << LTC3351_MASK_ALARM_IIN_OC_OFFSET) | //SIZE: 1, MAX: 1, DEFAULT: 0
                         (0u << LTC3351_MASK_ALARM_ICHG_UC_OFFSET) | //SIZE: 1, MAX: 1, DEFAULT: 0
                         (0u << LTC3351_MASK_ALARM_DTEMP_HOT_OFFSET) | //SIZE: 1, MAX: 1, DEFAULT: 0
                         (0u << LTC3351_MASK_ALARM_DTEMP_COLD_OFFSET) | //SIZE: 1, MAX: 1, DEFAULT: 0
                         (0u << LTC3351_MASK_ALARM_VOUT_OV_OFFSET) | //SIZE: 1, MAX: 1, DEFAULT: 0
                         (0u << LTC3351_MASK_ALARM_VOUT_UV_OFFSET) | //SIZE: 1, MAX: 1, DEFAULT: 0
                         (0u << LTC3351_MASK_ALARM_VCAP_OV_OFFSET) | //SIZE: 1, MAX: 1, DEFAULT: 0
                         (0u << LTC3351_MASK_ALARM_VCAP_UV_OFFSET) | //SIZE: 1, MAX: 1, DEFAULT: 0
                         (0u << LTC3351_MASK_ALARM_VIN_OV_OFFSET) | //SIZE: 1, MAX: 1, DEFAULT: 0
                         (0u << LTC3351_MASK_ALARM_VIN_UV_OFFSET) | //SIZE: 1, MAX: 1, DEFAULT: 0
                         (0u << LTC3351_MASK_ALARM_GPI_OV_OFFSET) | //SIZE: 1, MAX: 1, DEFAULT: 0
                         (0u << LTC3351_MASK_ALARM_GPI_UV_OFFSET)); //SIZE: 1, MAX: 1, DEFAULT: 0

  //monitor_status_mask_reg: Mask Monitor Status Register: Writing a one to any bit in this register enables a rising edge of its respective bit in monitor_status_reg to trigger an SMBALERT.
  //  mask_mon_cap_precharge: Set the SMBALERT when there is a rising edge on mon_cap_precharge
  //    LTC3351_MASK_MON_CAP_PRECHARGE_DEFAULT: 0
  //  mask_mon_shunting: Set the SMBALERT when there is a rising edge on mon_shunting
  //    LTC3351_MASK_MON_SHUNTING_DEFAULT: 0
  //  mask_mon_balancing: Set the SMBALERT when there is a rising edge on mon_balancing
  //    LTC3351_MASK_MON_BALANCING_DEFAULT: 0
  //  mask_mon_power_returned: Set the SMBALERT when there is a rising edge on mon_power_returned
  //    LTC3351_MASK_MON_POWER_RETURNED_DEFAULT: 0
  //  mask_mon_power_failed: Set the SMBALERT when there is a rising edge on mon_power_failed
  //    LTC3351_MASK_MON_POWER_FAILED_DEFAULT: 0
  //  mask_mon_esr_meas_active: Set the SMBALERT when there is a rising edge on mon_esr_meas_active
  //    LTC3351_MASK_MON_ESR_MEAS_ACTIVE_DEFAULT: 0
  //  mask_mon_cap_meas_active: Set the SMBALERT when there is a rising edge on mon_cap_meas_active
  //    LTC3351_MASK_MON_CAP_MEAS_ACTIVE_DEFAULT: 0
  //  mask_mon_disable_charger: Set the SMBALERT when there is a rising edge on mon_disable_charger
  //    LTC3351_MASK_MON_DISABLE_CHARGER_DEFAULT: 0
  //  mask_mon_meas_failed: Set the SMBALERT when there is a rising edge on mon_meas_failed
  //    LTC3351_MASK_MON_MEAS_FAILED_DEFAULT: 0
  //  mask_mon_esr_done: Set the SMBALERT when there is a rising edge on mon_esr_done
  //    LTC3351_MASK_MON_ESR_DONE_DEFAULT: 0
  //  mask_mon_cap_done: Set the SMBALERT when there is a rising edge on mon_cap_done
  //    LTC3351_MASK_MON_CAP_DONE_DEFAULT: 0
  //  mask_mon_capesr_pending: Set the SMBALERT when there is a rising edge on mon_capesr_pending
  //    LTC3351_MASK_MON_CAPESR_PENDING_DEFAULT: 0
  //  mask_mon_meas_active: Set the SMBALERT when there is a rising edge on mon_meas_active
  //    LTC3351_MASK_MON_MEAS_ACTIVE_DEFAULT: 0
  LTC3351_write_register(chip, LTC3351_MONITOR_STATUS_MASK_REG,
                         (0u << LTC3351_MASK_MON_CAP_PRECHARGE_OFFSET) | //SIZE: 1, MAX: 1, DEFAULT: 0
                         (0u << LTC3351_MASK_MON_SHUNTING_OFFSET) | //SIZE: 1, MAX: 1, DEFAULT: 0
                         (0u << LTC3351_MASK_MON_BALANCING_OFFSET) | //SIZE: 1, MAX: 1, DEFAULT: 0
                         (0u << LTC3351_MASK_MON_POWER_RETURNED_OFFSET) | //SIZE: 1, MAX: 1, DEFAULT: 0
                         (0u << LTC3351_MASK_MON_POWER_FAILED_OFFSET) | //SIZE: 1, MAX: 1, DEFAULT: 0
                         (0u << LTC3351_MASK_MON_ESR_MEAS_ACTIVE_OFFSET) | //SIZE: 1, MAX: 1, DEFAULT: 0
                         (0u << LTC3351_MASK_MON_CAP_MEAS_ACTIVE_OFFSET) | //SIZE: 1, MAX: 1, DEFAULT: 0
                         (0u << LTC3351_MASK_MON_DISABLE_CHARGER_OFFSET) | //SIZE: 1, MAX: 1, DEFAULT: 0
                         (0u << LTC3351_MASK_MON_MEAS_FAILED_OFFSET) | //SIZE: 1, MAX: 1, DEFAULT: 0
                         (0u << LTC3351_MASK_MON_ESR_DONE_OFFSET) | //SIZE: 1, MAX: 1, DEFAULT: 0
                         (0u << LTC3351_MASK_MON_CAP_DONE_OFFSET) | //SIZE: 1, MAX: 1, DEFAULT: 0
                         (0u << LTC3351_MASK_MON_CAPESR_PENDING_OFFSET) | //SIZE: 1, MAX: 1, DEFAULT: 0
                         (0u << LTC3351_MASK_MON_MEAS_ACTIVE_OFFSET)); //SIZE: 1, MAX: 1, DEFAULT: 0

  //vcapfb_dac_reg
  //  vcapfb_dac: VCAP Regulation Reference: This register is used to program the capacitor voltage feedback loop's reference voltage. Only bits 3:0 are active. VCAPFB_DAC = 37.5mV * vcapfb_dac + 637.5mV
  //    LTC3351_VCAPFB_DAC_DEFAULT: 10
  //    LTC3351_VCAPFB_DAC_FORMAT_R2U: This is used to program the capacitor voltage feedback loop's reference voltage. Only bits 3:0 are active. CAPFBREF = 37.5mV * vcapfb_dac + 637.5mV. MIN:0.6375, MAX:1.2.
  LTC3351_write_register(chip, LTC3351_VCAPFB_DAC_REG, LTC3351_VCAPFB_DAC_FORMAT_R2U(1.0125)); //SIZE: 4, MAX: 15, DEFAULT: 10

  //vshunt_reg
  //  vshunt: Shunt Voltage Register: This register programs the shunt voltage for each capacitor in the stack. When set below 3.6V, the charger will limit current and the active shunts will shunt current to prevent this voltage from being exceeded. As a capacitor voltage nears this level, the charge current will be reduced. Current will be shunted when the capacitor voltage is within 25mV of vshunt. Vshunt should be programmed at least 50mV higher than the intended final balanced individual capacitor voltage. When programmed above 3.6V no current will be shunted, however the charge current will be reduced as described. 182.8µV per LSB.
  //    LTC3351_VSHUNT_DEFAULT: 14744
  //    (uint16_t)LTC3351_CELL_FORMAT_R2I: Measured voltage between CAP pins or CAP1 and CAPRTN. MIN:-5.9899904, MAX:5.9898076.
  LTC3351_write_register(chip, LTC3351_VSHUNT_REG, (uint16_t)LTC3351_CELL_FORMAT_R2I(2.6952032)); //SIZE: 16, MAX: 65535, DEFAULT: 14744

  //adc_vin_ch_en_reg
  //  adc_vin_vcap4_en: Enables ADC measurement of vcap4 while in charging mode. This bit must be set for capacitance and ESR measurement if there are four capacitors in the stack
  //    LTC3351_ADC_VIN_VCAP4_EN_DEFAULT: 1
  //  adc_vin_vcap3_en: Enables ADC measurement of vcap3 while in charging mode. This bit must be set for capacitance and ESR measurement if there are three or more capacitors in the stack
  //    LTC3351_ADC_VIN_VCAP3_EN_DEFAULT: 1
  //  adc_vin_vcap2_en: Enables ADC measurement of vcap2 while in charging mode. This bit must be set for capacitance and ESR measurement if there are two or more capacitors in the stack.
  //    LTC3351_ADC_VIN_VCAP2_EN_DEFAULT: 1
  //  adc_vin_vcap1_en: Enables ADC measurement of vcap1 while in charging mode. This bit must be set for capacitance and ESR measurement.
  //    LTC3351_ADC_VIN_VCAP1_EN_DEFAULT: 1
  //  adc_vin_vin_en: Enables ADC measurement of vin while in charging mode.
  //    LTC3351_ADC_VIN_VIN_EN_DEFAULT: 0
  //  adc_vin_vcap_en: Enables ADC measurement of vcap while in charging mode.
  //    LTC3351_ADC_VIN_VCAP_EN_DEFAULT: 0
  //  adc_vin_vout_en: Enables ADC measurement of vout while in charging mode.
  //    LTC3351_ADC_VIN_VOUT_EN_DEFAULT: 0
  //  adc_vin_iin_en: Enables ADC measurement of input current while in charging mode.
  //    LTC3351_ADC_VIN_IIN_EN_DEFAULT: 0
  //  adc_vin_gpi_en: Enables ADC measurement of GPI (general purpose input) while in charging mode.
  //    LTC3351_ADC_VIN_GPI_EN_DEFAULT: 0
  //  adc_vin_dtemp_en: Enables ADC measurement of die temperature while in charging mode.
  //    LTC3351_ADC_VIN_DTEMP_EN_DEFAULT: 0
  //  adc_vin_ichg_en: Enables ADC measurement of charge current while in charging mode.
  //    LTC3351_ADC_VIN_ICHG_EN_DEFAULT: 1
  LTC3351_write_register(chip, LTC3351_ADC_VIN_CH_EN_REG,
                         (1u << LTC3351_ADC_VIN_VCAP4_EN_OFFSET) | //SIZE: 1, MAX: 1, DEFAULT: 1
                         (1u << LTC3351_ADC_VIN_VCAP3_EN_OFFSET) | //SIZE: 1, MAX: 1, DEFAULT: 1
                         (1u << LTC3351_ADC_VIN_VCAP2_EN_OFFSET) | //SIZE: 1, MAX: 1, DEFAULT: 1
                         (1u << LTC3351_ADC_VIN_VCAP1_EN_OFFSET) | //SIZE: 1, MAX: 1, DEFAULT: 1
                         (0u << LTC3351_ADC_VIN_VIN_EN_OFFSET) | //SIZE: 1, MAX: 1, DEFAULT: 0
                         (0u << LTC3351_ADC_VIN_VCAP_EN_OFFSET) | //SIZE: 1, MAX: 1, DEFAULT: 0
                         (0u << LTC3351_ADC_VIN_VOUT_EN_OFFSET) | //SIZE: 1, MAX: 1, DEFAULT: 0
                         (0u << LTC3351_ADC_VIN_IIN_EN_OFFSET) | //SIZE: 1, MAX: 1, DEFAULT: 0
                         (0u << LTC3351_ADC_VIN_GPI_EN_OFFSET) | //SIZE: 1, MAX: 1, DEFAULT: 0
                         (0u << LTC3351_ADC_VIN_DTEMP_EN_OFFSET) | //SIZE: 1, MAX: 1, DEFAULT: 0
                         (1u << LTC3351_ADC_VIN_ICHG_EN_OFFSET)); //SIZE: 1, MAX: 1, DEFAULT: 1

  //adc_backup_ch_en_reg
  //  adc_backup_vcap4_en: Enables ADC measurement of vcap4 while in backup mode.
  //    LTC3351_ADC_BACKUP_VCAP4_EN_DEFAULT: 0
  //  adc_backup_vcap3_en: Enables ADC measurement of vcap3 while in backup mode.
  //    LTC3351_ADC_BACKUP_VCAP3_EN_DEFAULT: 0
  //  adc_backup_vcap2_en: Enables ADC measurement of vcap2 while in backup mode.
  //    LTC3351_ADC_BACKUP_VCAP2_EN_DEFAULT: 0
  //  adc_backup_vcap1_en: Enables ADC measurement of vcap1 while in backup mode.
  //    LTC3351_ADC_BACKUP_VCAP1_EN_DEFAULT: 0
  //  adc_backup_vin_en: Enables ADC measurement of vin while in backup mode.
  //    LTC3351_ADC_BACKUP_VIN_EN_DEFAULT: 0
  //  adc_backup_vcap_en: Enables ADC measurement of vcap while in backup mode.
  //    LTC3351_ADC_BACKUP_VCAP_EN_DEFAULT: 0
  //  adc_backup_vout_en: Enables ADC measurement of vout while in backup mode.
  //    LTC3351_ADC_BACKUP_VOUT_EN_DEFAULT: 0
  //  adc_backup_iin_en: Enables ADC measurement of input current while in backup mode.
  //    LTC3351_ADC_BACKUP_IIN_EN_DEFAULT: 0
  //  adc_backup_gpi_en: Enables ADC measurement of GPI (general purpose input) while in backup mode.
  //    LTC3351_ADC_BACKUP_GPI_EN_DEFAULT: 0
  //  adc_backup_dtemp_en: Enables ADC measurement of die temperature while in backup mode.
  //    LTC3351_ADC_BACKUP_DTEMP_EN_DEFAULT: 0
  //  adc_backup_ichg_en: Enables ADC measurement of charge current while in backup mode.
  //    LTC3351_ADC_BACKUP_ICHG_EN_DEFAULT: 0
  LTC3351_write_register(chip, LTC3351_ADC_BACKUP_CH_EN_REG,
                         (0u << LTC3351_ADC_BACKUP_VCAP4_EN_OFFSET) | //SIZE: 1, MAX: 1, DEFAULT: 0
                         (0u << LTC3351_ADC_BACKUP_VCAP3_EN_OFFSET) | //SIZE: 1, MAX: 1, DEFAULT: 0
                         (0u << LTC3351_ADC_BACKUP_VCAP2_EN_OFFSET) | //SIZE: 1, MAX: 1, DEFAULT: 0
                         (0u << LTC3351_ADC_BACKUP_VCAP1_EN_OFFSET) | //SIZE: 1, MAX: 1, DEFAULT: 0
                         (0u << LTC3351_ADC_BACKUP_VIN_EN_OFFSET) | //SIZE: 1, MAX: 1, DEFAULT: 0
                         (0u << LTC3351_ADC_BACKUP_VCAP_EN_OFFSET) | //SIZE: 1, MAX: 1, DEFAULT: 0
                         (0u << LTC3351_ADC_BACKUP_VOUT_EN_OFFSET) | //SIZE: 1, MAX: 1, DEFAULT: 0
                         (0u << LTC3351_ADC_BACKUP_IIN_EN_OFFSET) | //SIZE: 1, MAX: 1, DEFAULT: 0
                         (0u << LTC3351_ADC_BACKUP_GPI_EN_OFFSET) | //SIZE: 1, MAX: 1, DEFAULT: 0
                         (0u << LTC3351_ADC_BACKUP_DTEMP_EN_OFFSET) | //SIZE: 1, MAX: 1, DEFAULT: 0
                         (0u << LTC3351_ADC_BACKUP_ICHG_EN_OFFSET)); //SIZE: 1, MAX: 1, DEFAULT: 0

  //adc_wait_vin_reg
  //  adc_wait_vin: Sets the wait time between ADC measurement groups while in charging mode. The LSB of this register has a weight of 400uS. The ADC measures all enabled channels then waits this time before measuring all channels again. The ADC data is used for balancing and shunting, increasing this time reduces the shunt and balancer update rate and is not typically recommended if shunting or balancing is enabled. If shunting or measuring capacitance/ESR this time may be ignored by the ADC. 400uS per LSB
  //    LTC3351_ADC_WAIT_VIN_DEFAULT: 100
  LTC3351_write_register(chip, LTC3351_ADC_WAIT_VIN_REG, 100u); //SIZE: 16, MAX: 65535, DEFAULT: 100

  //adc_wait_backup_reg
  //  adc_wait_backup: Sets the wait time between ADC measurement groups while in backup mode. The LSB of this register has a weight of 400uS. The ADC measures all enabled channels then waits this time before measuring all channels again. 400uS per LSB
  //    LTC3351_ADC_WAIT_BACKUP_DEFAULT: 100
  LTC3351_write_register(chip, LTC3351_ADC_WAIT_BACKUP_REG, 100u); //SIZE: 16, MAX: 65535, DEFAULT: 100

  //gpi_uv_lvl_reg
  //  gpi_uv_lvl: General Purpose Input Under Voltage Level: This is an alarm threshold for the GPI pin. If enabled, the GPI pin voltage falling below this level will trigger an alarm and an SMBALERT. 182.8µV per LSB
  //    LTC3351_GPI_UV_LVL_DEFAULT: 0
  //    (uint16_t)LTC3351_CELL_FORMAT_R2I: Measured voltage between CAP pins or CAP1 and CAPRTN. MIN:-5.9899904, MAX:5.9898076.
  LTC3351_write_register(chip, LTC3351_GPI_UV_LVL_REG, (uint16_t)LTC3351_CELL_FORMAT_R2I(0.0)); //SIZE: 16, MAX: 65535, DEFAULT: 0

  //gpi_ov_lvl_reg
  //  gpi_ov_lvl: General Purpose Input Over Voltage Level: This is an alarm threshold for the GPI pin. If enabled, the GPI pin voltage rising above this level will trigger an alarm and an SMBALERT. 182.8µV per LSB 
  //    LTC3351_GPI_OV_LVL_DEFAULT: 0
  //    (uint16_t)LTC3351_CELL_FORMAT_R2I: Measured voltage between CAP pins or CAP1 and CAPRTN. MIN:-5.9899904, MAX:5.9898076.
  LTC3351_write_register(chip, LTC3351_GPI_OV_LVL_REG, (uint16_t)LTC3351_CELL_FORMAT_R2I(0.0)); //SIZE: 16, MAX: 65535, DEFAULT: 0

  //vin_uv_lvl_reg
  //  vin_uv_lvl: VIN Under Voltage Level: This is an alarm threshold for the input voltage. If enabled, the input pin voltage falling below this level will trigger an alarm and an SMBALERT. 2.19mV per LSB 
  //    LTC3351_VIN_UV_LVL_DEFAULT: 0
  //    (uint16_t)LTC3351_VIN_FORMAT_R2I: Measured input voltage. MIN:-71.76192, MAX:71.75973.
  LTC3351_write_register(chip, LTC3351_VIN_UV_LVL_REG, (uint16_t)LTC3351_VIN_FORMAT_R2I(0.0)); //SIZE: 16, MAX: 65535, DEFAULT: 0

  //vin_ov_lvl_reg
  //  vin_ov_lvl: VIN Over Voltage Level: This is an alarm threshold for the input voltage. If enabled, the input pin voltage rising above this level will trigger an alarm and an SMBALERT. 2.19mV per LSB 
  //    LTC3351_VIN_OV_LVL_DEFAULT: 0
  //    (uint16_t)LTC3351_VIN_FORMAT_R2I: Measured input voltage. MIN:-71.76192, MAX:71.75973.
  LTC3351_write_register(chip, LTC3351_VIN_OV_LVL_REG, (uint16_t)LTC3351_VIN_FORMAT_R2I(0.0)); //SIZE: 16, MAX: 65535, DEFAULT: 0

  //vcap_uv_lvl_reg
  //  vcap_uv_lvl: VCAP Under Voltage Level: This is an alarm threshold for the capacitor stack voltage. If enabled, the VCAP pin voltage falling below this level will trigger an alarm and an SMBALERT. 1.46mV per LSB 
  //    LTC3351_VCAP_UV_LVL_DEFAULT: 0
  //    (uint16_t)LTC3351_VCAP_FORMAT_R2I: Measured capacitor stack voltage. MIN:-47.84128, MAX:47.83982.
  LTC3351_write_register(chip, LTC3351_VCAP_UV_LVL_REG, (uint16_t)LTC3351_VCAP_FORMAT_R2I(0.0)); //SIZE: 16, MAX: 65535, DEFAULT: 0

  //vcap_ov_lvl_reg
  //  vcap_ov_lvl: VCAP Over Voltage Level: This is an alarm threshold for the capacitor stack voltage. If enabled, the VCAP pin voltage rising above this level will trigger an alarm and an SMBALERT. 1.46mV per LSB
  //    LTC3351_VCAP_OV_LVL_DEFAULT: 0
  //    (uint16_t)LTC3351_VCAP_FORMAT_R2I: Measured capacitor stack voltage. MIN:-47.84128, MAX:47.83982.
  LTC3351_write_register(chip, LTC3351_VCAP_OV_LVL_REG, (uint16_t)LTC3351_VCAP_FORMAT_R2I(0.0)); //SIZE: 16, MAX: 65535, DEFAULT: 0

  //vout_uv_lvl_reg
  //  vout_uv_lvl: VOUT Under Voltage Level: This is an alarm threshold for the output voltage. If enabled, the VOUT pin voltage falling below this level will trigger an alarm and an SMBALERT. 2.19mV per LSB 
  //    LTC3351_VOUT_UV_LVL_DEFAULT: 0
  //    (uint16_t)LTC3351_VOUT_FORMAT_R2I: Measured output voltage MIN:-71.76192, MAX:71.75973.
  LTC3351_write_register(chip, LTC3351_VOUT_UV_LVL_REG, (uint16_t)LTC3351_VOUT_FORMAT_R2I(0.0)); //SIZE: 16, MAX: 65535, DEFAULT: 0

  //vout_ov_lvl_reg
  //  vout_ov_lvl: VOUT Over Voltage Level: This is an alarm threshold for the output voltage. If enabled, the VOUT pin voltage rising above this level will trigger an alarm and an SMBALERT. 2.19mV per LSB
  //    LTC3351_VOUT_OV_LVL_DEFAULT: 0
  //    (uint16_t)LTC3351_VOUT_FORMAT_R2I: Measured output voltage MIN:-71.76192, MAX:71.75973.
  LTC3351_write_register(chip, LTC3351_VOUT_OV_LVL_REG, (uint16_t)LTC3351_VOUT_FORMAT_R2I(0.0)); //SIZE: 16, MAX: 65535, DEFAULT: 0

  //dtemp_cold_lvl_reg
  //  dtemp_cold_lvl: Die Temperature Cold Level: This is an alarm threshold for the die temperature. If enabled, the die temperature falling below this level will trigger an alarm and an SMBALERT. Temperature = 0.0295C per LSB - 274C
  //    LTC3351_DTEMP_COLD_LVL_DEFAULT: 0
  //    (uint16_t)LTC3351_DTEMP_FORMAT_R2I: Measured die temperature. MIN:-1243.9328, MAX:695.903200001.
  LTC3351_write_register(chip, LTC3351_DTEMP_COLD_LVL_REG, (uint16_t)LTC3351_DTEMP_FORMAT_R2I(-274.0)); //SIZE: 16, MAX: 65535, DEFAULT: 0

  //dtemp_hot_lvl_reg
  //  dtemp_hot_lvl: Die Temperature Hot Level: This is an alarm threshold for the die temperature. If enabled, the die temperature rising above this level will trigger an alarm and an SMBALERT. Temperature = 0.0295C per LSB - 274C
  //    LTC3351_DTEMP_HOT_LVL_DEFAULT: 0
  //    (uint16_t)LTC3351_DTEMP_FORMAT_R2I: Measured die temperature. MIN:-1243.9328, MAX:695.903200001.
  LTC3351_write_register(chip, LTC3351_DTEMP_HOT_LVL_REG, (uint16_t)LTC3351_DTEMP_FORMAT_R2I(-274.0)); //SIZE: 16, MAX: 65535, DEFAULT: 0

  //ichg_uc_lvl_reg
  //  ichg_uc_lvl: Charge Undercurrent Level: This is an alarm threshold for the charge current. If enabled, the charge current falling below this level will trigger an alarm and an SMBALERT. 1.955µV/Rsnsc per LSB
  //    LTC3351_ICHG_UC_LVL_DEFAULT: 0
  //    (uint16_t)LTC3351_ICHARGE_FORMAT_R2I: Measured Charge Current. MIN:-10.6769066667, MAX:10.6765808333.
  LTC3351_write_register(chip, LTC3351_ICHG_UC_LVL_REG, (uint16_t)LTC3351_ICHARGE_FORMAT_R2I(0.0)); //SIZE: 16, MAX: 65535, DEFAULT: 0

  //iin_oc_lvl_reg
  //  iin_oc_lvl: Input Overcurrent Level: This is an alarm threshold for the input current. If enabled, the input current rising above this level will trigger an alarm and an SMBALERT. 1.955µV/Rsnsi per LSB
  //    LTC3351_IIN_OC_LVL_DEFAULT: 0
  //    (uint16_t)LTC3351_IIN_FORMAT_R2I: Measured input current. MIN:-4.00384, MAX:4.0037178125.
  LTC3351_write_register(chip, LTC3351_IIN_OC_LVL_REG, (uint16_t)LTC3351_IIN_FORMAT_R2I(0.0)); //SIZE: 16, MAX: 65535, DEFAULT: 0

  //cap_uv_lvl_reg
  //  cap_uv_lvl: Capacitor Under Voltage Level: This is an alarm threshold for each individual capacitor voltage in the stack. If enabled, any capacitor voltage falling below this level will trigger an alarm and an SMBALERT. 182.8µV per LSB.
  //    LTC3351_CAP_UV_LVL_DEFAULT: 0
  //    (uint16_t)LTC3351_CELL_FORMAT_R2I: Measured voltage between CAP pins or CAP1 and CAPRTN. MIN:-5.9899904, MAX:5.9898076.
  LTC3351_write_register(chip, LTC3351_CAP_UV_LVL_REG, (uint16_t)LTC3351_CELL_FORMAT_R2I(0.0)); //SIZE: 16, MAX: 65535, DEFAULT: 0

  //cap_ov_lvl_reg
  //  cap_ov_lvl: Capacitor Over Voltage Level: This is an alarm threshold for each individual capacitor in the stack. If enabled, any capacitor voltage rising above this level will trigger an alarm and an SMBALERT. 182.8µV per LSB 
  //    LTC3351_CAP_OV_LVL_DEFAULT: 0
  //    (uint16_t)LTC3351_CELL_FORMAT_R2I: Measured voltage between CAP pins or CAP1 and CAPRTN. MIN:-5.9899904, MAX:5.9898076.
  LTC3351_write_register(chip, LTC3351_CAP_OV_LVL_REG, (uint16_t)LTC3351_CELL_FORMAT_R2I(0.0)); //SIZE: 16, MAX: 65535, DEFAULT: 0

  //cap_lo_lvl_reg
  //  cap_lo_lvl: Capacitance Low Level: This is an alarm threshold for the measured stack capacitance. If the measured stack capacitance is less than this level it will trigger an alarm and an SMBALERT, if enabled. When ctl_cap_scale is set to 1, capacitance is 3.36µF * RT/RTST per LSB. When ctl_cap_scale is set to 0 it is 336µF * RT/RTST per LSB.
  //    LTC3351_CAP_LO_LVL_DEFAULT: 0
  //    LTC3351_CAP_FORMAT_R2U: Measured capacitor stack capacitance value. When CTL_CAP_SCALE_VALUE is set to 1, capacitance is 3.36µF * RT/RTST per LSB. When CTL_CAP_SCALE_VALUE is set to 0 it is 336µF * RT/RTST per LSB. MIN:0.0, MAX:130.116763636.
  LTC3351_write_register(chip, LTC3351_CAP_LO_LVL_REG, LTC3351_CAP_FORMAT_R2U(0.0)); //SIZE: 16, MAX: 65535, DEFAULT: 0

  //esr_hi_lvl_reg
  //  esr_hi_lvl: ESR High Level: This is an alarm threshold for the measured stack ESR. If enabled, a measurement of stack ESR exceeding this level will trigger an alarm and an SMBALERT. Rsnsc/64 per LSB.
  //    LTC3351_ESR_HI_LVL_DEFAULT: 0
  //    LTC3351_ESR_FORMAT_R2U: Measured capacitor stack equivalent series resistance (ESR) value. MIN:0.0, MAX:6.14390625.
  LTC3351_write_register(chip, LTC3351_ESR_HI_LVL_REG, LTC3351_ESR_FORMAT_R2U(0.0)); //SIZE: 16, MAX: 65535, DEFAULT: 0

  //esr_i_on_settling_reg
  //  esr_i_on_settling: Time to allow the charging current to settle before measuring the charge voltage and current for ESR. Each LSB is 1024 switcher periods.
  //    LTC3351_ESR_I_ON_SETTLING_DEFAULT: 2
  LTC3351_write_register(chip, LTC3351_ESR_I_ON_SETTLING_REG, 2u); //SIZE: 16, MAX: 65535, DEFAULT: 2

  //esr_i_off_settling_reg
  //  esr_i_off_settling: Time to wait after turning the charge current off before measuring the charge voltage and current for ESR. Each LSB is 1024 switcher periods.
  //    LTC3351_ESR_I_OFF_SETTLING_DEFAULT: 8
  LTC3351_write_register(chip, LTC3351_ESR_I_OFF_SETTLING_REG, 8u); //SIZE: 16, MAX: 65535, DEFAULT: 8

  //esr_i_override_reg
  //  esr_i_override: This value overrides the LTC3351's adaptive test current selection for the ESR test. If this register is non-zero, the lower 8 bits will be used as an 8 bit DAC value to set the charge current during the ESR test. Typically this register will not need to be set. ITEST = 32mV * (esr_i_override[7:0] + 1) / 256 / Rsnsc
  //    LTC3351_ESR_I_OVERRIDE_DEFAULT: 0
  LTC3351_write_register(chip, LTC3351_ESR_I_OVERRIDE_REG, 0u); //SIZE: 16, MAX: 65535, DEFAULT: 0

  //cap_i_on_settling_reg
  //  cap_i_on_settling: Time to wait after turning the test current on before measuring the first voltage of the capacitance measurement. Each LSB is 1024 switcher periods.
  //    LTC3351_CAP_I_ON_SETTLING_DEFAULT: 8
  LTC3351_write_register(chip, LTC3351_CAP_I_ON_SETTLING_REG, 8u); //SIZE: 16, MAX: 65535, DEFAULT: 8

  //cap_delta_v_setting_reg
  //  cap_delta_v_setting: The target delta V for the capacitance test. The scale is 182.8µV per LSB. The default is approximately 100mV.
  //    LTC3351_CAP_DELTA_V_SETTING_DEFAULT: 550
  LTC3351_write_register(chip, LTC3351_CAP_DELTA_V_SETTING_REG, 550u); //SIZE: 16, MAX: 65535, DEFAULT: 550

  //min_boost_cap_voltage_reg
  //  min_boost_cap_voltage: If this register is non-zero, it sets the minimum capacitor voltage the boost will operate at. If any capacitor voltage falls below this value in boost mode the boost will be forced off, the boost will not turn back on even if the capacitor voltage rises above this voltage. Only after input power returns will the boost be re-enabled. This prevents the boost from cycling on and off many times once the capacitors' voltage has discharged to the point it can no longer support the system load through the boost. To use this feature vcap[1:num_caps+1] measurements must be enabled in backup mode, see adc_backup_ch_en_reg. Also the capacitor voltages are only measured as often as set by adc_wait_backup.
  //    LTC3351_MIN_BOOST_CAP_VOLTAGE_DEFAULT: 0
  //    (uint16_t)LTC3351_CELL_FORMAT_R2I: Measured voltage between CAP pins or CAP1 and CAPRTN. MIN:-5.9899904, MAX:5.9898076.
  LTC3351_write_register(chip, LTC3351_MIN_BOOST_CAP_VOLTAGE_REG, (uint16_t)LTC3351_CELL_FORMAT_R2I(0.0)); //SIZE: 16, MAX: 65535, DEFAULT: 0

  //min_vout_hs_disable_reg
  //  min_vout_hs_disable: If this register is non-zero, it sets the minimum voltage VOUT is allowed to reach while the HotSwap is disabled. If the voltage falls below this level the ctl_hotswap_disable bit will be cleared, re-enabling the HotSwap controller.  To use this feature the VOUT measurement must be enabled in boost mode, see adc_backup_ch_en_reg. Also the VOUT voltage is only measured as often as set by adc_wait_backup. 
  //    LTC3351_MIN_VOUT_HS_DISABLE_DEFAULT: 0
  //    (uint16_t)LTC3351_VOUT_FORMAT_R2I: Measured output voltage MIN:-71.76192, MAX:71.75973.
  LTC3351_write_register(chip, LTC3351_MIN_VOUT_HS_DISABLE_REG, (uint16_t)LTC3351_VOUT_FORMAT_R2I(0.0)); //SIZE: 16, MAX: 65535, DEFAULT: 0

  //alarm_reg: Alarms Register: A one in any bit in the register indicates its respective alarm has triggered. All bits are active high. Alarms are cleared by clearing (writing 0) the appropriate bit in this register. Setting (writing 1) bits has no effect. For example to clear the alarm_gpi_uv alarm, write 0xFFFD.
  //  alarm_esr_hi: ESR high alarm
  //    LTC3351_ALARM_ESR_HI_DEFAULT: 0
  //  alarm_cap_lo: Capacitance low alarm
  //    LTC3351_ALARM_CAP_LO_DEFAULT: 0
  //  alarm_cap_ov: Capacitor Over Voltage alarm
  //    LTC3351_ALARM_CAP_OV_DEFAULT: 0
  //  alarm_cap_uv: Capacitor Under Voltage alarm
  //    LTC3351_ALARM_CAP_UV_DEFAULT: 0
  //  alarm_iin_oc: Input overcurrent alarm
  //    LTC3351_ALARM_IIN_OC_DEFAULT: 0
  //  alarm_ichg_uc: Charge undercurrent alarm
  //    LTC3351_ALARM_ICHG_UC_DEFAULT: 0
  //  alarm_dtemp_hot: Die temperature hot alarm
  //    LTC3351_ALARM_DTEMP_HOT_DEFAULT: 0
  //  alarm_dtemp_cold: Die temperature cold alarm
  //    LTC3351_ALARM_DTEMP_COLD_DEFAULT: 0
  //  alarm_vout_ov: VOUT Over Voltage alarm
  //    LTC3351_ALARM_VOUT_OV_DEFAULT: 0
  //  alarm_vout_uv: VOUT Under Voltage alarm
  //    LTC3351_ALARM_VOUT_UV_DEFAULT: 0
  //  alarm_vcap_ov: VCAP Over Voltage alarm
  //    LTC3351_ALARM_VCAP_OV_DEFAULT: 0
  //  alarm_vcap_uv: VCAP Under Voltage alarm
  //    LTC3351_ALARM_VCAP_UV_DEFAULT: 0
  //  alarm_vin_ov: VIN Over Voltage alarm
  //    LTC3351_ALARM_VIN_OV_DEFAULT: 0
  //  alarm_vin_uv: VIN Under Voltage alarm
  //    LTC3351_ALARM_VIN_UV_DEFAULT: 0
  //  alarm_gpi_ov: GPI Over Voltage alarm
  //    LTC3351_ALARM_GPI_OV_DEFAULT: 0
  //  alarm_gpi_uv: GPI Under Voltage alarm
  //    LTC3351_ALARM_GPI_UV_DEFAULT: 0
  LTC3351_write_register(chip, LTC3351_ALARM_REG,
                         (0u << LTC3351_ALARM_ESR_HI_OFFSET) | //SIZE: 1, MAX: 1, DEFAULT: 0
                         (0u << LTC3351_ALARM_CAP_LO_OFFSET) | //SIZE: 1, MAX: 1, DEFAULT: 0
                         (0u << LTC3351_ALARM_CAP_OV_OFFSET) | //SIZE: 1, MAX: 1, DEFAULT: 0
                         (0u << LTC3351_ALARM_CAP_UV_OFFSET) | //SIZE: 1, MAX: 1, DEFAULT: 0
                         (0u << LTC3351_ALARM_IIN_OC_OFFSET) | //SIZE: 1, MAX: 1, DEFAULT: 0
                         (0u << LTC3351_ALARM_ICHG_UC_OFFSET) | //SIZE: 1, MAX: 1, DEFAULT: 0
                         (0u << LTC3351_ALARM_DTEMP_HOT_OFFSET) | //SIZE: 1, MAX: 1, DEFAULT: 0
                         (0u << LTC3351_ALARM_DTEMP_COLD_OFFSET) | //SIZE: 1, MAX: 1, DEFAULT: 0
                         (0u << LTC3351_ALARM_VOUT_OV_OFFSET) | //SIZE: 1, MAX: 1, DEFAULT: 0
                         (0u << LTC3351_ALARM_VOUT_UV_OFFSET) | //SIZE: 1, MAX: 1, DEFAULT: 0
                         (0u << LTC3351_ALARM_VCAP_OV_OFFSET) | //SIZE: 1, MAX: 1, DEFAULT: 0
                         (0u << LTC3351_ALARM_VCAP_UV_OFFSET) | //SIZE: 1, MAX: 1, DEFAULT: 0
                         (0u << LTC3351_ALARM_VIN_OV_OFFSET) | //SIZE: 1, MAX: 1, DEFAULT: 0
                         (0u << LTC3351_ALARM_VIN_UV_OFFSET) | //SIZE: 1, MAX: 1, DEFAULT: 0
                         (0u << LTC3351_ALARM_GPI_OV_OFFSET) | //SIZE: 1, MAX: 1, DEFAULT: 0
                         (0u << LTC3351_ALARM_GPI_UV_OFFSET)); //SIZE: 1, MAX: 1, DEFAULT: 0

}
