/******************************************************************************
*
* LTC3351: Super Capacitor Backup Controller with HotSwap
*
* @verbatim
* LTC3351 supercap charger with hotswap
* @endverbatim
*
* http://www.linear.com/product/LTC3351
*
* http://www.linear.com/product/LTC3351#demoboards
*
* REVISION HISTORY
* $Revision: 164 $
* $Date: 2018-04-02 18:29:52 -0400 (Mon, 02 Apr 2018) $
*
* Copyright (c) 2018 Analog Devices, Inc.
*
* All rights reserved.
*
* Redistribution and use in source and binary forms, with or without
* modification,
* are permitted provided that the following conditions are met:
*  - Redistributions of source code must retain the above copyright
*    notice, this list of conditions and the following disclaimer.
*  - Redistributions in binary form must reproduce the above copyright
*    notice, this list of conditions and the following disclaimer in
*    the documentation and/or other materials provided with the
*    distribution.
*  - Neither the name of Analog Devices, Inc. nor the names of its
*    contributors may be used to endorse or promote products derived
*    from this software without specific prior written permission.
*  - The use of this software may or may not infringe the patent rights
*    of one or more patent holders.  This license does not release you
*    from the requirement that you obtain separate licenses from these
*    patent holders to use this software.
*  - Use of the software either in source or binary form, must be run
*    on or directly connected to an Analog Devices Inc. component.
*
* THIS SOFTWARE IS PROVIDED BY ANALOG DEVICES "AS IS" AND ANY EXPRESS OR
* IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, NON-INFRINGEMENT,
* MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
* IN NO EVENT SHALL ANALOG DEVICES BE LIABLE FOR ANY DIRECT, INDIRECT,
* INCIDENTAL,SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
* LIMITED TO, INTELLECTUAL PROPERTY RIGHTS, PROCUREMENT OF SUBSTITUTE GOODS
* OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
* HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
* LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
* OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
* DAMAGE.
*
******************************************************************************/


/*! @file
 *  @ingroup LTC3351
 *  @brief LTC3351 communication library example
 *
 *  This is an example program to demonstrate use of the API. The
 *  my_write_register and my_read_register functions simply display data and
 *  return constant data to allow compiling and demonstration without hardware.
 */

#include "LTC3351.h"
#include "LTC3351_pec.h"
#include <stdio.h>

/*! Example of hardware port configuration. Modify structure contents to match hardware requirements. Passed to read and write functions. */
struct port_configuration
{
  FILE *stream; //!< fprintf stream pointer.
};

/*!
This  is  an  example  of an smbus_read_register function. In a real application
this  function  would  do  an SMBus read_word transaction of slave address
address, and command code command_code, and place the returned data in *data. It
should  return  a  0  for success and a non-zero error code for failure. The API
functions  will  return  this error code in the event of an error. If you do not
need error handling, it should simply always return 0.
*/
int my_read_register(uint8_t address,              //!< Target IC's 7-bit SMBus address
                     uint8_t command_code,         //!< Command Code to be read from
                     uint16_t *data,               //!< Pointer to data destination
                     struct port_configuration *pc //!< Pointer to additional implementation-specific port configuration struct, if required.
                    )
{
  fprintf(pc->stream, "Read word Transaction:\n");
  fprintf(pc->stream, "\tAddress: %d (%#04X)\n",address,address);
  fprintf(pc->stream, "\tCommand_code: %d (%#04X)\n",command_code,command_code);
  *data = rand() % (1u << 16);
  fprintf(pc->stream, "\tRaw read fake data: %d (%#06X)\n",*data,*data);
  fprintf(pc->stream, "\tPEC: %d (%#04X)\n",pec_read_word(address,command_code,*data),pec_read_word(address,command_code,*data));
  return 0;
}

/*!
This  is  an  example of an smbus_write_register function. In a real application
this  function  would  do an SMBus write_word transaction of slave address
address, and command  code command_code, writing data.  It should return a 0 for
success  and  a  non-zero  error code for failure. The API functions will return
this  error code in the event of an error. If you do not need error handling, it
should simply always return 0.
*/
int my_write_register(uint8_t address,              //!< Target IC's 7-bit SMBus address
                      uint8_t command_code,         //!< Command Code to be written to
                      uint16_t data,                //!< Data to be written
                      struct port_configuration *pc //!< Pointer to additional implementation-specific port configuration struct, if required.
                     )
{
  fprintf(pc->stream, "Write word Transaction:\n");
  fprintf(pc->stream, "\tAddress: %d (%#04X)\n",address,address);
  fprintf(pc->stream, "\tCommandCode: %d (%#04X)\n",command_code,command_code);
  fprintf(pc->stream, "\tRaw Data: %d (%#06X)\n",data,data);
  fprintf(pc->stream, "\tPEC: %d (%#04X)\n",pec_write_word(address,command_code,data),pec_write_word(address,command_code,data));
  return 0;
}

int main(void)
{
  uint16_t data;

  struct port_configuration pc =
  {
    .stream = stderr
  };

  LTC3351_chip_cfg_t LTC3351 = 
  {
    .address = LTC3351_ADDR_09,
    .read_register = my_read_register,
    .write_register = my_write_register,
    .port_configuration = &pc
  };

  /*! the API functions can be used to read and write individual bit fields
  within a command code */
  printf("Using Read Register for the LTC3351_CTL_START_CAP_ESR_MEAS bit field\n");
  LTC3351_read_register(&LTC3351, LTC3351_CTL_START_CAP_ESR_MEAS, &data);
  printf("\tRead: %d\n",data);
  

  printf("\nUsing Read/Modify/Write Register for the LTC3351_CTL_START_CAP_ESR_MEAS bit field\n");
  LTC3351_write_register(&LTC3351, LTC3351_CTL_START_CAP_ESR_MEAS, 1);
  

  /*! the API functions can also be used to read and write whole command codes */
  printf("\nUsing Read Register for the LTC3351_CTL_REG command code\n");
  LTC3351_read_register(&LTC3351, LTC3351_CTL_REG, &data);
  printf("\tRead: %d\n",data);
  
  printf("\nUsing Read/Modify/Write Register for the LTC3351_CTL_REG command code\n");
  LTC3351_write_register(&LTC3351, LTC3351_CTL_REG, 5808);
  
  printf("\nUsing Read Register for the LTC3351_CTL_REG command code\n");
  LTC3351_read_register(&LTC3351, LTC3351_CTL_REG, &data);
  printf("\tRead: %d\n",data);
  printf("\tDecoding bit field icharge_jeita_2_bf from command code data: %d\n", LTC3351_CTL_START_CAP_ESR_MEAS_DECODE(data));

  // Example showing how to modify LTC3351 power-on defaults with an inexpensive µC.
  printf("\nRe-writing default values for all registers (modify LTC3351_reg_config.c to change defaults).\n");
  configure_LTC3351_reg(&LTC3351);
  // Alternatively, write each bit-field individually.
  printf("\nRe-writing default values for all bit fields (uses more program memory; modify LTC3351_bf_config.c to change defaults).\n");
  configure_LTC3351_bf(&LTC3351);

  return EXIT_SUCCESS;
}
