//! @todo Review this file?  Document with Doxygen?  Time permitting...
/*
LT8491 USB to Serial Controller

This file contains the routines to emulate the DC590B USB to Serial Converter. All commands
are supported except Uxxy the Write Port D bus. Added the 'D' delay ms command.
With this program, the Linduino can be used by the QuikEval program running on a PC
to communicate with QuikEval compatible demo boards.

The Kxy bit bang command uses the following pin mappings :
0-Linduino 2
1-Linduino 3
2-Linduino 4
3-Linduino 5
4-Linduino 6
5-Linduino 7

REVISION HISTORY
$Revision: 3659 $
$Date: 2015-07-01 10:19:20 -0700 (Wed, 01 Jul 2015) $

Copyright (c) 2013, Linear Technology Corp.(LTC)
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

1. Redistributions of source code must retain the above copyright notice, this
   list of conditions and the following disclaimer.
2. Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

The views and conclusions contained in the software and documentation are those
of the authors and should not be interpreted as representing official policies,
either expressed or implied, of Linear Technology Corp.

The Linear Technology Linduino is not affiliated with the official Arduino team.
However, the Linduino is only possible because of the Arduino team's commitment
to the open-source community.  Please, visit http://www.arduino.cc and
http://store.arduino.cc , and consider a purchase that will help fund their
ongoing work.
 */

#include <Arduino.h>
#include <stdint.h>
#include "Linduino.h"
#include "QuikEval_EEPROM.h"
#include "LT_SPI.h"
#include "UserInterface.h"
#include "LT_I2C.h"
#include <Wire.h>
#include <SPI.h>

#define ACK_PIN       2

// timeouts
#define READ_TIMEOUT  20
#define MISO_TIMEOUT  1000

// recording mode constants
#define RECORDING_SIZE 50
const byte off = 0;
const byte playback = 1;

// serial mode constants
const byte spi_mode = 0;
const byte i2c_mode = 1;
const byte i2c_auxiliary_mode = 2;

// hex conversion constants
char hex_digits[16]=
{
  '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'A', 'B', 'C', 'D', 'E', 'F'
};

// spi clock divider
const char spi_divider = SPI_CLOCK_DIV32;  // configure the spi port for 4MHz SCK (500kHz@div32??)

static uint8_t ackState;

// global variables
byte serial_mode = spi_mode;  // current serial mode
byte recording_mode = off;        // recording mode off
char id_string[44]="LT8491_I2C_Interface,---------------------\n\0"; // id string
char hex_to_byte_buffer[5]=
{
  '0', 'x', '0', '0', '\0'
};               // buffer for ASCII hex to byte conversion
char byte_to_hex_buffer[3]=
{
  '\0','\0','\0'
};                     // buffer for byte to ASCII hex conversion
char recording_buffer[RECORDING_SIZE]=
{
  '\0'
}; // buffer for saving recording loop
byte recording_index = 0;                // index to the recording buffer

byte i2c_addr = 0;

char get_char();

void byte_to_hex(byte value)
// convert a byte to two hex characters
{
  byte_to_hex_buffer[0]=hex_digits[value>>4];        // get upper nibble
  byte_to_hex_buffer[1]=hex_digits[(value & 0x0F)];  // get lower nibble
  byte_to_hex_buffer[2]='\0';                        // add NULL at end
}

byte read_hex()
// read 2 hex characters from the serial buffer and convert
// them to a byte
{
  byte data;
  hex_to_byte_buffer[2]=get_char();
  hex_to_byte_buffer[3]=get_char();
  data = strtol(hex_to_byte_buffer, NULL, 0);
  return(data);
}

char get_char()
// get the next character either from the serial port
// or the recording buffer
{
  char command='\0';
  if (recording_mode != playback)
  {
    // read a command from the serial port
    while (Serial.available() <= 0);
    return(Serial.read());
  }
  else
  {
    // read a command from the recording buffer
    if (recording_index < RECORDING_SIZE)
    {
      command = recording_buffer[recording_index++];
      // disregard loop commands during playback
      if (command == 'w') command='\1';
      if (command == 't') command='\1';
      if (command == 'v') command='\1';
      if (command == 'u') command='\1';
    }
    else
      command = '\0';
    if (command == '\0')
    {
      recording_index = 0;
      recording_mode = off;
    }
    return(command);
  }
}
int i = 0;
unsigned char pseudo_reset = 0;

void PrintHex8(uint8_t *data, uint8_t length) // prints 8-bit data in hex with leading zeroes
{
     char tmp[16];
       for (int i=0; i<length; i++) { 
         sprintf(tmp, "%.2X",data[i]); 
         Serial.print(tmp);
       }
}

void PrintHex16(uint16_t *data, uint8_t length) // prints 16-bit data in hex with leading zeroes
{
       char tmp[16];
       for (int i=0; i<length; i++)
       { 
         sprintf(tmp, "%.4X",data[i]); 
         Serial.print(tmp);
       }
}

void setup()
// Setup the program
{
  digitalWrite(QUIKEVAL_GPIO, LOW);
  digitalWrite(QUIKEVAL_CS, HIGH);
  digitalWrite(ACK_PIN, LOW);
  digitalWrite(3, LOW);
  digitalWrite(4, LOW);
  digitalWrite(5, LOW);
  digitalWrite(6, LOW);
  digitalWrite(7, LOW);
  pinMode(QUIKEVAL_GPIO, OUTPUT);
  pinMode(QUIKEVAL_CS, OUTPUT);
  pinMode(ACK_PIN, OUTPUT);
  pinMode(3, OUTPUT);
  pinMode(4, OUTPUT);
  pinMode(5, OUTPUT);
  pinMode(6, OUTPUT);
  pinMode(7, OUTPUT);

  Serial.begin(115200);          // enable the serial port for 115200 baud

  quikeval_SPI_init();
  quikeval_SPI_connect();   // Connect SPI to main data port

  quikeval_I2C_init();           // Configure the EEPROM I2C port for 100kHz SCK
  Serial.print("hello\n");
  Serial.flush();
}

void loop()
{
  byte tx_data;
  byte rx_data;
  uint16_t rx_data_word;
  byte pin_value;
  int delay_value;
  int pin;
  char command;
  int byte_count;
  long delay_count;
  String inString = "";
  float sclFreq = 0;
  float tempTWBR[4] = {0};
  float tempTWBRremainder = 0;
  float tempTWBRremainderPrev = 1;
  byte prescalar;
  byte success;

  typedef union
  {
    uint8_t   data_8[4];
    uint16_t  data_16[2];
    uint32_t  data_32;
  }traffic_t;

  traffic_t i2cTraffic;
  
  command = get_char();
  switch (command)
  {
    case 'a':
      i2c_addr = read_hex();
      //Serial.print(i2c_addr);
      //Serial.print('\n');
      //Serial.flush();
      break;
    case 'b':
      command = read_hex();
      ackState = i2c_read_byte_data(i2c_addr, command, &rx_data);
      digitalWrite(ACK_PIN, ackState);
      PrintHex8(&rx_data, 1);
      //Serial.print(rx_data, HEX);
      //Serial.print("\n\0");
      //Serial.flush();
      break;
    case 'B':
      command = read_hex();
      tx_data = read_hex();
      ackState = i2c_write_byte_data(i2c_addr, command, tx_data);
      digitalWrite(ACK_PIN, ackState);
      break;
    case 'W':
      command = read_hex();
      ackState = i2c_read_word_data(i2c_addr, command, &rx_data_word);
      digitalWrite(ACK_PIN, ackState);
      PrintHex16(&rx_data_word, 1);
      //Serial.print(rx_data_word, HEX);
      //Serial.print("\n\0");
      //Serial.flush();
      break;
    //write word
    case 'V':
      command = read_hex();
      i2cTraffic.data_8[0] = read_hex();
      i2cTraffic.data_8[1] = read_hex();
      ackState = i2c_write_word_data(i2c_addr, command, i2cTraffic.data_16[0]);
      digitalWrite(ACK_PIN, ackState);
      /*
      Serial.print("----> ");
      Serial.print(i2cTraffic.data_16[0], HEX);
      Serial.print("\n\0");
      Serial.flush();
      */
      break;
    //read long
    case 'e':
      command = read_hex();
      i2c_read_block_data(i2c_addr, command, 4, i2cTraffic.data_8);
      PrintHex8(i2cTraffic.data_8, 4);
      break;
    case 'E':
      command = read_hex();
      i2cTraffic.data_8[0] = read_hex();
      i2cTraffic.data_8[1] = read_hex();
      i2cTraffic.data_8[2] = read_hex();
      i2cTraffic.data_8[3] = read_hex();
      i2c_write_block_data(i2c_addr, command, 4, i2cTraffic.data_8);
      /*
      Serial.print("----> "); 
      Serial.print(i2cTraffic.data_32, HEX);
      Serial.print("\n\0");
      Serial.flush();
      */
      break;
    case 'D':
      // delay milliseconds
      delay_value = read_hex();
      delay_value<<=8;
      delay_value|=read_hex();
      delay(delay_value);
      break;
    case 'g':
      // IO pin low
      output_low(QUIKEVAL_GPIO);
      break;
    case 'G':
      // IO pin high
      output_high(QUIKEVAL_GPIO);
      break;
    case 'H':     // wait for MISO to go high with a timeout
      delay_count = 0;
      while (1)
      {
        if (input(MISO)==1) break;   // MISO is high so quit
        if (delay_count++>MISO_TIMEOUT)
        {
          //Serial.print('T');         // timeout occurred. Print 'T'
          break;
        }
        else delay(1);
      }
      break;
    case 'i':
      // send controller id string
      pseudo_reset = 0;
      Serial.print(id_string);
      Serial.print('\0');
      Serial.flush();
      break;
    case 'I':
      // get controller id string
      quikeval_SPI_connect();   // Connect SPI to main data port
      pseudo_reset = 0;
      byte_count = read_quikeval_id_string(&ui_buffer[0]);
      if (byte_count!=0)
      {
        Serial.print(ui_buffer);
        //Serial.print("LTC4261,Cls,D4261,01,01,DC,DC998A,--------------\n\0");
        Serial.print('\0');
        Serial.flush();
      }
      break;
    case 'K':
      // Bang pin. The pin assignments are :
      // 0: PIND2, Arduino 2
      // 1: PIND3, Arduino 3
      // 2: PIND4, Arduino 4
      // 3: PIND5, Arduino 5
      // 4: PIND6, Arduino 6
      // 5: PIND6, Arduino 7
      pin_value = get_char();          // read the value
      pin = get_char()-0x30;         // read the pin
      if (pin_value == '0') digitalWrite(pin+2, LOW);
      else digitalWrite(pin+2, HIGH);
      break;
    case 'L':
      // wait for MISO to go low with a timeout
      delay_count = 0;
      while (1)
      {
        if (input(MISO)==0) break;   // MISO is low so quit
        if (delay_count++>MISO_TIMEOUT)
        {
          //Serial.print('T');         // timeout occurred. Print 'T'
          break;
        }
        else delay(1);
      }
      break;
    //added to set the I2C clock speed
    case 'l':
      delay(1);
      while(Serial.available() > 0)
      {
        int inChar = Serial.read();
        inString += (char)inChar;
      }
      //Serial.print("Input String: ");
      //Serial.print(inString);
      //Serial.print("\tAfter Conversion to float: ");
      sclFreq = inString.toFloat();
      if((sclFreq <= 400000) && (sclFreq >= 490))
      {
        //Serial.println(sclFreq);
        inString = "";
        //check for best fit
        for(uint8_t i = 0; i < 4; i++)
        {
          
          if(i == 0)
          {
            prescalar = 1;
          }
          else
          {
            prescalar = (prescalar << 2);
          }
          //Serial.print("Prescalar: ");
          //Serial.println(prescalar);
          tempTWBR[i] = (-8*(sclFreq - 1000000))/(sclFreq*prescalar);
          //Serial.print("tempTWBR[");
          //Serial.print(i);
          //Serial.print("] ");
          //Serial.println(tempTWBR[i]);
          if(tempTWBR[i] > 255)
          {
            tempTWBR[i] = -1;
          }
          
          if(tempTWBR[i] != -1)
          {
            //tempTWBRremainder = 1 - (tempTWBR[i] - (long)tempTWBR[i]);
            tempTWBRremainder = ((long)tempTWBR[i] + 1) - tempTWBR[i];
            if(tempTWBRremainder > 0.5)
            {
              tempTWBRremainder = 1 - tempTWBRremainder;
            }
            //Serial.print("tempTWBRremainder ");
            //Serial.println(tempTWBRremainder);
            if(tempTWBRremainder == 0)
            {
              TWBR = (byte)tempTWBR[i];
              TWSR = i;  
              break;       
            }
            else
            {
              if(tempTWBRremainder < tempTWBRremainderPrev)
              {
                TWBR = (byte)(0.5 + tempTWBR[i]);
                TWSR = i;
                tempTWBRremainderPrev = tempTWBRremainder;
              }
            }
          }
        }
        tempTWBRremainderPrev = 1;
        //Serial.print("TWSR: ");
        //Serial.print(TWSR);
        //Serial.print(" TWBR: ");
        //Serial.println(TWBR);
      }
      else
      {
        Serial.println("Invalid Frequency");
      }
      Serial.flush();
      break;
    case 'M':
      // change the serial mode
      command = get_char();
      switch (command)
      {
        case 'I':
          // I2C mode
          serial_mode = i2c_mode;
          // enable_main_I2C();
          quikeval_I2C_connect();
          break;
        case 'S':
          // spi mode
          serial_mode = spi_mode;
          // Need to send command to disable LTC4302
          // enable_main_SPI();
          quikeval_SPI_connect();
          break;
        case 'X':
          // axillary I2C mode - no hardware action necessary, always available.
          serial_mode = i2c_auxiliary_mode;
          quikeval_SPI_connect();
          break;
      }
      //delay(1);
      break;
    case 'p':
      // I2C stop
      if (serial_mode!=spi_mode)   TWCR=(1<<TWINT) | (1<<TWEN) | (1<<TWSTO);  // I2C stop //i2c_stop();
      // if(serial_mode == i2c_auxiliary_mode) i2c_stop();
      break;
    case 'P':
      // ping
      Serial.print('P');
      delay(5);
      break;
    case 'Q':
      // Read byte in I2C mode only. Add ACK
      switch (serial_mode)
      {
        case i2c_mode:
          rx_data = i2c_read(WITH_ACK);
          byte_to_hex(rx_data);
          Serial.print(byte_to_hex_buffer);
          break;
        case i2c_auxiliary_mode:
          rx_data = i2c_read(WITH_ACK);
          byte_to_hex(rx_data);
          Serial.print(byte_to_hex_buffer);
          break;
      }
      break;

    case 'r':
      rx_data = spi_read(0);
      byte_to_hex(rx_data);
      Serial.print(byte_to_hex_buffer);
      break;

    case 'R':
      // Read byte, add NACK in I2C mode
      switch (serial_mode)
      {
        case spi_mode:
          rx_data = spi_read(0);
          byte_to_hex(rx_data);
          Serial.print(byte_to_hex_buffer);
          break;
        case i2c_mode:
          rx_data = i2c_read(WITH_NACK);
          byte_to_hex(rx_data);
          Serial.print(byte_to_hex_buffer);
          break;
        case i2c_auxiliary_mode:
          rx_data = i2c_read(WITH_NACK);
          byte_to_hex(rx_data);
          Serial.print(byte_to_hex_buffer);
          break;
      }
      break;
    case 's':   // I2C start
      if (serial_mode == i2c_mode) i2c_start();
      if (serial_mode == i2c_auxiliary_mode) i2c_start();
      break;
    case 'S':   // send byte
      tx_data = read_hex();
      switch (serial_mode)
      {
        case spi_mode:
          spi_write(tx_data);
          break;
        case i2c_mode:
          if (i2c_write(tx_data)==1) Serial.print('N');
          break;
        case i2c_auxiliary_mode:
          if (i2c_write(tx_data)==1) Serial.print('N');
          break;
      }
      break;
    case 't':   // recording loop
      recording_index = 0;
      do
      {
        command = get_char();
        if (command == 'u')   // stop recording
        {
          recording_buffer[recording_index]='\0';
          recording_index = 0;
          break;
        }
        else            // add character to recording buffer
        {
          if (recording_index < RECORDING_SIZE) recording_buffer[recording_index++]=command;
        }
      }
      while (1);
      break;
    case 'T':   // transceive byte
      tx_data = read_hex();
      if (serial_mode == spi_mode)
      {
        rx_data = spi_read(tx_data);
        byte_to_hex(rx_data);
        Serial.print(byte_to_hex_buffer);
      }
      break;
    case 'v':    // echo recording loop
      Serial.print(recording_buffer);
      break;
    case 'w':
      recording_mode = playback;
      break;
    case 'x':
      output_low(QUIKEVAL_CS);
      break;
    case 'X':
      output_high(QUIKEVAL_CS);
      break;
    case 'Z':    // line feed
      Serial.print('\n');
      Serial.flush();
      break;
    case 0x80:    // Reset
      if (pseudo_reset == 0)
      {
        delay(500); // The delay is needed for older GUI's
        Serial.print("hello\n");
        pseudo_reset = 1;
      }
      break;
  }
}
