// **********************************************************************************
// **********************************************************************************
// ----------------------------------------------------------------------------------
// ################
// ##   ###########   Analog Devices Inc.
// ##      ########
// ##         #####   Copyright (c) 2019 Analog Devices Inc. All rights reserved.
// ##            ##   This file is the confidential and proprietary property of ADI.
// ##         #####   Possession or use of this file requires a written license.
// ##      ########   The licensing information may be found at: www.analog.com
// ##   ###########
// ################
// ----------------------------------------------------------------------------------
// Description:       UART software register controlled
// ----------------------------------------------------------------------------------
// **********************************************************************************
// **********************************************************************************

`timescale 1ps/1ps

module uart (

    // uart interface

    output  wire            uart_tx_data,
    input   wire            uart_tx_ready,
    input   wire            uart_rx_data,
    output  reg             uart_rx_ready = 'd0,

    // uart clock

    input   wire            uart_clk,

    // axilite interface

    input   wire            axilite_clk,
    input   wire            axilite_resetn,
    input   wire            axilite_swreset,
    output  reg             axilite_swreset_clr = 'd0,
    input   wire            axilite_enable,
    input   wire  [31:0]    axilite_clk_period,
    input   wire  [ 1:0]    axilite_bit_width,
    input   wire            axilite_stop_bits,
    input   wire            axilite_parity_enable,
    input   wire            axilite_parity_type,
    input   wire            axilite_rdy_polarity,
    input   wire            axilite_rdy_mode,
    input   wire            axilite_loopback,
    input   wire            axilite_tx_wr,
    input   wire  [ 7:0]    axilite_tx_wdata,
    output  wire            axilite_tx_rdy,
    input   wire            axilite_rx_rd,
    output  wire  [ 7:0]    axilite_rx_rdata,
    input   wire            axilite_rx_rdy,
    output  wire            axilite_tx_full,
    output  wire            axilite_tx_empty,
    output  wire            axilite_rx_full,
    output  wire            axilite_rx_empty,
    output  wire            axilite_rx_perror,
    output  wire            axilite_rx_ovf,
    output  wire  [31:0]    axilite_clk_mon_count);

    // internal registers

    reg           [ 3:0]    uart_rx_cnt = 'd0;
    reg                     uart_rx_lvl = 'd0;
    reg                     uart_rx_ovf = 'd0;
    reg                     uart_rx_wr = 'd0;
    reg           [ 7:0]    uart_rx_wdata = 'd0;
    reg           [ 3:0]    uart_rx_wcnt = 'd0;
    reg           [10:0]    uart_rx_pdata = 'd0;
    reg                     uart_rx_perror = 'd0;
    reg                     uart_tx_rd = 'd0;
    reg                     uart_tx_state = 'd0;
    reg           [ 3:0]    uart_tx_count = 'd0;
    reg                     uart_tx_data_out = 'd0;
    reg           [11:0]    uart_tx_pdata = 'd0;
    reg                     uart_clken = 'd0;
    reg           [31:0]    uart_clkcnt = 'd0;
    reg           [ 3:0]    axilite_rx_rcnt = 'd0;
    reg                     axilite_swresetn = 'd0;

    // internal signals

    wire                    uart_rx_valid;
    wire                    uart_rx_stop;
    wire                    uart_rx_start;
    wire                    uart_rx_perr;
    wire                    uart_rx_pexp;
    wire                    uart_rx_prcv;
    wire          [ 7:0]    uart_rx_data_in;
    wire          [ 4:0]    uart_rx_wcnt_5;
    wire          [ 4:0]    uart_rx_rcnt_5;
    wire          [ 4:0]    uart_rx_cnt_5;
    wire          [ 3:0]    uart_rx_rcnt;
    wire                    uart_tx_start;
    wire          [ 4:0]    uart_ctrl;
    wire          [ 3:0]    uart_length;
    wire          [ 7:0]    uart_tx_rdata;
    wire                    uart_tx_empty;
    wire                    uart_rx_full;
    wire                    uart_rx_sdata;
    wire                    uart_rx_rdy;
    wire                    uart_tx_rdy;
    wire          [31:0]    uart_clk_period;
    wire          [ 1:0]    uart_bit_width;
    wire                    uart_stop_bits;
    wire                    uart_parity_enable;
    wire                    uart_parity_type;
    wire                    uart_rdy_polarity;
    wire                    uart_rdy_mode;
    wire                    uart_resetn;
    wire                    rx_data_in;
    wire                    tx_ready_in;
    wire                    axilite_uart_resetn;

    // receive data select function

    function [7:0] rx_dsel;
        input [ 4:0] ctrl;
        input [11:0] din;
        reg   [ 7:0] dout;
        begin
            case (ctrl[3:0])
                4'b0000: dout = {3'd0, din[10:6]};
                4'b1000: dout = {3'd0, din[ 9:5]};
                4'b0100: dout = {3'd0, din[ 9:5]};
                4'b1100: dout = {3'd0, din[ 8:4]};
                4'b0001: dout = {2'd0, din[10:5]};
                4'b1001: dout = {2'd0, din[ 9:4]};
                4'b0101: dout = {2'd0, din[ 9:4]};
                4'b1101: dout = {2'd0, din[ 8:3]};
                4'b0010: dout = {1'd0, din[10:4]};
                4'b1010: dout = {1'd0, din[ 9:3]};
                4'b0110: dout = {1'd0, din[ 9:3]};
                4'b1110: dout = {1'd0, din[ 8:2]};
                4'b0011: dout = din[10:3];
                4'b1011: dout = din[ 9:2];
                4'b0111: dout = din[ 9:2];
                default: dout = din[ 8:1];
            endcase
            rx_dsel = dout;
        end
    endfunction

    // transmit parity control function

    function [3:0] tx_psel;
        input [ 4:0] ctrl;
        input [ 7:0] din;
        reg   [ 3:0] dout;
        begin
            case (ctrl[1:0])
                2'b00: begin
                    dout[3:1] = 3'b111;
                    dout[0:0] = ^{ctrl[4], din[4:0]} | ~ctrl[3];
                end
                2'b01: begin
                    dout[3:2] = 2'b11;
                    dout[1:1] = ^{ctrl[4], din[5:0]} | ~ctrl[3];
                    dout[0:0] = din[5:5];
                end
                2'b10: begin
                    dout[3:3] = 1'b1;
                    dout[2:2] = ^{ctrl[4], din[6:0]} | ~ctrl[3];
                    dout[1:0] = din[6:5];
                end
                default: begin
                    dout[3:3] = ^{ctrl[4], din[7:0]} | ~ctrl[3];
                    dout[2:0] = din[7:5];
                end
            endcase
            tx_psel = dout;
        end
    endfunction

    // receive

    assign uart_rx_valid = ~uart_rx_start & uart_rx_stop & uart_rx_sdata;
    assign uart_rx_stop = ~uart_stop_bits | uart_rx_pdata[10];
    assign uart_rx_start = (uart_length == 'h7) ? uart_rx_pdata[5] :
        ((uart_length == 'h8) ? uart_rx_pdata[4] :
        ((uart_length == 'h9) ? uart_rx_pdata[3] :
        ((uart_length == 'ha) ? uart_rx_pdata[2] :
        ((uart_length == 'hb) ? uart_rx_pdata[1] :
        ((uart_length == 'hc) ? uart_rx_pdata[0] : 1'd1)))));
    assign uart_rx_perr = uart_parity_enable & (uart_rx_pexp ^ uart_rx_prcv);
    assign uart_rx_pexp = ^{uart_parity_type, uart_rx_data_in};
    assign uart_rx_prcv = (uart_stop_bits == 1'd1) ? uart_rx_pdata[9] :
        uart_rx_pdata[10];
    assign uart_rx_data_in = rx_dsel(uart_ctrl, uart_rx_pdata);

    always @(negedge uart_resetn or posedge uart_clk) begin
        if (uart_resetn == 1'b0) begin
            uart_rx_ready <= 1'd0;
            uart_rx_cnt <= 4'd0;
            uart_rx_lvl <= 1'd0;
            uart_rx_ovf <= 1'd0;
            uart_rx_wr <= 1'd0;
            uart_rx_wdata <= 8'd0;
            uart_rx_wcnt <= 4'd0;
            uart_rx_pdata <= 11'h7ff;
            uart_rx_perror <= 1'd0;
        end else begin
            uart_rx_ready <= (uart_rdy_mode == 1'd1) ? uart_rx_rdy :
                (uart_rx_lvl ^ uart_rdy_polarity);
            uart_rx_cnt <= uart_rx_cnt_5[3:0];
            if (uart_rx_cnt >= 4'd12) begin
                uart_rx_lvl <= 1'b1;
            end else if (uart_rx_cnt <= 4'd8) begin
                uart_rx_lvl <= 1'b0;
            end
            uart_rx_ovf <= uart_clken & uart_rx_valid & uart_rx_full;
            uart_rx_wr <= uart_clken & uart_rx_valid & ~uart_rx_full;
            uart_rx_wdata <= uart_rx_data_in;
            if (uart_rx_wr == 1'd1) begin
                uart_rx_wcnt <= uart_rx_wcnt + 1'd1;
            end
            if (uart_clken == 1'd1) begin
                if (uart_rx_valid == 1'd1) begin
                    uart_rx_pdata <= 11'h7ff;
                    uart_rx_perror <= uart_rx_perr;
                end else begin
                    uart_rx_pdata <= {uart_rx_sdata, uart_rx_pdata[10:1]};
                    uart_rx_perror <= 1'd0;
                end
            end
        end
    end

    assign uart_rx_wcnt_5 = {1'b1, uart_rx_wcnt};
    assign uart_rx_rcnt_5 = {1'b0, uart_rx_rcnt};
    assign uart_rx_cnt_5 = uart_rx_wcnt_5 - uart_rx_rcnt_5;

    always @(negedge axilite_resetn or posedge axilite_clk) begin
        if (axilite_resetn == 1'b0) begin
            axilite_rx_rcnt <= 4'd0;
        end else begin
            if (axilite_swresetn == 1'b0) begin
                axilite_rx_rcnt <= 4'd0;
            end else if (axilite_rx_rd == 1'd1) begin
                axilite_rx_rcnt <= axilite_rx_rcnt + 1'd1;
            end
        end
    end

    // transmit

    assign uart_tx_data = uart_tx_data_out;
    assign uart_tx_start = ~(uart_tx_empty | (uart_tx_rdy ^ uart_rdy_polarity));

    always @(negedge uart_resetn or posedge uart_clk) begin
        if (uart_resetn == 1'b0) begin
            uart_tx_rd <= 1'd0;
            uart_tx_state <= 1'd0;
            uart_tx_count <= 4'd0;
            uart_tx_data_out <= 1'd1;
            uart_tx_pdata <= 12'hfff;
        end else begin
            if (uart_tx_state == 1'd1) begin
                uart_tx_rd <= 1'd0;
                if (uart_clken == 1'd1) begin
                    if (uart_tx_count >= uart_length) begin
                        uart_tx_state <= 1'd0;
                        uart_tx_count <= 4'd0;
                    end else begin
                        uart_tx_state <= 1'd1;
                        uart_tx_count <= uart_tx_count + 1'd1;
                    end
                    uart_tx_data_out <= uart_tx_pdata[0];
                    uart_tx_pdata <= {1'd1, uart_tx_pdata[11:1]};
                end
            end else begin
                uart_tx_rd <= uart_tx_start;
                uart_tx_state <= uart_tx_start;
                uart_tx_count <= 4'd0;
                uart_tx_data_out <= 1'd1;
                uart_tx_pdata[11:10] <= 2'b11;
                uart_tx_pdata[ 9: 6] <= tx_psel(uart_ctrl, uart_tx_rdata);
                uart_tx_pdata[ 5: 1] <= uart_tx_rdata[4:0];
                uart_tx_pdata[ 0: 0] <= 1'd0;
            end
        end
    end

    // uart length (start + word + parity + stop) and control

    assign uart_ctrl = {uart_parity_type, uart_parity_enable,
        uart_stop_bits, uart_bit_width};
    assign uart_length = uart_bit_width + uart_stop_bits +
        uart_parity_enable + 3'd7;

    // bit clock

    always @(negedge uart_resetn or posedge uart_clk) begin
        if (uart_resetn == 1'b0) begin
            uart_clken <= 1'd0;
            uart_clkcnt <= 32'd1;
        end else begin
            if (uart_clkcnt >= uart_clk_period) begin
                uart_clken <= 1'd1;
                uart_clkcnt <= 32'd1;
            end else begin
                uart_clken <= 1'd0;
                uart_clkcnt <= uart_clkcnt + 1'd1;
            end
        end
    end

    // loopback

    assign rx_data_in = (axilite_loopback == 1'd1) ? uart_tx_data : uart_rx_data;
    assign tx_ready_in = (axilite_loopback == 1'd1) ? uart_rx_ready : uart_tx_ready;

    // software reset

    assign axilite_uart_resetn = axilite_swresetn & axilite_enable;

    always @(negedge axilite_resetn or posedge axilite_clk) begin
        if (axilite_resetn == 1'b0) begin
            axilite_swresetn <= 1'd0;
            axilite_swreset_clr <= 1'd0;
        end else begin
            axilite_swresetn <= ~axilite_swreset;
            axilite_swreset_clr <= axilite_swreset;
        end
    end

    // instantiations

    mem_fifo #(.DATA_WIDTH (8)) i_tx_fifo (
        .wr_clk                         (axilite_clk),
        .wr_resetn                      (axilite_swresetn),
        .wr_valid                       (axilite_tx_wr),
        .wr_data                        (axilite_tx_wdata),
        .wr_limit                       (axilite_tx_full),
        .rd_clk                         (uart_clk),
        .rd_resetn                      (uart_resetn),
        .rd_read                        (uart_tx_rd),
        .rd_valid                       (),
        .rd_data                        (uart_tx_rdata),
        .rd_empty                       (uart_tx_empty));

    mem_fifo #(.DATA_WIDTH (8)) i_rx_fifo (
        .wr_clk                         (uart_clk),
        .wr_resetn                      (uart_resetn),
        .wr_valid                       (uart_rx_wr),
        .wr_data                        (uart_rx_wdata),
        .wr_limit                       (uart_rx_full),
        .rd_clk                         (axilite_clk),
        .rd_resetn                      (axilite_swresetn),
        .rd_read                        (axilite_rx_rd),
        .rd_valid                       (),
        .rd_data                        (axilite_rx_rdata),
        .rd_empty                       (axilite_rx_empty));

    cdc #(.DATA_WIDTH(3)) i_cdc (
        .src_data                       ({rx_data_in,
                                            axilite_rx_rdy,
                                            tx_ready_in}),
        .dest_resetn                    (uart_resetn),
        .dest_clk                       (uart_clk),
        .dest_data                      ({uart_rx_sdata,
                                            uart_rx_rdy,
                                            uart_tx_rdy}));

    cdc_cntrl #(.DATA_WIDTH(43)) i_cntrl (
        .src_resetn                     (axilite_resetn),
        .src_clk                        (axilite_clk),
        .src_data                       ({axilite_clk_period,
                                            axilite_bit_width,
                                            axilite_stop_bits,
                                            axilite_parity_enable,
                                            axilite_parity_type,
                                            axilite_rdy_polarity,
                                            axilite_rdy_mode,
                                            axilite_rx_rcnt}),
        .dest_resetn                    (uart_resetn),
        .dest_clk                       (uart_clk),
        .dest_data                      ({uart_clk_period,
                                            uart_bit_width,
                                            uart_stop_bits,
                                            uart_parity_enable,
                                            uart_parity_type,
                                            uart_rdy_polarity,
                                            uart_rdy_mode,
                                            uart_rx_rcnt}));

    cdc_resetp i_resetn (
        .src_clk                        (axilite_clk),
        .src_resetn                     (axilite_uart_resetn),
        .dest_clk                       (uart_clk),
        .dest_resetp                    (uart_resetn));

    cdc #(.DATA_WIDTH(3)) i_fifo_status (
        .src_data                       ({uart_rx_full,
                                            uart_tx_empty,
                                            tx_ready_in}),
        .dest_resetn                    (axilite_swresetn),
        .dest_clk                       (axilite_clk),
        .dest_data                      ({axilite_rx_full,
                                            axilite_tx_empty,
                                            axilite_tx_rdy}));

    cdc_event #(.DATA_WIDTH(2)) i_event (
        .src_resetn                     (uart_resetn),
        .src_clk                        (uart_clk),
        .src_data                       ({uart_rx_perror,
                                            uart_rx_ovf}),
        .dest_resetn                    (axilite_resetn),
        .dest_clk                       (axilite_clk),
        .dest_data                      ({axilite_rx_perror,
                                            axilite_rx_ovf}));

    clk_mon i_clk_mon (
        .clk                            (uart_clk),
        .axilite_resetn                 (axilite_resetn),
        .axilite_clk                    (axilite_clk),
        .axilite_clk_mon_count          (axilite_clk_mon_count));

endmodule

// **********************************************************************************
// **********************************************************************************
