// **********************************************************************************
// **********************************************************************************
// ----------------------------------------------------------------------------------
// ################
// ##   ###########   Analog Devices Inc.
// ##      ########
// ##         #####   Copyright (c) 2019 Analog Devices Inc. All rights reserved.
// ##            ##   This file is the confidential and proprietary property of ADI.
// ##         #####   Possession or use of this file requires a written license.
// ##      ########   The licensing information may be found at: www.analog.com
// ##   ###########
// ################
// ----------------------------------------------------------------------------------
// Description:       TDD DMA enable
// ----------------------------------------------------------------------------------
// **********************************************************************************
// **********************************************************************************

`timescale 1ps/1ps

module tdd_dma_enable (

    // source

    input   wire            resetn,
    input   wire            clk,
    input   wire            sync,
    output  reg             enable = 'd0,

    // tdd

    input   wire            tdd_resetn,
    input   wire            tdd_clk,
    input   wire            tdd_genb,
    input   wire            tdd_state,
    input   wire            tdd_last,
    input   wire  [ 31:0]   tdd_clk_cnt,
    input   wire  [ 31:0]   tdd_frm_cnt,

    // axilite

    input   wire            axilite_clk,
    input   wire            axilite_resetn,
    output  wire  [ 11:0]   axilite_clk_ratio,
    input   wire  [ 31:0]   axilite_frame_period,
    input   wire  [ 31:0]   axilite_pri_assert,
    input   wire  [ 31:0]   axilite_pri_deassert,
    input   wire  [ 31:0]   axilite_sec_assert,
    input   wire  [ 31:0]   axilite_sec_deassert,
    input   wire  [ 31:0]   axilite_pri_frame_assert,
    input   wire  [ 31:0]   axilite_pri_frame_deassert,
    input   wire  [ 31:0]   axilite_sec_frame_assert,
    input   wire  [ 31:0]   axilite_sec_frame_deassert,
    input   wire  [  1:0]   axilite_mode,
    input   wire            axilite_sync);

    // internal registers

    reg           [  3:0]   scale = 'd0;
    reg                     toggle = 'd0;
    reg                     pri_sync_enable = 'd0;
    reg                     pri_sync_enable_d = 'd0;
    reg           [ 35:0]   pri_count = 'd0;
    reg                     pri_sync = 'd0;
    reg                     sec_sync_enable = 'd0;
    reg                     sec_sync_enable_d = 'd0;
    reg           [ 35:0]   sec_count = 'd0;
    reg                     sec_sync = 'd0;
    reg                     tdd_toggle_d = 'd0;
    reg           [ 11:0]   tdd_ratio = 'd0;
    reg           [ 11:0]   tdd_scale = 'd0;
    reg                     tdd_pri_enable = 'd0;
    reg                     tdd_sec_enable = 'd0;
    reg           [ 31:0]   axilite_pri_diff_f = 'd0;
    reg           [ 31:0]   axilite_pri_diff_a = 'd0;
    reg           [ 31:0]   axilite_pri_diff_d = 'd0;
    reg           [ 31:0]   axilite_pri_len = 'd0;
    reg           [ 31:0]   axilite_sec_diff_f = 'd0;
    reg           [ 31:0]   axilite_sec_diff_a = 'd0;
    reg           [ 31:0]   axilite_sec_diff_d = 'd0;
    reg           [ 31:0]   axilite_sec_len = 'd0;

    // internal signals

    wire                    pri_enable;
    wire                    sec_enable;
    wire          [ 11:0]   ratio;
    wire                    genb;
    wire          [ 35:0]   pri_len;
    wire          [ 35:0]   sec_len;
    wire          [  1:0]   mode;
    wire                    sel;
    wire                    tdd_toggle;
    wire          [ 31:0]   tdd_pri_assert;
    wire          [ 31:0]   tdd_pri_deassert;
    wire          [ 31:0]   tdd_sec_assert;
    wire          [ 31:0]   tdd_sec_deassert;
    wire          [ 31:0]   tdd_pri_frame_assert;
    wire          [ 31:0]   tdd_pri_frame_deassert;
    wire          [ 31:0]   tdd_sec_frame_assert;
    wire          [ 31:0]   tdd_sec_frame_deassert;
    wire                    tdd_int_resetn;

    // tdd enable

    always @(negedge resetn or posedge clk) begin
        if (resetn == 1'b0) begin
            enable <= 'd0;
        end else begin
            if (mode[1] == 1'd0) begin
                enable <= mode[0] & genb;
            end else if (sel == 1'd1) begin
                enable <= pri_sync | sec_sync;
            end else begin
                enable <= pri_enable | sec_enable;
            end
        end
    end

    assign pri_len[3:0] = 4'd0;
    assign sec_len[3:0] = 4'd0;

    always @(negedge resetn or posedge clk) begin
        if (resetn == 1'b0) begin
            scale <= 4'd0;
            toggle <= 1'd0;
            pri_sync_enable <= 'd0;
            pri_sync_enable_d <= 'd0;
            pri_count <= 'd0;
            pri_sync <= 'd0;
            sec_sync_enable <= 'd0;
            sec_sync_enable_d <= 'd0;
            sec_count <= 'd0;
            sec_sync <= 'd0;
        end else begin
            scale <= scale + 1'd1;
            if (scale == 4'hf) begin
                toggle <= ~toggle;
            end
            if (sync == 1'd1) begin
                pri_sync_enable <= pri_enable;
            end
            pri_sync_enable_d <= pri_sync_enable;
            if ((pri_sync_enable == 1'd1) && (pri_sync_enable_d == 1'd0)) begin
                pri_count <= 36'd0;
                pri_sync <= 1'd1;
            end else if (pri_sync == 1'd1) begin
                pri_count <= pri_count + ratio;
                if (pri_count >= pri_len) begin
                    pri_sync <= 1'd0;
                end
            end
            if (sync == 1'd1) begin
                sec_sync_enable <= sec_enable;
            end
            sec_sync_enable_d <= sec_sync_enable;
            if ((sec_sync_enable == 1'd1) && (sec_sync_enable_d == 1'd0)) begin
                sec_count <= 36'd0;
                sec_sync <= 1'd1;
            end else if (sec_sync == 1'd1) begin
                sec_count <= sec_count + ratio;
                if (sec_count >= sec_len) begin
                    sec_sync <= 1'd0;
                end
            end
        end
    end

    always @(negedge tdd_int_resetn or posedge tdd_clk) begin
        if (tdd_int_resetn == 1'b0) begin
            tdd_toggle_d <= 'b0;
            tdd_ratio <= 'd0;
            tdd_scale <= 'd0;
            tdd_pri_enable <= 'd0;
            tdd_sec_enable <= 'd0;
        end else begin
            tdd_toggle_d <= tdd_toggle;
            if (((tdd_toggle_d ^ tdd_toggle) == 1'b1) &&
                (tdd_state == 1'd0)) begin
                tdd_ratio <= tdd_scale;
            end
            if ((tdd_toggle_d ^ tdd_toggle) == 1'b1) begin
                tdd_scale <= 12'd1;
            end else begin
                tdd_scale <= tdd_scale + 1'd1;
            end
            if (((tdd_frm_cnt >= tdd_pri_frame_assert) &&
                (tdd_frm_cnt <= tdd_pri_frame_deassert)) ||
                ((tdd_frm_cnt >= tdd_sec_frame_assert) &&
                (tdd_frm_cnt <= tdd_sec_frame_deassert))) begin
                if (tdd_clk_cnt == tdd_pri_deassert) begin
                    tdd_pri_enable <= 1'd0;
                end else if (tdd_clk_cnt == tdd_pri_assert) begin
                    tdd_pri_enable <= tdd_state & ~tdd_last;
                end
                if (tdd_clk_cnt == tdd_sec_deassert) begin
                    tdd_sec_enable <= 1'd0;
                end else if (tdd_clk_cnt == tdd_sec_assert) begin
                    tdd_sec_enable <= tdd_state & ~tdd_last;
                end
            end
        end
    end

    always @(negedge axilite_resetn or posedge axilite_clk) begin
        if (axilite_resetn == 1'b0) begin
            axilite_pri_diff_f <= 'd0;
            axilite_pri_diff_a <= 'd0;
            axilite_pri_diff_d <= 'd0;
            axilite_pri_len <= 'd0;
            axilite_sec_diff_f <= 'd0;
            axilite_sec_diff_a <= 'd0;
            axilite_sec_diff_d <= 'd0;
            axilite_sec_len <= 'd0;
        end else begin
            axilite_pri_diff_f <= axilite_frame_period - axilite_pri_assert;
            axilite_pri_diff_a <= axilite_pri_diff_f + axilite_pri_deassert;
            axilite_pri_diff_d <= axilite_pri_deassert - axilite_pri_assert;
            if (axilite_pri_assert > axilite_pri_deassert) begin
                axilite_pri_len <= axilite_pri_diff_a;
            end else begin
                axilite_pri_len <= axilite_pri_diff_d;
            end
            axilite_sec_diff_f <= axilite_frame_period - axilite_sec_assert;
            axilite_sec_diff_a <= axilite_sec_diff_f + axilite_sec_deassert;
            axilite_sec_diff_d <= axilite_sec_deassert - axilite_sec_assert;
            if (axilite_sec_assert > axilite_sec_deassert) begin
                axilite_sec_len <= axilite_sec_diff_a;
            end else begin
                axilite_sec_len <= axilite_sec_diff_d;
            end
        end
    end

    // instantiations

    cdc_resetp i_cdc_resetp (
        .src_clk          (tdd_clk),
        .src_resetn       (tdd_resetn),
        .dest_clk         (tdd_clk),
        .dest_resetp      (tdd_int_resetn));

    cdc #(.DATA_WIDTH(2)) i_cdc (
        .src_data         ({tdd_pri_enable,
                            tdd_sec_enable}),
        .dest_resetn      (resetn),
        .dest_clk         (clk),
        .dest_data        ({pri_enable,
                            sec_enable}));

    cdc_cntrl #(.DATA_WIDTH(13)) i_cdc_cntrl_0 (
        .src_resetn       (tdd_int_resetn),
        .src_clk          (tdd_clk),
        .src_data         ({tdd_ratio,
                            tdd_genb}),
        .dest_resetn      (resetn),
        .dest_clk         (clk),
        .dest_data        ({ratio,
                            genb}));

    cdc_cntrl #(.DATA_WIDTH(67)) i_cdc_cntrl_1 (
        .src_resetn       (axilite_resetn),
        .src_clk          (axilite_clk),
        .src_data         ({axilite_pri_len,
                            axilite_sec_len,
                            axilite_mode,
                            axilite_sync}),
        .dest_resetn      (resetn),
        .dest_clk         (clk),
        .dest_data        ({pri_len[35:4],
                            sec_len[35:4],
                            mode,
                            sel}));

    cdc_cntrl #(.DATA_WIDTH(12)) i_axilite_cdc_cntrl (
        .src_resetn       (tdd_int_resetn),
        .src_clk          (tdd_clk),
        .src_data         (tdd_ratio),
        .dest_resetn      (axilite_resetn),
        .dest_clk         (axilite_clk),
        .dest_data        (axilite_clk_ratio));

    cdc #(.DATA_WIDTH(1)) i_tdd_cdc (
        .src_data         (toggle),
        .dest_resetn      (tdd_int_resetn),
        .dest_clk         (tdd_clk),
        .dest_data        (tdd_toggle));

    cdc_cntrl #(.DATA_WIDTH(256)) i_tdd_cdc_cntrl (
        .src_resetn       (axilite_resetn),
        .src_clk          (axilite_clk),
        .src_data         ({axilite_pri_assert,
                            axilite_pri_deassert,
                            axilite_sec_assert,
                            axilite_sec_deassert,
                            axilite_pri_frame_assert,
                            axilite_pri_frame_deassert,
                            axilite_sec_frame_assert,
                            axilite_sec_frame_deassert}),
        .dest_resetn      (tdd_int_resetn),
        .dest_clk         (tdd_clk),
        .dest_data        ({tdd_pri_assert,
                            tdd_pri_deassert,
                            tdd_sec_assert,
                            tdd_sec_deassert,
                            tdd_pri_frame_assert,
                            tdd_pri_frame_deassert,
                            tdd_sec_frame_assert,
                            tdd_sec_frame_deassert}));

endmodule

// **********************************************************************************
// **********************************************************************************
