// **********************************************************************************
// **********************************************************************************
// ----------------------------------------------------------------------------------
// ################
// ##   ###########   Analog Devices Inc.
// ##      ########
// ##         #####   Copyright (c) 2019 Analog Devices Inc. All rights reserved.
// ##            ##   This file is the confidential and proprietary property of ADI.
// ##         #####   Possession or use of this file requires a written license.
// ##      ########   The licensing information may be found at: www.analog.com
// ##   ###########
// ################
// ----------------------------------------------------------------------------------
// Description:       SPI Slave, software register controlled
// ----------------------------------------------------------------------------------
// **********************************************************************************
// **********************************************************************************

`timescale 1ps/1ps

module sspi #(parameter SSN_WIDTH = 1) (

    // spi interface
 
    input   wire  [(SSN_WIDTH-1):0]   sspi_ssn,
    input   wire                      sspi_sclk,
    input   wire                      sspi_mosi,
    output  wire                      sspi_miso_out,
    output  wire                      sspi_miso_enb,

    // axi interface

    input   wire                      axilite_clk,
    input   wire                      axilite_resetn,
    input   wire                      axilite_swreset,
    output  reg                       axilite_swreset_clr = 'd0,
    output  wire  [   7:0]            axilite_ss,
    input   wire                      axilite_enable,
    input   wire                      axilite_bidir_enb,
    input   wire  [   3:0]            axilite_bidir_count,
    input   wire                      axilite_mosi_edge_sel,
    input   wire                      axilite_miso_edge_sel,
    input   wire                      axilite_mosi_rd,
    output  wire  [   7:0]            axilite_mosi_rddata,
    output  wire                      axilite_mosi_full,
    output  wire                      axilite_mosi_empty,
    input   wire                      axilite_miso_wr,
    input   wire  [   7:0]            axilite_miso_wrdata,
    output  wire                      axilite_miso_full,
    output  wire                      axilite_miso_empty);

    // internal registers
 
    reg           [  2:0]             scount_p = 'd0;
    reg           [  3:0]             bcount_p = 'd0;
    reg                               mosi_toggle_p = 'd0;
    reg           [  6:0]             mosi_data_in_p = 'd0;
    reg                               miso_sel_p = 'd0;
    reg                               miso_toggle_p = 'd0;
    reg           [  7:0]             mosi_data_p = 'd0;
    reg           [  2:0]             scount_n = 'd0;
    reg           [  3:0]             bcount_n = 'd0;
    reg                               mosi_toggle_n = 'd0;
    reg           [  6:0]             mosi_data_in_n = 'd0;
    reg                               miso_sel_n = 'd0;
    reg                               miso_toggle_n = 'd0;
    reg           [  7:0]             mosi_data_n = 'd0;
    reg                               axilite_mosi_toggle_d = 1'd0;
    reg                               axilite_mosi_wr = 1'd0;
    reg           [  7:0]             axilite_mosi_wrdata = 8'd0;
    reg                               axilite_enable_d = 1'd0;
    reg                               axilite_miso_toggle_d = 1'd0;
    reg                               axilite_miso_rd = 1'd0;
    reg                               axilite_miso_rdsel = 1'd0;
    reg           [  7:0]             axilite_miso_rddata_0 = 8'd0;
    reg           [  7:0]             axilite_miso_rddata_1 = 8'd0;
    reg                               axilite_swresetn = 1'd0;

    // internal signals
 
    wire                              ssn;
    wire          [  3:0]             bcount;
    wire                              mosi_toggle;
    wire          [  7:0]             mosi_data;
    wire                              miso_toggle;
    wire                              miso_sel;
    wire          [  2:0]             miso_count;
    wire          [  7:0]             miso_data;
    wire                              axilite_mosi_toggle;
    wire                              axilite_mosi_toggle_p;
    wire                              axilite_miso_toggle;
    wire                              axilite_miso_toggle_p;
    wire                              axilite_miso_rdvalid;
    wire          [  7:0]             axilite_miso_rddata;

    // ssn

    assign ssn = &sspi_ssn;

    // enable

    assign bcount = (axilite_miso_edge_sel == 1'b1) ? bcount_n : bcount_p;
    assign sspi_miso_enb = (bcount >= axilite_bidir_count) ? 1'd1 : ~axilite_bidir_enb;

    // mosi

    assign mosi_toggle = (axilite_mosi_edge_sel == 1'b1) ? mosi_toggle_n : mosi_toggle_p;
    assign mosi_data = (axilite_mosi_edge_sel == 1'b1) ? mosi_data_n : mosi_data_p;

    // miso
 
    assign miso_toggle = (axilite_miso_edge_sel == 1'b1) ? miso_toggle_n : miso_toggle_p;
    assign miso_sel = (axilite_miso_edge_sel == 1'b1) ? miso_sel_n : miso_sel_p;
    assign miso_count = (axilite_miso_edge_sel == 1'b1) ? scount_n : scount_p;
    assign miso_data = (miso_sel == 1'd1) ? axilite_miso_rddata_1 : axilite_miso_rddata_0;
    assign sspi_miso_out = (miso_count == 7) ? miso_data[0] : ((miso_count == 6) ? miso_data[1] :
        ((miso_count == 5) ? miso_data[2] : ((miso_count == 4) ? miso_data[3] :
        ((miso_count == 3) ? miso_data[4] : ((miso_count == 2) ? miso_data[5] :
        ((miso_count == 1) ? miso_data[6] : (miso_data[7] & ~ssn)))))));

    // rising edge sampling

    always @(posedge sspi_sclk or posedge ssn) begin
        if (ssn == 1'b1) begin
            scount_p <= 3'd0;
            bcount_p <= 4'd0;
            mosi_toggle_p <= 1'd0;
            mosi_data_in_p <= 7'd0;
            miso_sel_p <= 1'd0;
            miso_toggle_p <= 1'd0;
        end else begin
            if (scount_p >= 'd7) begin
                scount_p <= 3'd0;
            end else begin
                scount_p <= scount_p + 1'd1;
            end
            if ((scount_p >= 'd7) && (bcount_p < 'hf)) begin
                bcount_p <= bcount_p + 1'd1;
            end
            if (scount_p >= 'd7) begin
                mosi_toggle_p <= ~mosi_toggle_p;
            end
            mosi_data_in_p <= {mosi_data_in_p[5:0], sspi_mosi};
            if (scount_p >= 'd7) begin
                miso_sel_p <= ~miso_sel_p;
            end
            if (scount_p == 'd0) begin
                miso_toggle_p <= ~miso_toggle_p;
            end
        end
    end

    always @(posedge sspi_sclk) begin
        if (scount_p >= 'd7) begin
            mosi_data_p <= {mosi_data_in_p, sspi_mosi};
        end
    end

    // falling edge sampling

    always @(negedge sspi_sclk or posedge ssn) begin
        if (ssn == 1'b1) begin
            scount_n <= 3'd0;
            bcount_n <= 4'd0;
            mosi_toggle_n <= 1'd0;
            mosi_data_in_n <= 7'd0;
            miso_sel_n <= 1'd0;
            miso_toggle_n <= 1'd0;
        end else begin
            if (scount_n >= 'd7) begin
                scount_n <= 3'd0;
            end else begin
                scount_n <= scount_n + 1'd1;
            end
            if ((scount_n >= 'd7) && (bcount_n < 'hf)) begin
                bcount_n <= bcount_n + 1'd1;
            end
            if (scount_n >= 'd7) begin
                mosi_toggle_n <= ~mosi_toggle_n;
            end
            mosi_data_in_n <= {mosi_data_in_n[5:0], sspi_mosi};
            if (scount_n >= 'd7) begin
                miso_sel_n <= ~miso_sel_n;
            end
            if (scount_n == 'd0) begin
                miso_toggle_n <= ~miso_toggle_n;
            end
        end
    end

    always @(negedge sspi_sclk) begin
        if (scount_n >= 'd7) begin
            mosi_data_n <= {mosi_data_in_n, sspi_mosi};
        end
    end

    // mosi data
 
    assign axilite_mosi_toggle_p = axilite_mosi_toggle_d ^ axilite_mosi_toggle;

    always @(posedge axilite_clk or negedge axilite_resetn) begin
        if (axilite_resetn == 1'd0) begin
            axilite_mosi_toggle_d <= 1'd0;
            axilite_mosi_wr <= 1'd0;
            axilite_mosi_wrdata <= 8'd0;
        end else begin
            axilite_mosi_toggle_d <= axilite_mosi_toggle;
            axilite_mosi_wr <= axilite_mosi_toggle_p & axilite_enable;
            if (axilite_mosi_toggle_p == 1'd1) begin
                axilite_mosi_wrdata <= mosi_data;
            end
        end
    end

    // miso data

    assign axilite_miso_toggle_p = axilite_miso_toggle_d ^ axilite_miso_toggle;

    always @(posedge axilite_clk or negedge axilite_resetn) begin
        if (axilite_resetn == 1'd0) begin
            axilite_enable_d <= 1'd0;
            axilite_miso_toggle_d <= 1'd0;
            axilite_miso_rd <= 1'd0;
            axilite_miso_rdsel <= 1'd0;
            axilite_miso_rddata_0 <= 8'd0;
            axilite_miso_rddata_1 <= 8'd0;
        end else begin
            axilite_enable_d <= axilite_enable;
            axilite_miso_toggle_d <= axilite_miso_toggle;
            axilite_miso_rd <= (axilite_enable & ~axilite_enable_d) | axilite_miso_toggle_p;
            if (axilite_enable == 1'd0) begin
                axilite_miso_rdsel <= 1'd0;
                axilite_miso_rddata_0 <= 8'hde;
                axilite_miso_rddata_1 <= 8'had;
            end else if (axilite_miso_rdvalid == 1'd1) begin
                axilite_miso_rdsel <= ~axilite_miso_rdsel;
                if (axilite_miso_rdsel == 1'd0) begin
                    axilite_miso_rddata_0 <= axilite_miso_rddata;
                end
                if (axilite_miso_rdsel == 1'd1) begin
                    axilite_miso_rddata_1 <= axilite_miso_rddata;
                end
            end
        end
    end

    // software reset
 
    always @(negedge axilite_resetn or posedge axilite_clk) begin
        if (axilite_resetn == 1'b0) begin
            axilite_swresetn <= 1'd0;
            axilite_swreset_clr <= 1'd0;
        end else begin
            axilite_swresetn <= ~axilite_swreset;
            axilite_swreset_clr <= axilite_swreset;
        end
    end

    // ssn maximum is 7

    assign axilite_ss[7:SSN_WIDTH] = 'd0;

    // instantiations

    cdc #(.DATA_WIDTH(2)) i_cdc_toggle (
        .src_data             ({mosi_toggle, miso_toggle}),
        .dest_resetn          (axilite_resetn),
        .dest_clk             (axilite_clk),
        .dest_data            ({axilite_mosi_toggle, axilite_miso_toggle}));

    cdc #(.DATA_WIDTH(SSN_WIDTH)) i_cdc_ss (
        .src_data             (~sspi_ssn),
        .dest_resetn          (axilite_resetn),
        .dest_clk             (axilite_clk),
        .dest_data            (axilite_ss[(SSN_WIDTH-1):0]));

    mem_fifo #(
        .WRITE_LIMIT          (14),
        .ADDRESS_WIDTH        (4),
        .DATA_WIDTH           (8))
    i_mosi_fifo (
        .wr_clk               (axilite_clk),
        .wr_resetn            (axilite_swresetn),
        .wr_valid             (axilite_mosi_wr),
        .wr_data              (axilite_mosi_wrdata),
        .wr_limit             (axilite_mosi_full),
        .rd_clk               (axilite_clk),
        .rd_resetn            (axilite_swresetn),
        .rd_read              (axilite_mosi_rd),
        .rd_valid             (),
        .rd_data              (axilite_mosi_rddata),
        .rd_empty             (axilite_mosi_empty));

    mem_fifo #(
        .WRITE_LIMIT          (14),
        .ADDRESS_WIDTH        (4),
        .DATA_WIDTH           (8))
    i_miso_fifo (
        .wr_clk               (axilite_clk),
        .wr_resetn            (axilite_swresetn),
        .wr_valid             (axilite_miso_wr),
        .wr_data              (axilite_miso_wrdata),
        .wr_limit             (axilite_miso_full),
        .rd_clk               (axilite_clk),
        .rd_resetn            (axilite_swresetn),
        .rd_read              (axilite_miso_rd),
        .rd_valid             (axilite_miso_rdvalid),
        .rd_data              (axilite_miso_rddata),
        .rd_empty             (axilite_miso_empty));

endmodule

// **********************************************************************************
// **********************************************************************************
