// **********************************************************************************
// **********************************************************************************
// ----------------------------------------------------------------------------------
// ################
// ##   ###########   Analog Devices Inc.
// ##      ########
// ##         #####   Copyright (c) 2019 Analog Devices Inc. All rights reserved.
// ##            ##   This file is the confidential and proprietary property of ADI.
// ##         #####   Possession or use of this file requires a written license.
// ##      ########   The licensing information may be found at: www.analog.com
// ##   ###########
// ################
// ----------------------------------------------------------------------------------
// Description:       QSPI software register controlled
// ----------------------------------------------------------------------------------
// **********************************************************************************
// **********************************************************************************

`timescale 1ps/1ps

module qspi #(parameter SSN_WIDTH = 1) (

    // qspi interface

    input   wire  [(SSN_WIDTH-1):0]   qspi_ssn,
    output  wire                      qspi_ssn_enb,
    output  wire  [(SSN_WIDTH-1):0]   qspi_ssn_out,
    input   wire                      qspi_sclk,
    output  wire                      qspi_sclk_enb,
    output  wire                      qspi_sclk_out,
    input   wire  [ 3:0]              qspi_sdata,
    output  wire                      qspi_sdata_enb,
    output  wire  [ 3:0]              qspi_sdata_out,

    // axi interface

    input   wire                      axilite_clk,
    input   wire                      axilite_resetn,
    input   wire                      axilite_swreset,
    output  reg                       axilite_swreset_clr = 'd0,
    output  wire  [ 7:0]              axilite_ss,
    input   wire  [ 7:0]              axilite_ss_out,
    input   wire                      axilite_enable,
    input   wire                      axilite_master,
    input   wire  [ 3:0]              axilite_clk_period,
    input   wire  [ 7:0]              axilite_bidir_count,
    input   wire                      axilite_tx_edge_sel,
    input   wire                      axilite_rx_edge_sel,
    input   wire  [ 7:0]              axilite_tx_count,
    input   wire  [ 7:0]              axilite_rx_count,
    input   wire                      axilite_rx_rd,
    output  wire  [ 3:0]              axilite_rx_rddata,
    output  wire                      axilite_rx_full,
    output  wire                      axilite_rx_empty,
    input   wire                      axilite_tx_wr,
    input   wire  [ 3:0]              axilite_tx_wrdata,
    output  wire                      axilite_tx_full,
    output  wire                      axilite_tx_empty);

    // internal registers

    reg           [ 7:0]              qspi_rx_count = 'd0;
    reg           [ 7:0]              qspi_tx_count = 'd0;
    reg           [ 7:0]              axilite_ssn = 'd0;
    reg           [ 7:0]              axilite_tx_xfer_count = 'd0;
    reg           [ 7:0]              axilite_rx_xfer_count = 'd0;
    reg           [ 1:0]              axilite_rwn = 'd0;
    reg                               axilite_sdata_enb = 'd0;
    reg           [ 3:0]              axilite_sdata = 'd0;
    reg                               axilite_sclk = 'd0;
    reg                               axilite_active = 'd0;
    reg           [ 3:0]              axilite_clk_count = 'd0;
    reg                               axilite_m_rx_wr = 'd0;
    reg           [ 3:0]              axilite_m_rx_wrdata = 'd0;
    reg                               axilite_swresetn = 1'd0;

    // internal signals

    wire                              qspi_resetn;
    wire                              qspi_enable;
    wire                              qspi_rx_clk;
    wire                              qspi_rx_wr;
    wire                              qspi_rx_full;
    wire                              qspi_tx_clk;
    wire                              qspi_tx_rd;
    wire          [ 3:0]              qspi_tx_rddata;
    wire                              qspi_tx_empty;
    wire                              qspi_tx_mem_read;
    wire                              qspi_tx_mem_valid;
    wire          [ 3:0]              qspi_tx_mem_data;
    wire                              qspi_tx_mem_empty;
    wire                              axilite_sclk_sel;
    wire                              axilite_s_rx_rd;
    wire                              axilite_s_rx_rdvalid;
    wire          [ 3:0]              axilite_s_rx_rddata;
    wire                              axilite_s_rx_empty;
    wire                              axilite_s_tx_wr;
    wire          [ 3:0]              axilite_s_tx_wrdata;
    wire                              axilite_s_tx_full;
    wire                              axilite_m_tx_wr;
    wire          [ 3:0]              axilite_m_tx_wrdata;
    wire                              axilite_m_rx_full;
    wire                              axilite_m_rx_rd;
    wire          [ 3:0]              axilite_m_rx_rddata;
    wire                              axilite_m_rx_empty;
    wire                              axilite_m_tx_rd;
    wire          [ 3:0]              axilite_m_tx_rddata;
    wire                              axilite_m_tx_empty;
    wire                              axilite_m_tx_full;
    wire                              axilite_m_tx_mem_read;
    wire                              axilite_m_tx_mem_valid;
    wire          [ 3:0]              axilite_m_tx_mem_data;
    wire                              axilite_m_tx_mem_empty;

    // reset and io signals

    assign qspi_resetn = ~(&qspi_ssn);
    assign qspi_enable = axilite_enable & ~axilite_master;
    assign qspi_ssn_enb = axilite_master;
    assign qspi_ssn_out = axilite_ssn[(SSN_WIDTH-1):0];
    assign qspi_sclk_enb = axilite_master;
    assign qspi_sclk_out = axilite_sclk;
    assign qspi_sdata_enb = (axilite_master == 1'b1) ? axilite_sdata_enb : qspi_tx_rd;
    assign qspi_sdata_out = (axilite_master == 1'b1) ? axilite_sdata : qspi_tx_rddata;

    // axilite fifo signals

    assign axilite_tx_full = (axilite_master == 1'b1) ?
        axilite_m_tx_full : axilite_s_tx_full;
    assign axilite_tx_empty = (axilite_master == 1'b1) ?
        axilite_m_tx_empty : axilite_s_tx_empty;

    assign axilite_rx_full = (axilite_master == 1'b1) ?
        axilite_m_rx_full : axilite_s_rx_full;
    assign axilite_rx_empty = (axilite_master == 1'b1) ?
        axilite_m_rx_empty : axilite_s_rx_empty;

    assign axilite_rx_rddata = (axilite_master == 1'b1) ?
        axilite_m_rx_rddata : axilite_s_rx_rddata;

    // slave, data input

    assign qspi_rx_clk = axilite_rx_edge_sel ^ qspi_sclk;
    assign qspi_rx_wr = (qspi_rx_count >= axilite_bidir_count) ? 1'b0 : qspi_enable;

    assign axilite_s_rx_rd = ~axilite_master & axilite_rx_rd;

    always @(posedge qspi_rx_clk or negedge qspi_resetn) begin
        if (qspi_resetn == 1'b0) begin
            qspi_rx_count <= 8'd0;
        end else begin
            qspi_rx_count <= qspi_rx_count + 1'b1;
        end
    end

    // slave, data output

    assign qspi_tx_clk = axilite_enable & (axilite_tx_edge_sel ^ qspi_sclk);
    assign qspi_tx_rd = (qspi_tx_count > axilite_bidir_count) ? qspi_enable : 1'b0;

    assign axilite_s_tx_wr = ~axilite_master & axilite_tx_wr;
    assign axilite_s_tx_wrdata = axilite_tx_wrdata;

    always @(posedge qspi_tx_clk or negedge qspi_resetn) begin
        if (qspi_resetn == 1'b0) begin
            qspi_tx_count <= 8'd0;
        end else begin
            qspi_tx_count <= qspi_tx_count + 1'b1;
        end
    end

    // master, count handler

    always @(negedge axilite_resetn or posedge axilite_clk) begin
        if (axilite_resetn == 1'b0) begin
            axilite_ssn <= 8'hff;
            axilite_tx_xfer_count <= 8'd0;
            axilite_rx_xfer_count <= 8'd0;
            axilite_rwn <= 2'd3;
        end else begin
            if ((axilite_enable == 1'b1) && (axilite_master == 1'b1)) begin
                axilite_ssn <= ~axilite_ss_out;
                if (axilite_m_tx_rd == 1'b1) begin
                    axilite_tx_xfer_count <= axilite_tx_xfer_count + 1'b1;
                end
                if (axilite_m_rx_wr == 1'b1) begin
                    axilite_rx_xfer_count <= axilite_rx_xfer_count + 1'b1;
                end
                if (axilite_rx_xfer_count >= axilite_rx_count) begin
                    axilite_rwn <= 2'd2;
                end else if (axilite_tx_xfer_count >= axilite_tx_count) begin
                    axilite_rwn <= 2'd1;
                end else begin
                    axilite_rwn <= 2'd0;
                end
            end else begin
                axilite_ssn <= 8'hff;
                axilite_tx_xfer_count <= 8'd0;
                axilite_rx_xfer_count <= 8'd1;
                axilite_rwn <= 2'd3;
            end
        end
    end

    // master, data handler

    assign axilite_sclk_sel = axilite_sclk ^ axilite_tx_edge_sel;
    assign axilite_m_rx_rd = axilite_master & axilite_rx_rd;
    assign axilite_m_tx_wr = axilite_master & axilite_tx_wr;
    assign axilite_m_tx_wrdata = axilite_tx_wrdata;
    assign axilite_m_tx_rd = (axilite_rwn == 2'd0) ?
        ~(axilite_active | axilite_m_tx_empty) : 1'd0;

    always @(negedge axilite_resetn or posedge axilite_clk) begin
        if (axilite_resetn == 1'b0) begin
            axilite_sdata_enb <= 1'b0;
            axilite_sdata <= 4'd0;
            axilite_sclk <= 1'd0;
            axilite_active <= 1'd0;
            axilite_clk_count <= 4'd0;
            axilite_m_rx_wr <= 1'b0;
            axilite_m_rx_wrdata <= 4'd0;
        end else begin
            if (axilite_active == 1'b1) begin
                axilite_sdata_enb <= axilite_sdata_enb;
                axilite_sdata <= axilite_sdata;
                if (axilite_clk_count >= axilite_clk_period) begin
                    if (axilite_sclk_sel == 1'b1) begin
                        axilite_sclk <= ~axilite_sclk;
                    end
                    axilite_active <= axilite_sclk_sel;
                    axilite_clk_count <= 4'd1;
                end else begin
                    axilite_sclk <= axilite_sclk;
                    axilite_active <= axilite_active;
                    axilite_clk_count <= axilite_clk_count + 1'd1;
                end
                if ((axilite_clk_count >= axilite_clk_period) && (axilite_sdata_enb == 1'd0)) begin
                    axilite_m_rx_wr <= axilite_sclk ^ axilite_rx_edge_sel;
                    axilite_m_rx_wrdata <= qspi_sdata;
                end else begin
                    axilite_m_rx_wr <= 1'b0;
                    axilite_m_rx_wrdata <= 4'd0;
                end
            end else if (axilite_rwn == 2'd0) begin
                axilite_sdata_enb <= ~axilite_m_tx_empty;
                axilite_sdata <= axilite_m_tx_rddata;
                if (axilite_m_tx_empty == 1'b0) begin
                    axilite_sclk <= ~axilite_sclk;
                end
                axilite_active <= ~axilite_m_tx_empty;
                axilite_clk_count <= 4'd1;
                axilite_m_rx_wr <= 1'b0;
                axilite_m_rx_wrdata <= 4'd0;
            end else if (axilite_rwn == 2'd1) begin
                axilite_sdata_enb <= 1'b0;
                axilite_sdata <= 4'd0;
                if (axilite_m_rx_full == 1'b0) begin
                    axilite_sclk <= ~axilite_sclk;
                end
                axilite_active <= ~axilite_m_rx_full;
                axilite_clk_count <= 4'd1;
                axilite_m_rx_wr <= 1'b0;
                axilite_m_rx_wrdata <= 4'd0;
            end else begin
                axilite_sdata_enb <= 1'b0;
                axilite_sdata <= 4'd0;
                axilite_sclk <= axilite_tx_edge_sel;
                axilite_active <= 1'b0;
                axilite_clk_count <= 4'd0;
                axilite_m_rx_wr <= 1'b0;
                axilite_m_rx_wrdata <= 4'd0;
            end
        end
    end

    // software reset

    always @(negedge axilite_resetn or posedge axilite_clk) begin
        if (axilite_resetn == 1'b0) begin
            axilite_swresetn <= 1'd0;
            axilite_swreset_clr <= 1'd0;
        end else begin
            axilite_swresetn <= ~axilite_swreset;
            axilite_swreset_clr <= axilite_swreset;
        end
    end

    // slave slect, maximum width is 6

    assign axilite_ss[7:SSN_WIDTH] = 'd0;

    // instantiations

    cdc #(.DATA_WIDTH(SSN_WIDTH)) i_cdc_ss (
        .src_data             (~qspi_ssn),
        .dest_resetn          (axilite_resetn),
        .dest_clk             (axilite_clk),
        .dest_data            (axilite_ss[(SSN_WIDTH-1):0]));

    cdc #(.DATA_WIDTH(2)) i_cdc_flags (
        .src_data             ({qspi_tx_empty,
                                qspi_rx_full}),
        .dest_resetn          (axilite_resetn),
        .dest_clk             (axilite_clk),
        .dest_data            ({axilite_s_tx_empty,
                                axilite_s_rx_full}));

    mem_fifo #(
        .WRITE_LIMIT          (14),
        .ADDRESS_WIDTH        (4),
        .DATA_WIDTH           (4))
    i_s_rx_fifo (
        .wr_clk               (qspi_rx_clk),
        .wr_resetn            (qspi_resetn),
        .wr_valid             (qspi_rx_wr),
        .wr_data              (qspi_sdata),
        .wr_limit             (qspi_rx_full),
        .rd_clk               (axilite_clk),
        .rd_resetn            (axilite_swresetn),
        .rd_read              (axilite_s_rx_rd),
        .rd_valid             (axilite_s_rx_rdvalid),
        .rd_data              (axilite_s_rx_rddata),
        .rd_empty             (axilite_s_rx_empty));

    mem_fifo #(
        .WRITE_LIMIT          (14),
        .ADDRESS_WIDTH        (4),
        .DATA_WIDTH           (4))
    i_s_tx_fifo (
        .wr_clk               (axilite_clk),
        .wr_resetn            (axilite_swresetn),
        .wr_valid             (axilite_s_tx_wr),
        .wr_data              (axilite_s_tx_wrdata),
        .wr_limit             (axilite_s_tx_full),
        .rd_clk               (qspi_tx_clk),
        .rd_resetn            (qspi_resetn),
        .rd_read              (qspi_tx_mem_read),
        .rd_valid             (qspi_tx_mem_valid),
        .rd_data              (qspi_tx_mem_data),
        .rd_empty             (qspi_tx_mem_empty));

    mem_zrd #(.DATA_WIDTH(4)) i_s_tx_zrd (
        .rd_clk               (qspi_tx_clk),
        .rd_resetn            (qspi_resetn),
        .rd_read              (qspi_tx_rd),
        .rd_valid             (),
        .rd_data              (qspi_tx_rddata),
        .rd_empty             (qspi_tx_empty),
        .rd_mem_read          (qspi_tx_mem_read),
        .rd_mem_valid         (qspi_tx_mem_valid),
        .rd_mem_data          (qspi_tx_mem_data),
        .rd_mem_empty         (qspi_tx_mem_empty));

    mem_fifo #(
        .WRITE_LIMIT          (14),
        .ADDRESS_WIDTH        (4),
        .DATA_WIDTH           (4))
    i_m_rx_fifo (
        .wr_clk               (axilite_clk),
        .wr_resetn            (axilite_swresetn),
        .wr_valid             (axilite_m_rx_wr),
        .wr_data              (axilite_m_rx_wrdata),
        .wr_limit             (axilite_m_rx_full),
        .rd_clk               (axilite_clk),
        .rd_resetn            (axilite_swresetn),
        .rd_read              (axilite_m_rx_rd),
        .rd_valid             (),
        .rd_data              (axilite_m_rx_rddata),
        .rd_empty             (axilite_m_rx_empty));

    mem_fifo #(
        .WRITE_LIMIT          (14),
        .ADDRESS_WIDTH        (4),
        .DATA_WIDTH           (4))
    i_m_tx_fifo (
        .wr_clk               (axilite_clk),
        .wr_resetn            (axilite_swresetn),
        .wr_valid             (axilite_m_tx_wr),
        .wr_data              (axilite_m_tx_wrdata),
        .wr_limit             (axilite_m_tx_full),
        .rd_clk               (axilite_clk),
        .rd_resetn            (axilite_swresetn),
        .rd_read              (axilite_m_tx_mem_read),
        .rd_valid             (axilite_m_tx_mem_valid),
        .rd_data              (axilite_m_tx_mem_data),
        .rd_empty             (axilite_m_tx_mem_empty));

    mem_zrd #(.DATA_WIDTH(4)) i_m_tx_zrd (
        .rd_clk               (axilite_clk),
        .rd_resetn            (axilite_swresetn),
        .rd_read              (axilite_m_tx_rd),
        .rd_valid             (),
        .rd_data              (axilite_m_tx_rddata),
        .rd_empty             (axilite_m_tx_empty),
        .rd_mem_read          (axilite_m_tx_mem_read),
        .rd_mem_valid         (axilite_m_tx_mem_valid),
        .rd_mem_data          (axilite_m_tx_mem_data),
        .rd_mem_empty         (axilite_m_tx_mem_empty));

endmodule

// **********************************************************************************
// **********************************************************************************
