// **********************************************************************************
// **********************************************************************************
// ----------------------------------------------------------------------------------
// ################
// ##   ###########   Analog Devices Inc.
// ##      ########
// ##         #####   Copyright (c) 2019 Analog Devices Inc. All rights reserved.
// ##            ##   This file is the confidential and proprietary property of ADI.
// ##         #####   Possession or use of this file requires a written license.
// ##      ########   The licensing information may be found at: www.analog.com
// ##   ###########
// ################
// ----------------------------------------------------------------------------------
// Description:       FIFO, simple dual port
// ----------------------------------------------------------------------------------
// **********************************************************************************
// **********************************************************************************

`timescale 1ps/1ps

module mem_fifo #(

    parameter   WRITE_LIMIT = 14,
    parameter   ADDRESS_WIDTH = 4,
    parameter   DATA_WIDTH = 16) (

    input   wire                        wr_clk,
    input   wire                        wr_resetn,
    input   wire                        wr_valid,
    input   wire  [(DATA_WIDTH-1):0]    wr_data,
    output  reg                         wr_limit = 'd0,

    input   wire                        rd_clk,
    input   wire                        rd_resetn,
    input   wire                        rd_read,
    output  reg                         rd_valid = 'd0,
    output  reg   [(DATA_WIDTH-1):0]    rd_data = 'd0,
    output  reg                         rd_empty = 'd0);

    // internal registers
 
    reg           [(ADDRESS_WIDTH-1):0] wr_wraddr = 'd0;
    reg           [(ADDRESS_WIDTH-1):0] wr_wraddr_g = 'd0;
    reg           [(ADDRESS_WIDTH-1):0] wr_rdaddr = 'd0;
    reg           [(ADDRESS_WIDTH-1):0] rd_rdaddr = 'd0;
    reg           [(ADDRESS_WIDTH-1):0] rd_rdaddr_g = 'd0;
    reg           [(ADDRESS_WIDTH-1):0] rd_wraddr = 'd0;
    reg                                 rd_valid_d = 'd0;

    // internal signals
 
    wire          [ADDRESS_WIDTH:0]     wr_diff;
    wire                                wr_enable;
    wire          [(ADDRESS_WIDTH-1):0] wr_rdaddr_g;
    wire          [ADDRESS_WIDTH:0]     rd_diff;
    wire                                rd_limit;
    wire          [(ADDRESS_WIDTH-1):0] rd_wraddr_g;
    wire          [(DATA_WIDTH-1):0]    rd_data_d;

    // binary to gray

    function [(ADDRESS_WIDTH-1):0]  b2g;
        input [(ADDRESS_WIDTH-1):0] din;
        reg   [(ADDRESS_WIDTH-1):0] dout;
        integer i;
        begin
            dout[(ADDRESS_WIDTH-1)] = din[(ADDRESS_WIDTH-1)];
            for (i = (ADDRESS_WIDTH-1); i > 0; i = i - 1) begin
                dout[(i-1)] = din[i] ^ din[(i-1)];
            end
            b2g = dout;
        end
    endfunction

    // gray to binary

    function [(ADDRESS_WIDTH-1):0]  g2b;
        input [(ADDRESS_WIDTH-1):0] din;
        reg   [(ADDRESS_WIDTH-1):0] dout;
        integer i;
        begin
            dout[(ADDRESS_WIDTH-1)] = din[(ADDRESS_WIDTH-1)];
            for (i = (ADDRESS_WIDTH-1); i > 0; i = i - 1) begin
                dout[(i-1)] = dout[i] ^ din[(i-1)];
            end
            g2b = dout;
        end
    endfunction

    // write
 
    assign wr_diff = {1'd1, wr_wraddr} - {1'd0, wr_rdaddr};
    assign wr_enable = wr_valid & ~wr_limit;

    always @(negedge wr_resetn or posedge wr_clk) begin
        if (wr_resetn == 1'd0) begin
            wr_wraddr <= 'd0;
            wr_wraddr_g <= 'd0;
            wr_rdaddr <= 'd0;
            wr_limit <= 1'd0;
        end else begin
            if (wr_enable == 1'd1) begin
                wr_wraddr <= wr_wraddr + 1'd1;
            end
            wr_wraddr_g <= b2g(wr_wraddr);
            wr_rdaddr <= g2b(wr_rdaddr_g);
            if (wr_diff[(ADDRESS_WIDTH-1):0] > WRITE_LIMIT) begin
                wr_limit <= 1'b1;
            end else begin
                wr_limit <= 1'b0;
            end
        end
    end

    // read

    assign rd_diff = {1'd1, rd_wraddr} - {1'd0, rd_rdaddr};
    assign rd_limit = |rd_diff[(ADDRESS_WIDTH-1):0];

    always @(negedge rd_resetn or posedge rd_clk) begin
        if (rd_resetn == 1'd0) begin
            rd_rdaddr <= 'd0;
            rd_rdaddr_g <= 'd0;
            rd_wraddr <= 'd0;
            rd_empty <= 1'd1;
            rd_valid_d <= 1'd0;
            rd_valid <= 1'd0;
            rd_data <= 'd0;
        end else begin
            if ((rd_read == 1'd1) && (rd_limit == 1'd1)) begin
                rd_rdaddr <= rd_rdaddr + 1'd1;
            end
            rd_rdaddr_g <= b2g(rd_rdaddr);
            rd_wraddr <= g2b(rd_wraddr_g);
            rd_empty <= ~rd_limit;
            rd_valid_d <= rd_read & rd_limit;
            rd_valid <= rd_valid_d;
            rd_data <= rd_data_d;
        end
    end

    // instantiations
 
    cdc #(.DATA_WIDTH(ADDRESS_WIDTH)) i_rd_cdc (
        .src_data             (wr_wraddr_g),
        .dest_resetn          (rd_resetn),
        .dest_clk             (rd_clk),
        .dest_data            (rd_wraddr_g));

    cdc #(.DATA_WIDTH(ADDRESS_WIDTH)) i_wr_cdc (
        .src_data             (rd_rdaddr_g),
        .dest_resetn          (wr_resetn),
        .dest_clk             (wr_clk),
        .dest_data            (wr_rdaddr_g));

    mem_sdp #(
        .DATA_WIDTH           (DATA_WIDTH),
        .ADDRESS_WIDTH        (ADDRESS_WIDTH))
    i_mem (
        .wr_clk               (wr_clk),
        .wr_enable            (wr_enable),
        .wr_addr              (wr_wraddr),
        .wr_data              (wr_data),
        .rd_clk               (rd_clk),
        .rd_enable            (1'd1),
        .rd_addr              (rd_rdaddr),
        .rd_data              (rd_data_d));

endmodule

// **********************************************************************************
// **********************************************************************************
