// **********************************************************************************
// **********************************************************************************
// ----------------------------------------------------------------------------------
// ################
// ##   ###########   Analog Devices Inc.
// ##      ########
// ##         #####   Copyright (c) 2019 Analog Devices Inc. All rights reserved.
// ##            ##   This file is the confidential and proprietary property of ADI.
// ##         #####   Possession or use of this file requires a written license.
// ##      ########   The licensing information may be found at: www.analog.com
// ##   ###########
// ################
// ----------------------------------------------------------------------------------
// Description:       I2S software register controlled
// ----------------------------------------------------------------------------------
// **********************************************************************************
// **********************************************************************************

`timescale 1ps/1ps

module i2s (

    // i2s interface

    input   wire            i2s_sclk,
    output  wire            i2s_sclk_enb,
    output  wire            i2s_sclk_out,
    input   wire            i2s_ws,
    output  wire            i2s_ws_enb,
    output  wire            i2s_ws_out,
    input   wire            i2s_msdi,
    output  wire            i2s_msdi_enb,
    output  wire            i2s_msdi_out,
    input   wire            i2s_msdo,
    output  wire            i2s_msdo_enb,
    output  wire            i2s_msdo_out,

    // axi interface

    input   wire            axilite_clk,
    input   wire            axilite_resetn,
    input   wire            axilite_swreset,
    output  reg             axilite_swreset_clr = 'd0,
    input   wire            axilite_enable,
    input   wire            axilite_master,
    input   wire  [ 3:0]    axilite_clk_period,
    input   wire  [ 3:0]    axilite_no_of_bytes,
    input   wire            axilite_rx_rd,
    output  wire  [ 7:0]    axilite_rx_rddata,
    output  wire            axilite_rx_full,
    output  wire            axilite_rx_empty,
    input   wire            axilite_tx_wr,
    input   wire  [ 7:0]    axilite_tx_wrdata,
    output  wire            axilite_tx_full,
    output  wire            axilite_tx_empty);

    // internal registers

    reg                     i2s_ws_d = 'd0;
    reg           [ 2:0]    i2s_bit_count = 'd0;
    reg                     i2s_rx_wr = 'd0;
    reg           [ 7:0]    i2s_rx_wrdata = 'd0;
    reg                     i2s_tx_rd = 'd0;
    reg           [ 7:0]    i2s_tx_data = 'd0;
    reg                     axilite_sclk = 'd0;
    reg           [ 3:0]    axilite_clk_count = 'd0;
    reg           [ 2:0]    axilite_bit_count = 'd0;
    reg           [ 3:0]    axilite_byte_count = 'd0;
    reg                     axilite_ws = 'd0;
    reg                     axilite_swresetn = 1'd0;

    // internal signals

    wire                    i2s_sdi;
    wire                    i2s_sresetn;
    wire                    i2s_rx_full;
    wire          [ 7:0]    i2s_tx_rddata;
    wire                    i2s_tx_empty;

    // master and slave, data handler
 
    assign i2s_sdi = (axilite_master == 1'd1) ? i2s_msdi : i2s_msdo;

    assign i2s_msdi_enb = ~axilite_master & axilite_enable;
    assign i2s_msdi_out = i2s_tx_data[7];

    assign i2s_msdo_enb = axilite_master & axilite_enable;
    assign i2s_msdo_out = i2s_tx_data[7];

    assign i2s_ws_p = i2s_ws ^ i2s_ws_d;

    always @(posedge i2s_sclk or negedge i2s_sresetn) begin
        if (i2s_sresetn == 1'b0) begin
            i2s_ws_d <= 1'd0;
            i2s_bit_count <= 3'd0;
            i2s_rx_wr <= 1'd0;
            i2s_rx_wrdata <= 8'd0;
            i2s_tx_rd <= 1'd0;
            i2s_tx_data <= 8'd0;
        end else begin
            i2s_ws_d <= i2s_ws;
            if (i2s_ws_p == 1'b1) begin
                i2s_bit_count <= 3'd1;
            end else begin
                i2s_bit_count <= i2s_bit_count + 1'd1;
            end
            i2s_rx_wr <= ~(|i2s_bit_count) & ~i2s_rx_full;
            if (i2s_bit_count == 3'd1) begin
                i2s_rx_wrdata <= {7'd0, i2s_sdi};
            end else begin
                i2s_rx_wrdata <= {i2s_rx_wrdata[6:0], i2s_sdi};
            end
            i2s_tx_rd <= (&i2s_bit_count) & ~i2s_tx_empty;
            if (i2s_bit_count == 3'd0) begin
                i2s_tx_data <= i2s_tx_rddata;
            end else begin
                i2s_tx_data <= {i2s_tx_data[6:0], 1'd0};
            end
        end
    end

    // master, data handler

    assign i2s_sclk_enb = axilite_enable & axilite_master;
    assign i2s_sclk_out = axilite_sclk;
    assign i2s_ws_enb = axilite_enable & axilite_master;
    assign i2s_ws_out = axilite_ws;

    always @(negedge axilite_resetn or posedge axilite_clk) begin
        if (axilite_resetn == 1'b0) begin
            axilite_sclk <= 1'd0;
            axilite_clk_count <= 4'd0;
            axilite_bit_count <= 3'd0;
            axilite_byte_count <= 4'd0;
            axilite_ws <= 1'd0;
        end else begin
            if (axilite_enable == 1'b1) begin
                if (axilite_clk_count >= axilite_clk_period) begin
                    axilite_sclk <= ~axilite_sclk;
                end
                if (axilite_clk_count >= axilite_clk_period) begin
                    axilite_clk_count <= 4'd0;
                end else begin
                    axilite_clk_count <= axilite_clk_count + 1'd1;
                end
                if ((axilite_clk_count >= axilite_clk_period) && (axilite_sclk == 1'b0)) begin
                    axilite_bit_count <= axilite_bit_count + 1'd1;
                    if (axilite_bit_count == 3'd0) begin
                        if (axilite_byte_count >= axilite_no_of_bytes) begin
                            axilite_byte_count <= 4'd0;
                        end else begin
                            axilite_byte_count <= axilite_byte_count + 1'b1;
                        end
                    end
                    if ((axilite_bit_count == 3'd0) && (axilite_byte_count == 3'd0)) begin
                        axilite_ws <= ~axilite_ws;
                    end
                end
            end else begin
                axilite_sclk <= 1'd0;
                axilite_clk_count <= 4'd0;
                axilite_bit_count <= 3'd0;
                axilite_byte_count <= 4'd0;
                axilite_ws <= 1'd0;
            end
        end
    end

    // software reset

    always @(negedge axilite_resetn or posedge axilite_clk) begin
        if (axilite_resetn == 1'b0) begin
            axilite_swresetn <= 1'd0;
            axilite_swreset_clr <= 1'd0;
        end else begin
            axilite_swresetn <= ~axilite_swreset;
            axilite_swreset_clr <= axilite_swreset;
        end
    end

    // instantiations

    cdc_resetp i_cdc_resetp_sys_200m (
        .src_clk              (axilite_clk),
        .src_resetn           (axilite_swresetn),
        .dest_clk             (i2s_sclk),
        .dest_resetp          (i2s_sresetn));

    cdc #(.DATA_WIDTH(2)) i_cdc_flags (
        .src_data             ({i2s_tx_empty,
                                i2s_rx_full}),
        .dest_resetn          (axilite_resetn),
        .dest_clk             (axilite_clk),
        .dest_data            ({axilite_tx_empty,
                                axilite_rx_full}));
    mem_fifo #(
        .WRITE_LIMIT          (14),
        .ADDRESS_WIDTH        (4),
        .DATA_WIDTH           (8))
    i_rx_fifo (
        .wr_clk               (i2s_sclk),
        .wr_resetn            (i2s_sresetn),
        .wr_valid             (i2s_rx_wr),
        .wr_data              (i2s_rx_wrdata),
        .wr_limit             (i2s_rx_full),
        .rd_clk               (axilite_clk),
        .rd_resetn            (axilite_swresetn),
        .rd_read              (axilite_rx_rd),
        .rd_valid             (),
        .rd_data              (axilite_rx_rddata),
        .rd_empty             (axilite_rx_empty));

    mem_fifo #(
        .WRITE_LIMIT          (14),
        .ADDRESS_WIDTH        (4),
        .DATA_WIDTH           (8))
    i_tx_fifo (
        .wr_clk               (axilite_clk),
        .wr_resetn            (axilite_swresetn),
        .wr_valid             (axilite_tx_wr),
        .wr_data              (axilite_tx_wrdata),
        .wr_limit             (axilite_tx_full),
        .rd_clk               (i2s_sclk),
        .rd_resetn            (i2s_sresetn),
        .rd_read              (i2s_tx_rd),
        .rd_valid             (),
        .rd_data              (i2s_tx_rddata),
        .rd_empty             (i2s_tx_empty));

endmodule

// **********************************************************************************
// **********************************************************************************
