// **********************************************************************************
// **********************************************************************************
// ----------------------------------------------------------------------------------
// ################
// ##   ###########   Analog Devices Inc.
// ##      ########
// ##         #####   Copyright (c) 2019 Analog Devices Inc. All rights reserved.
// ##            ##   This file is the confidential and proprietary property of ADI.
// ##         #####   Possession or use of this file requires a written license.
// ##      ########   The licensing information may be found at: www.analog.com
// ##   ###########
// ################
// ----------------------------------------------------------------------------------
// Description:       Transmit DMA Core
// ----------------------------------------------------------------------------------
// **********************************************************************************
// **********************************************************************************

`timescale 1ps/1ps

module axi_txdma #(

    parameter integer INSTANCE_ID = 0,
    parameter integer AXIS_DW = 64,
    parameter integer AXIS_SW = 8,
    parameter integer AXIMM_DW = 512) (

    // data interface, valid must follow read with desired latency

    input   wire                            tx_clk,
    input   wire                            tx_resetn,
    input   wire                            tx_tsenb,
    input   wire                            tx_tstrig,
    input   wire                            tx_tsdsync,
    input   wire                            tx_tsrd,
    output  reg                             tx_tsav = 'd0,
    output  reg                             tx_tsvld = 'd0,
    output  reg   [(AXIS_DW-1):0]           tx_tsd = 'd0,

    // asynchronous events

    input   wire  [  3:0]                   trig,
    output  reg                             tx_mtrig_out = 'd0,
    input   wire                            tx_mtrig_in,

    // axi master interface

    input   wire                            aximm_clk,
    input   wire                            aximm_resetn,
    output  wire                            aximm_intc_cntrl,
    output  wire  [ 39:0]                   aximm_araddr,
    output  wire  [  7:0]                   aximm_arlen,
    output  wire  [  2:0]                   aximm_arsize,
    output  wire  [  1:0]                   aximm_arburst,
    output  wire  [  2:0]                   aximm_arprot,
    output  wire  [  3:0]                   aximm_arcache,
    output  wire  [  3:0]                   aximm_aruser,
    output  wire                            aximm_arvalid,
    input   wire                            aximm_arready,
    input   wire  [(AXIMM_DW-1):0]          aximm_rdata,
    input   wire  [  1:0]                   aximm_rresp,
    input   wire                            aximm_rlast,
    input   wire                            aximm_rvalid,
    output  wire                            aximm_rready,

    // axi lite interface

    input   wire                            axilite_clk,
    input   wire                            axilite_resetn,
    input   wire                            axilite_awvalid,
    input   wire  [ 15:0]                   axilite_awaddr,
    output  wire                            axilite_awready,
    input   wire                            axilite_wvalid,
    input   wire  [ 31:0]                   axilite_wdata,
    output  wire                            axilite_wready,
    output  wire                            axilite_bvalid,
    output  wire  [  1:0]                   axilite_bresp,
    input   wire                            axilite_bready,
    input   wire                            axilite_arvalid,
    input   wire  [ 15:0]                   axilite_araddr,
    output  wire                            axilite_arready,
    output  wire                            axilite_rvalid,
    output  wire  [  1:0]                   axilite_rresp,
    output  wire  [ 31:0]                   axilite_rdata,
    input   wire                            axilite_rready,
    output  reg                             axilite_intr = 'd0);

    // constants

    localparam  VERSION = {8'd0, 8'd6, 8'd4, 8'd0};

    // internal registers

    reg           [  4:0]                   tx_trig_d = 'd0;
    reg           [  6:0]                   tx_trig_enable = 'd0;
    reg                                     tx_mtrig_fall = 'd0;
    reg                                     tx_mtrig = 'd0;
    reg                                     tx_active = 'd0;
    reg                                     aximm_swresetn = 'd0;
    reg                                     aximm_swreset_d = 'd0;
    reg           [  5:0]                   aximm_swreset_count = 'd0;
    reg                                     aximm_mmq_event = 'd0;
    reg                                     aximm_start_req_d = 'd0;
    reg                                     aximm_stop_req_d = 'd0;
    reg                                     aximm_start = 'd0;
    reg                                     aximm_stop = 'd0;
    reg                                     aximm_active = 'd0;
    reg                                     aximm_status = 'd0;
    reg                                     aximm_slave_err = 'd0;
    reg                                     aximm_addrdec_err = 'd0;
    reg                                     aximm_tlast_err = 'd0;
    reg                                     aximm_length_err = 'd0;
    reg           [ 11:0]                   aximm_ack_buf_cnt = 'd0;
    reg           [ 11:0]                   aximm_req_tag = 'd0;
    reg           [ 39:0]                   aximm_req_start_addr = 40'd0;
    reg           [ 11:0]                   aximm_req_buf_cnt = 'd0;
    reg                                     aximm_req_valid = 'd0;
    reg                                     aximm_req_active = 'd0;
    reg           [  3:0]                   aximm_qcnt = 'd0;
    reg           [ 31:0]                   axilite_mmq_status = 'd0;
    reg                                     axilite_busy_d = 'd0;
    reg           [ 32:0]                   axilite_data_xfer_count = 'd0;
    reg           [ 39:0]                   axilite_start_addr = 'd0;
    reg                                     axilite_cdc_state_d = 'd0;
    reg                                     axilite_cdc_state_clr_d = 'd0;
    reg                                     axilite_cdc_state_int = 'd0;

    // internal signals

    wire                                    tx_trig_valid;
    wire                                    tx_itrig;
    wire          [  4:0]                   tx_trig;
    wire          [ 11:0]                   tx_trig_mode;
    wire                                    tx_master_trig;
    wire                                    tx_start_sync;
    wire                                    tx_cyclic_sync;
    wire                                    tx_cdc_state;
    wire                                    tx_mm_active;
    wire                                    tx_underflow;
    wire                                    tx_ready;
    wire                                    tx_enable;
    wire                                    tx_valid;
    wire          [(AXIS_DW-1):0]           tx_data;
    wire                                    tx_last;
    wire                                    tx_int_resetn;
    wire                                    tx_es_valid;
    wire                                    tx_es_ready;
    wire          [(AXIS_DW-1):0]           tx_es_data;
    wire                                    tx_es_last;
    wire                                    aximm_ack_ready;
    wire          [  3:0]                   aximm_ack_tag_p1;
    wire                                    aximm_done;
    wire          [ 79:0]                   aximm_req_data;
    wire                                    aximm_qreq;
    wire                                    aximm_qack;
    wire                                    aximm_swreset;
    wire          [ 11:0]                   aximm_trig_mode;
    wire                                    aximm_master_trig;
    wire                                    aximm_start_sync;
    wire                                    aximm_cyclic_sync;
    wire                                    aximm_cdc_state;
    wire                                    aximm_start_req;
    wire                                    aximm_stop_req;
    wire                                    aximm_cyclic;
    wire          [  3:0]                   aximm_qthreshold;
    wire          [ 39:0]                   aximm_start_addr;
    wire          [ 22:0]                   aximm_length;
    wire          [ 11:0]                   aximm_num_of_buffers;
    wire                                    aximm_valid;
    wire          [(AXIS_DW-1):0]           aximm_data;
    wire                                    aximm_last;
    wire                                    aximm_ready;
    wire                                    aximm_err;
    wire                                    aximm_req_ready;
    wire                                    aximm_ack_valid;
    wire          [  7:0]                   aximm_ack_data;
    wire                                    axilite_mmq_event;
    wire                                    axilite_mmq_tlimit_wr;
    wire          [ 31:0]                   axilite_mmq_tlimit;
    wire          [  5:0]                   axilite_intr_status;
    wire          [  5:0]                   axilite_intr_enable;
    wire          [  5:0]                   axilite_intr_active;
    wire                                    axilite_underflow;
    wire                                    axilite_swreset_clr;
    wire                                    axilite_busy;
    wire                                    axilite_status;
    wire          [  3:0]                   axilite_qcnt;
    wire                                    axilite_length_err;
    wire                                    axilite_tlast_err;
    wire                                    axilite_addrdec_err;
    wire                                    axilite_slave_err;
    wire          [ 11:0]                   axilite_req_buf_cnt;
    wire          [ 11:0]                   axilite_ack_buf_cnt;
    wire                                    axilite_swreset;
    wire                                    axilite_start;
    wire                                    axilite_stop;
    wire                                    axilite_cyclic;
    wire          [  3:0]                   axilite_qthreshold;
    wire          [ 39:0]                   axilite_offset_addr;
    wire          [ 39:0]                   axilite_base_addr;
    wire          [ 22:0]                   axilite_length;
    wire          [ 11:0]                   axilite_num_of_buffers;
    wire          [ 11:0]                   axilite_trig_mode;
    wire                                    axilite_master_trig;
    wire                                    axilite_start_sync;
    wire                                    axilite_cyclic_sync;
    wire                                    axilite_cdc_state;
    wire                                    axilite_cdc_state_clr;
    wire                                    axilite_cdc_state_clr_inv;
    wire          [ 31:0]                   axilite_clk_mon_count;
    wire                                    axilite_intc_cntrl;
    wire                                    axilite_wrreq;
    wire          [ 15:0]                   axilite_wraddr;
    wire          [ 31:0]                   axilite_wrdata;
    wire                                    axilite_wrack;
    wire                                    axilite_rdreq;
    wire          [ 15:0]                   axilite_rdaddr;
    wire          [ 31:0]                   axilite_rddata;
    wire                                    axilite_rdack;

    // interrupts

    assign axilite_intr_active = axilite_intr_status & axilite_intr_enable;
    assign axilite_intr_status[0] = ~axilite_busy;

    always @(negedge axilite_resetn or posedge axilite_clk) begin
        if (axilite_resetn == 1'b0) begin
            axilite_intr <= 1'd0;
        end else begin
            axilite_intr <= | axilite_intr_active;
        end
    end

    // device interface

    assign tx_underflow = tx_active & tx_ready & ~tx_valid;

    generate
    for (genvar n = 0; n < 5; n = n + 1) begin: g_tx_trig

    always @(negedge tx_int_resetn or posedge tx_clk) begin
        if (tx_int_resetn == 1'b0) begin
            tx_trig_d[n] <= 1'd0;
            tx_trig_enable[n] <= 1'd0;
        end else begin
            tx_trig_d[n] <= tx_trig[n];
            if (tx_valid == 1'd0) begin
                tx_trig_enable[n] <= 1'd0;
            end else if (tx_trig_enable[n] == 1'd0) begin
                case (tx_trig_mode[((n*2)+1):(n*2)])
                    2'd3: tx_trig_enable[n] <= tx_trig[n] ^ tx_trig_d[n];
                    2'd2: tx_trig_enable[n] <= ~tx_trig[n] & tx_trig_d[n];
                    2'd1: tx_trig_enable[n] <= tx_trig[n] & ~tx_trig_d[n];
                    default: tx_trig_enable[n] <= 1'd1;
                endcase
            end
        end
    end

    end
    endgenerate

    always @(negedge tx_int_resetn or posedge tx_clk) begin
        if (tx_int_resetn == 1'b0) begin
            tx_trig_enable[5] <= 1'd0;
            tx_trig_enable[6] <= 1'd0;
        end else begin
            if ((tx_start_sync == 1'd0) || ((tx_valid == 1'd1) &&
                (tx_tsenb == 1'd0))) begin
                tx_trig_enable[5] <= 1'd1;
            end else if (tx_valid == 1'd0) begin
                tx_trig_enable[5] <= 1'd0;
            end
            if ((tx_cyclic_sync == 1'd0) || ((tx_valid == 1'd1) &&
                (tx_tsenb == 1'd0))) begin
                tx_trig_enable[6] <= 1'd1;
            end else if ((tx_enable == 1'd1) && (tx_last == 1'd1)) begin
                tx_trig_enable[6] <= 1'd0;
            end
        end
    end

    assign tx_trig_valid = & tx_trig_enable;
    assign tx_itrig = (tx_master_trig == 1'd1) ? tx_mtrig : tx_trig_valid;

    always @(negedge tx_int_resetn or negedge tx_clk) begin
        if (tx_int_resetn == 1'b0) begin
            tx_mtrig_fall <= 1'd0;
        end else begin
            tx_mtrig_fall <= tx_mtrig_in;
        end
    end

    always @(negedge tx_int_resetn or posedge tx_clk) begin
        if (tx_int_resetn == 1'b0) begin
            tx_mtrig_out <= 1'd0;
            tx_mtrig <= 1'd0;
        end else begin
            if (tx_tsdsync == 1'd1) begin
                tx_mtrig_out <= tx_trig_valid;
            end
            tx_mtrig <= tx_mtrig_fall;
        end
    end

    assign tx_ready = tx_tsrd & tx_itrig;
    assign tx_enable = tx_valid & tx_ready;

    always @(negedge tx_int_resetn or posedge tx_clk) begin
        if (tx_int_resetn == 1'b0) begin
            tx_tsav <= 1'd0;
            tx_tsvld <= 1'd0;
            tx_tsd <= 'd0;
            tx_active <= 1'd0;
        end else begin
            tx_tsav <= tx_valid;
            tx_tsvld <= tx_tsrd;
            if (tx_tsrd == 1'd1) begin
                if (tx_enable == 1'd1) begin
                    tx_tsd <= tx_data;
                end else begin
                    tx_tsd <= 'd0;
                end
            end
            if (tx_active == 1'd1) begin
                tx_active <= tx_mm_active;
            end else begin
                tx_active <= tx_mm_active & tx_valid;
            end
        end
    end

    // aximm interface

    assign aximm_ack_ready = 1'b1;
    assign aximm_ack_tag_p1 = aximm_ack_data[3:0] + 1'd1;
    assign aximm_done = (aximm_ack_tag_p1 == aximm_req_tag[3:0]) ?
        (~aximm_req_active & aximm_ack_valid &
        (~aximm_cyclic | aximm_stop)) : 1'd0;
    assign aximm_req_data[79:76] = 4'd0;
    assign aximm_req_data[75:72] = aximm_req_tag[3:0];
    assign aximm_req_data[71:32] = aximm_req_start_addr;
    assign aximm_req_data[31] = 1'b0;
    assign aximm_req_data[30] = 1'b1;
    assign aximm_req_data[29:24] = 6'd0;
    assign aximm_req_data[23] = 1'b1;
    assign aximm_req_data[22:0] = aximm_length;
    assign aximm_qreq = aximm_req_valid & aximm_req_ready;
    assign aximm_qack = aximm_ack_valid & aximm_ack_ready;

    always @(negedge aximm_resetn or posedge aximm_clk) begin
        if (aximm_resetn == 1'b0) begin
            aximm_swresetn <= 1'd0;
            aximm_swreset_d <= 1'd0;
            aximm_swreset_count <= 6'd0;
        end else begin
            aximm_swresetn <= ~aximm_swreset_count[5];
            aximm_swreset_d <= aximm_swreset;
            if (aximm_swreset_count[5] == 1'd1) begin
                aximm_swreset_count <= aximm_swreset_count + 1'd1;
            end else if ((aximm_swreset == 1'd1) && (aximm_swreset_d == 1'd0)) begin
                aximm_swreset_count <= 6'h20;
            end
        end
    end

    always @(negedge aximm_swresetn or posedge aximm_clk) begin
        if (aximm_swresetn == 1'b0) begin
            aximm_mmq_event <= 1'b0;
        end else begin
            if ((aximm_valid == 1'b1) && (aximm_ready == 1'b1)) begin
                aximm_mmq_event <= 1'b1;
            end else begin
                aximm_mmq_event <= 1'b0;
            end
        end
    end

    always @(negedge aximm_swresetn or posedge aximm_clk) begin
        if (aximm_swresetn == 1'b0) begin
            aximm_start_req_d <= 1'd0;
            aximm_stop_req_d <= 1'd0;
            aximm_start <= 1'd0;
            aximm_stop <= 1'd0;
            aximm_active <= 1'd0;
            aximm_status <= 1'b1;
            aximm_slave_err <= 1'b0;
            aximm_addrdec_err <= 1'b0;
            aximm_tlast_err <= 1'b0;
            aximm_length_err <= 1'b0;
            aximm_ack_buf_cnt <= 12'b0;
            aximm_req_tag <= 12'd0;
            aximm_req_start_addr <= 40'd0;
            aximm_req_buf_cnt <= 12'd0;
            aximm_req_valid <= 1'b0;
            aximm_req_active <= 1'b0;
            aximm_qcnt <= 4'd0;
        end else begin
            aximm_start_req_d <= aximm_start_req;
            aximm_stop_req_d <= aximm_stop_req;
            aximm_start <= aximm_start_req & ~aximm_start_req_d;
            if (aximm_stop == 1'b1) begin
                aximm_stop <= ~aximm_done;
            end else begin
                aximm_stop <= aximm_stop_req & ~aximm_stop_req_d;
            end
            if (aximm_active == 1'b1) begin
                aximm_active <= ~aximm_done;
            end else begin
                aximm_active <= aximm_start;
            end
            if (aximm_ack_valid == 1'b1) begin
                aximm_status <= aximm_status & aximm_ack_data[7];
                aximm_slave_err <= aximm_slave_err | aximm_ack_data[6];
                aximm_addrdec_err <= aximm_addrdec_err | aximm_ack_data[5];
                aximm_tlast_err <= aximm_tlast_err | aximm_ack_data[4];
                aximm_length_err <= aximm_length_err | aximm_err;
                if (aximm_ack_buf_cnt >= aximm_num_of_buffers) begin
                    aximm_ack_buf_cnt <= 12'd0;
                end else begin
                    aximm_ack_buf_cnt <= aximm_ack_buf_cnt + 1'd1;
                end
            end else if (aximm_start == 1'b1) begin
                aximm_status <= 1'b1;
                aximm_slave_err <= 1'b0;
                aximm_addrdec_err <= 1'b0;
                aximm_tlast_err <= 1'b0;
                aximm_length_err <= 1'b0;
                aximm_ack_buf_cnt <= 12'd0;
            end
            if (aximm_req_active == 1'b1) begin
                if ((aximm_req_valid == 1'b1) && (aximm_req_ready == 1'b1)) begin
                    aximm_req_tag <= aximm_req_tag + 1'b1;
                    if (aximm_req_buf_cnt >= aximm_num_of_buffers) begin
                        aximm_req_start_addr <= aximm_start_addr;
                        aximm_req_buf_cnt <= 12'd0;
                    end else begin
                        aximm_req_start_addr <= aximm_req_start_addr + aximm_length;
                        aximm_req_buf_cnt <= aximm_req_buf_cnt + 1'd1;
                    end
                end
                if ((aximm_req_valid == 1'b0) || (aximm_req_ready == 1'b1)) begin
                    aximm_req_valid <= (aximm_qcnt > aximm_qthreshold) ? 1'b0 :
                        ((aximm_req_buf_cnt >= aximm_num_of_buffers) ?
                        (~aximm_req_valid | (aximm_cyclic & ~aximm_stop)) : 1'b1);
                end
                if ((aximm_req_valid == 1'b1) && (aximm_req_ready == 1'b1) &&
                    (aximm_req_buf_cnt >= aximm_num_of_buffers)) begin
                    aximm_req_active <= aximm_cyclic & ~aximm_stop;
                end
            end else if (aximm_start == 1'd1) begin
                aximm_req_tag <= 12'd0;
                aximm_req_start_addr <= aximm_start_addr;
                aximm_req_buf_cnt <= 12'd0;
                aximm_req_valid <= 1'b1;
                aximm_req_active <= 1'b1;
            end
            if ((aximm_qreq == 1'b1) && (aximm_qack == 1'b0)) begin
                aximm_qcnt <= aximm_qcnt + 1'b1;
            end else if ((aximm_qreq == 1'b0) && (aximm_qack == 1'b1)) begin
                aximm_qcnt <= aximm_qcnt - 1'b1;
            end
        end
    end

    assign tx_trig[4] = tx_tstrig;

    // start address & cdc state

    assign axilite_cdc_state_clr_inv = ~axilite_cdc_state_clr;

    always @(negedge axilite_resetn or posedge axilite_clk) begin
        if (axilite_resetn == 1'b0) begin
            axilite_start_addr <= 'd0;
            axilite_cdc_state_d <= 'd0;
            axilite_cdc_state_clr_d <= 'd0;
            axilite_cdc_state_int <= 'd0;
        end else begin
            axilite_start_addr <= axilite_base_addr + axilite_offset_addr;
            axilite_cdc_state_d <= axilite_cdc_state;
            axilite_cdc_state_clr_d <= axilite_cdc_state_clr;
            if ((axilite_cdc_state_clr_d == 1'd0) &&
                (axilite_cdc_state_clr == 1'd1)) begin
                axilite_cdc_state_int <= 1'd0;
            end else if ((axilite_cdc_state_d == 1'd0) &&
                (axilite_cdc_state == 1'd1)) begin
                axilite_cdc_state_int <= 1'd1;
            end
        end
    end

    always @(negedge axilite_resetn or posedge axilite_clk) begin
        if (axilite_resetn == 1'b0) begin
            axilite_mmq_status <= 32'd0;
        end else begin
            if ((axilite_mmq_event == 1'd1) || (axilite_mmq_tlimit_wr == 1'd1)) begin
                axilite_mmq_status <= axilite_mmq_tlimit;
            end else if (axilite_mmq_status > 0) begin
                axilite_mmq_status <= axilite_mmq_status - 1'd1;
            end
        end
    end

    always @(negedge axilite_resetn or posedge axilite_clk) begin
        if (axilite_resetn == 1'b0) begin
            axilite_busy_d <= 1'd0;
            axilite_data_xfer_count <= 33'd0;
        end else begin
            axilite_busy_d <= axilite_busy;
            if ((axilite_busy_d == 1'd0) && (axilite_busy == 1'd1)) begin
                axilite_data_xfer_count <= 33'd0;
            end else if (axilite_data_xfer_count[32] == 1'd1) begin
                axilite_data_xfer_count <= {33{1'd1}};
            end else if (axilite_busy == 1'd1) begin
                axilite_data_xfer_count <= axilite_data_xfer_count + 1'd1;
            end
        end
    end

    assign tx_valid = tx_es_valid;
    assign tx_data = tx_es_data;
    assign tx_last = tx_es_last;
    assign tx_es_ready = tx_ready;

    // instantiations

    clk_mon i_clk_mon (
        .clk                                (aximm_clk),
        .axilite_resetn                     (axilite_resetn),
        .axilite_clk                        (axilite_clk),
        .axilite_clk_mon_count              (axilite_clk_mon_count));

    cdc #(.DATA_WIDTH(4)) i_cdc_tx (
        .src_data                           (trig),
        .dest_resetn                        (tx_int_resetn),
        .dest_clk                           (tx_clk),
        .dest_data                          (tx_trig[3:0]));

    cdc_cntrl #(.DATA_WIDTH(17)) i_cdc_cntrl_tx (
        .src_resetn                         (aximm_resetn),
        .src_clk                            (aximm_clk),
        .src_data                           ({aximm_active,
                                                aximm_trig_mode,
                                                aximm_master_trig,
                                                aximm_start_sync,
                                                aximm_cyclic_sync,
                                                aximm_cdc_state}),
        .dest_resetn                        (tx_int_resetn),
        .dest_clk                           (tx_clk),
        .dest_data                          ({tx_mm_active,
                                                tx_trig_mode,
                                                tx_master_trig,
                                                tx_start_sync,
                                                tx_cyclic_sync,
                                                tx_cdc_state}));

    cdc_cntrl #(.DATA_WIDTH(100)) i_cdc_cntrl_aximm (
        .src_resetn                         (axilite_resetn),
        .src_clk                            (axilite_clk),
        .src_data                           ({axilite_swreset,
                                                axilite_intc_cntrl,
                                                axilite_trig_mode,
                                                axilite_master_trig,
                                                axilite_start_sync,
                                                axilite_cyclic_sync,
                                                axilite_cdc_state_int,
                                                axilite_start,
                                                axilite_stop,
                                                axilite_cyclic,
                                                axilite_qthreshold,
                                                axilite_start_addr,
                                                axilite_length,
                                                axilite_num_of_buffers}),
        .dest_resetn                        (aximm_resetn),
        .dest_clk                           (aximm_clk),
        .dest_data                          ({aximm_swreset,
                                                aximm_intc_cntrl,
                                                aximm_trig_mode,
                                                aximm_master_trig,
                                                aximm_start_sync,
                                                aximm_cyclic_sync,
                                                aximm_cdc_state,
                                                aximm_start_req,
                                                aximm_stop_req,
                                                aximm_cyclic,
                                                aximm_qthreshold,
                                                aximm_start_addr,
                                                aximm_length,
                                                aximm_num_of_buffers}));

    cdc #(.DATA_WIDTH(1)) i_cdc_axilite (
        .src_data                           (tx_cdc_state),
        .dest_resetn                        (axilite_resetn),
        .dest_clk                           (axilite_clk),
        .dest_data                          (axilite_cdc_state_clr));

    cdc_event #(.DATA_WIDTH(1)) i_cdc_event_tx (
        .src_resetn                         (tx_int_resetn),
        .src_clk                            (tx_clk),
        .src_data                           (tx_underflow),
        .dest_resetn                        (axilite_resetn),
        .dest_clk                           (axilite_clk),
        .dest_data                          (axilite_underflow));

    cdc_event #(.DATA_WIDTH(1)) i_cdc_event_aximm (
        .src_resetn                         (aximm_resetn),
        .src_clk                            (aximm_clk),
        .src_data                           (aximm_mmq_event),
        .dest_resetn                        (axilite_resetn),
        .dest_clk                           (axilite_clk),
        .dest_data                          (axilite_mmq_event));

    cdc_cntrl #(.DATA_WIDTH(35)) i_cdc_cntrl_axilite (
        .src_resetn                         (aximm_resetn),
        .src_clk                            (aximm_clk),
        .src_data                           ({aximm_swresetn,
                                                aximm_active,
                                                aximm_status,
                                                aximm_qcnt,
                                                aximm_length_err,
                                                aximm_tlast_err,
                                                aximm_addrdec_err,
                                                aximm_slave_err,
                                                aximm_req_buf_cnt,
                                                aximm_ack_buf_cnt}),
        .dest_resetn                        (axilite_resetn),
        .dest_clk                           (axilite_clk),
        .dest_data                          ({axilite_swreset_clr,
                                                axilite_busy,
                                                axilite_status,
                                                axilite_qcnt,
                                                axilite_length_err,
                                                axilite_tlast_err,
                                                axilite_addrdec_err,
                                                axilite_slave_err,
                                                axilite_req_buf_cnt,
                                                axilite_ack_buf_cnt}));

    cdc_resetp i_cdc_resetp (
        .src_clk                            (tx_clk),
        .src_resetn                         (tx_resetn),
        .dest_clk                           (tx_clk),
        .dest_resetp                        (tx_int_resetn));

    axis_data_fifo_v2_0_15_top #(
        .C_FAMILY                           ("zynq"),
        .C_AXIS_TDATA_WIDTH                 (AXIS_DW),
        .C_AXIS_TID_WIDTH                   (1),
        .C_AXIS_TDEST_WIDTH                 (1),
        .C_AXIS_TUSER_WIDTH                 (1),
        .C_AXIS_SIGNAL_SET                  (32'h1b),
        .C_FIFO_DEPTH                       (512),
        .C_FIFO_MODE                        (1),
        .C_IS_ACLK_ASYNC                    (1),
        .C_SYNCHRONIZER_STAGE               (3),
        .C_ACLKEN_CONV_MODE                 (0),
        .C_ECC_MODE                         (0),
        .C_FIFO_MEMORY_TYPE                 ("auto"),
        .C_USE_ADV_FEATURES                 ("1000"),
        .C_PROG_EMPTY_THRESH                (5),
        .C_PROG_FULL_THRESH                 (11))
    i_es (
        .s_axis_aclk                        (aximm_clk),
        .s_axis_aresetn                     (aximm_swresetn),
        .s_axis_aclken                      (1'b1),
        .s_axis_tvalid                      (aximm_valid),
        .s_axis_tready                      (aximm_ready),
        .s_axis_tdata                       (aximm_data),
        .s_axis_tstrb                       ({AXIS_SW{1'b1}}),
        .s_axis_tkeep                       ({AXIS_SW{1'b1}}),
        .s_axis_tlast                       (aximm_last),
        .s_axis_tid                         (1'b0),
        .s_axis_tdest                       (1'b0),
        .s_axis_tuser                       (1'b0),
        .almost_full                        (),
        .prog_full                          (),
        .axis_wr_data_count                 (),
        .injectsbiterr                      (1'b0),
        .injectdbiterr                      (1'b0),
        .m_axis_aclk                        (tx_clk),
        .m_axis_aclken                      (1'b1),
        .m_axis_tvalid                      (tx_es_valid),
        .m_axis_tready                      (tx_es_ready),
        .m_axis_tdata                       (tx_es_data),
        .m_axis_tstrb                       (),
        .m_axis_tkeep                       (),
        .m_axis_tlast                       (tx_es_last),
        .m_axis_tid                         (),
        .m_axis_tdest                       (),
        .m_axis_tuser                       (),
        .almost_empty                       (),
        .prog_empty                         (),
        .axis_rd_data_count                 (),
        .sbiterr                            (),
        .dbiterr                            ());

    axi_datamover #(
        .C_INCLUDE_MM2S                     (1),
        .C_M_AXI_MM2S_ARID                  (0),
        .C_M_AXI_MM2S_ID_WIDTH              (1),
        .C_M_AXI_MM2S_ADDR_WIDTH            (40),
        .C_M_AXI_MM2S_DATA_WIDTH            (AXIMM_DW),
        .C_M_AXIS_MM2S_TDATA_WIDTH          (AXIS_DW),
        .C_INCLUDE_MM2S_STSFIFO             (1),
        .C_MM2S_STSCMD_FIFO_DEPTH           (4),
        .C_MM2S_STSCMD_IS_ASYNC             (0),
        .C_INCLUDE_MM2S_DRE                 (0),
        .C_MM2S_BURST_SIZE                  (16),
        .C_MM2S_BTT_USED                    (23),
        .C_MM2S_ADDR_PIPE_DEPTH             (3),
        .C_MM2S_INCLUDE_SF                  (1),
        .C_INCLUDE_S2MM                     (0),
        .C_M_AXI_S2MM_AWID                  (1),
        .C_M_AXI_S2MM_ID_WIDTH              (4),
        .C_M_AXI_S2MM_ADDR_WIDTH            (40),
        .C_M_AXI_S2MM_DATA_WIDTH            (32),
        .C_S_AXIS_S2MM_TDATA_WIDTH          (32),
        .C_INCLUDE_S2MM_STSFIFO             (0),
        .C_S2MM_STSCMD_FIFO_DEPTH           (4),
        .C_S2MM_STSCMD_IS_ASYNC             (0),
        .C_INCLUDE_S2MM_DRE                 (0),
        .C_S2MM_BURST_SIZE                  (16),
        .C_S2MM_BTT_USED                    (16),
        .C_S2MM_SUPPORT_INDET_BTT           (0),
        .C_S2MM_ADDR_PIPE_DEPTH             (3),
        .C_S2MM_INCLUDE_SF                  (0),
        .C_ENABLE_CACHE_USER                (0),
        .C_ENABLE_SKID_BUF                  ("11111"),
        .C_ENABLE_MM2S_TKEEP                (1),
        .C_ENABLE_S2MM_TKEEP                (1),
        .C_ENABLE_S2MM_ADV_SIG              (0),
        .C_ENABLE_MM2S_ADV_SIG              (0),
        .C_MICRO_DMA                        (0),
        .C_CMD_WIDTH                        (80),
        .C_FAMILY                           ("zynq"))
    i_mm (
        .m_axi_mm2s_aclk                    (aximm_clk),
        .m_axi_mm2s_aresetn                 (aximm_swresetn),
        .mm2s_halt                          (1'b0),
        .mm2s_halt_cmplt                    (),
        .mm2s_err                           (aximm_err),
        .m_axis_mm2s_cmdsts_aclk            (aximm_clk),
        .m_axis_mm2s_cmdsts_aresetn         (aximm_swresetn),
        .s_axis_mm2s_cmd_tvalid             (aximm_req_valid),
        .s_axis_mm2s_cmd_tready             (aximm_req_ready),
        .s_axis_mm2s_cmd_tdata              (aximm_req_data),
        .m_axis_mm2s_sts_tvalid             (aximm_ack_valid),
        .m_axis_mm2s_sts_tready             (aximm_ack_ready),
        .m_axis_mm2s_sts_tdata              (aximm_ack_data),
        .m_axis_mm2s_sts_tkeep              (),
        .m_axis_mm2s_sts_tlast              (),
        .mm2s_allow_addr_req                (1'b1),
        .mm2s_addr_req_posted               (),
        .mm2s_rd_xfer_cmplt                 (),
        .m_axi_mm2s_arid                    (),
        .m_axi_mm2s_araddr                  (aximm_araddr),
        .m_axi_mm2s_arlen                   (aximm_arlen),
        .m_axi_mm2s_arsize                  (aximm_arsize),
        .m_axi_mm2s_arburst                 (aximm_arburst),
        .m_axi_mm2s_arprot                  (aximm_arprot),
        .m_axi_mm2s_arcache                 (aximm_arcache),
        .m_axi_mm2s_aruser                  (aximm_aruser),
        .m_axi_mm2s_arvalid                 (aximm_arvalid),
        .m_axi_mm2s_arready                 (aximm_arready),
        .m_axi_mm2s_rdata                   (aximm_rdata),
        .m_axi_mm2s_rresp                   (aximm_rresp),
        .m_axi_mm2s_rlast                   (aximm_rlast),
        .m_axi_mm2s_rvalid                  (aximm_rvalid),
        .m_axi_mm2s_rready                  (aximm_rready),
        .m_axis_mm2s_tdata                  (aximm_data),
        .m_axis_mm2s_tkeep                  (),
        .m_axis_mm2s_tlast                  (aximm_last),
        .m_axis_mm2s_tvalid                 (aximm_valid),
        .m_axis_mm2s_tready                 (aximm_ready),
        .mm2s_dbg_sel                       (4'd0),
        .mm2s_dbg_data                      (),
        .m_axi_s2mm_aclk                    (1'b0),
        .m_axi_s2mm_aresetn                 (1'b1),
        .s2mm_halt                          (1'b0),
        .s2mm_halt_cmplt                    (),
        .s2mm_err                           (),
        .m_axis_s2mm_cmdsts_awclk           (1'b0),
        .m_axis_s2mm_cmdsts_aresetn         (1'b1),
        .s_axis_s2mm_cmd_tvalid             (1'b0),
        .s_axis_s2mm_cmd_tready             (),
        .s_axis_s2mm_cmd_tdata              (80'd0),
        .m_axis_s2mm_sts_tvalid             (),
        .m_axis_s2mm_sts_tready             (1'b0),
        .m_axis_s2mm_sts_tdata              (),
        .m_axis_s2mm_sts_tkeep              (),
        .m_axis_s2mm_sts_tlast              (),
        .s2mm_allow_addr_req                (1'b1),
        .s2mm_addr_req_posted               (),
        .s2mm_wr_xfer_cmplt                 (),
        .s2mm_ld_nxt_len                    (),
        .s2mm_wr_len                        (),
        .m_axi_s2mm_awid                    (),
        .m_axi_s2mm_awaddr                  (),
        .m_axi_s2mm_awlen                   (),
        .m_axi_s2mm_awsize                  (),
        .m_axi_s2mm_awburst                 (),
        .m_axi_s2mm_awprot                  (),
        .m_axi_s2mm_awcache                 (),
        .m_axi_s2mm_awuser                  (),
        .m_axi_s2mm_awvalid                 (),
        .m_axi_s2mm_awready                 (1'b0),
        .m_axi_s2mm_wdata                   (),
        .m_axi_s2mm_wstrb                   (),
        .m_axi_s2mm_wlast                   (),
        .m_axi_s2mm_wvalid                  (),
        .m_axi_s2mm_wready                  (1'b0),
        .m_axi_s2mm_bresp                   (2'd0),
        .m_axi_s2mm_bvalid                  (1'b0),
        .m_axi_s2mm_bready                  (),
        .s_axis_s2mm_tdata                  (32'd0),
        .s_axis_s2mm_tkeep                  (4'd0),
        .s_axis_s2mm_tlast                  (1'b0),
        .s_axis_s2mm_tvalid                 (1'b0),
        .s_axis_s2mm_tready                 (),
        .s2mm_dbg_sel                       (4'd0),
        .s2mm_dbg_data                      ());

    axi_dma_top_regs #(.ID(0)) i_regs (
        .axilite_version                    (VERSION),
        .axilite_instance_id                (INSTANCE_ID),
        .axilite_scratch                    (),
        .axilite_timer                      (),
        .axilite_aximm_width                (AXIMM_DW),
        .axilite_axis_width                 (AXIS_DW),
        .axilite_dma_tx1_rx0                ('d1),
        .axilite_swreset                    (axilite_swreset),
        .axilite_swreset_clr                (axilite_swreset_clr),
        .axilite_start                      (axilite_start),
        .axilite_start_clr                  (~axilite_busy),
        .axilite_stop                       (axilite_stop),
        .axilite_stop_clr                   (~axilite_busy),
        .axilite_cyclic                     (axilite_cyclic),
        .axilite_qthreshold                 (axilite_qthreshold),
        .axilite_busy                       (axilite_busy),
        .axilite_status                     (axilite_status),
        .axilite_ovf_unf                    (axilite_underflow),
        .axilite_intr_ovf_unf               (axilite_intr_status[1]),
        .axilite_length_error               (axilite_length_err),
        .axilite_intr_length_error          (axilite_intr_status[2]),
        .axilite_tlast_error                (axilite_tlast_err),
        .axilite_intr_tlast_error           (axilite_intr_status[3]),
        .axilite_addrdec_error              (axilite_addrdec_err),
        .axilite_intr_addrdec_error         (axilite_intr_status[4]),
        .axilite_slave_error                (axilite_slave_err),
        .axilite_intr_slave_error           (axilite_intr_status[5]),
        .axilite_req_buf_num                (axilite_req_buf_cnt),
        .axilite_ack_buf_num                (axilite_ack_buf_cnt),
        .axilite_start_addr_low             (axilite_offset_addr[31:0]),
        .axilite_start_addr_high            (axilite_offset_addr[39:32]),
        .axilite_base_addr_low              (axilite_base_addr[31:0]),
        .axilite_base_addr_high             (axilite_base_addr[39:32]),
        .axilite_length                     (axilite_length),
        .axilite_num_of_buffers             (axilite_num_of_buffers),
        .axilite_trigger_mode_0             (axilite_trig_mode[1:0]),
        .axilite_trigger_mode_1             (axilite_trig_mode[3:2]),
        .axilite_trigger_mode_2             (axilite_trig_mode[5:4]),
        .axilite_trigger_mode_3             (axilite_trig_mode[7:6]),
        .axilite_trigger_mode_4             (axilite_trig_mode[9:8]),
        .axilite_trigger_mode_5             (axilite_trig_mode[11:10]),
        .axilite_start_sync                 (axilite_start_sync),
        .axilite_cyclic_sync                (axilite_cyclic_sync),
        .axilite_cdc_state                  (axilite_cdc_state),
        .axilite_cdc_state_clr              (axilite_cdc_state_clr_inv),
        .axilite_master_trigger             (axilite_master_trig),
        .axilite_mmq_tlimit_wr              (axilite_mmq_tlimit_wr),
        .axilite_mmq_tlimit_data            (axilite_mmq_tlimit),
        .axilite_mmq_status                 (axilite_mmq_status),
        .axilite_mmq_count                  (axilite_qcnt),
        .axilite_intr_done_enable           (axilite_intr_enable[0]),
        .axilite_intr_ovf_unf_enable        (axilite_intr_enable[1]),
        .axilite_intr_length_error_enable   (axilite_intr_enable[2]),
        .axilite_intr_tlast_error_enable    (axilite_intr_enable[3]),
        .axilite_intr_addrdec_error_enable  (axilite_intr_enable[4]),
        .axilite_intr_slave_error_enable    (axilite_intr_enable[5]),
        .axilite_clk_mon_count              (axilite_clk_mon_count),
        .axilite_data_xfer_count            (axilite_data_xfer_count[31:0]),
        .axilite_intc_cntrl                 (axilite_intc_cntrl),
        .axilite_clk                        (axilite_clk),
        .axilite_resetn                     (axilite_resetn),
        .axilite_wrreq                      (axilite_wrreq),
        .axilite_wraddr                     (axilite_wraddr),
        .axilite_wrdata                     (axilite_wrdata),
        .axilite_wrack                      (axilite_wrack),
        .axilite_rdreq                      (axilite_rdreq),
        .axilite_rdaddr                     (axilite_rdaddr),
        .axilite_rddata                     (axilite_rddata),
        .axilite_rdack                      (axilite_rdack));

    axilite_slave_if #(.AXI_ADDRESS_WIDTH(16)) i_axilite_if (
        .axilite_clk                        (axilite_clk),
        .axilite_resetn                     (axilite_resetn),
        .axilite_awvalid                    (axilite_awvalid),
        .axilite_awaddr                     (axilite_awaddr),
        .axilite_awready                    (axilite_awready),
        .axilite_wvalid                     (axilite_wvalid),
        .axilite_wdata                      (axilite_wdata),
        .axilite_wready                     (axilite_wready),
        .axilite_bvalid                     (axilite_bvalid),
        .axilite_bresp                      (axilite_bresp),
        .axilite_bready                     (axilite_bready),
        .axilite_arvalid                    (axilite_arvalid),
        .axilite_araddr                     (axilite_araddr),
        .axilite_arready                    (axilite_arready),
        .axilite_rvalid                     (axilite_rvalid),
        .axilite_rresp                      (axilite_rresp),
        .axilite_rdata                      (axilite_rdata),
        .axilite_rready                     (axilite_rready),
        .axilite_wrreq                      (axilite_wrreq),
        .axilite_wraddr                     (axilite_wraddr),
        .axilite_wrdata                     (axilite_wrdata),
        .axilite_wrack                      (axilite_wrack),
        .axilite_rdreq                      (axilite_rdreq),
        .axilite_rdaddr                     (axilite_rdaddr),
        .axilite_rddata                     (axilite_rddata),
        .axilite_rdack                      (axilite_rdack));

endmodule

// **********************************************************************************
// **********************************************************************************
