// **********************************************************************************
// **********************************************************************************
// ----------------------------------------------------------------------------------
// ################
// ##   ###########   Analog Devices Inc.
// ##      ########
// ##         #####   Copyright (c) 2019 Analog Devices Inc. All rights reserved.
// ##            ##   This file is the confidential and proprietary property of ADI.
// ##         #####   Possession or use of this file requires a written license.
// ##      ########   The licensing information may be found at: www.analog.com
// ##   ###########
// ################
// ----------------------------------------------------------------------------------
// Description:       System Id
// ----------------------------------------------------------------------------------
// **********************************************************************************
// **********************************************************************************

`timescale 1ps/1ps

module axi_sysid #(

    parameter integer INSTANCE_ID = 0,
    parameter integer INTR_WIDTH = 8,
    parameter integer STATUS_WIDTH = 8,
    parameter integer CONTROL_WIDTH = 8,
    parameter integer GPIO_IN_WIDTH = 8,
    parameter integer GPIO_OUT_WIDTH = 8,
    parameter integer PWM_OUT_WIDTH = 8,
    parameter [ 31:0] BOARD_ID = 32'd0,
    parameter [ 31:0] MEMORY_ID = 32'd0,
    parameter [ 31:0] SYS_VERSION = 32'd0,
    parameter [ 63:0] SYS_PARAMS = 64'd0,
    parameter [255:0] SYS_INFO_0 = 256'd0,
    parameter [255:0] SYS_INFO_1 = 256'd0,
    parameter [ 63:0] SYS_BUILD = 64'd0) (

    // peripheral interface
 
    input   wire  [(INTR_WIDTH-1):0]        axilite_sys_in_intr,
    output  reg                             axilite_sys_intr = 'd0,

    // system interface

    input   wire  [(STATUS_WIDTH-1):0]      status,
    output  wire  [(CONTROL_WIDTH-1):0]     control,
    input   wire  [(GPIO_IN_WIDTH-1):0]     gpio_in,
    output  wire  [(GPIO_OUT_WIDTH-1):0]    gpio_out,
    output  wire  [(PWM_OUT_WIDTH-1):0]     pwm_out,

    // memory interface

    input   wire  [ 11:0]                   pg,
    output  wire  [ 11:0]                   pg_sel,

    // axilite interface

    input   wire                            axilite_clk,
    input   wire                            axilite_resetn,
    input   wire                            axilite_awvalid,
    input   wire  [ 11:0]                   axilite_awaddr,
    output  wire                            axilite_awready,
    input   wire                            axilite_wvalid,
    input   wire  [ 31:0]                   axilite_wdata,
    output  wire                            axilite_wready,
    output  wire                            axilite_bvalid,
    output  wire  [  1:0]                   axilite_bresp,
    input   wire                            axilite_bready,
    input   wire                            axilite_arvalid,
    input   wire  [ 11:0]                   axilite_araddr,
    output  wire                            axilite_arready,
    output  wire                            axilite_rvalid,
    output  wire  [  1:0]                   axilite_rresp,
    output  wire  [ 31:0]                   axilite_rdata,
    input   wire                            axilite_rready,
    output  reg                             axilite_intr = 'd0);

    // constants

    localparam  VERSION = {8'd0, 8'd6, 8'd2, 8'd1};
    localparam  PWM_INT_WIDTH = (PWM_OUT_WIDTH < 8) ? PWM_OUT_WIDTH : 8;

    // internal registers

    reg           [  3:0]                   axilite_gpio_in_deb[0:(GPIO_IN_WIDTH-1)];
    reg           [(GPIO_IN_WIDTH-1):0]     axilite_gpio_in = 'd0;
    reg           [(PWM_INT_WIDTH-1):0]     axilite_pwm_data = 'd0;
    reg           [  1:0]                   axilite_pwm_state[0:(PWM_INT_WIDTH-1)];
    reg           [ 15:0]                   axilite_pwm_count[0:(PWM_INT_WIDTH-1)];
    reg                                     axilite_pwm_update = 'd0;
    reg           [ 31:0]                   axilite_pwm_prescale = 'd0;

    // internal signals
 
    wire          [ 31:0]                   axilite_sys_intr_active;
    wire          [ 31:0]                   axilite_sys_intr_status;
    wire          [ 31:0]                   axilite_sys_intr_enable;
    wire          [ 15:0]                   axilite_intr_active;
    wire          [ 15:0]                   axilite_intr_status;
    wire          [ 15:0]                   axilite_intr_enable;
    wire          [ 15:0]                   axilite_pwm_limit[0:(PWM_INT_WIDTH-1)];
    wire          [(GPIO_IN_WIDTH-1):0]     axilite_gpio_in_raw;
    wire          [  7:0]                   axilite_gpio_in_state;
    wire          [  7:0]                   axilite_gpio_out;
    wire          [ 31:0]                   axilite_pwm_prescale_count;
    wire          [  7:0]                   axilite_pwm_enable;
    wire          [  7:0]                   axilite_pwm_out;
    wire          [ 15:0]                   axilite_pwm_delay_count[0:7];
    wire          [ 15:0]                   axilite_pwm_high_count[0:7];
    wire          [ 15:0]                   axilite_pwm_low_count[0:7];
    wire          [(STATUS_WIDTH-1):0]      axilite_status;
    wire          [  7:0]                   axilite_status_state;
    wire          [  7:0]                   axilite_control_state;
    wire                                    axilite_wrreq;
    wire          [ 11:0]                   axilite_wraddr;
    wire          [ 31:0]                   axilite_wrdata;
    wire                                    axilite_wrack;
    wire                                    axilite_rdreq;
    wire          [ 11:0]                   axilite_rdaddr;
    wire          [ 31:0]                   axilite_rddata;
    wire                                    axilite_rdack;

    // sys interrupts

    always @(negedge axilite_resetn or posedge axilite_clk) begin
        if (axilite_resetn == 1'b0) begin
            axilite_sys_intr <= 1'd0;
        end else begin
            axilite_sys_intr <= | axilite_sys_intr_active;
        end
    end

    assign axilite_sys_intr_active = axilite_sys_intr_status & axilite_sys_intr_enable;

    generate
    if (INTR_WIDTH >= 32) begin

    assign axilite_sys_intr_status = axilite_sys_in_intr[31:0];

    end else begin

    assign axilite_sys_intr_status[31:INTR_WIDTH] = 'd0;
    assign axilite_sys_intr_status[(INTR_WIDTH-1):0] = axilite_sys_in_intr;

    end
    endgenerate

    // interrupts
 
    always @(negedge axilite_resetn or posedge axilite_clk) begin
        if (axilite_resetn == 1'b0) begin
            axilite_intr <= 1'd0;
        end else begin
            axilite_intr <= | axilite_intr_active;
        end
    end

    assign axilite_intr_active = axilite_intr_status & axilite_intr_enable;
    assign axilite_intr_status = {axilite_status_state, axilite_gpio_in_state};

    // gpio inputs (max width 8)

    generate
    if (GPIO_IN_WIDTH >= 8) begin

    assign axilite_gpio_in_state = axilite_gpio_in[7:0];

    end else begin

    assign axilite_gpio_in_state[7:GPIO_IN_WIDTH] = 'd0;
    assign axilite_gpio_in_state[(GPIO_IN_WIDTH-1):0] = axilite_gpio_in;

    end

    for (genvar n = 0; n < GPIO_IN_WIDTH; n = n + 1) begin: g_gpio_in
    always @(negedge axilite_resetn or posedge axilite_clk) begin
        if (axilite_resetn == 1'b0) begin
            axilite_gpio_in_deb[n] <= 4'd0;
            axilite_gpio_in[n] <= 1'd0;
        end else begin
            axilite_gpio_in_deb[n] <= {axilite_gpio_in_deb[n][2:0], axilite_gpio_in_raw[n]};
            if (axilite_gpio_in_deb[n] == 4'hf) begin
                axilite_gpio_in[n] <= 1'b1;
            end else if (axilite_gpio_in_deb[n] == 4'h0) begin
                axilite_gpio_in[n] <= 1'b0;
            end
        end
    end
    end

    endgenerate

    // gpio outputs (max width 8)

    generate
    if (GPIO_OUT_WIDTH > 8) begin

    assign gpio_out[(GPIO_OUT_WIDTH-1):8] = 'd0;
    assign gpio_out[7:0] = axilite_gpio_out;

    end else begin

    assign gpio_out = axilite_gpio_out[(GPIO_OUT_WIDTH-1):0];

    end
    endgenerate

    // pwm outputs (max width 8)

    generate
    if (PWM_OUT_WIDTH > 8) begin

    assign pwm_out[(PWM_OUT_WIDTH-1):8] = 'd0;
    assign pwm_out[7:0] = axilite_pwm_data;

    end else begin

    assign pwm_out = axilite_pwm_data[(PWM_INT_WIDTH-1):0];

    end
    endgenerate

    generate
    for (genvar n = 0; n < PWM_INT_WIDTH; n = n + 1) begin: g_pwm_out

    assign axilite_pwm_limit[n] = (axilite_pwm_state[n] == 2'd0) ? axilite_pwm_delay_count[n] :
        ((axilite_pwm_state[n] == 2'd1) ? axilite_pwm_high_count[n] : axilite_pwm_low_count[n]);

    always @(negedge axilite_resetn or posedge axilite_clk) begin
        if (axilite_resetn == 1'b0) begin
            axilite_pwm_data[n] <= 1'd0;
            axilite_pwm_state[n] <= 2'd0;
            axilite_pwm_count[n] <= 16'd0;
        end else begin
            if (axilite_pwm_enable[n] == 1'd0) begin
                axilite_pwm_data[n] <= axilite_pwm_out[n];
                axilite_pwm_state[n] <= 2'd0;
                axilite_pwm_count[n] <= 16'd0;
            end else if (axilite_pwm_update == 1'd1) begin
                axilite_pwm_data[n] <= axilite_pwm_state[n][1];
                if (axilite_pwm_count[n] < axilite_pwm_limit[n]) begin
                    axilite_pwm_state[n] <= axilite_pwm_state[n];
                    axilite_pwm_count[n] <= axilite_pwm_count[n] + 1'd1;
                end else begin
                    axilite_pwm_state[n] <= {axilite_pwm_state[n][0], ~axilite_pwm_state[n][0]};
                    axilite_pwm_count[n] <= 16'd0;
                end
            end
        end
    end

    end
    endgenerate

    // pwm outputs timer update

    always @(negedge axilite_resetn or posedge axilite_clk) begin
        if (axilite_resetn == 1'b0) begin
            axilite_pwm_update <= 'h0;
            axilite_pwm_prescale <= 'h0;
        end else begin
            if (axilite_pwm_prescale >= axilite_pwm_prescale_count) begin
                axilite_pwm_update <= 1'd1;
                axilite_pwm_prescale <= 32'd0;
            end else begin
                axilite_pwm_update <= 1'd0;
                axilite_pwm_prescale <= axilite_pwm_prescale + 1'd1;
            end
        end
    end

    // status inputs (max width 8)

    generate
    if (STATUS_WIDTH >= 8) begin

    assign axilite_status_state = axilite_status[7:0];

    end else begin

    assign axilite_status_state[7:STATUS_WIDTH] = 'd0;
    assign axilite_status_state[(STATUS_WIDTH-1):0] = axilite_status;

    end
    endgenerate

    // control outputs (max width 8)

    generate
    if (CONTROL_WIDTH > 8) begin

    assign control[(CONTROL_WIDTH-1):8] = 'd0;
    assign control[7:0] = axilite_control_state;

    end else begin

    assign control = axilite_control_state[(CONTROL_WIDTH-1):0];

    end
    endgenerate

    // instantiations

    cdc #(.DATA_WIDTH(STATUS_WIDTH + GPIO_IN_WIDTH)) i_cdc (
        .src_data                           ({status, gpio_in}),
        .dest_resetn                        (axilite_resetn),
        .dest_clk                           (axilite_clk),
        .dest_data                          ({axilite_status, axilite_gpio_in_raw}));

    axi_sysid_top_regs i_regs (
        .axilite_version                    (VERSION),
        .axilite_instance_id                (INSTANCE_ID),
        .axilite_scratch                    (),
        .axilite_timer                      (),
        .axilite_board_id                   (BOARD_ID),
        .axilite_gpio_in                    (axilite_gpio_in_state),
        .axilite_gpio_out                   (axilite_gpio_out),
        .axilite_status_state               (axilite_status_state),
        .axilite_control_state              (axilite_control_state),
        .axilite_intr_gpio_in_enable        (axilite_intr_enable[7:0]),
        .axilite_intr_status_state_enable   (axilite_intr_enable[15:8]),
        .axilite_sys_version                (SYS_VERSION),
        .axilite_sys_params_0               (SYS_PARAMS[63:32]),
        .axilite_sys_params_1               (SYS_PARAMS[31:0]),
        .axilite_sys_build_0                (SYS_BUILD[63:32]),
        .axilite_sys_build_1                (SYS_BUILD[31:0]),
        .axilite_sys_info_0_0               (SYS_INFO_0[((32*7)+31):(32*7)]),
        .axilite_sys_info_0_1               (SYS_INFO_0[((32*6)+31):(32*6)]),
        .axilite_sys_info_0_2               (SYS_INFO_0[((32*5)+31):(32*5)]),
        .axilite_sys_info_0_3               (SYS_INFO_0[((32*4)+31):(32*4)]),
        .axilite_sys_info_0_4               (SYS_INFO_0[((32*3)+31):(32*3)]),
        .axilite_sys_info_0_5               (SYS_INFO_0[((32*2)+31):(32*2)]),
        .axilite_sys_info_0_6               (SYS_INFO_0[((32*1)+31):(32*1)]),
        .axilite_sys_info_0_7               (SYS_INFO_0[((32*0)+31):(32*0)]),
        .axilite_sys_info_1_0               (SYS_INFO_1[((32*7)+31):(32*7)]),
        .axilite_sys_info_1_1               (SYS_INFO_1[((32*6)+31):(32*6)]),
        .axilite_sys_info_1_2               (SYS_INFO_1[((32*5)+31):(32*5)]),
        .axilite_sys_info_1_3               (SYS_INFO_1[((32*4)+31):(32*4)]),
        .axilite_sys_info_1_4               (SYS_INFO_1[((32*3)+31):(32*3)]),
        .axilite_sys_info_1_5               (SYS_INFO_1[((32*2)+31):(32*2)]),
        .axilite_sys_info_1_6               (SYS_INFO_1[((32*1)+31):(32*1)]),
        .axilite_sys_info_1_7               (SYS_INFO_1[((32*0)+31):(32*0)]),
        .axilite_pwm_prescale_count         (axilite_pwm_prescale_count),
        .axilite_pwm_enable                 (axilite_pwm_enable),
        .axilite_pwm_out                    (axilite_pwm_out),
        .axilite_pwm_delay_count_0          (axilite_pwm_delay_count[0]),
        .axilite_pwm_high_count_0           (axilite_pwm_high_count[0]),
        .axilite_pwm_low_count_0            (axilite_pwm_low_count[0]),
        .axilite_pwm_delay_count_1          (axilite_pwm_delay_count[1]),
        .axilite_pwm_high_count_1           (axilite_pwm_high_count[1]),
        .axilite_pwm_low_count_1            (axilite_pwm_low_count[1]),
        .axilite_pwm_delay_count_2          (axilite_pwm_delay_count[2]),
        .axilite_pwm_high_count_2           (axilite_pwm_high_count[2]),
        .axilite_pwm_low_count_2            (axilite_pwm_low_count[2]),
        .axilite_pwm_delay_count_3          (axilite_pwm_delay_count[3]),
        .axilite_pwm_high_count_3           (axilite_pwm_high_count[3]),
        .axilite_pwm_low_count_3            (axilite_pwm_low_count[3]),
        .axilite_pwm_delay_count_4          (axilite_pwm_delay_count[4]),
        .axilite_pwm_high_count_4           (axilite_pwm_high_count[4]),
        .axilite_pwm_low_count_4            (axilite_pwm_low_count[4]),
        .axilite_pwm_delay_count_5          (axilite_pwm_delay_count[5]),
        .axilite_pwm_high_count_5           (axilite_pwm_high_count[5]),
        .axilite_pwm_low_count_5            (axilite_pwm_low_count[5]),
        .axilite_pwm_delay_count_6          (axilite_pwm_delay_count[6]),
        .axilite_pwm_high_count_6           (axilite_pwm_high_count[6]),
        .axilite_pwm_low_count_6            (axilite_pwm_low_count[6]),
        .axilite_pwm_delay_count_7          (axilite_pwm_delay_count[7]),
        .axilite_pwm_high_count_7           (axilite_pwm_high_count[7]),
        .axilite_pwm_low_count_7            (axilite_pwm_low_count[7]),
        .axilite_sys_intr_status            (axilite_sys_intr),
        .axilite_sys_intr_0_status          (axilite_sys_intr_status),
        .axilite_sys_intr_0_enable          (axilite_sys_intr_enable),
        .axilite_mem_id                     (MEMORY_ID),
        .axilite_mem_pg                     (pg),
        .axilite_mem_pg_sel                 (pg_sel),
        .axilite_clk                        (axilite_clk),
        .axilite_resetn                     (axilite_resetn),
        .axilite_wrreq                      (axilite_wrreq),
        .axilite_wraddr                     (axilite_wraddr),
        .axilite_wrdata                     (axilite_wrdata),
        .axilite_wrack                      (axilite_wrack),
        .axilite_rdreq                      (axilite_rdreq),
        .axilite_rdaddr                     (axilite_rdaddr),
        .axilite_rddata                     (axilite_rddata),
        .axilite_rdack                      (axilite_rdack));

    axilite_slave_if #(
        .AXI_ADDRESS_WIDTH                  (12))
    i_axilite_if (
        .axilite_clk                        (axilite_clk),
        .axilite_resetn                     (axilite_resetn),
        .axilite_awvalid                    (axilite_awvalid),
        .axilite_awaddr                     (axilite_awaddr),
        .axilite_awready                    (axilite_awready),
        .axilite_wvalid                     (axilite_wvalid),
        .axilite_wdata                      (axilite_wdata),
        .axilite_wready                     (axilite_wready),
        .axilite_bvalid                     (axilite_bvalid),
        .axilite_bresp                      (axilite_bresp),
        .axilite_bready                     (axilite_bready),
        .axilite_arvalid                    (axilite_arvalid),
        .axilite_araddr                     (axilite_araddr),
        .axilite_arready                    (axilite_arready),
        .axilite_rvalid                     (axilite_rvalid),
        .axilite_rresp                      (axilite_rresp),
        .axilite_rdata                      (axilite_rdata),
        .axilite_rready                     (axilite_rready),
        .axilite_wrreq                      (axilite_wrreq),
        .axilite_wraddr                     (axilite_wraddr),
        .axilite_wrdata                     (axilite_wrdata),
        .axilite_wrack                      (axilite_wrack),
        .axilite_rdreq                      (axilite_rdreq),
        .axilite_rdaddr                     (axilite_rdaddr),
        .axilite_rddata                     (axilite_rddata),
        .axilite_rdack                      (axilite_rdack));

endmodule

// **********************************************************************************
// **********************************************************************************
