// **********************************************************************************
// **********************************************************************************
// ----------------------------------------------------------------------------------
// ################
// ##   ###########   Analog Devices Inc.
// ##      ########
// ##         #####   Copyright (c) 2019 Analog Devices Inc. All rights reserved.
// ##            ##   This file is the confidential and proprietary property of ADI.
// ##         #####   Possession or use of this file requires a written license.
// ##      ########   The licensing information may be found at: www.analog.com
// ##   ###########
// ################
// ----------------------------------------------------------------------------------
// Description:       Receive DMA Core
// ----------------------------------------------------------------------------------
// **********************************************************************************
// **********************************************************************************

`timescale 1ps/1ps

module axi_rxdma_core #(

    parameter [3:0] MODULE_ID = 0,
    parameter integer INSTANCE_ID = 0,
    parameter integer RSD_DW = 64,
    parameter integer AXIS_DW = 64,
    parameter integer AXIMM_SF = 1,
    parameter integer AXIMM_DW = 512,
    parameter integer AXIMM_SW = 64) (

    // ssi interface

    input   wire                            rx_clk,
    input   wire                            rx_resetn,
    input   wire                            rx_rsenb,
    input   wire                            rx_rstrig,
    input   wire                            rx_rsdsync,
    input   wire                            rx_rsvld,
    input   wire  [(RSD_DW-1):0]            rx_rsd,

    // asynchronous events

    input   wire  [  3:0]                   trig,
    output  reg                             rx_mtrig_out = 'd0,
    input   wire                            rx_mtrig_in,

    // axi master interface

    input   wire                            aximm_clk,
    input   wire                            aximm_resetn,
    output  wire                            aximm_intc_cntrl,
    output  wire  [ 39:0]                   aximm_awaddr,
    output  wire  [  7:0]                   aximm_awlen,
    output  wire  [  2:0]                   aximm_awsize,
    output  wire  [  1:0]                   aximm_awburst,
    output  wire  [  2:0]                   aximm_awprot,
    output  wire  [  3:0]                   aximm_awcache,
    output  wire                            aximm_awvalid,
    input   wire                            aximm_awready,
    output  wire  [(AXIMM_DW-1):0]          aximm_wdata,
    output  wire  [(AXIMM_SW-1):0]          aximm_wstrb,
    output  wire                            aximm_wlast,
    output  wire                            aximm_wvalid,
    input   wire                            aximm_wready,
    input   wire  [  1:0]                   aximm_bresp,
    input   wire                            aximm_bvalid,
    output  wire                            aximm_bready,

    // axilite interface

    input   wire                            axilite_clk,
    input   wire                            axilite_resetn,
    input   wire                            axilite_wrreq,
    input   wire  [ 15:0]                   axilite_wraddr,
    input   wire  [ 31:0]                   axilite_wrdata,
    output  wire                            axilite_wrack,
    input   wire                            axilite_rdreq,
    input   wire  [ 15:0]                   axilite_rdaddr,
    output  wire  [ 31:0]                   axilite_rddata,
    output  wire                            axilite_rdack,
    output  reg                             axilite_intr = 'd0);

    // constants

    localparam  VERSION = {8'd0, 8'd6, 8'd4, 8'd0};
    localparam  AXIS_SIZE = AXIS_DW/8;

    // internal registers

    reg           [  4:0]                   rx_trig_d = 'd0;
    reg           [  5:0]                   rx_trig_enable = 'd0;
    reg                                     rx_senb_d = 'd0;
    reg                                     rx_mtrig_fall = 'd0;
    reg                                     rx_mtrig = 'd0;
    reg                                     aximm_resetn_int = 'd0;
    reg                                     aximm_swresetn = 'd0;
    reg                                     aximm_swreset_d = 'd0;
    reg           [  5:0]                   aximm_swreset_count = 'd0;
    reg                                     aximm_mmq_event = 'd0;
    reg                                     aximm_start_req_d = 'd0;
    reg                                     aximm_stop_req_d = 'd0;
    reg                                     aximm_start = 'd0;
    reg                                     aximm_stop = 'd0;
    reg                                     aximm_active = 'd0;
    reg                                     aximm_enable = 'd0;
    reg           [ 11:0]                   aximm_xfer_tag = 'd0;
    reg                                     aximm_status = 'd0;
    reg                                     aximm_slave_err = 'd0;
    reg                                     aximm_addrdec_err = 'd0;
    reg                                     aximm_tlast_err = 'd0;
    reg                                     aximm_length_err = 'd0;
    reg           [ 11:0]                   aximm_ack_buf_cnt = 'd0;
    reg           [ 11:0]                   aximm_req_tag = 'd0;
    reg           [ 39:0]                   aximm_req_start_addr = 40'd0;
    reg           [ 11:0]                   aximm_req_buf_cnt = 4'd0;
    reg                                     aximm_req_valid = 'd0;
    reg                                     aximm_req_active = 'd0;
    reg           [  3:0]                   aximm_qcnt = 'd0;
    reg           [ 22:0]                   aximm_rdcnt = 'd0;
    reg           [ 31:0]                   axilite_mmq_status = 'd0;
    reg                                     axilite_busy_d = 'd0;
    reg           [ 32:0]                   axilite_data_xfer_count = 'd0;
    reg           [ 39:0]                   axilite_start_addr = 'd0;
    reg                                     axilite_cdc_state_d = 'd0;
    reg                                     axilite_cdc_state_clr_d = 'd0;
    reg                                     axilite_cdc_state_int = 'd0;

    // internal signals

    wire                                    rx_trig_valid;
    wire                                    rx_itrig;
    wire          [  4:0]                   rx_trig;
    wire                                    rx_enable;
    wire          [ 11:0]                   rx_trig_mode;
    wire                                    rx_master_trig;
    wire                                    rx_start_sync;
    wire                                    rx_cdc_state;
    wire                                    rx_overflow;
    wire                                    rx_valid;
    wire          [(AXIS_DW-1):0]           rx_data;
    wire                                    rx_full;
    wire                                    rx_int_resetn;
    wire                                    aximm_int_valid;
    wire                                    aximm_int_ready;
    wire                                    aximm_ack_ready;
    wire          [  3:0]                   aximm_ack_tag_p1;
    wire                                    aximm_done;
    wire                                    aximm_last;
    wire                                    aximm_last_valid;
    wire          [ 79:0]                   aximm_req_data;
    wire                                    aximm_qreq;
    wire                                    aximm_qack;
    wire                                    aximm_swreset;
    wire          [ 11:0]                   aximm_trig_mode;
    wire                                    aximm_master_trig;
    wire                                    aximm_start_sync;
    wire                                    aximm_cdc_state;
    wire                                    aximm_start_req;
    wire                                    aximm_stop_req;
    wire                                    aximm_cyclic;
    wire          [  3:0]                   aximm_qthreshold;
    wire          [ 39:0]                   aximm_start_addr;
    wire          [ 22:0]                   aximm_length;
    wire          [ 11:0]                   aximm_num_of_buffers;
    wire                                    aximm_valid;
    wire          [(AXIS_DW-1):0]           aximm_data;
    wire                                    aximm_err;
    wire                                    aximm_req_ready;
    wire                                    aximm_ack_valid;
    wire          [  7:0]                   aximm_ack_data;
    wire                                    aximm_ready;
    wire                                    axilite_mmq_event;
    wire                                    axilite_mmq_tlimit_wr;
    wire          [ 31:0]                   axilite_mmq_tlimit;
    wire          [  5:0]                   axilite_intr_status;
    wire          [  5:0]                   axilite_intr_enable;
    wire          [  5:0]                   axilite_intr_active;
    wire                                    axilite_overflow;
    wire                                    axilite_swreset_clr;
    wire                                    axilite_busy;
    wire                                    axilite_status;
    wire          [  3:0]                   axilite_qcnt;
    wire                                    axilite_length_err;
    wire                                    axilite_tlast_err;
    wire                                    axilite_addrdec_err;
    wire                                    axilite_slave_err;
    wire          [ 11:0]                   axilite_req_buf_cnt;
    wire          [ 11:0]                   axilite_ack_buf_cnt;
    wire                                    axilite_swreset;
    wire                                    axilite_start;
    wire                                    axilite_stop;
    wire                                    axilite_cyclic;
    wire          [  3:0]                   axilite_qthreshold;
    wire          [ 39:0]                   axilite_offset_addr;
    wire          [ 39:0]                   axilite_base_addr;
    wire          [ 22:0]                   axilite_length;
    wire          [ 11:0]                   axilite_num_of_buffers;
    wire          [ 11:0]                   axilite_trig_mode;
    wire                                    axilite_master_trig;
    wire                                    axilite_start_sync;
    wire                                    axilite_cdc_state;
    wire                                    axilite_cdc_state_clr;
    wire                                    axilite_cdc_state_clr_inv;
    wire          [ 31:0]                   axilite_clk_mon_count;
    wire                                    axilite_intc_cntrl;

    // interrupts

    assign axilite_intr_active = axilite_intr_status & axilite_intr_enable;
    assign axilite_intr_status[0] = ~axilite_busy;

    always @(negedge axilite_resetn or posedge axilite_clk) begin
        if (axilite_resetn == 1'b0) begin
            axilite_intr <= 1'd0;
        end else begin
            axilite_intr <= | axilite_intr_active;
        end
    end

    // device interface

    assign rx_overflow = rx_enable & rx_full;

    generate
    for (genvar n = 0; n < 5; n = n + 1) begin: g_rx_trig

    always @(negedge rx_int_resetn or posedge rx_clk) begin
        if (rx_int_resetn == 1'b0) begin
            rx_trig_d[n] <= 1'd0;
            rx_trig_enable[n] <= 1'd0;
        end else begin
            rx_trig_d[n] <= rx_trig[n];
            if (rx_enable == 1'd0) begin
                rx_trig_enable[n] <= 1'd0;
            end else if (rx_trig_enable[n] == 1'd0) begin
                case (rx_trig_mode[((n*2)+1):(n*2)])
                    2'd3: rx_trig_enable[n] <= rx_trig[n] ^ rx_trig_d[n];
                    2'd2: rx_trig_enable[n] <= ~rx_trig[n] & rx_trig_d[n];
                    2'd1: rx_trig_enable[n] <= rx_trig[n] & ~rx_trig_d[n];
                    default: rx_trig_enable[n] <= 1'd1;
                endcase
            end
        end
    end

    end
    endgenerate

    always @(negedge rx_int_resetn or posedge rx_clk) begin
        if (rx_int_resetn == 1'b0) begin
            rx_senb_d <= 1'd0;
            rx_trig_enable[5] <= 1'd0;
        end else begin
            rx_senb_d <= rx_rsenb;
            if ((rx_start_sync == 1'd0) || ((rx_enable == 1'd1) &&
                (rx_rsenb == 1'd1) && (rx_senb_d == 1'd0))) begin
                rx_trig_enable[5] <= 1'd1;
            end else if (rx_enable == 1'd0) begin
                rx_trig_enable[5] <= 1'd0;
            end
        end
    end

    assign rx_trig_valid = & rx_trig_enable;
    assign rx_itrig = (rx_master_trig == 1'd1) ? rx_mtrig : rx_trig_valid;

    always @(negedge rx_int_resetn or negedge rx_clk) begin
        if (rx_int_resetn == 1'b0) begin
            rx_mtrig_fall <= 1'd0;
        end else begin
            rx_mtrig_fall <= rx_mtrig_in;
        end
    end

    always @(negedge rx_int_resetn or posedge rx_clk) begin
        if (rx_int_resetn == 1'b0) begin
            rx_mtrig_out <= 1'd0;
            rx_mtrig <= 1'd0;
        end else begin
            if (rx_rsdsync == 1'd1) begin
                rx_mtrig_out <= rx_trig_valid;
            end
            rx_mtrig <= rx_mtrig_fall;
        end
    end

    // aximm interface

    always @(posedge aximm_clk) begin
        aximm_resetn_int <= aximm_resetn;
    end

    assign aximm_valid = aximm_int_valid & aximm_enable;
    assign aximm_int_ready = aximm_ready | ~aximm_active;
    assign aximm_ack_ready = 1'b1;
    assign aximm_ack_tag_p1 = aximm_ack_data[3:0] + 1'd1;
    assign aximm_done = (aximm_ack_tag_p1 == aximm_req_tag[3:0]) ?
        (~aximm_req_active & aximm_ack_valid &
        (~aximm_cyclic | aximm_stop)) : 1'd0;
    assign aximm_last = (aximm_rdcnt >= aximm_length) ? aximm_active : 1'b0;
    assign aximm_last_valid = aximm_valid & aximm_ready & aximm_last;
    assign aximm_req_data[79:76] = 4'd0;
    assign aximm_req_data[75:72] = aximm_req_tag[3:0];
    assign aximm_req_data[71:32] = aximm_req_start_addr;
    assign aximm_req_data[31] = 1'b0;
    assign aximm_req_data[30] = 1'b1;
    assign aximm_req_data[29:24] = 6'd0;
    assign aximm_req_data[23] = 1'b1;
    assign aximm_req_data[22:0] = aximm_length;
    assign aximm_qreq = aximm_req_valid & aximm_req_ready;
    assign aximm_qack = aximm_ack_valid & aximm_ack_ready;

    always @(negedge aximm_resetn_int or posedge aximm_clk) begin
        if (aximm_resetn_int == 1'b0) begin
            aximm_swresetn <= 1'd0;
            aximm_swreset_d <= 1'd0;
            aximm_swreset_count <= 6'd0;
        end else begin
            aximm_swresetn <= ~aximm_swreset_count[5];
            aximm_swreset_d <= aximm_swreset;
            if (aximm_swreset_count[5] == 1'd1) begin
                aximm_swreset_count <= aximm_swreset_count + 1'd1;
            end else if ((aximm_swreset == 1'd1) && (aximm_swreset_d == 1'd0)) begin
                aximm_swreset_count <= 6'h20;
            end
        end
    end

    always @(negedge aximm_swresetn or posedge aximm_clk) begin
        if (aximm_swresetn == 1'b0) begin
            aximm_mmq_event <= 1'b0;
        end else begin
            if ((aximm_valid == 1'b1) && (aximm_ready == 1'b1)) begin
                aximm_mmq_event <= 1'b1;
            end else begin
                aximm_mmq_event <= 1'b0;
            end
        end
    end

    always @(negedge aximm_swresetn or posedge aximm_clk) begin
        if (aximm_swresetn == 1'b0) begin
            aximm_start_req_d <= 1'd0;
            aximm_stop_req_d <= 1'd0;
            aximm_start <= 1'd0;
            aximm_stop <= 1'd0;
            aximm_active <= 1'd0;
            aximm_enable <= 1'd0;
            aximm_xfer_tag <= 12'd0;
            aximm_status <= 1'b1;
            aximm_slave_err <= 1'b0;
            aximm_addrdec_err <= 1'b0;
            aximm_tlast_err <= 1'b0;
            aximm_length_err <= 1'b0;
            aximm_ack_buf_cnt <= 12'b0;
            aximm_req_tag <= 12'd0;
            aximm_req_start_addr <= 40'd0;
            aximm_req_buf_cnt <= 12'd0;
            aximm_req_valid <= 1'b0;
            aximm_req_active <= 1'b0;
            aximm_qcnt <= 4'd0;
            aximm_rdcnt <= 'd0;
        end else begin
            aximm_start_req_d <= aximm_start_req;
            aximm_stop_req_d <= aximm_stop_req;
            aximm_start <= aximm_start_req & ~aximm_start_req_d;
            if (aximm_stop == 1'b1) begin
                aximm_stop <= ~aximm_done;
            end else begin
                aximm_stop <= aximm_stop_req & ~aximm_stop_req_d;
            end
            if (aximm_active == 1'b1) begin
                aximm_active <= ~aximm_done;
                if ((aximm_xfer_tag >= aximm_req_tag) && (aximm_last_valid == 1'b1)) begin
                    aximm_enable <= aximm_cyclic & ~aximm_stop;
                end
            end else begin
                aximm_active <= aximm_start;
                aximm_enable <= aximm_start;
            end
            if (aximm_active == 1'b1) begin
                if (aximm_last_valid == 1'b1) begin
                    aximm_xfer_tag <= aximm_xfer_tag + 1'd1;
                end
            end else if (aximm_start == 1'b1) begin
                aximm_xfer_tag <= 12'd1;
            end
            if (aximm_ack_valid == 1'b1) begin
                aximm_status <= aximm_status & aximm_ack_data[7];
                aximm_slave_err <= aximm_slave_err | aximm_ack_data[6];
                aximm_addrdec_err <= aximm_addrdec_err | aximm_ack_data[5];
                aximm_tlast_err <= aximm_tlast_err | aximm_ack_data[4];
                aximm_length_err <= aximm_length_err | aximm_err;
                if (aximm_ack_buf_cnt >= aximm_num_of_buffers) begin
                    aximm_ack_buf_cnt <= 12'd0;
                end else begin
                    aximm_ack_buf_cnt <= aximm_ack_buf_cnt + 1'd1;
                end
            end else if (aximm_start == 1'b1) begin
                aximm_status <= 1'b1;
                aximm_slave_err <= 1'b0;
                aximm_addrdec_err <= 1'b0;
                aximm_tlast_err <= 1'b0;
                aximm_length_err <= 1'b0;
                aximm_ack_buf_cnt <= 12'd0;
            end
            if (aximm_req_active == 1'b1) begin
                if ((aximm_req_valid == 1'b1) && (aximm_req_ready == 1'b1)) begin
                    aximm_req_tag <= aximm_req_tag + 1'b1;
                    if (aximm_req_buf_cnt >= aximm_num_of_buffers) begin
                        aximm_req_start_addr <= aximm_start_addr;
                        aximm_req_buf_cnt <= 12'd0;
                    end else begin
                        aximm_req_start_addr <= aximm_req_start_addr + aximm_length;
                        aximm_req_buf_cnt <= aximm_req_buf_cnt + 1'd1;
                    end
                end
                if ((aximm_req_valid == 1'b0) || (aximm_req_ready == 1'b1)) begin
                    aximm_req_valid <= (aximm_qcnt > aximm_qthreshold) ? 1'b0 :
                        ((aximm_req_buf_cnt >= aximm_num_of_buffers) ?
                        (~aximm_req_valid | (aximm_cyclic & ~aximm_stop)) : 1'b1);
                end
                if ((aximm_req_valid == 1'b1) && (aximm_req_ready == 1'b1) &&
                    (aximm_req_buf_cnt >= aximm_num_of_buffers)) begin
                    aximm_req_active <= aximm_cyclic & ~aximm_stop;
                end
            end else if (aximm_start == 1'd1) begin
                aximm_req_tag <= 12'd0;
                aximm_req_start_addr <= aximm_start_addr;
                aximm_req_buf_cnt <= 12'd0;
                aximm_req_valid <= 1'b1;
                aximm_req_active <= 1'b1;
            end
            if ((aximm_qreq == 1'b1) && (aximm_qack == 1'b0)) begin
                aximm_qcnt <= aximm_qcnt + 1'b1;
            end else if ((aximm_qreq == 1'b0) && (aximm_qack == 1'b1)) begin
                aximm_qcnt <= aximm_qcnt - 1'b1;
            end
            if (aximm_active == 1'b0) begin
                aximm_rdcnt <= AXIS_SIZE;
            end else if ((aximm_valid == 1'd1) && (aximm_ready == 1'd1)) begin
                if (aximm_rdcnt >= aximm_length) begin
                    aximm_rdcnt <= AXIS_SIZE;
                end else begin
                    aximm_rdcnt <= aximm_rdcnt + AXIS_SIZE;
                end
            end
        end
    end

    assign rx_trig[4] = rx_rstrig;

    // start address & cdc state

    assign axilite_cdc_state_clr_inv = ~axilite_cdc_state_clr;

    always @(negedge axilite_resetn or posedge axilite_clk) begin
        if (axilite_resetn == 1'b0) begin
            axilite_start_addr <= 'd0;
            axilite_cdc_state_d <= 'd0;
            axilite_cdc_state_clr_d <= 'd0;
            axilite_cdc_state_int <= 'd0;
        end else begin
            axilite_start_addr <= axilite_base_addr + axilite_offset_addr;
            axilite_cdc_state_d <= axilite_cdc_state;
            axilite_cdc_state_clr_d <= axilite_cdc_state_clr;
            if ((axilite_cdc_state_clr_d == 1'd0) &&
                (axilite_cdc_state_clr == 1'd1)) begin
                axilite_cdc_state_int <= 1'd0;
            end else if ((axilite_cdc_state_d == 1'd0) &&
                (axilite_cdc_state == 1'd1)) begin
                axilite_cdc_state_int <= 1'd1;
            end
        end
    end

    always @(negedge axilite_resetn or posedge axilite_clk) begin
        if (axilite_resetn == 1'b0) begin
            axilite_mmq_status <= 32'd0;
        end else begin
            if ((axilite_mmq_event == 1'd1) || (axilite_mmq_tlimit_wr == 1'd1)) begin
                axilite_mmq_status <= axilite_mmq_tlimit;
            end else if (axilite_mmq_status > 0) begin
                axilite_mmq_status <= axilite_mmq_status - 1'd1;
            end
        end
    end

    always @(negedge axilite_resetn or posedge axilite_clk) begin
        if (axilite_resetn == 1'b0) begin
            axilite_busy_d <= 1'd0;
            axilite_data_xfer_count <= 33'd0;
        end else begin
            axilite_busy_d <= axilite_busy;
            if ((axilite_busy_d == 1'd0) && (axilite_busy == 1'd1)) begin
                axilite_data_xfer_count <= 33'd0;
            end else if (axilite_data_xfer_count[32] == 1'd1) begin
                axilite_data_xfer_count <= {33{1'd1}};
            end else if (axilite_busy == 1'd1) begin
                axilite_data_xfer_count <= axilite_data_xfer_count + 1'd1;
            end
        end
    end

    // instantiations

    clk_mon i_clk_mon (
        .clk                                (aximm_clk),
        .axilite_resetn                     (axilite_resetn),
        .axilite_clk                        (axilite_clk),
        .axilite_clk_mon_count              (axilite_clk_mon_count));

    cdc #(.DATA_WIDTH(4)) i_cdc_rx (
        .src_data                           (trig),
        .dest_resetn                        (rx_int_resetn),
        .dest_clk                           (rx_clk),
        .dest_data                          (rx_trig[3:0]));

    cdc_cntrl #(.DATA_WIDTH(16)) i_cdc_cntrl_rx (
        .src_resetn                         (aximm_resetn_int),
        .src_clk                            (aximm_clk),
        .src_data                           ({aximm_enable,
                                                aximm_trig_mode,
                                                aximm_master_trig,
                                                aximm_start_sync,
                                                aximm_cdc_state}),
        .dest_resetn                        (rx_int_resetn),
        .dest_clk                           (rx_clk),
        .dest_data                          ({rx_enable,
                                                rx_trig_mode,
                                                rx_master_trig,
                                                rx_start_sync,
                                                rx_cdc_state}));

    cdc_cntrl #(.DATA_WIDTH(99)) i_cdc_cntrl_aximm (
        .src_resetn                         (axilite_resetn),
        .src_clk                            (axilite_clk),
        .src_data                           ({axilite_swreset,
                                                axilite_intc_cntrl,
                                                axilite_trig_mode,
                                                axilite_master_trig,
                                                axilite_start_sync,
                                                axilite_cdc_state_int,
                                                axilite_start,
                                                axilite_stop,
                                                axilite_cyclic,
                                                axilite_qthreshold,
                                                axilite_start_addr,
                                                axilite_length,
                                                axilite_num_of_buffers}),
        .dest_resetn                        (aximm_resetn_int),
        .dest_clk                           (aximm_clk),
        .dest_data                          ({aximm_swreset,
                                                aximm_intc_cntrl,
                                                aximm_trig_mode,
                                                aximm_master_trig,
                                                aximm_start_sync,
                                                aximm_cdc_state,
                                                aximm_start_req,
                                                aximm_stop_req,
                                                aximm_cyclic,
                                                aximm_qthreshold,
                                                aximm_start_addr,
                                                aximm_length,
                                                aximm_num_of_buffers}));

    cdc_event #(.DATA_WIDTH(1)) i_cdc_event_rx (
        .src_resetn                         (rx_int_resetn),
        .src_clk                            (rx_clk),
        .src_data                           (rx_overflow),
        .dest_resetn                        (axilite_resetn),
        .dest_clk                           (axilite_clk),
        .dest_data                          (axilite_overflow));

    cdc_event #(.DATA_WIDTH(1)) i_cdc_event_aximm (
        .src_resetn                         (aximm_resetn_int),
        .src_clk                            (aximm_clk),
        .src_data                           (aximm_mmq_event),
        .dest_resetn                        (axilite_resetn),
        .dest_clk                           (axilite_clk),
        .dest_data                          (axilite_mmq_event));

    cdc #(.DATA_WIDTH(1)) i_cdc_axilite (
        .src_data                           (rx_cdc_state),
        .dest_resetn                        (axilite_resetn),
        .dest_clk                           (axilite_clk),
        .dest_data                          (axilite_cdc_state_clr));

    cdc_cntrl #(.DATA_WIDTH(35)) i_cdc_cntrl_axilite (
        .src_resetn                         (aximm_resetn_int),
        .src_clk                            (aximm_clk),
        .src_data                           ({aximm_swresetn,
                                                aximm_active,
                                                aximm_status,
                                                aximm_qcnt,
                                                aximm_length_err,
                                                aximm_tlast_err,
                                                aximm_addrdec_err,
                                                aximm_slave_err,
                                                aximm_req_buf_cnt,
                                                aximm_ack_buf_cnt}),
        .dest_resetn                        (axilite_resetn),
        .dest_clk                           (axilite_clk),
        .dest_data                          ({axilite_swreset_clr,
                                                axilite_busy,
                                                axilite_status,
                                                axilite_qcnt,
                                                axilite_length_err,
                                                axilite_tlast_err,
                                                axilite_addrdec_err,
                                                axilite_slave_err,
                                                axilite_req_buf_cnt,
                                                axilite_ack_buf_cnt}));

    cdc_resetp i_cdc_resetp (
        .src_clk                            (rx_clk),
        .src_resetn                         (rx_resetn),
        .dest_clk                           (rx_clk),
        .dest_resetp                        (rx_int_resetn));

    upsize #(
        .IW                                 (RSD_DW),
        .OW                                 (AXIS_DW))
    i_upsize (
        .resetn                             (rx_int_resetn),
        .clk                                (rx_clk),
        .start                              (rx_itrig),
        .valid_in                           (rx_rsvld),
        .data_in                            (rx_rsd),
        .valid_out                          (rx_valid),
        .data_out                           (rx_data));

    mem_zdfifo #(
        .WRITE_LIMIT                        (252),
        .ADDRESS_WIDTH                      (8),
        .DATA_WIDTH                         (AXIS_DW))
    i_es (
        .wr_clk                             (rx_clk),
        .wr_resetn                          (rx_int_resetn),
        .wr_valid                           (rx_valid),
        .wr_data                            (rx_data),
        .wr_limit                           (rx_full),
        .rd_clk                             (aximm_clk),
        .rd_resetn                          (aximm_swresetn),
        .rd_read                            (aximm_int_ready),
        .rd_valid                           (aximm_int_valid),
        .rd_data                            (aximm_data),
        .rd_empty                           ());

    axi_datamover #(
        .C_INCLUDE_MM2S                     (0),
        .C_M_AXI_MM2S_ARID                  (0),
        .C_M_AXI_MM2S_ID_WIDTH              (1),
        .C_M_AXI_MM2S_ADDR_WIDTH            (40),
        .C_M_AXI_MM2S_DATA_WIDTH            (AXIMM_DW),
        .C_M_AXIS_MM2S_TDATA_WIDTH          (AXIS_DW),
        .C_INCLUDE_MM2S_STSFIFO             (1),
        .C_MM2S_STSCMD_FIFO_DEPTH           (4),
        .C_MM2S_STSCMD_IS_ASYNC             (0),
        .C_INCLUDE_MM2S_DRE                 (0),
        .C_MM2S_BURST_SIZE                  (16),
        .C_MM2S_BTT_USED                    (23),
        .C_MM2S_ADDR_PIPE_DEPTH             (4),
        .C_MM2S_INCLUDE_SF                  (AXIMM_SF),
        .C_INCLUDE_S2MM                     (1),
        .C_M_AXI_S2MM_AWID                  (0),
        .C_M_AXI_S2MM_ID_WIDTH              (1),
        .C_M_AXI_S2MM_ADDR_WIDTH            (40),
        .C_M_AXI_S2MM_DATA_WIDTH            (AXIMM_DW),
        .C_S_AXIS_S2MM_TDATA_WIDTH          (AXIS_DW),
        .C_INCLUDE_S2MM_STSFIFO             (1),
        .C_S2MM_STSCMD_FIFO_DEPTH           (4),
        .C_S2MM_STSCMD_IS_ASYNC             (0),
        .C_INCLUDE_S2MM_DRE                 (0),
        .C_S2MM_BURST_SIZE                  (16),
        .C_S2MM_BTT_USED                    (23),
        .C_S2MM_SUPPORT_INDET_BTT           (0),
        .C_S2MM_ADDR_PIPE_DEPTH             (4),
        .C_S2MM_INCLUDE_SF                  (AXIMM_SF),
        .C_ENABLE_CACHE_USER                (0),
        .C_ENABLE_SKID_BUF                  ("11111"),
        .C_ENABLE_MM2S_TKEEP                (0),
        .C_ENABLE_S2MM_TKEEP                (0),
        .C_ENABLE_S2MM_ADV_SIG              (0),
        .C_ENABLE_MM2S_ADV_SIG              (0),
        .C_MICRO_DMA                        (0),
        .C_CMD_WIDTH                        (80),
        .C_FAMILY                           ("zynq"))
    i_mm (
        .m_axi_mm2s_aclk                    (1'd0),
        .m_axi_mm2s_aresetn                 (1'd1),
        .mm2s_halt                          (1'd0),
        .mm2s_halt_cmplt                    (),
        .mm2s_err                           (),
        .m_axis_mm2s_cmdsts_aclk            (1'd0),
        .m_axis_mm2s_cmdsts_aresetn         (1'd1),
        .s_axis_mm2s_cmd_tvalid             (1'd0),
        .s_axis_mm2s_cmd_tready             (),
        .s_axis_mm2s_cmd_tdata              (80'd0),
        .m_axis_mm2s_sts_tvalid             (),
        .m_axis_mm2s_sts_tready             (1'd0),
        .m_axis_mm2s_sts_tdata              (),
        .m_axis_mm2s_sts_tkeep              (),
        .m_axis_mm2s_sts_tlast              (),
        .mm2s_allow_addr_req                (1'd1),
        .mm2s_addr_req_posted               (),
        .mm2s_rd_xfer_cmplt                 (),
        .m_axi_mm2s_arid                    (),
        .m_axi_mm2s_araddr                  (),
        .m_axi_mm2s_arlen                   (),
        .m_axi_mm2s_arsize                  (),
        .m_axi_mm2s_arburst                 (),
        .m_axi_mm2s_arprot                  (),
        .m_axi_mm2s_arcache                 (),
        .m_axi_mm2s_aruser                  (),
        .m_axi_mm2s_arvalid                 (),
        .m_axi_mm2s_arready                 ('d0),
        .m_axi_mm2s_rdata                   ('d0),
        .m_axi_mm2s_rresp                   ('d0),
        .m_axi_mm2s_rlast                   ('d0),
        .m_axi_mm2s_rvalid                  ('d0),
        .m_axi_mm2s_rready                  (),
        .m_axis_mm2s_tdata                  (),
        .m_axis_mm2s_tkeep                  (),
        .m_axis_mm2s_tlast                  (),
        .m_axis_mm2s_tvalid                 (),
        .m_axis_mm2s_tready                 ('d0),
        .mm2s_dbg_sel                       ('d0),
        .mm2s_dbg_data                      (),
        .m_axi_s2mm_aclk                    (aximm_clk),
        .m_axi_s2mm_aresetn                 (aximm_swresetn),
        .s2mm_halt                          ('d0),
        .s2mm_halt_cmplt                    (),
        .s2mm_err                           (aximm_err),
        .m_axis_s2mm_cmdsts_awclk           (aximm_clk),
        .m_axis_s2mm_cmdsts_aresetn         (aximm_swresetn),
        .s_axis_s2mm_cmd_tvalid             (aximm_req_valid),
        .s_axis_s2mm_cmd_tready             (aximm_req_ready),
        .s_axis_s2mm_cmd_tdata              (aximm_req_data),
        .m_axis_s2mm_sts_tvalid             (aximm_ack_valid),
        .m_axis_s2mm_sts_tready             (aximm_ack_ready),
        .m_axis_s2mm_sts_tdata              (aximm_ack_data),
        .m_axis_s2mm_sts_tkeep              (),
        .m_axis_s2mm_sts_tlast              (),
        .s2mm_allow_addr_req                (1'd1),
        .s2mm_addr_req_posted               (),
        .s2mm_wr_xfer_cmplt                 (),
        .s2mm_ld_nxt_len                    (),
        .s2mm_wr_len                        (),
        .m_axi_s2mm_awid                    (),
        .m_axi_s2mm_awaddr                  (aximm_awaddr),
        .m_axi_s2mm_awlen                   (aximm_awlen),
        .m_axi_s2mm_awsize                  (aximm_awsize),
        .m_axi_s2mm_awburst                 (aximm_awburst),
        .m_axi_s2mm_awprot                  (aximm_awprot),
        .m_axi_s2mm_awcache                 (aximm_awcache),
        .m_axi_s2mm_awuser                  (),
        .m_axi_s2mm_awvalid                 (aximm_awvalid),
        .m_axi_s2mm_awready                 (aximm_awready),
        .m_axi_s2mm_wdata                   (aximm_wdata),
        .m_axi_s2mm_wstrb                   (aximm_wstrb),
        .m_axi_s2mm_wlast                   (aximm_wlast),
        .m_axi_s2mm_wvalid                  (aximm_wvalid),
        .m_axi_s2mm_wready                  (aximm_wready),
        .m_axi_s2mm_bresp                   (aximm_bresp),
        .m_axi_s2mm_bvalid                  (aximm_bvalid),
        .m_axi_s2mm_bready                  (aximm_bready),
        .s_axis_s2mm_tdata                  (aximm_data),
        .s_axis_s2mm_tkeep                  ({AXIS_SIZE{1'b1}}),
        .s_axis_s2mm_tlast                  (aximm_last),
        .s_axis_s2mm_tvalid                 (aximm_valid),
        .s_axis_s2mm_tready                 (aximm_ready),
        .s2mm_dbg_sel                       (4'd0),
        .s2mm_dbg_data                      ());

    axi_dma_top_regs #(.ID(MODULE_ID)) i_regs (
        .axilite_version                    (VERSION),
        .axilite_instance_id                (INSTANCE_ID),
        .axilite_scratch                    (),
        .axilite_timer                      (),
        .axilite_aximm_width                (AXIMM_DW),
        .axilite_axis_width                 (AXIS_DW),
        .axilite_dma_tx1_rx0                ('d0),
        .axilite_swreset                    (axilite_swreset),
        .axilite_swreset_clr                (axilite_swreset_clr),
        .axilite_start                      (axilite_start),
        .axilite_start_clr                  (~axilite_busy),
        .axilite_stop                       (axilite_stop),
        .axilite_stop_clr                   (~axilite_busy),
        .axilite_cyclic                     (axilite_cyclic),
        .axilite_qthreshold                 (axilite_qthreshold),
        .axilite_busy                       (axilite_busy),
        .axilite_status                     (axilite_status),
        .axilite_ovf_unf                    (axilite_overflow),
        .axilite_intr_ovf_unf               (axilite_intr_status[1]),
        .axilite_length_error               (axilite_length_err),
        .axilite_intr_length_error          (axilite_intr_status[2]),
        .axilite_tlast_error                (axilite_tlast_err),
        .axilite_intr_tlast_error           (axilite_intr_status[3]),
        .axilite_addrdec_error              (axilite_addrdec_err),
        .axilite_intr_addrdec_error         (axilite_intr_status[4]),
        .axilite_slave_error                (axilite_slave_err),
        .axilite_intr_slave_error           (axilite_intr_status[5]),
        .axilite_req_buf_num                (axilite_req_buf_cnt),
        .axilite_ack_buf_num                (axilite_ack_buf_cnt),
        .axilite_start_addr_low             (axilite_offset_addr[31:0]),
        .axilite_start_addr_high            (axilite_offset_addr[39:32]),
        .axilite_base_addr_low              (axilite_base_addr[31:0]),
        .axilite_base_addr_high             (axilite_base_addr[39:32]),
        .axilite_length                     (axilite_length),
        .axilite_num_of_buffers             (axilite_num_of_buffers),
        .axilite_trigger_mode_0             (axilite_trig_mode[1:0]),
        .axilite_trigger_mode_1             (axilite_trig_mode[3:2]),
        .axilite_trigger_mode_2             (axilite_trig_mode[5:4]),
        .axilite_trigger_mode_3             (axilite_trig_mode[7:6]),
        .axilite_trigger_mode_4             (axilite_trig_mode[9:8]),
        .axilite_trigger_mode_5             (axilite_trig_mode[11:10]),
        .axilite_start_sync                 (axilite_start_sync),
        .axilite_cyclic_sync                (),
        .axilite_cdc_state                  (axilite_cdc_state),
        .axilite_cdc_state_clr              (axilite_cdc_state_clr_inv),
        .axilite_master_trigger             (axilite_master_trig),
        .axilite_mmq_tlimit_wr              (axilite_mmq_tlimit_wr),
        .axilite_mmq_tlimit_data            (axilite_mmq_tlimit),
        .axilite_mmq_status                 (axilite_mmq_status),
        .axilite_mmq_count                  (axilite_qcnt),
        .axilite_intr_done_enable           (axilite_intr_enable[0]),
        .axilite_intr_ovf_unf_enable        (axilite_intr_enable[1]),
        .axilite_intr_length_error_enable   (axilite_intr_enable[2]),
        .axilite_intr_tlast_error_enable    (axilite_intr_enable[3]),
        .axilite_intr_addrdec_error_enable  (axilite_intr_enable[4]),
        .axilite_intr_slave_error_enable    (axilite_intr_enable[5]),
        .axilite_clk_mon_count              (axilite_clk_mon_count),
        .axilite_data_xfer_count            (axilite_data_xfer_count[31:0]),
        .axilite_intc_cntrl                 (axilite_intc_cntrl),
        .axilite_clk                        (axilite_clk),
        .axilite_resetn                     (axilite_resetn),
        .axilite_wrreq                      (axilite_wrreq),
        .axilite_wraddr                     (axilite_wraddr),
        .axilite_wrdata                     (axilite_wrdata),
        .axilite_wrack                      (axilite_wrack),
        .axilite_rdreq                      (axilite_rdreq),
        .axilite_rdaddr                     (axilite_rdaddr),
        .axilite_rddata                     (axilite_rddata),
        .axilite_rdack                      (axilite_rdack));

endmodule

// **********************************************************************************
// **********************************************************************************
