// **********************************************************************************
// **********************************************************************************
// ----------------------------------------------------------------------------------
// ################
// ##   ###########   Analog Devices Inc.
// ##      ########
// ##         #####   Copyright (c) 2019 Analog Devices Inc. All rights reserved.
// ##            ##   This file is the confidential and proprietary property of ADI.
// ##         #####   Possession or use of this file requires a written license.
// ##      ########   The licensing information may be found at: www.analog.com
// ##   ###########
// ################
// ----------------------------------------------------------------------------------
// Description:       Simple SPI Master
// ----------------------------------------------------------------------------------
// **********************************************************************************
// **********************************************************************************

`timescale 1ps/1ps

module axi_mspi #(

    parameter integer INSTANCE_ID = 0,
    parameter integer NUM_OF_SLAVES = 8,
    parameter integer SCLK_WIDTH = 1,
    parameter integer MOSI_WIDTH = 1,
    parameter integer MISO_WIDTH = 1) (

    // spi interface

    output  wire  [(NUM_OF_SLAVES-1):0] spi_ssn,
    output  wire  [(SCLK_WIDTH-1):0]    spi_sclk,
    inout   wire  [(MOSI_WIDTH-1):0]    spi_mosi,
    input   wire  [(MISO_WIDTH-1):0]    spi_miso,

    // trigger sources

    input   wire  [  3:0]               trig,
    input   wire                        spi_trig,

    // dma interface

    output  wire                        spi_miso_resetn,
    output  wire                        spi_miso_rsenb,
    output  wire                        spi_miso_rstrig,
    output  wire                        spi_miso_rsdsync,
    output  reg                         spi_miso_rsvld = 'd0,
    output  wire  [  7:0]               spi_miso_rsd,

    output  wire                        spi_mosi_resetn,
    output  wire                        spi_mosi_tsenb,
    output  wire                        spi_mosi_tstrig,
    output  wire                        spi_mosi_tsdsync,
    output  reg                         spi_mosi_tsrd = 'd0,
    input   wire                        spi_mosi_tsav,
    input   wire                        spi_mosi_tsvld,
    input   wire  [  7:0]               spi_mosi_tsd,

    // spi clock

    input   wire                        spi_clk,

    // axilite interface

    input   wire                        axilite_clk,
    input   wire                        axilite_resetn,
    input   wire                        axilite_awvalid,
    input   wire  [ 11:0]               axilite_awaddr,
    output  wire                        axilite_awready,
    input   wire                        axilite_wvalid,
    input   wire  [ 31:0]               axilite_wdata,
    output  wire                        axilite_wready,
    output  wire                        axilite_bvalid,
    output  wire  [  1:0]               axilite_bresp,
    input   wire                        axilite_bready,
    input   wire                        axilite_arvalid,
    input   wire  [ 11:0]               axilite_araddr,
    output  wire                        axilite_arready,
    output  wire                        axilite_rvalid,
    output  wire  [  1:0]               axilite_rresp,
    output  wire  [ 31:0]               axilite_rdata,
    input   wire                        axilite_rready,
    output  reg                         axilite_intr = 1'd0);

    // constants

    localparam  VERSION = {8'd0, 8'd6, 8'd1, 8'd0};

    // internal registers
    
    reg                                 spi_mosi_in_resetn = 'd0;
    reg           [(MOSI_WIDTH-1):0]    spi_mosi_in = 'd0;
    reg                                 spi_miso_in_resetn = 'd0;
    reg           [(MISO_WIDTH-1):0]    spi_miso_in = 'd0;
    reg                                 spi_miso_wr = 'd0;
    reg           [  7:0]               spi_miso_wrdata = 'd0;
    reg                                 spi_in1_out0 = 'd0;
    reg                                 spi_mosi_dma_state = 'd0;
    reg                                 spi_mosi_dma_empty = 'd0;
    reg           [  7:0]               spi_mosi_dma_data = 'd0;
    reg                                 spi_mosi_rd = 'd0;
    reg                                 spi_mosi_dma_rd = 'd0;
    reg           [(NUM_OF_SLAVES-1):0] spi_data_out = 'd0;
    reg           [  6:0]               spi_data_out_p = 'd0;
    reg                                 spi_sclk_out = 'd0;
    reg           [(SCLK_WIDTH-1):0]    spi_sclk_out_p = 'd0;
    reg                                 spi_ssn_state = 1'd1;
    reg           [(NUM_OF_SLAVES-1):0] spi_ssn_out_p = 'd0;
    reg           [ 19:0]               spi_frm_count = 'd0;
    reg                                 spi_frm_state = 'd0;
    reg           [ 19:0]               spi_ssn_count = 'd0;
    reg                                 spi_ssn_status = 'd0;
    reg           [ 16:0]               spi_ssn_delay_count = 'd0;
    reg                                 spi_ssn_enable = 'd0;
    reg                                 spi_bit_state = 'd0;
    reg                                 spi_bit_valid = 'd0;
    reg           [  4:0]               spi_bit_count = 'd0;
    reg           [ 15:0]               spi_bit_clk_count = 'd0;
    reg           [ 22:0]               spi_byte_count = 'd0;
    reg                                 spi_sclk_d = 1'd0;
    reg           [ 31:0]               spi_ssn_sclk_count = 'd0;
    reg           [  4:0]               spi_trig_d = 'd0;
    reg           [  4:0]               spi_trig_enable = 'd0;
    reg                                 spi_swresetn = 'd0;
    reg                                 axilite_swresetn = 'd0;

    // internal signals
 
    wire          [(NUM_OF_SLAVES-1):0] spi_mosi_in_p;
    wire          [(NUM_OF_SLAVES-1):0] spi_miso_in_p;
    wire          [(NUM_OF_SLAVES-1):0] spi_data_in_bus;
    wire                                spi_data_in;
    wire                                spi_miso_wr_valid;
    wire                                spi_dma_status;
    wire                                spi_dma_enable;
    wire                                spi_fifo_enable;
    wire                                spi_enable;
    wire                                spi_trig_valid;
    wire          [  7:0]               spi_mosi_rddata;
    wire                                spi_mosi_empty;
    wire                                spi_miso_full;
    wire          [  7:0]               spi_ssn_disable;
    wire          [  7:0]               spi_sclk_disable;
    wire          [  7:0]               spi_mosi_disable;
    wire          [  4:0]               spi_trig_p;
    wire          [  7:0]               spi_slv_sel_n;
    wire                                spi_sw_enable;
    wire                                spi_auto;
    wire          [ 19:0]               spi_num_of_bytes;
    wire          [ 19:0]               spi_num_of_frames;
    wire          [ 15:0]               spi_sclk_period;
    wire                                spi_sclk_init;
    wire                                spi_mosi_bidir_mode;
    wire          [  3:0]               spi_mosi_bidir_offset;
    wire                                spi_miso_fall1_rise0;
    wire          [  7:0]               spi_miso_select;
    wire                                spi_miso_enable;
    wire          [  9:0]               spi_trig_mode;
    wire                                spi_dma1_fifo0;
    wire                                spi_swreset_clr;
    wire                                spi_swreset;
    wire                                spi_resetn;
    wire          [  5:0]               axilite_intr_active;
    wire          [  5:0]               axilite_intr_status;
    wire          [  7:0]               axilite_slv_sel_n;
    wire                                axilite_sw_enable;
    wire                                axilite_auto;
    wire                                axilite_frm_state;
    wire          [ 19:0]               axilite_num_of_bytes;
    wire          [ 19:0]               axilite_num_of_frames;
    wire          [ 15:0]               axilite_sclk_period;
    wire                                axilite_sclk_init;
    wire                                axilite_mosi_bidir_mode;
    wire          [  3:0]               axilite_mosi_bidir_offset;
    wire                                axilite_miso_fall1_rise0;
    wire          [  7:0]               axilite_miso_select;
    wire                                axilite_miso_enable;
    wire          [  9:0]               axilite_trig_mode;
    wire                                axilite_dma1_fifo0;
    wire                                axilite_swreset;
    wire                                axilite_swreset_clr;
    wire                                axilite_mosi_wr;
    wire          [  7:0]               axilite_mosi_wrdata;
    wire                                axilite_mosi_empty;
    wire                                axilite_mosi_full;
    wire                                axilite_miso_rd;
    wire          [  7:0]               axilite_miso_rddata;
    wire                                axilite_miso_empty;
    wire                                axilite_miso_full;
    wire          [ 22:0]               axilite_byte_count;
    wire                                axilite_ssn_state;
    wire          [ 31:0]               axilite_ssn_sclk_count;
    wire                                axilite_dma_status;
    wire          [  5:0]               axilite_intr_enable;
    wire          [  7:0]               axilite_ssn_disable;
    wire          [  7:0]               axilite_sclk_disable;
    wire          [  7:0]               axilite_mosi_disable;
    wire                                axilite_wrreq;
    wire          [ 11:0]               axilite_wraddr;
    wire          [ 31:0]               axilite_wrdata;
    wire                                axilite_wrack;
    wire                                axilite_rdreq;
    wire          [ 11:0]               axilite_rdaddr;
    wire          [ 31:0]               axilite_rddata;
    wire                                axilite_rdack;

    // interrupts

    assign axilite_intr_active = axilite_intr_status & axilite_intr_enable;
    assign axilite_intr_status[5] = ~axilite_frm_state;
    assign axilite_intr_status[4] = ~axilite_dma_status;
    assign axilite_intr_status[3] = axilite_miso_empty;
    assign axilite_intr_status[2] = axilite_miso_full;
    assign axilite_intr_status[1] = axilite_mosi_empty;
    assign axilite_intr_status[0] = axilite_mosi_full;

    always @(negedge axilite_resetn or posedge axilite_clk) begin
        if (axilite_resetn == 1'b0) begin
            axilite_intr <= 1'd0;
        end else begin
            axilite_intr <= | axilite_intr_active;
        end
    end

    // data in

    always @(negedge spi_swresetn or posedge spi_clk) begin
        if (spi_swresetn == 1'b0) begin
            spi_mosi_in_resetn <= 1'd0;
        end else begin
            spi_mosi_in_resetn <= spi_mosi_bidir_mode;
        end
    end

    always @(negedge spi_mosi_in_resetn or posedge spi_clk) begin
        if (spi_mosi_in_resetn == 1'b0) begin
            spi_mosi_in <= 'd0;
        end else if (spi_miso_wr_valid == 1'd1) begin
            spi_mosi_in <= spi_mosi;
        end
    end

    assign spi_mosi_in_p = (MOSI_WIDTH == 1) ? {NUM_OF_SLAVES{spi_mosi_in}} :
        spi_mosi_in[(NUM_OF_SLAVES-1):0];

    always @(negedge spi_swresetn or posedge spi_clk) begin
        if (spi_swresetn == 1'b0) begin
            spi_miso_in_resetn <= 1'd0;
        end else begin
            spi_miso_in_resetn <= ~spi_mosi_bidir_mode;
        end
    end

    always @(negedge spi_miso_in_resetn or posedge spi_clk) begin
        if (spi_miso_in_resetn == 1'b0) begin
            spi_miso_in <= 'd0;
        end else if (spi_miso_wr_valid == 1'd1) begin
            spi_miso_in <= spi_miso;
        end
    end

    assign spi_miso_in_p = (MISO_WIDTH == 1) ? {NUM_OF_SLAVES{spi_miso_in}} :
        spi_miso_in[(NUM_OF_SLAVES-1):0];

    // data in mux
 
    assign spi_data_in_bus = (spi_mosi_in_p | spi_miso_in_p) &
        spi_miso_select[(NUM_OF_SLAVES-1):0];
    assign spi_data_in = |spi_data_in_bus;

    always @(negedge spi_swresetn or posedge spi_clk) begin
        if (spi_swresetn == 1'b0) begin
            spi_miso_rsvld <= 1'd0;
            spi_miso_wr <= 1'd0;
            spi_miso_wrdata <= 8'd0;
        end else begin
            if ((spi_bit_valid == 1'b1) && (spi_bit_count >= 5'd18) &&
                (spi_miso_enable == 1'd1)) begin
                spi_miso_rsvld <= spi_dma1_fifo0;
                spi_miso_wr <= ~spi_dma1_fifo0;
            end else begin
                spi_miso_rsvld <= 1'd0;
                spi_miso_wr <= 1'd0;
            end
            if ((spi_bit_valid == 1'd1) && (spi_miso_wr_valid == 1'd1)) begin
                spi_miso_wrdata <= {spi_miso_wrdata[6:0], spi_data_in};
            end
        end
    end

    assign spi_miso_rsenb = 1'd1;
    assign spi_miso_rstrig = 1'd1;
    assign spi_miso_rsdsync = 1'd0;

    assign spi_miso_resetn = spi_swresetn;
    assign spi_miso_rsd = spi_miso_wrdata;
    assign spi_miso_wr_valid = spi_miso_fall1_rise0 ^ spi_bit_count[0];

    // data out
 
    generate
    for (genvar i = 0; i < MOSI_WIDTH; i = i + 1) begin: g_mosi
    assign spi_mosi[i] = ((spi_in1_out0 == 1'b1) || (spi_mosi_disable[i] == 1'b1)) ?
        1'bz : spi_data_out[i];
    end
    endgenerate

    always @(negedge spi_swresetn or posedge spi_clk) begin
        if (spi_swresetn == 1'b0) begin
            spi_in1_out0 <= 1'd0;
        end else begin
            if (spi_ssn_state == 1'd1) begin
                spi_in1_out0 <= 1'd0;
            end else if ((spi_bit_valid == 1'b1) &&
                (spi_ssn_count == spi_mosi_bidir_offset) &&
                (spi_bit_count >= 5'd16)) begin
                spi_in1_out0 <= spi_mosi_bidir_mode;
            end
        end
    end

    always @(negedge spi_swresetn or posedge spi_clk) begin
        if (spi_swresetn == 1'b0) begin
            spi_mosi_dma_state <= 1'd0;
            spi_mosi_tsrd <= 1'd0;
            spi_mosi_dma_empty <= 1'd1;
            spi_mosi_dma_data <= 8'd0;
        end else begin
            if (spi_mosi_dma_state == 1'd0) begin
                spi_mosi_tsrd <= spi_mosi_tsav;
                spi_mosi_dma_state <= spi_mosi_tsav;
                spi_mosi_dma_empty <= 1'd1;
                spi_mosi_dma_data <= 8'd0;
            end else begin
                spi_mosi_tsrd <= 1'd0;
                spi_mosi_dma_state <= ~spi_mosi_dma_rd;
                if (spi_mosi_tsvld == 1'd1) begin
                    spi_mosi_dma_empty <= 1'd0;
                    spi_mosi_dma_data <= spi_mosi_tsd;
                end else if (spi_mosi_dma_rd == 1'd1) begin
                    spi_mosi_dma_empty <= 1'd1;
                    spi_mosi_dma_data <= 8'd0;
                end
            end
        end
    end

    always @(negedge spi_swresetn or posedge spi_clk) begin
        if (spi_swresetn == 1'b0) begin
            spi_mosi_rd <= 1'd0;
            spi_mosi_dma_rd <= 1'd0;
            spi_data_out <= {NUM_OF_SLAVES{1'd0}};
            spi_data_out_p <= 7'd0;
        end else begin
            if (spi_bit_count == 5'd0) begin
                spi_mosi_rd <= ~spi_dma1_fifo0 & spi_bit_valid;
                spi_mosi_dma_rd <= spi_dma1_fifo0 & spi_bit_valid;
            end else begin
                spi_mosi_rd <= 1'd0;
                spi_mosi_dma_rd <= 1'd0;
            end
            if ((spi_bit_valid == 1'd1) && (spi_bit_count[0] == 1'd0)) begin
                if (spi_bit_count[4:1] != 4'd0) begin
                    spi_data_out <= {NUM_OF_SLAVES{spi_data_out_p[6]}};
                    spi_data_out_p <= {spi_data_out_p[5:0], 1'd0};
                end else if (spi_dma1_fifo0 == 1'd1) begin
                    spi_data_out <= {NUM_OF_SLAVES{spi_mosi_dma_data[7]}};
                    spi_data_out_p <=  spi_mosi_dma_data[6:0];
                end else begin
                    spi_data_out <= {NUM_OF_SLAVES{spi_mosi_rddata[7]}};
                    spi_data_out_p <=  spi_mosi_rddata[6:0];
                end
            end
        end
    end

    assign spi_mosi_tsenb = 1'd1;
    assign spi_mosi_tstrig = 1'd1;
    assign spi_mosi_tsdsync = 1'd0;

    assign spi_mosi_resetn = spi_swresetn;

    // clock

    generate
    for (genvar i = 0; i < SCLK_WIDTH; i = i + 1) begin: g_sclk
    assign spi_sclk[i] = (spi_sclk_disable[i] == 1'd1) ? 1'bz : spi_sclk_out_p[i];
    end
    endgenerate

    always @(negedge spi_swresetn or posedge spi_clk) begin
        if (spi_swresetn == 1'b0) begin
            spi_sclk_out <= 1'd1;
            spi_sclk_out_p <= {SCLK_WIDTH{1'd0}};
        end else begin
            if (spi_bit_state == 1'b0) begin
                spi_sclk_out <= ~spi_sclk_init;
                spi_sclk_out_p <= {SCLK_WIDTH{spi_sclk_init}};
            end else if ((spi_bit_valid == 1'd1) &&
                (spi_bit_count >= 5'd1) &&
                (spi_bit_count <= 5'd16)) begin
                spi_sclk_out <= ~spi_sclk_out;
                spi_sclk_out_p <= {SCLK_WIDTH{spi_sclk_out}};
            end
        end
    end

    // chip select
 
    generate
    for (genvar i = 0; i < NUM_OF_SLAVES; i = i + 1) begin: g_ssn
    assign spi_ssn[i] = (spi_ssn_disable[i] == 1'd1) ? 1'bz : spi_ssn_out_p[i];
    end
    endgenerate

    always @(negedge spi_swresetn or posedge spi_clk) begin
        if (spi_swresetn == 1'b0) begin
            spi_ssn_state <= 1'd1;
            spi_ssn_out_p <= {NUM_OF_SLAVES{1'b1}};
        end else begin
            if ((spi_auto == 1'd0) || (spi_ssn_status == 1'd1)) begin
                spi_ssn_state <= & spi_slv_sel_n;
                spi_ssn_out_p <= spi_slv_sel_n[(NUM_OF_SLAVES-1):0];
            end else begin
                spi_ssn_state <= 1'd1;
                spi_ssn_out_p <= {NUM_OF_SLAVES{1'b1}};
            end
        end
    end

    // transfer control and counts

    assign spi_dma_status = spi_mosi_tsav;
    assign spi_dma_enable = ~spi_dma1_fifo0 | ~spi_mosi_dma_empty;
    assign spi_fifo_enable = spi_dma1_fifo0 | ~(spi_mosi_empty | spi_miso_full);
    assign spi_enable = spi_dma_enable & spi_fifo_enable;

    always @(negedge spi_swresetn or posedge spi_clk) begin
        if (spi_swresetn == 1'b0) begin
            spi_frm_count <= 'd0;
            spi_frm_state <= 'd0;
        end else begin
            if (spi_frm_state == 1'd0) begin
                spi_frm_count <= 'd0;
                spi_frm_state <= spi_enable & spi_sw_enable & spi_trig_valid;
            end else if ((spi_bit_valid == 1'd1) && (spi_bit_count >= 5'd18) &&
                (spi_ssn_count >= spi_num_of_bytes)) begin
                spi_frm_count <= spi_frm_count + 1'd1;
                if (spi_frm_count >= spi_num_of_frames) begin
                    spi_frm_state <= 1'd0;
                end
            end
        end
    end

    always @(negedge spi_swresetn or posedge spi_clk) begin
        if (spi_swresetn == 1'b0) begin
            spi_ssn_count <= 'd0;
            spi_ssn_status <= 'd0;
            spi_ssn_delay_count <= 'd0;
            spi_ssn_enable <= 'd0;
        end else begin
            if (spi_ssn_status == 1'd0) begin
                spi_ssn_count <= 'd0;
                spi_ssn_status <= spi_ssn_enable & spi_frm_state;
            end else if ((spi_bit_valid == 1'd1) && (spi_bit_count >= 5'd18)) begin
                spi_ssn_count <= spi_ssn_count + 1'd1;
                if (spi_ssn_count >= spi_num_of_bytes) begin
                    spi_ssn_status <= 1'd0;
                end
            end
            if (spi_ssn_status == 1'd0) begin
                spi_ssn_delay_count <= spi_ssn_delay_count + 1'd1;
                if ((spi_ssn_delay_count[16:1] >= spi_sclk_period) &&
                    (spi_ssn_delay_count[0] == 1'd1)) begin
                    spi_ssn_enable <= 1'd1;
                end
            end else begin
                spi_ssn_delay_count <= 17'd0;
                spi_ssn_enable <= 1'd0;
            end
        end
    end

    always @(negedge spi_swresetn or posedge spi_clk) begin
        if (spi_swresetn == 1'b0) begin
            spi_bit_state <= 'd0;
            spi_bit_valid <= 'd0;
            spi_bit_count <= 'd0;
            spi_bit_clk_count <= 'd0;
        end else begin
            if (spi_bit_state == 1'b0) begin
                spi_bit_state <= spi_enable & spi_ssn_status;
            end else if ((spi_bit_clk_count >= spi_sclk_period) &&
                (spi_bit_count >= 5'd18)) begin
                spi_bit_state <= 1'd0;
            end
            if (spi_bit_state == 1'b0) begin
                spi_bit_valid <= spi_enable & spi_ssn_status;
                spi_bit_count <= 5'd0;
            end else if ((spi_bit_clk_count >= spi_sclk_period) &&
                (spi_bit_count < 5'd18)) begin
                spi_bit_valid <= 1'd1;
                spi_bit_count <= spi_bit_count + 1'd1;
            end else begin
                spi_bit_valid <= 1'd0;
                spi_bit_count <= spi_bit_count;
            end
            if ((spi_bit_clk_count < spi_sclk_period) &&
                (spi_bit_state == 1'b1)) begin
                spi_bit_clk_count <= spi_bit_clk_count + 1'd1;
            end else begin
                spi_bit_clk_count <= 16'd0;
            end
        end
    end

    always @(negedge spi_swresetn or posedge spi_clk) begin
        if (spi_swresetn == 1'b0) begin
            spi_byte_count <= 'd0;
            spi_sclk_d <= 1'd0;
            spi_ssn_sclk_count <= 32'd0;
        end else begin
            if ((spi_bit_valid == 1'd1) && (spi_bit_count >= 5'd18)) begin
                spi_byte_count <= spi_byte_count + 1'd1;
            end
            spi_sclk_d <= spi_sclk_out;
            if ((spi_ssn_state == 1'd0) && ((spi_sclk_out ^ spi_sclk_d) == 1'd1)) begin
                spi_ssn_sclk_count <= spi_ssn_sclk_count + 1'd1;
            end
        end
    end

    // triggers
 
    assign spi_trig_valid = & spi_trig_enable;

    generate
    for (genvar n = 0; n < 5; n = n + 1) begin: g_trig

    always @(negedge spi_swresetn or posedge spi_clk) begin
        if (spi_swresetn == 1'b0) begin
            spi_trig_d[n] <= 1'd0;
            spi_trig_enable[n] <= 1'd0;
        end else begin
            spi_trig_d[n] <= spi_trig_p[n];
            if ((spi_frm_state == 1'd1) || (spi_sw_enable == 1'd0)) begin
                spi_trig_enable[n] <= 1'd0;
            end else if (spi_trig_enable[n] == 1'd0) begin
                case (spi_trig_mode[((n*2)+1):(n*2)])
                    2'd3: spi_trig_enable[n] <= spi_trig_p[n] ^ spi_trig_d[n];
                    2'd2: spi_trig_enable[n] <= ~spi_trig_p[n] & spi_trig_d[n];
                    2'd1: spi_trig_enable[n] <= spi_trig_p[n] & ~spi_trig_d[n];
                    default: spi_trig_enable[n] <= 1'd1;
                endcase
            end
        end
    end

    end
    endgenerate

    // software reset
 
    assign spi_swreset_clr = spi_swreset;

    always @(negedge spi_resetn or posedge spi_clk) begin
        if (spi_resetn == 1'b0) begin
            spi_swresetn <= 1'b0;
        end else begin
            spi_swresetn <= ~spi_swreset;
        end
    end

    always @(negedge axilite_resetn or posedge axilite_clk) begin
        if (axilite_resetn == 1'b0) begin
            axilite_swresetn <= 1'b0;
        end else begin
            axilite_swresetn <= ~axilite_swreset;
        end
    end

    // instantiations
 
    cdc_resetp #(.ACTIVE_RISE1_FALL0(1)) i_resetn (
        .src_clk                        (axilite_clk),
        .src_resetn                     (axilite_resetn),
        .dest_clk                       (spi_clk),
        .dest_resetp                    (spi_resetn));

    cdc #(.DATA_WIDTH(5)) i_trig (
        .src_data                       ({spi_trig, trig}),
        .dest_resetn                    (spi_swresetn),
        .dest_clk                       (spi_clk),
        .dest_data                      (spi_trig_p));

    mem_fifo #(
        .WRITE_LIMIT                    (14),
        .ADDRESS_WIDTH                  (4),
        .DATA_WIDTH                     (8))
    i_mosi_fifo (
        .wr_clk                         (axilite_clk),
        .wr_resetn                      (axilite_swresetn),
        .wr_valid                       (axilite_mosi_wr),
        .wr_data                        (axilite_mosi_wrdata),
        .wr_limit                       (axilite_mosi_full),
        .rd_clk                         (spi_clk),
        .rd_resetn                      (spi_swresetn),
        .rd_read                        (spi_mosi_rd),
        .rd_valid                       (),
        .rd_data                        (spi_mosi_rddata),
        .rd_empty                       (spi_mosi_empty));

    mem_fifo #(
        .WRITE_LIMIT                    (14),
        .ADDRESS_WIDTH                  (4),
        .DATA_WIDTH                     (8))
    i_miso_fifo (
        .wr_clk                         (spi_clk),
        .wr_resetn                      (spi_swresetn),
        .wr_valid                       (spi_miso_wr),
        .wr_data                        (spi_miso_wrdata),
        .wr_limit                       (spi_miso_full),
        .rd_clk                         (axilite_clk),
        .rd_resetn                      (axilite_swresetn),
        .rd_read                        (axilite_miso_rd),
        .rd_valid                       (),
        .rd_data                        (axilite_miso_rddata),
        .rd_empty                       (axilite_miso_empty));

    cdc_cntrl #(.DATA_WIDTH(61)) i_cdc_cntrl_axilite (
        .src_resetn                     (spi_resetn),
        .src_clk                        (spi_clk),
        .src_data                       ({spi_mosi_empty,
                                            spi_miso_full,
                                            spi_frm_state,
                                            spi_swreset_clr,
                                            spi_byte_count,
                                            spi_ssn_state,
                                            spi_ssn_sclk_count,
                                            spi_dma_status}),
        .dest_resetn                    (axilite_resetn),
        .dest_clk                       (axilite_clk),
        .dest_data                      ({axilite_mosi_empty,
                                            axilite_miso_full,
                                            axilite_frm_state,
                                            axilite_swreset_clr,
                                            axilite_byte_count,
                                            axilite_ssn_state,
                                            axilite_ssn_sclk_count,
                                            axilite_dma_status}));

    cdc_cntrl #(.DATA_WIDTH(118)) i_cdc_cntrl_spi (
        .src_resetn                     (axilite_resetn),
        .src_clk                        (axilite_clk),
        .src_data                       ({axilite_slv_sel_n,
                                            axilite_sw_enable,
                                            axilite_auto,
                                            axilite_num_of_bytes,
                                            axilite_num_of_frames,
                                            axilite_sclk_period,
                                            axilite_sclk_init,
                                            axilite_mosi_bidir_mode,
                                            axilite_mosi_bidir_offset,
                                            axilite_miso_fall1_rise0,
                                            axilite_miso_select,
                                            axilite_miso_enable,
                                            axilite_trig_mode,
                                            axilite_dma1_fifo0,
                                            axilite_swreset,
                                            axilite_ssn_disable,
                                            axilite_sclk_disable,
                                            axilite_mosi_disable}),
        .dest_resetn                    (spi_resetn),
        .dest_clk                       (spi_clk),
        .dest_data                      ({spi_slv_sel_n,
                                            spi_sw_enable,
                                            spi_auto,
                                            spi_num_of_bytes,
                                            spi_num_of_frames,
                                            spi_sclk_period,
                                            spi_sclk_init,
                                            spi_mosi_bidir_mode,
                                            spi_mosi_bidir_offset,
                                            spi_miso_fall1_rise0,
                                            spi_miso_select,
                                            spi_miso_enable,
                                            spi_trig_mode,
                                            spi_dma1_fifo0,
                                            spi_swreset,
                                            spi_ssn_disable,
                                            spi_sclk_disable,
                                            spi_mosi_disable}));

    axi_mspi_top_regs i_regs (
        .axilite_version                (VERSION),
        .axilite_instance_id            (INSTANCE_ID),
        .axilite_scratch                (),
        .axilite_timer                  (),
        .axilite_slave_select_n         (axilite_slv_sel_n),
        .axilite_enable                 (axilite_sw_enable),
        .axilite_auto                   (axilite_auto),
        .axilite_busy                   (axilite_frm_state),
        .axilite_num_of_bytes           (axilite_num_of_bytes),
        .axilite_num_of_frames          (axilite_num_of_frames),
        .axilite_sclk_period            (axilite_sclk_period),
        .axilite_sclk_init              (axilite_sclk_init),
        .axilite_mosi_bidir_mode        (axilite_mosi_bidir_mode),
        .axilite_mosi_bidir_offset      (axilite_mosi_bidir_offset),
        .axilite_miso_fall1_rise0       (axilite_miso_fall1_rise0),
        .axilite_miso_select            (axilite_miso_select),
        .axilite_miso_enable            (axilite_miso_enable),
        .axilite_trigger_mode_0         (axilite_trig_mode[1:0]),
        .axilite_trigger_mode_1         (axilite_trig_mode[3:2]),
        .axilite_trigger_mode_2         (axilite_trig_mode[5:4]),
        .axilite_trigger_mode_3         (axilite_trig_mode[7:6]),
        .axilite_trigger_mode_4         (axilite_trig_mode[9:8]),
        .axilite_dma1_fifo0             (axilite_dma1_fifo0),
        .axilite_swreset                (axilite_swreset),
        .axilite_swreset_clr            (axilite_swreset_clr),
        .axilite_mosi_data_wr           (axilite_mosi_wr),
        .axilite_mosi_data_data         (axilite_mosi_wrdata),
        .axilite_miso_data_rd           (axilite_miso_rd),
        .axilite_miso_data_data         (axilite_miso_rddata),
        .axilite_mosi_full              (axilite_mosi_full),
        .axilite_mosi_empty             (axilite_mosi_empty),
        .axilite_miso_full              (axilite_miso_full),
        .axilite_miso_empty             (axilite_miso_empty),
        .axilite_byte_count             (axilite_byte_count),
        .axilite_ssn_state              (axilite_ssn_state),
        .axilite_ssn_sclk_count         (axilite_ssn_sclk_count),
        .axilite_dma_status             (axilite_dma_status),
        .axilite_intr_mosi_full_enable  (axilite_intr_enable[0]),
        .axilite_intr_mosi_empty_enable (axilite_intr_enable[1]),
        .axilite_intr_miso_full_enable  (axilite_intr_enable[2]),
        .axilite_intr_miso_empty_enable (axilite_intr_enable[3]),
        .axilite_intr_dma_done_enable   (axilite_intr_enable[4]),
        .axilite_intr_xfer_done_enable  (axilite_intr_enable[5]),
        .axilite_ssn_disable            (axilite_ssn_disable),
        .axilite_sclk_disable           (axilite_sclk_disable),
        .axilite_mosi_disable           (axilite_mosi_disable),
        .axilite_clk                    (axilite_clk),
        .axilite_resetn                 (axilite_resetn),
        .axilite_wrreq                  (axilite_wrreq),
        .axilite_wraddr                 (axilite_wraddr),
        .axilite_wrdata                 (axilite_wrdata),
        .axilite_wrack                  (axilite_wrack),
        .axilite_rdreq                  (axilite_rdreq),
        .axilite_rdaddr                 (axilite_rdaddr),
        .axilite_rddata                 (axilite_rddata),
        .axilite_rdack                  (axilite_rdack));

    axilite_slave_if #(
        .AXI_ADDRESS_WIDTH              (12))
    i_axilite_if (
        .axilite_clk                    (axilite_clk),
        .axilite_resetn                 (axilite_resetn),
        .axilite_awvalid                (axilite_awvalid),
        .axilite_awaddr                 (axilite_awaddr),
        .axilite_awready                (axilite_awready),
        .axilite_wvalid                 (axilite_wvalid),
        .axilite_wdata                  (axilite_wdata),
        .axilite_wready                 (axilite_wready),
        .axilite_bvalid                 (axilite_bvalid),
        .axilite_bresp                  (axilite_bresp),
        .axilite_bready                 (axilite_bready),
        .axilite_arvalid                (axilite_arvalid),
        .axilite_araddr                 (axilite_araddr),
        .axilite_arready                (axilite_arready),
        .axilite_rvalid                 (axilite_rvalid),
        .axilite_rresp                  (axilite_rresp),
        .axilite_rdata                  (axilite_rdata),
        .axilite_rready                 (axilite_rready),
        .axilite_wrreq                  (axilite_wrreq),
        .axilite_wraddr                 (axilite_wraddr),
        .axilite_wrdata                 (axilite_wrdata),
        .axilite_wrack                  (axilite_wrack),
        .axilite_rdreq                  (axilite_rdreq),
        .axilite_rdaddr                 (axilite_rdaddr),
        .axilite_rddata                 (axilite_rddata),
        .axilite_rdack                  (axilite_rdack));

endmodule

// **********************************************************************************
// **********************************************************************************
