#!/usr/bin/perl
## ##################################################################################
## ##################################################################################
## ----------------------------------------------------------------------------------
## ################
## ##   ###########   Analog Devices Inc.
## ##      ########
## ##         #####   Copyright (c) 2019 Analog Devices Inc. All rights reserved.
## ##            ##   This file is the confidential and proprietary property of ADI.
## ##         #####   Possession or use of this file requires a written license.
## ##      ########   The licensing information may be found at: www.analog.com
## ##   ###########
## ################
## ----------------------------------------------------------------------------------
## Description:       DMA registers
## ----------------------------------------------------------------------------------
## ##################################################################################
## ##################################################################################

use autodie;
require 'dooku.pl';

$addr_width = 16;
$addr_width_used = 12;

$no_of_instances = 1;
$instance_base_addr = 0x0;
$instance_offset_addr = 0x0;

setdooku(@ARGV);
module(0x0, 'axi_dma_top');

define('OFFSET', 0x0000, 'Address offset');

register(0x000, 'REG_VERSION');
field(0, 32, RO, 0x00060000, 'VERSION', 'This is read as the current version of this IP core.');

register(0x004, 'REG_INSTANCE_ID');
field(0, 32, RO, 0x0, 'INSTANCE_ID', 'These bits Indicate the instance ID, in any multiple
    instances system, instance id 0x0 is the master core.');

register(0x008, 'REG_SCRATCH');
field(0, 32, RW, 0x0, 'SCRATCH', 'The general purpose scratch register');

register(0x00c, 'REG_TIMER');
field(0, 32, RW, 0x0, 'TIMER', 'The general purpose timer register');

register(0x010, 'REG_AXIMM_WIDTH');
field(0, 32, RO, 0x200, 'AXIMM_WIDTH', 'These bits Indicate AXIMM data width, ideally
    it is best to set the length to an integer multiple of this width in bytes. This allows
    minimal realignment and byte enables.');

register(0x014, 'REG_AXIS_WIDTH');
field(0, 32, RO, 0x8, 'AXIS_WIDTH', 'These bits Indicate AXIS data width,
    the dma length must be an integer multiple of this width.');

register(0x018, 'REG_DMA_TYPE');
field(0, 1, RO, 0x0, 'DMA_TX1_RX0', 'These bits indicate dma type.');

register(0x020, 'REG_SWRESET');
field(0, 1, RW1HC, 0x0, 'SWRESET', 'If set, the core is reset at the AXIMM interface.
    In all hardware error cases, a software reset is required to resume normal
    operation. Hardware clears this bit once the internal reset is completed.');

register(0x024, 'REG_CONTROL');
field(0, 1, RW1HC, 0x0, 'START', 'If set interpreted as a request to start the data transfer.
    All DMA related registers must be set properly before asserting this bit.');
field(1, 1, RW1HC, 0x0, 'STOP', 'If set, while the DMA BUSY bit is set, interpreted as a
    request to stop the data transfer. This may take a while depending on the queue threshold
    programmed and cyclic mode. All pending bursts is cleared.');
field(2, 1, RW,    0x0, 'CYCLIC', 'This bit may be asserted to indicate a cyclic data
    transfer, software may split up contigous memory as segments to provide a seamless
    capture, however it must be capable of finishing read or write before data transfer repeats.');
field(4, 4, RW,    0x0, 'QTHRESHOLD', 'The transaction queue threshold, this limits the
    maximum number of posted transactions within the current data transfer.');

register(0x028, 'REG_BUSY');
field(0, 1, RO,   0x0, 'BUSY', 'If asserted indicates that dma is running or has stopped
    because of an error as indicated by the STATUS fields below. This bit is asserted
    low to indicate a successful completion of DMA, provided there are no overflow or underflow.');

register(0x02c, 'REG_STATUS');
field(0, 1, RO,   0x0, 'STATUS', 'If asserted indicates that the AXI bus transaction
    has completed successfully, this is not an indication of the entire transfer.
    This bit is accumulative');
field(1, 1, RW1C, 0x0, 'OVF_UNF', 'If asserted indicates an overflow or underflow, data transfer
    may complete, but most likely data is corrupted.');
field(2, 1, RW1C, 0x0, 'LENGTH_ERROR', 'If asserted indicates an invalid transfer
    length, recovery requires a soft-reset.');
field(3, 1, RW1C, 0x0, 'TLAST_ERROR', 'If asserted indicates an internal LAST flag
    error, recovery requires a soft-reset.');
field(4, 1, RW1C, 0x0, 'ADDRDEC_ERROR', 'If asserted indicates an address decode error,
    likely that the slave is not addressable, recovery requires a soft-reset.');
field(5, 1, RW1C, 0x0, 'SLAVE_ERROR', 'If asserted indicates a slave error, this is
    the AXI-RESPONSE received from the slave, recovery requires a soft-reset.');

define('XFER_STATUS', 0x01, 'Transfer status');
define('XFER_OVF_UNF', 0x02, 'Transfer overflow or underflow');
define('XFER_LENGTH_ERROR', 0x04, 'Transfer invalid length');
define('XFER_TLAST_ERROR', 0x08, 'Transfer missing or incorrect size');
define('XFER_ADDRDEC_ERROR', 0x10, 'Transfer invalid address');
define('XFER_SLAVE_ERROR', 0x20, 'Transfer bus errors indicated by slave');

register(0x030, 'REG_REQ_BUF_NUM');
field(0, 12, RO,   0x0, 'REQ_BUF_NUM', 'The current requested buffer number.');

register(0x034, 'REG_ACK_BUF_NUM');
field(0, 12, RO,   0x0, 'ACK_BUF_NUM', 'The current completed buffer number.');

register(0x080, 'REG_START_ADDR_LOW');
field(0, 32, RW, 0x0, 'START_ADDR_LOW', 'start address low order bits');

register(0x084, 'REG_START_ADDR_HIGH');
field(0, 8, RW, 0x0, 'START_ADDR_HIGH', 'start address high order bits');

register(0x088, 'REG_BASE_ADDR_LOW');
field(0, 32, RW, 0x80000000, 'BASE_ADDR_LOW', 'base address low order bits');

register(0x08c, 'REG_BASE_ADDR_HIGH');
field(0, 8, RW, 0x0, 'BASE_ADDR_HIGH', 'base address high order bits');

register(0x0a0, 'REG_LENGTH');
field(0, 23, RW, 0x0, 'LENGTH', 'indicates length in bytes');

register(0x0a4, 'REG_NUM_OF_BUFERS');
field(0, 12, RW, 0x0, 'NUM_OF_BUFFERS', 'indicates number of buffers.');

$field_base = 0;
$field_offset = 2;
$field_width = 2;
$field_type = "RW";
$field_default = 0x0;
$field_count = 6;
$field_index = 0;

register(0x0c0, 'REG_TRIGGERS');
for ($field_index = 0; $field_index < $field_count; $field_index++) {
    field_loop("TRIGGER_MODE_%d", "The DMA may optionally be controlled through
        the cores trigger inputs. There are four programmable trigger inputs, these bits
        control each one of the triggers independently. The data transfer is controlled
        as an AND function of all the trigger modes. In all modes, the duration of data
        transfer is controlled by the length.");
}

define('TRIGGER_DISABLE', 0x000, 'This trigger source is disabled');
define('TRIGGER_RISING_EDGE', 0x001, 'Transfer begins at the rising edge of this input');
define('TRIGGER_FALLING_EDGE', 0x002, 'Transfer begins at the falling edge of this input');
define('TRIGGER_EDGE', 0x003, 'Transfer begins at either edge of this input');

register(0x0c4, 'REG_FLAGS');
field(0, 1, RW, 0x0, 'START_SYNC', 'The DMA also allows samples transferred
    in relation with enable signals. These modes allow per enable transfer control.
    If this bit is set, data transfer begins at the next rising edge of enable.
    If set to 0x0, data transfer begins as soon as possible.');
field(1, 1, RW, 0x0, 'CYCLIC_SYNC', 'If set to 0x1, data transfer at each cycle
    begins only with the rising edge of enable. This could cause data hit, depending
    on the DMA length and enable duration (transmit only).');

register(0x100, 'REG_CDC_STATE');
field(0, 1, RW1HC, 0x0, 'CDC_STATE', 'If set, cleared when CDC transfer is complete.');

register(0x140, 'REG_MASTER_TRIGGER');
field(0, 1, RW, 0x0, 'MASTER_TRIGGER', 'If set to 0x1, interface uses master trigger,
    this signal is based on board configuration and must be sourced from another instance
    of the same core.');

register(0x160, 'REG_MMQ_TLIMIT');
field(0, 32, MW, 0x0, 'MMQ_TLIMIT', 'The MMQ timer limit.');

register(0x164, 'REG_MMQ_STATUS');
field(0, 32, RO, 0x0, 'MMQ_STATUS', 'The MMQ status data.');

register(0x168, 'REG_MMQ_COUNT');
field(0, 4, RO, 0x0, 'MMQ_COUNT', 'The MMQ count (level).');

register(0x204, 'REG_INTR_ENABLE');
field(0, 1, RW, 0x0, 'INTR_DONE_ENABLE', 'Interrupt enable for INTR_DONE.');
field(1, 1, RW, 0x0, 'INTR_OVF_UNF_ENABLE', 'Interrupt enable for INTR_OVF_UNF.');
field(2, 1, RW, 0x0, 'INTR_LENGTH_ERROR_ENABLE', 'Interrupt enable for INTR_LENGTH_ERROR.');
field(3, 1, RW, 0x0, 'INTR_TLAST_ERROR_ENABLE', 'Interrupt enable for INTR_TLAST_ERROR.');
field(4, 1, RW, 0x0, 'INTR_ADDRDEC_ERROR_ENABLE', 'Interrupt enable for INTR_ADDRDEC_ERROR.');
field(5, 1, RW, 0x0, 'INTR_SLAVE_ERROR_ENABLE', 'Interrupt enable for INTR_SLAVE_ERROR.');

register(0x300, 'REG_CLK_MON_COUNT');
field(0, 32, RO, 0x0, 'CLK_MON_COUNT', 'Clock monitor frequency count. If read 0x0,
    indicates a loss of clock. The format is 16.16 x 100MHz');

register(0x304, 'REG_DATA_XFER_COUNT');
field(0, 32, RO, 0x0, 'DATA_XFER_COUNT', 'Data transfer count. Indicates time taken
    to complete the transfer. The format is 32.0 x 10ns.');

register(0x340, 'REG_INTC_CNTRL');
field(0, 1, RW, 0x0, 'INTC_CNTRL', 'Interconnect control, internal use only.');

endregisters();
endmodule();
generate();

## ##################################################################################
## ##################################################################################
