// **********************************************************************************
// **********************************************************************************
// ----------------------------------------------------------------------------------
// ################
// ##   ###########   Analog Devices Inc.
// ##      ########
// ##         #####   Copyright (c) 2019 Analog Devices Inc. All rights reserved.
// ##            ##   This file is the confidential and proprietary property of ADI.
// ##         #####   Possession or use of this file requires a written license.
// ##      ########   The licensing information may be found at: www.analog.com
// ##   ###########
// ################
// ----------------------------------------------------------------------------------
// Description:       AXI_ADRV9001, transmit interface (single channel)
// ----------------------------------------------------------------------------------
// **********************************************************************************
// **********************************************************************************

`timescale 1ps/1ps

module axi_adrv9001_tx_if #(

    parameter GPIO2_CMOS1_LVDS0 = 0,
    parameter DEVICE_TYPE = "ULTRASCALE",
    parameter IODELAY_GROUP = "IODELAY_GROUP") (

    // system interface

    input   wire            tx_resetn,
    input   wire            tx_hs_clk,
    input   wire            tx_clk,
    input   wire  [  7:0]   tx_pdata,
    output  wire            tx_data_p,
    output  wire            tx_data_n,
    input   wire            tx_buf_enable,
    input   wire  [  2:0]   tx_mcs_delay,

    // axilite interface

    input   wire            axilite_resetn,
    input   wire            axilite_clk,
    input   wire  [  4:0]   axilite_delay_wrdata,
    output  wire  [  4:0]   axilite_delay_rddata);

    // internal registers
 
    reg                     tx_delay_enable = 'd0;
    reg                     tx_delay_rdwr = 'd0;
    reg           [  4:0]   tx_delay_wrdata_d = 'd0;
    reg                     tx_buf_disable = 'd0;
    reg           [  7:0]   tx_pdata_d = 'd0;
    reg           [  7:0]   tx_mdata = 'd0;

    // internal signals
 
    wire                    tx_ddata;
    wire                    tx_sdata;
    wire                    tx_tdata;
    wire          [  4:0]   tx_delay_wrdata;
    wire          [  4:0]   tx_delay_rddata;
    wire          [  3:0]   unused;

    // load delay value

    always @(negedge tx_resetn or posedge tx_clk) begin
        if (tx_resetn == 1'b0) begin
            tx_delay_wrdata_d <= 5'd0;
            tx_delay_enable <= 1'd0;
            tx_delay_rdwr <= 1'b0;
        end else begin
            tx_delay_wrdata_d <= tx_delay_wrdata;
            tx_delay_enable <= tx_delay_rdwr;
            if (tx_delay_wrdata_d == tx_delay_wrdata) begin
                tx_delay_rdwr <= 1'b0;
            end else begin
                tx_delay_rdwr <= 1'b1;
            end
        end
    end

    // buffer enable

    always @(negedge tx_resetn or posedge tx_clk) begin
        if (tx_resetn == 1'b0) begin
            tx_buf_disable <= 1'd1;
        end else begin
            tx_buf_disable <= ~tx_buf_enable;
        end
    end

    // latency (delay setting)

    always @(negedge tx_resetn or posedge tx_clk) begin
        if (tx_resetn == 1'b0) begin
            tx_pdata_d <= 8'd0;
            tx_mdata <= 8'd0;
        end else begin
            tx_pdata_d <= tx_pdata;
            case (tx_mcs_delay)
                3'd7: tx_mdata <= {tx_pdata_d[6:0], tx_pdata[7:7]};
                3'd6: tx_mdata <= {tx_pdata_d[5:0], tx_pdata[7:6]};
                3'd5: tx_mdata <= {tx_pdata_d[4:0], tx_pdata[7:5]};
                3'd4: tx_mdata <= {tx_pdata_d[3:0], tx_pdata[7:4]};
                3'd3: tx_mdata <= {tx_pdata_d[2:0], tx_pdata[7:3]};
                3'd2: tx_mdata <= {tx_pdata_d[1:0], tx_pdata[7:2]};
                3'd1: tx_mdata <= {tx_pdata_d[0:0], tx_pdata[7:1]};
                default: tx_mdata <= tx_pdata;
            endcase
        end
    end

    // instantiations

    generate
    if (GPIO2_CMOS1_LVDS0 == 2) begin

    assign tx_ddata = 1'd0;
    assign tx_tdata = 1'd0;

    end else if (DEVICE_TYPE == "ULTRASCALE") begin
        
    OSERDESE3 #(
        .SIM_DEVICE             ("ULTRASCALE_PLUS"),
        .INIT                   (1'b0),
        .DATA_WIDTH             (8),
        .IS_RST_INVERTED        (1'b0),
        .IS_CLK_INVERTED        (1'b0),
        .IS_CLKDIV_INVERTED     (1'b0))
    i_serdes (
        .CLK                    (tx_hs_clk),
        .CLKDIV                 (tx_clk),
        .RST                    (~tx_resetn),
        .D                      ({tx_mdata[0],
                                    tx_mdata[1],
                                    tx_mdata[2],
                                    tx_mdata[3],
                                    tx_mdata[4],
                                    tx_mdata[5],
                                    tx_mdata[6],
                                    tx_mdata[7]}),
        .T                      (tx_buf_disable),
        .OQ                     (tx_ddata),
        .T_OUT                  (tx_tdata));

    end else begin

    OSERDESE2 #(
        .DATA_RATE_OQ           ("DDR"),
        .DATA_RATE_TQ           ("SDR"),
        .DATA_WIDTH             (8),
        .TRISTATE_WIDTH         (1),
        .SERDES_MODE            ("MASTER"))
    i_serdes (
        .CLK                    (tx_hs_clk),
        .CLKDIV                 (tx_clk),
        .RST                    (~tx_resetn),
        .D1                     (tx_mdata[7]),
        .D2                     (tx_mdata[6]),
        .D3                     (tx_mdata[5]),
        .D4                     (tx_mdata[4]),
        .D5                     (tx_mdata[3]),
        .D6                     (tx_mdata[2]),
        .D7                     (tx_mdata[1]),
        .D8                     (tx_mdata[0]),
        .OQ                     (tx_ddata),
        .T1                     (tx_buf_disable),
        .T2                     (tx_buf_disable),
        .T3                     (tx_buf_disable),
        .T4                     (tx_buf_disable),
        .TQ                     (tx_tdata),
        .OCE                    (1'b1),
        .TCE                    (1'b1),
        .TBYTEIN                (1'b0),
        .SHIFTIN1               (1'b0),
        .SHIFTIN2               (1'b0),
        .SHIFTOUT1              (),
        .SHIFTOUT2              (),
        .OFB                    (),
        .TFB                    (),
        .TBYTEOUT               ());

    end
    endgenerate

    // io delay
 
    generate
    if (GPIO2_CMOS1_LVDS0 == 2) begin

    assign tx_sdata = tx_ddata;
    assign axilite_delay_rddata = 5'd0;
    assign tx_delay_wrdata = 5'd0;
    assign tx_delay_rddata = 5'd0;
    assign unused = 4'd0;

    end else if (DEVICE_TYPE == "ULTRASCALE") begin

    (* IODELAY_GROUP = IODELAY_GROUP *)
    ODELAYE3 #(
        .SIM_DEVICE             ("ULTRASCALE_PLUS"),
        .CASCADE                ("NONE"),
        .DELAY_FORMAT           ("COUNT"),
        .DELAY_TYPE             ("VAR_LOAD"),
        .DELAY_VALUE            (0),
        .IS_CLK_INVERTED        (1'b0),
        .IS_RST_INVERTED        (1'b0),
        .REFCLK_FREQUENCY       (300.0),
        .UPDATE_MODE            ("ASYNC"))
    i_delay (
        .CASC_IN                (1'd0),
        .CASC_RETURN            (1'd0),
        .CASC_OUT               (),
        .INC                    (1'd0),
        .RST                    (1'd0),
        .EN_VTC                 (1'd0),
        .ODATAIN                (tx_ddata),
        .DATAOUT                (tx_sdata),
        .CLK                    (tx_clk),
        .CE                     (tx_delay_enable),
        .LOAD                   (tx_delay_rdwr),
        .CNTVALUEIN             ({tx_delay_wrdata, 4'd0}),
        .CNTVALUEOUT            ({tx_delay_rddata, unused}));

    cdc #(.DATA_WIDTH(5)) i_tx_cdc (
        .src_data               (axilite_delay_wrdata),
        .dest_resetn            (tx_resetn),
        .dest_clk               (tx_clk),
        .dest_data              (tx_delay_wrdata));

    cdc #(.DATA_WIDTH(5)) i_axilite_cdc (
        .src_data               (tx_delay_wrdata),
        .dest_resetn            (axilite_resetn),
        .dest_clk               (axilite_clk),
        .dest_data              (axilite_delay_rddata));

    end else begin

    assign tx_sdata = tx_ddata;
    assign axilite_delay_rddata = 5'd0;
    assign tx_delay_wrdata = 5'd0;
    assign tx_delay_rddata = 5'd0;
    assign unused = 4'd0;

    end
    endgenerate

    // io buffer
 
    generate
    if (GPIO2_CMOS1_LVDS0 == 1) begin

    OBUFT i_obuf (
        .T                      (tx_tdata),
        .I                      (tx_sdata),
        .O                      (tx_data_p));

    assign tx_data_n = 1'bz;

    end else if (GPIO2_CMOS1_LVDS0 == 0) begin

    OBUFTDS i_obuf (
        .T                      (tx_tdata),
        .I                      (tx_sdata),
        .O                      (tx_data_p),
        .OB                     (tx_data_n));

    end else begin

    assign tx_data_p = 1'd0;
    assign tx_data_n = 1'd0;

    end
    endgenerate

endmodule

// **********************************************************************************
// **********************************************************************************
