// **********************************************************************************
// **********************************************************************************
// ----------------------------------------------------------------------------------
// ################
// ##   ###########   Analog Devices Inc.
// ##      ########
// ##         #####   Copyright (c) 2019 Analog Devices Inc. All rights reserved.
// ##            ##   This file is the confidential and proprietary property of ADI.
// ##         #####   Possession or use of this file requires a written license.
// ##      ########   The licensing information may be found at: www.analog.com
// ##   ###########
// ################
// ----------------------------------------------------------------------------------
// Description:       AXI_ADRV9001, transmit data generator
// ----------------------------------------------------------------------------------
// **********************************************************************************
// **********************************************************************************

`timescale 1ps/1ps

module axi_adrv9001_tx_dgen (

    // transmit interface

    input   wire            tx_clk,
    input   wire            tx_resetn,
    input   wire            tx_read,
    output  reg             tx_valid = 'd0,
    output  reg   [ 63:0]   tx_data = 'd0,
    input   wire            tx_lb_valid,
    input   wire  [ 63:0]   tx_lb_data,
    input   wire  [ 31:0]   tx_pattern,
    input   wire            tx_pattern_pulse,
    input   wire            tx_pattern_start,
    output  reg             tx_pattern_done = 'd0,
    output  reg   [ 31:0]   tx_pattern_latency = 'd0,
    input   wire  [  2:0]   tx_num_of_bits,
    input   wire  [  3:0]   tx_data_sel);

    // internal registers
 
    reg           [  1:0]   tx_tone32_cnt = 'd0;
    reg           [ 63:0]   tx_patt32 = 'd0;
    reg           [ 63:0]   tx_nibb32 = 'd0;
    reg           [ 63:0]   tx_ramp32 = 'd0;
    reg           [ 63:0]   tx_pn1532 = 'd0;
    reg           [ 63:0]   tx_pn0732 = 'd0;
    reg           [ 63:0]   tx_patt24 = 'd0;
    reg           [ 63:0]   tx_ramp24 = 'd0;
    reg           [ 63:0]   tx_patt16 = 'd0;
    reg           [ 63:0]   tx_ramp16 = 'd0;
    reg           [ 63:0]   tx_ramp08 = 'd0;
    reg           [ 63:0]   tx_patt08 = 'd0;
    reg           [ 63:0]   tx_patt02 = 'd0;
    reg                     tx_pattern_ins_d = 'd0;
    reg                     tx_pattern_ins = 'd0;
    reg                     tx_pattern_mon_d = 'd0;
    reg                     tx_pattern_mon = 'd0;
    reg                     tx_lat_valid = 'd0;
    reg                     tx_lat_stop = 'd0;
    reg           [  3:0]   tx_lat_incr = 'd0;
    reg                     tx_latxx_valid = 'd0;
    reg                     tx_lat32_stop = 'd0;
    reg           [  3:0]   tx_lat32_incr = 'd0;
    reg                     tx_lat16_stop = 'd0;
    reg           [  3:0]   tx_lat16_incr = 'd0;
    reg                     tx_lat08_stop = 'd0;
    reg           [  3:0]   tx_lat08_incr = 'd0;

    // internal signals

    wire          [ 63:0]   tx_tone32;
    wire          [ 63:0]   tx_pn1532_fn;
    wire          [ 63:0]   tx_pn0732_fn;
    wire          [ 63:0]   tx_ramp02;
    wire                    tx_pattern_ins_p;
    wire                    tx_pattern_mon_p;
    wire          [ 31:0]   tx_lat32_patt;
    wire          [ 15:0]   tx_lat16_patt;
    wire          [  7:0]   tx_lat08_patt;

    // pn7 function

    function [31:0] pn7;
        input [31:0] din;
        reg   [31:0] dout;
        begin
            dout[31] = din[20] ^ din[18] ^ din[19] ^ din[17];
            dout[30] = din[19] ^ din[17] ^ din[18] ^ din[16];
            dout[29] = din[18] ^ din[16] ^ din[17] ^ din[22] ^ din[21];
            dout[28] = din[17] ^ din[22] ^ din[16] ^ din[20];
            dout[27] = din[16] ^ din[22] ^ din[19];
            dout[26] = din[22] ^ din[18];
            dout[25] = din[21] ^ din[17];
            dout[24] = din[20] ^ din[16];
            dout[23] = din[19] ^ din[22] ^ din[21];
            dout[22] = din[18] ^ din[21] ^ din[20];
            dout[21] = din[17] ^ din[20] ^ din[19];
            dout[20] = din[16] ^ din[19] ^ din[18];
            dout[19] = din[22] ^ din[18] ^ din[21] ^ din[17];
            dout[18] = din[21] ^ din[17] ^ din[20] ^ din[16];
            dout[17] = din[20] ^ din[16] ^ din[19] ^ din[22] ^ din[21];
            dout[16] = din[19] ^ din[22] ^ din[18] ^ din[20];
            dout[15] = din[22] ^ din[21];
            dout[14] = din[21] ^ din[20];
            dout[13] = din[20] ^ din[19];
            dout[12] = din[19] ^ din[18];
            dout[11] = din[18] ^ din[17];
            dout[10] = din[17] ^ din[16];
            dout[ 9] = din[16] ^ din[22] ^ din[21];
            dout[ 8] = din[22] ^ din[20];
            dout[ 7] = din[21] ^ din[19];
            dout[ 6] = din[20] ^ din[18];
            dout[ 5] = din[19] ^ din[17];
            dout[ 4] = din[18] ^ din[16];
            dout[ 3] = din[17] ^ din[22] ^ din[21];
            dout[ 2] = din[16] ^ din[21] ^ din[20];
            dout[ 1] = din[22] ^ din[20] ^ din[21] ^ din[19];
            dout[ 0] = din[21] ^ din[19] ^ din[20] ^ din[18];
            pn7 = dout;
        end
    endfunction

    // pn15 function

    function [31:0] pn15;
        input [31:0] din;
        reg   [31:0] dout;
        begin
            dout[31] = din[29] ^ din[27];
            dout[30] = din[28] ^ din[26];
            dout[29] = din[27] ^ din[25];
            dout[28] = din[26] ^ din[24];
            dout[27] = din[25] ^ din[23];
            dout[26] = din[24] ^ din[22];
            dout[25] = din[23] ^ din[21];
            dout[24] = din[22] ^ din[20];
            dout[23] = din[21] ^ din[19];
            dout[22] = din[20] ^ din[18];
            dout[21] = din[19] ^ din[17];
            dout[20] = din[18] ^ din[16];
            dout[19] = din[17] ^ din[30] ^ din[29];
            dout[18] = din[16] ^ din[29] ^ din[28];
            dout[17] = din[30] ^ din[28] ^ din[29] ^ din[27];
            dout[16] = din[29] ^ din[27] ^ din[28] ^ din[26];
            dout[15] = din[30] ^ din[29];
            dout[14] = din[29] ^ din[28];
            dout[13] = din[28] ^ din[27];
            dout[12] = din[27] ^ din[26];
            dout[11] = din[26] ^ din[25];
            dout[10] = din[25] ^ din[24];
            dout[ 9] = din[24] ^ din[23];
            dout[ 8] = din[23] ^ din[22];
            dout[ 7] = din[22] ^ din[21];
            dout[ 6] = din[21] ^ din[20];
            dout[ 5] = din[20] ^ din[19];
            dout[ 4] = din[19] ^ din[18];
            dout[ 3] = din[18] ^ din[17];
            dout[ 2] = din[17] ^ din[16];
            dout[ 1] = din[16] ^ din[30] ^ din[29];
            dout[ 0] = din[30] ^ din[28];
            pn15 = dout;
        end
    endfunction

    // transmit data sources
 
    always @(posedge tx_clk) begin
        tx_valid <= tx_read;
        if (tx_read == 1'b1) begin
            case ({tx_num_of_bits, tx_data_sel})
                7'b1000011: tx_data <= tx_patt32;
                7'b1000100: tx_data <= tx_nibb32;
                7'b1000101: tx_data <= tx_ramp32;
                7'b1000110: tx_data <= tx_pn1532;
                7'b1000111: tx_data <= tx_pn0732;
                7'b1001000: tx_data <= tx_tone32;
                7'b0110011: tx_data <= tx_patt24;
                7'b0110101: tx_data <= tx_ramp24;
                7'b0100011: tx_data <= tx_patt16;
                7'b0100101: tx_data <= tx_ramp16;
                7'b0010011: tx_data <= tx_patt08;
                7'b0010101: tx_data <= tx_ramp08;
                7'b0000011: tx_data <= tx_patt02;
                7'b0000101: tx_data <= tx_ramp02;
                default: tx_data <= 64'd0;
            endcase
        end
    end

    // 32bit (16I, 16Q) mode
 
    assign tx_tone32[63:32] = (tx_tone32_cnt == 2'd0) ? 32'h16a116a1 :
        ((tx_tone32_cnt == 2'd1) ? 32'he9f516a1 : ((tx_tone32_cnt == 2'd2) ?
        32'he9f5e9f5 : 32'h16a1e9f5));

    assign tx_tone32[31: 0] = (tx_tone32_cnt == 2'd0) ? 32'h20000000 :
        ((tx_tone32_cnt == 2'd1) ? 32'h00002000 : ((tx_tone32_cnt == 2'd2) ?
        32'he0000000 : 32'h0000e000));

    always @(negedge tx_resetn or posedge tx_clk) begin
        if (tx_resetn == 1'b0) begin
            tx_tone32_cnt <= 2'd0;
        end else if (tx_read == 1'b1) begin
            tx_tone32_cnt <= tx_tone32_cnt + 1'b1;
        end
    end

    always @(negedge tx_resetn or posedge tx_clk) begin
        if (tx_resetn == 1'b0) begin
            tx_patt32 <= 64'd0;
        end else if (tx_read == 1'b1) begin
            if (tx_pattern_pulse == 1'd0) begin
                tx_patt32[63:32] <= tx_pattern[31:0];
                tx_patt32[31: 0] <= tx_pattern[31:0];
            end else if (tx_pattern_ins == 1'd1) begin
                tx_patt32[63:32] <= 32'd0;
                tx_patt32[31: 0] <= tx_pattern[31:0];
            end else begin
                tx_patt32[63:32] <= 32'd0;
                tx_patt32[31: 0] <= 32'd0;
            end
        end
    end

    always @(negedge tx_resetn or posedge tx_clk) begin
        if (tx_resetn == 1'b0) begin
            tx_nibb32[63:60] <= 4'h1;
            tx_nibb32[59:56] <= 4'h1;
            tx_nibb32[55:52] <= 4'h1;
            tx_nibb32[51:48] <= 4'h1;
            tx_nibb32[47:44] <= 4'h1;
            tx_nibb32[43:40] <= 4'h1;
            tx_nibb32[39:36] <= 4'h1;
            tx_nibb32[35:32] <= 4'h1;
            tx_nibb32[31:28] <= 4'h0;
            tx_nibb32[27:24] <= 4'h0;
            tx_nibb32[23:20] <= 4'h0;
            tx_nibb32[19:16] <= 4'h0;
            tx_nibb32[15:12] <= 4'h0;
            tx_nibb32[11: 8] <= 4'h0;
            tx_nibb32[ 7: 4] <= 4'h0;
            tx_nibb32[ 3: 0] <= 4'h0;
        end else if (tx_read == 1'b1) begin
            tx_nibb32[63:60] <= tx_nibb32[3:0] + 2'h3;
            tx_nibb32[59:56] <= tx_nibb32[3:0] + 2'h3;
            tx_nibb32[55:52] <= tx_nibb32[3:0] + 2'h3;
            tx_nibb32[51:48] <= tx_nibb32[3:0] + 2'h3;
            tx_nibb32[47:44] <= tx_nibb32[3:0] + 2'h3;
            tx_nibb32[43:40] <= tx_nibb32[3:0] + 2'h3;
            tx_nibb32[39:36] <= tx_nibb32[3:0] + 2'h3;
            tx_nibb32[35:32] <= tx_nibb32[3:0] + 2'h3;
            tx_nibb32[31:28] <= tx_nibb32[3:0] + 2'h2;
            tx_nibb32[27:24] <= tx_nibb32[3:0] + 2'h2;
            tx_nibb32[23:20] <= tx_nibb32[3:0] + 2'h2;
            tx_nibb32[19:16] <= tx_nibb32[3:0] + 2'h2;
            tx_nibb32[15:12] <= tx_nibb32[3:0] + 2'h2;
            tx_nibb32[11: 8] <= tx_nibb32[3:0] + 2'h2;
            tx_nibb32[ 7: 4] <= tx_nibb32[3:0] + 2'h2;
            tx_nibb32[ 3: 0] <= tx_nibb32[3:0] + 2'h2;
        end
    end

    always @(negedge tx_resetn or posedge tx_clk) begin
        if (tx_resetn == 1'b0) begin
            tx_ramp32[63:48] <= 16'h1;
            tx_ramp32[47:32] <= 16'h1;
            tx_ramp32[31:16] <= 16'h0;
            tx_ramp32[15: 0] <= 16'h0;
        end else if (tx_read == 1'b1) begin
            tx_ramp32[63:48] <= tx_ramp32[15:0] + 2'h3;
            tx_ramp32[47:32] <= tx_ramp32[15:0] + 2'h3;
            tx_ramp32[31:16] <= tx_ramp32[15:0] + 2'h2;
            tx_ramp32[15: 0] <= tx_ramp32[15:0] + 2'h2;
        end
    end

    assign tx_pn1532_fn = pn15({tx_pn1532[47:32], tx_pn1532[15:0]});

    always @(negedge tx_resetn or posedge tx_clk) begin
        if (tx_resetn == 1'b0) begin
            tx_pn1532[63:48] <= 16'hffff;
            tx_pn1532[47:32] <= 16'hffff;
            tx_pn1532[31:16] <= 16'hffff;
            tx_pn1532[15: 0] <= 16'hffff;
        end else if (tx_read == 1'b1) begin
            tx_pn1532[63:48] <= tx_pn1532_fn[31:16];
            tx_pn1532[47:32] <= tx_pn1532_fn[31:16];
            tx_pn1532[31:16] <= tx_pn1532_fn[15: 0];
            tx_pn1532[15: 0] <= tx_pn1532_fn[15: 0];
        end
    end

    assign tx_pn0732_fn = pn7({tx_pn0732[47:32], tx_pn0732[15:0]});

    always @(negedge tx_resetn or posedge tx_clk) begin
        if (tx_resetn == 1'b0) begin
            tx_pn0732[63:48] <= 16'hffff;
            tx_pn0732[47:32] <= 16'hffff;
            tx_pn0732[31:16] <= 16'hffff;
            tx_pn0732[15: 0] <= 16'hffff;
        end else if (tx_read == 1'b1) begin
            tx_pn0732[63:48] <= tx_pn0732_fn[31:16];
            tx_pn0732[47:32] <= tx_pn0732_fn[31:16];
            tx_pn0732[31:16] <= tx_pn0732_fn[15: 0];
            tx_pn0732[15: 0] <= tx_pn0732_fn[15: 0];
        end
    end

    // 24bit (12I, 12Q) mode
 
    always @(negedge tx_resetn or posedge tx_clk) begin
        if (tx_resetn == 1'b0) begin
            tx_patt24 <= 64'd0;
        end else if (tx_read == 1'b1) begin
            if (tx_pattern_pulse == 1'd0) begin
                tx_patt24[63:48] <= {4'd0, tx_pattern[27:16]};
                tx_patt24[47:32] <= {4'd0, tx_pattern[11: 0]};
                tx_patt24[31:16] <= {4'd0, tx_pattern[27:16]};
                tx_patt24[15: 0] <= {4'd0, tx_pattern[11: 0]};
            end else if (tx_pattern_ins == 1'd1) begin
                tx_patt24[63:48] <= 16'd0;
                tx_patt24[47:32] <= 16'd0;
                tx_patt24[31:16] <= {4'd0, tx_pattern[27:16]};
                tx_patt24[15: 0] <= {4'd0, tx_pattern[11: 0]};
            end else begin
                tx_patt24[63:48] <= 16'd0;
                tx_patt24[47:32] <= 16'd0;
                tx_patt24[31:16] <= 16'd0;
                tx_patt24[15: 0] <= 16'd0;
            end
        end
    end

    always @(negedge tx_resetn or posedge tx_clk) begin
        if (tx_resetn == 1'b0) begin
            tx_ramp24[63:48] <= 16'h1;
            tx_ramp24[47:32] <= 16'h1;
            tx_ramp24[31:16] <= 16'h0;
            tx_ramp24[15: 0] <= 16'h0;
        end else if (tx_read == 1'b1) begin
            tx_ramp24[63:60] <= 4'd0;
            tx_ramp24[59:48] <= tx_ramp24[11:0] + 2'h3;
            tx_ramp24[47:44] <= 4'd0;
            tx_ramp24[43:32] <= tx_ramp24[11:0] + 2'h3;
            tx_ramp24[31:28] <= 4'd0;
            tx_ramp24[27:16] <= tx_ramp24[11:0] + 2'h2;
            tx_ramp24[15:12] <= 4'd0;
            tx_ramp24[11: 0] <= tx_ramp24[11:0] + 2'h2;
        end
    end

    // 16bit (16I) mode
 
    always @(negedge tx_resetn or posedge tx_clk) begin
        if (tx_resetn == 1'b0) begin
            tx_patt16 <= 64'd0;
        end else if (tx_read == 1'b1) begin
            if (tx_pattern_pulse == 1'd0) begin
                tx_patt16[63:48] <= tx_pattern[15:0];
                tx_patt16[47:32] <= tx_pattern[15:0];
                tx_patt16[31:16] <= tx_pattern[15:0];
                tx_patt16[15: 0] <= tx_pattern[15:0];
            end else if (tx_pattern_ins == 1'd1) begin
                tx_patt16[63:48] <= 16'd0;
                tx_patt16[47:32] <= 16'd0;
                tx_patt16[31:16] <= 16'd0;
                tx_patt16[15: 0] <= tx_pattern[15:0];
            end else begin
                tx_patt16[63:48] <= 16'd0;
                tx_patt16[47:32] <= 16'd0;
                tx_patt16[31:16] <= 16'd0;
                tx_patt16[15: 0] <= 16'd0;
            end
        end
    end

    always @(negedge tx_resetn or posedge tx_clk) begin
        if (tx_resetn == 1'b0) begin
            tx_ramp16[63:48] <= 16'h3;
            tx_ramp16[47:32] <= 16'h2;
            tx_ramp16[31:16] <= 16'h1;
            tx_ramp16[15: 0] <= 16'h0;
        end else if (tx_read == 1'b1) begin
            tx_ramp16[63:48] <= tx_ramp16[15:0] + 3'h7;
            tx_ramp16[47:32] <= tx_ramp16[15:0] + 3'h6;
            tx_ramp16[31:16] <= tx_ramp16[15:0] + 3'h5;
            tx_ramp16[15: 0] <= tx_ramp16[15:0] + 3'h4;
        end
    end

    // 8bit (8I) mode
 
    always @(negedge tx_resetn or posedge tx_clk) begin
        if (tx_resetn == 1'b0) begin
            tx_patt08 <= 64'd0;
        end else if (tx_read == 1'b1) begin
            if (tx_pattern_pulse == 1'd0) begin
                tx_patt08[63:56] <= tx_pattern[7:0];
                tx_patt08[55:48] <= tx_pattern[7:0];
                tx_patt08[47:40] <= tx_pattern[7:0];
                tx_patt08[39:32] <= tx_pattern[7:0];
                tx_patt08[31:24] <= tx_pattern[7:0];
                tx_patt08[23:16] <= tx_pattern[7:0];
                tx_patt08[15: 8] <= tx_pattern[7:0];
                tx_patt08[ 7: 0] <= tx_pattern[7:0];
            end else if (tx_pattern_ins == 1'd1) begin
                tx_patt08[63:56] <= 8'd0;
                tx_patt08[55:48] <= 8'd0;
                tx_patt08[47:40] <= 8'd0;
                tx_patt08[39:32] <= 8'd0;
                tx_patt08[31:24] <= 8'd0;
                tx_patt08[23:16] <= 8'd0;
                tx_patt08[15: 8] <= 8'd0;
                tx_patt08[ 7: 0] <= tx_pattern[7:0];
            end else begin
                tx_patt08[63:56] <= 8'd0;
                tx_patt08[55:48] <= 8'd0;
                tx_patt08[47:40] <= 8'd0;
                tx_patt08[39:32] <= 8'd0;
                tx_patt08[31:24] <= 8'd0;
                tx_patt08[23:16] <= 8'd0;
                tx_patt08[15: 8] <= 8'd0;
                tx_patt08[ 7: 0] <= 8'd0;
            end
        end
    end

    always @(negedge tx_resetn or posedge tx_clk) begin
        if (tx_resetn == 1'b0) begin
            tx_ramp08[63:56] <= 8'h7;
            tx_ramp08[55:48] <= 8'h6;
            tx_ramp08[47:40] <= 8'h5;
            tx_ramp08[39:32] <= 8'h4;
            tx_ramp08[31:24] <= 8'h3;
            tx_ramp08[23:16] <= 8'h2;
            tx_ramp08[15: 8] <= 8'h1;
            tx_ramp08[ 7: 0] <= 8'h0;
        end else if (tx_read == 1'b1) begin
            tx_ramp08[63:56] <= tx_ramp08[7:0] + 4'hf;
            tx_ramp08[55:48] <= tx_ramp08[7:0] + 4'he;
            tx_ramp08[47:40] <= tx_ramp08[7:0] + 4'hd;
            tx_ramp08[39:32] <= tx_ramp08[7:0] + 4'hc;
            tx_ramp08[31:24] <= tx_ramp08[7:0] + 4'hb;
            tx_ramp08[23:16] <= tx_ramp08[7:0] + 4'ha;
            tx_ramp08[15: 8] <= tx_ramp08[7:0] + 4'h9;
            tx_ramp08[ 7: 0] <= tx_ramp08[7:0] + 4'h8;
        end
    end

    // 2bit (2I) mode
 
    always @(negedge tx_resetn or posedge tx_clk) begin
        if (tx_resetn == 1'b0) begin
            tx_patt02 <= 64'd0;
        end else if (tx_read == 1'b1) begin
            if (tx_pattern_pulse == 1'd0) begin
                tx_patt02[63:56] <= {6'd0, tx_pattern[1:0]};
                tx_patt02[55:48] <= {6'd0, tx_pattern[1:0]};
                tx_patt02[47:40] <= {6'd0, tx_pattern[1:0]};
                tx_patt02[39:32] <= {6'd0, tx_pattern[1:0]};
                tx_patt02[31:24] <= {6'd0, tx_pattern[1:0]};
                tx_patt02[23:16] <= {6'd0, tx_pattern[1:0]};
                tx_patt02[15: 8] <= {6'd0, tx_pattern[1:0]};
                tx_patt02[ 7: 0] <= {6'd0, tx_pattern[1:0]};
            end else if (tx_pattern_ins == 1'd1) begin
                tx_patt02[63:56] <= 8'd0;
                tx_patt02[55:48] <= 8'd0;
                tx_patt02[47:40] <= 8'd0;
                tx_patt02[39:32] <= 8'd0;
                tx_patt02[31:24] <= 8'd0;
                tx_patt02[23:16] <= 8'd0;
                tx_patt02[15: 8] <= 8'd0;
                tx_patt02[ 7: 0] <= {6'd0, tx_pattern[1:0]};
            end else begin
                tx_patt02[63:56] <= 8'd0;
                tx_patt02[55:48] <= 8'd0;
                tx_patt02[47:40] <= 8'd0;
                tx_patt02[39:32] <= 8'd0;
                tx_patt02[31:24] <= 8'd0;
                tx_patt02[23:16] <= 8'd0;
                tx_patt02[15: 8] <= 8'd0;
                tx_patt02[ 7: 0] <= 8'd0;
            end
        end
    end

    assign tx_ramp02[63:56] = 8'h3;
    assign tx_ramp02[55:48] = 8'h2;
    assign tx_ramp02[47:40] = 8'h1;
    assign tx_ramp02[39:32] = 8'h0;
    assign tx_ramp02[31:24] = 8'h3;
    assign tx_ramp02[23:16] = 8'h2;
    assign tx_ramp02[15: 8] = 8'h1;
    assign tx_ramp02[ 7: 0] = 8'h0;

    // pattern latency done
 
    always @(posedge tx_clk) begin
        tx_pattern_done <= ~(tx_pattern_ins | tx_pattern_mon);
    end

    // pattern insert
 
    assign tx_pattern_ins_p = ~tx_pattern_ins_d & tx_pattern_start;

    always @(negedge tx_resetn or posedge tx_clk) begin
        if (tx_resetn == 1'b0) begin
            tx_pattern_ins_d <= 1'd0;
            tx_pattern_ins <= 1'd0;
        end else begin
            if (tx_read == 1'd1) begin
                tx_pattern_ins_d <= tx_pattern_start;
                if (tx_pattern_ins == 1'd1) begin
                    tx_pattern_ins <= 1'd0;
                end else if (tx_pattern_ins_p == 1'd1) begin
                    tx_pattern_ins <= 1'd1;
                end
            end
        end
    end

    // pattern monitor
 
    assign tx_pattern_mon_p = ~tx_pattern_mon_d & tx_pattern_start;

    always @(negedge tx_resetn or posedge tx_clk) begin
        if (tx_resetn == 1'd0) begin
            tx_pattern_mon_d <= 1'd0;
            tx_pattern_mon <= 1'd0;
            tx_pattern_latency <= 32'd0;
        end else begin
            if (tx_lat_valid == 1'd1) begin
                tx_pattern_mon_d <= tx_pattern_start;
                if (tx_pattern_mon == 1'd1) begin
                    tx_pattern_mon <= ~tx_lat_stop;
                    if (tx_pattern_latency[31] == 1'd0) begin
                        tx_pattern_latency <= tx_pattern_latency + tx_lat_incr;
                    end
                end else if (tx_pattern_mon_p == 1'd1) begin
                    tx_pattern_mon <= 1'd1;
                    tx_pattern_latency <= tx_lat_incr;
                end
            end
        end
    end

    always @(posedge tx_clk) begin
        tx_lat_valid <= tx_latxx_valid;
        case (tx_num_of_bits)
            3'b100, 3'b011: begin
                tx_lat_stop <= tx_lat32_stop;
                tx_lat_incr <= tx_lat32_incr;
            end
            3'b010: begin
                tx_lat_stop <= tx_lat16_stop;
                tx_lat_incr <= tx_lat16_incr;
            end
            3'b001, 3'b000: begin
                tx_lat_stop <= tx_lat08_stop;
                tx_lat_incr <= tx_lat08_incr;
            end
            default: begin
                tx_lat_stop <= 1'd0;
                tx_lat_incr <= 4'd0;
            end
        endcase
    end

    assign tx_lat32_patt[31:28] = (tx_num_of_bits == 3'd3) ? 4'd0 : tx_pattern[31:28];
    assign tx_lat32_patt[15:12] = (tx_num_of_bits == 3'd3) ? 4'd0 : tx_pattern[15:12];
    assign tx_lat32_patt[27:16] = tx_pattern[27:16];
    assign tx_lat32_patt[11: 0] = tx_pattern[11: 0];

    assign tx_lat16_patt = tx_pattern[15:0];

    assign tx_lat08_patt[7:2] = (tx_num_of_bits == 3'd0) ? 6'd0 : tx_pattern[7:2];
    assign tx_lat08_patt[1:0] = tx_pattern[1:0];

    always @(posedge tx_clk) begin
        tx_latxx_valid <= tx_lb_valid;
        if (tx_lb_data[31:0] == tx_lat32_patt) begin
            tx_lat32_stop <= 1'd1;
            tx_lat32_incr <= 4'd0;
        end else if (tx_lb_data[63:32] == tx_lat32_patt) begin
            tx_lat32_stop <= 1'd1;
            tx_lat32_incr <= 4'd1;
        end else begin
            tx_lat32_stop <= 1'd0;
            tx_lat32_incr <= 4'd2;
        end
        if (tx_lb_data[15:0] == tx_lat16_patt) begin
            tx_lat16_stop <= 1'd1;
            tx_lat16_incr <= 4'd0;
        end else if (tx_lb_data[31:16] == tx_lat16_patt) begin
            tx_lat16_stop <= 1'd1;
            tx_lat16_incr <= 4'd1;
        end else if (tx_lb_data[47:32] == tx_lat16_patt) begin
            tx_lat16_stop <= 1'd1;
            tx_lat16_incr <= 4'd2;
        end else if (tx_lb_data[63:48] == tx_lat16_patt) begin
            tx_lat16_stop <= 1'd1;
            tx_lat16_incr <= 4'd3;
        end else begin
            tx_lat16_stop <= 1'd0;
            tx_lat16_incr <= 4'd4;
        end
        if (tx_lb_data[7:0] == tx_lat08_patt) begin
            tx_lat08_stop <= 1'd1;
            tx_lat08_incr <= 4'd0;
        end else if (tx_lb_data[15:8] == tx_lat08_patt) begin
            tx_lat08_stop <= 1'd1;
            tx_lat08_incr <= 4'd1;
        end else if (tx_lb_data[23:16] == tx_lat08_patt) begin
            tx_lat08_stop <= 1'd1;
            tx_lat08_incr <= 4'd2;
        end else if (tx_lb_data[31:24] == tx_lat08_patt) begin
            tx_lat08_stop <= 1'd1;
            tx_lat08_incr <= 4'd3;
        end else if (tx_lb_data[39:32] == tx_lat08_patt) begin
            tx_lat08_stop <= 1'd1;
            tx_lat08_incr <= 4'd4;
        end else if (tx_lb_data[47:40] == tx_lat08_patt) begin
            tx_lat08_stop <= 1'd1;
            tx_lat08_incr <= 4'd5;
        end else if (tx_lb_data[55:48] == tx_lat08_patt) begin
            tx_lat08_stop <= 1'd1;
            tx_lat08_incr <= 4'd6;
        end else if (tx_lb_data[63:56] == tx_lat08_patt) begin
            tx_lat08_stop <= 1'd1;
            tx_lat08_incr <= 4'd7;
        end else begin
            tx_lat08_stop <= 1'd0;
            tx_lat08_incr <= 4'd8;
        end
    end

endmodule

// ***************************************************************************
// ***************************************************************************
