// **********************************************************************************
// **********************************************************************************
// ----------------------------------------------------------------------------------
// ################
// ##   ###########   Analog Devices Inc.
// ##      ########
// ##         #####   Copyright (c) 2019 Analog Devices Inc. All rights reserved.
// ##            ##   This file is the confidential and proprietary property of ADI.
// ##         #####   Possession or use of this file requires a written license.
// ##      ########   The licensing information may be found at: www.analog.com
// ##   ###########
// ################
// ----------------------------------------------------------------------------------
// Description:       AXI_ADRV9001, transmit interface
// ----------------------------------------------------------------------------------
// **********************************************************************************
// **********************************************************************************

`timescale 1ps/1ps

module axi_adrv9001_tx_clk #(

    parameter GPIO2_CMOS1_LVDS0 = 0,
    parameter REFCLK_DISABLE = 0,
    parameter DEVICE_TYPE = "ULTRASCALE",
    parameter MATCH_CLOCKING = 1,
    parameter IODELAY_GROUP = "IODELAY_GROUP") (

    // internal interface

    input   wire            rx_buf_clk,

    input   wire            tx_ref_clk_p,
    input   wire            tx_ref_clk_n,
    output  wire            tx_hs_clk,
    output  wire            tx_clk,

    output  wire            tx_clock_p,
    output  wire            tx_clock_n,

    // reset interface

    output  wire            tx_rst_clk,
    input   wire            tx_clk_resetn,
    input   wire            tx_serdes_resetn,

    // axi interface

    input   wire            axilite_clk,
    input   wire            axilite_resetn,
    input   wire  [  4:0]   axilite_wrdelay_int_c,
    output  wire  [  4:0]   axilite_rddelay_int_c,
    input   wire  [  4:0]   axilite_wrdelay_ext_c,
    output  wire  [  4:0]   axilite_rddelay_ext_c,
    input   wire  [  4:0]   axilite_wrdelay_c,
    output  wire  [  4:0]   axilite_rddelay_c,
    input   wire            axilite_buf_enable,
    input   wire            axilite_clk_enb,
    input   wire            axilite_clk_msb,
    input   wire            axilite_clk_sel);

    // internal registers
 
    reg           [  7:0]   tx_pclock = 'd0;
    reg           [  4:0]   axilite_wrdelay_d_int_c = 'd0;
    reg                     axilite_lddelay_int_c = 'd0;
    reg           [  4:0]   axilite_wrdelay_d_ext_c = 'd0;
    reg                     axilite_lddelay_ext_c = 'd0;
    reg                     axilite_clk0_sel = 1'b1;
    reg                     axilite_clk1_sel = 1'b0;

    // internal signals
 
    wire                    tx_ext_resetn;
    wire                    tx_buf_enable;
    wire                    tx_clk_msb;
    wire                    tx_ext_clk;
    wire                    tx_hs_ext_clk;
    wire                    tx_mr_clk;
    wire                    tx_mr_ext_clk;
    wire                    tx_delay_clk;
    wire                    tx_delay_ext_clk;
    wire                    tx_mux_clk;
    wire                    tx_ref_clk;
    wire                    tx_clk_resetn_int;

    // clock

    always @(negedge tx_ext_resetn or posedge tx_ext_clk) begin
        if (tx_ext_resetn == 1'b0) begin
            tx_pclock <= 8'd0;
        end else begin
            if (tx_clk_msb == 1'b0) begin
                tx_pclock <= 8'h55;
            end else begin
                tx_pclock <= 8'haa;
            end
        end
    end

    // internal delay

    always @(negedge axilite_resetn or posedge axilite_clk) begin
        if (axilite_resetn == 1'b0) begin
            axilite_wrdelay_d_int_c <= 5'd0;
            axilite_lddelay_int_c <= 1'b0;
        end else begin
            axilite_wrdelay_d_int_c <= axilite_wrdelay_int_c;
            if (axilite_wrdelay_d_int_c == axilite_wrdelay_int_c) begin
                axilite_lddelay_int_c <= 1'b0;
            end else begin
                axilite_lddelay_int_c <= 1'b1;
            end
        end
    end

    // external delay
 
    always @(negedge axilite_resetn or posedge axilite_clk) begin
        if (axilite_resetn == 1'b0) begin
            axilite_wrdelay_d_ext_c <= 5'd0;
            axilite_lddelay_ext_c <= 1'b0;
        end else begin
            axilite_wrdelay_d_ext_c <= axilite_wrdelay_ext_c;
            if (axilite_wrdelay_d_ext_c == axilite_wrdelay_ext_c) begin
                axilite_lddelay_ext_c <= 1'b0;
            end else begin
                axilite_lddelay_ext_c <= 1'b1;
            end
        end
    end

    // clock mux control
 
    always @(negedge axilite_resetn or posedge axilite_clk) begin
        if (axilite_resetn == 1'b0) begin
            axilite_clk0_sel <= 1'b1;
            axilite_clk1_sel <= 1'b0;
        end else begin
            axilite_clk0_sel <= ~axilite_clk_sel;
            axilite_clk1_sel <= axilite_clk_sel;
        end
    end

    // instantiations

    cdc #(.DATA_WIDTH(2)) i_cdc (
        .src_data                               ({axilite_buf_enable,
                                                    axilite_clk_msb}),
        .dest_resetn                            (tx_ext_resetn),
        .dest_clk                               (tx_ext_clk),
        .dest_data                              ({tx_buf_enable,
                                                    tx_clk_msb}));

    generate
    if (GPIO2_CMOS1_LVDS0 == 2) begin

    assign tx_clock_p = 1'd0;
    assign tx_clock_n = 1'd0;
    assign axilite_rddelay_c = 5'd0;

    end else begin

    axi_adrv9001_tx_if #(
        .GPIO2_CMOS1_LVDS0                      (GPIO2_CMOS1_LVDS0),
        .DEVICE_TYPE                            (DEVICE_TYPE),
        .IODELAY_GROUP                          (IODELAY_GROUP))
    i_pclock (
        .tx_resetn                              (tx_ext_resetn),
        .tx_hs_clk                              (tx_hs_ext_clk),
        .tx_clk                                 (tx_ext_clk),
        .tx_pdata                               (tx_pclock),
        .tx_data_p                              (tx_clock_p),
        .tx_data_n                              (tx_clock_n),
        .tx_buf_enable                          (tx_buf_enable),
        .tx_mcs_delay                           (3'd0),
        .axilite_resetn                         (axilite_resetn),
        .axilite_clk                            (axilite_clk),
        .axilite_delay_wrdata                   (axilite_wrdelay_c),
        .axilite_delay_rddata                   (axilite_rddelay_c));

    end
    endgenerate

    generate
    if (GPIO2_CMOS1_LVDS0 == 2) begin

    assign tx_clk_resetn_int = 'd0;
    assign tx_clk = 1'd0;
    assign tx_hs_clk = 1'd0;
    assign tx_ext_clk = 1'd0;
    assign tx_hs_ext_clk = 1'd0;
    assign tx_mr_clk = 1'd0;
    assign tx_mr_ext_clk = 1'd0;
    assign tx_delay_clk = 1'd0;
    assign tx_delay_ext_clk = 1'd0;
    assign axilite_rddelay_int_c = 5'd0;
    assign axilite_rddelay_ext_c = 5'd0;

    end else if (DEVICE_TYPE == "ULTRASCALE") begin

    assign tx_ext_resetn = tx_serdes_resetn;
    assign tx_rst_clk = tx_mux_clk;
    assign tx_ext_clk = tx_clk;
    assign tx_hs_ext_clk = tx_hs_clk;
    assign tx_mr_clk = 1'd0;
    assign tx_mr_ext_clk = 1'd0;
    assign tx_delay_clk = 1'd0;
    assign tx_delay_ext_clk = 1'd0;
    assign axilite_rddelay_int_c = 5'd0;
    assign axilite_rddelay_ext_c = 5'd0;

    FDCE #(
        .INIT                                   (1'd0))
    i_clk_resetn (
        .CE                                     (1'd1),
        .D                                      (1'd1),
        .C                                      (tx_rst_clk),
        .CLR                                    (~tx_clk_resetn),
        .Q                                      (tx_clk_resetn_int));

    BUFGCE_DIV #(
        .IS_CLR_INVERTED                        (1'b1),
        .BUFGCE_DIVIDE                          (4))
    i_clk (
        .CE                                     (axilite_clk_enb),
        .CLR                                    (tx_clk_resetn_int),
        .I                                      (tx_mux_clk),
        .O                                      (tx_clk));

    if (MATCH_CLOCKING == 1) begin

    BUFGCE_DIV #(
        .IS_CLR_INVERTED                        (1'b1),
        .BUFGCE_DIVIDE                          (1))
    i_hs_clk (
        .CE                                     (axilite_clk_enb),
        .CLR                                    (tx_clk_resetn_int),
        .I                                      (tx_mux_clk),
        .O                                      (tx_hs_clk));

    end else begin

    BUFGCE i_hs_clk (
        .CE                                     (axilite_clk_enb),
        .I                                      (tx_mux_clk),
        .O                                      (tx_hs_clk));

    end

    end else begin

    assign tx_clk_resetn_int = 'd0;

    cdc #(.DATA_WIDTH(1)) i_ext_resetn (
        .src_data                               (tx_serdes_resetn),
        .dest_resetn                            (1'b1),
        .dest_clk                               (tx_ext_clk),
        .dest_data                              (tx_ext_resetn));

    BUFR #(.BUFR_DIVIDE("4")) i_clk (
        .CE                                     (axilite_clk_enb),
        .CLR                                    (~tx_clk_resetn),
        .I                                      (tx_mr_clk),
        .O                                      (tx_clk));

    BUFR #(.BUFR_DIVIDE("4")) i_ext_clk (
        .CE                                     (axilite_clk_enb),
        .CLR                                    (1'd0),
        .I                                      (tx_mr_ext_clk),
        .O                                      (tx_ext_clk));

    BUFIO i_hs_clk (
        .I                                      (tx_mr_clk),
        .O                                      (tx_hs_clk));

    BUFIO i_hs_ext_clk (
        .I                                      (tx_mr_ext_clk),
        .O                                      (tx_hs_ext_clk));

    BUFMRCE i_mr_clk (
        .CE                                     (1'b1),
        .I                                      (tx_delay_clk),
        .O                                      (tx_mr_clk));

    BUFMRCE i_mr_ext_clk (
        .CE                                     (1'b1),
        .I                                      (tx_delay_ext_clk),
        .O                                      (tx_mr_ext_clk));

    assign tx_rst_clk = tx_delay_clk;

    (* IODELAY_GROUP = IODELAY_GROUP *)
    IDELAYE2 #(
        .CINVCTRL_SEL                           ("FALSE"),
        .DELAY_SRC                              ("DATAIN"),
        .HIGH_PERFORMANCE_MODE                  ("FALSE"),
        .IDELAY_TYPE                            ("VAR_LOAD"),
        .IDELAY_VALUE                           (0),
        .REFCLK_FREQUENCY                       (200.0),
        .PIPE_SEL                               ("FALSE"),
        .SIGNAL_PATTERN                         ("CLOCK"))
    i_delay_clk (
        .CE                                     (1'b0),
        .INC                                    (1'b0),
        .IDATAIN                                (1'b0),
        .LDPIPEEN                               (1'b0),
        .CINVCTRL                               (1'b0),
        .REGRST                                 (1'b0),
        .C                                      (axilite_clk),
        .DATAIN                                 (tx_mux_clk),
        .DATAOUT                                (tx_delay_clk),
        .LD                                     (axilite_lddelay_int_c),
        .CNTVALUEIN                             (axilite_wrdelay_int_c),
        .CNTVALUEOUT                            (axilite_rddelay_int_c));

    (* IODELAY_GROUP = IODELAY_GROUP *)
    IDELAYE2 #(
        .CINVCTRL_SEL                           ("FALSE"),
        .DELAY_SRC                              ("DATAIN"),
        .HIGH_PERFORMANCE_MODE                  ("FALSE"),
        .IDELAY_TYPE                            ("VAR_LOAD"),
        .IDELAY_VALUE                           (0),
        .REFCLK_FREQUENCY                       (200.0),
        .PIPE_SEL                               ("FALSE"),
        .SIGNAL_PATTERN                         ("CLOCK"))
    i_delay_ext_clk (
        .CE                                     (1'b0),
        .INC                                    (1'b0),
        .IDATAIN                                (1'b0),
        .LDPIPEEN                               (1'b0),
        .CINVCTRL                               (1'b0),
        .REGRST                                 (1'b0),
        .C                                      (axilite_clk),
        .DATAIN                                 (tx_mux_clk),
        .DATAOUT                                (tx_delay_ext_clk),
        .LD                                     (axilite_lddelay_ext_c),
        .CNTVALUEIN                             (axilite_wrdelay_ext_c),
        .CNTVALUEOUT                            (axilite_rddelay_ext_c));

    end
    endgenerate

    // transmit reference & receive clock mux
 
    generate
    if (GPIO2_CMOS1_LVDS0 == 2) begin

    assign tx_mux_clk = 1'd0;

    end else begin

    BUFGCTRL i_mux_clk (
        .IGNORE0                                (1'b0),
        .IGNORE1                                (1'b0),
        .CE0                                    (1'b1),
        .CE1                                    (1'b1),
        .S0                                     (axilite_clk0_sel),
        .S1                                     (axilite_clk1_sel),
        .I0                                     (tx_ref_clk),
        .I1                                     (rx_buf_clk),
        .O                                      (tx_mux_clk));

    end
    endgenerate

    // transmit reference clock

    generate
    if ((REFCLK_DISABLE == 0) && (GPIO2_CMOS1_LVDS0 == 0)) begin

    IBUFDS i_ibuf (
        .I                                      (tx_ref_clk_p),
        .IB                                     (tx_ref_clk_n),
        .O                                      (tx_ref_clk));

    end else if ((REFCLK_DISABLE == 0) && (GPIO2_CMOS1_LVDS0 == 1)) begin

    IBUF i_ibuf (
        .I                                      (tx_ref_clk_p),
        .O                                      (tx_ref_clk));

    end else begin

    assign tx_ref_clk = 1'd0;

    end
    endgenerate

endmodule

// **********************************************************************************
// **********************************************************************************
