// **********************************************************************************
// **********************************************************************************
// ----------------------------------------------------------------------------------
// ################
// ##   ###########   Analog Devices Inc.
// ##      ########
// ##         #####   Copyright (c) 2019 Analog Devices Inc. All rights reserved.
// ##            ##   This file is the confidential and proprietary property of ADI.
// ##         #####   Possession or use of this file requires a written license.
// ##      ########   The licensing information may be found at: www.analog.com
// ##   ###########
// ################
// ----------------------------------------------------------------------------------
// Description:       AXI_ADRV9001, transmit interface
// ----------------------------------------------------------------------------------
// **********************************************************************************
// **********************************************************************************

`timescale 1ps/1ps

module axi_adrv9001_tx #(

    parameter ID = 0,
    parameter GPIO2_CMOS1_LVDS0 = 0,
    parameter REFCLK_DISABLE = 0,
    parameter DEVICE_TYPE = "ULTRASCALE",
    parameter RESET_PIPE_LENGTH = 6,
    parameter MATCH_CLOCKING = 1,
    parameter IODELAY_GROUP = "IODELAY_GROUP",
    parameter IODELAY_CTRL_ENABLE = 1) (

    // transmit interface

    input   wire            tx_cssi_refclk_lssi_refclk_p,
    input   wire            tx_cssi_nc_lssi_refclk_n,
    output  wire            tx_cssi_clk_lssi_clk_p,
    output  wire            tx_cssi_nc_lssi_clk_n,
    output  wire            tx_cssi_strobe_lssi_strobe_p,
    output  wire            tx_cssi_nc_lssi_strobe_n,
    output  wire            tx_cssi_data1_lssi_data0_p,
    output  wire            tx_cssi_data0_lssi_data0_n,
    output  wire            tx_cssi_data3_lssi_data1_p,
    output  wire            tx_cssi_data2_lssi_data1_n,
    input   wire  [  9:0]   tx_cssi_gpio_in,
    output  wire  [  9:0]   tx_cssi_gpio_out,
    output  wire  [  9:0]   tx_cssi_gpio_enb,

    // internal interface

    input   wire            rx_buf_clk,
    input   wire            rx_clk,
    input   wire            rx_resetn,
    input   wire            rx_valid,
    input   wire  [ 63:0]   rx_data,
    output  wire            tx_clk,
    output  wire            tx_resetn,
    output  wire            tx_read,
    output  wire            tx_sync,
    input   wire            tx_valid,
    input   wire  [ 63:0]   tx_data,
    input   wire            tx_enable,
    output  wire  [  9:0]   tx_gpio_in,
    input   wire  [  9:0]   tx_gpio_out,
    input   wire  [  9:0]   tx_gpio_enb,

    // mcs & latency interface

    input   wire            mcs_clk_resetn,
    input   wire            mcs_phy_resetn,
    input   wire            mcs_dp_resetn,
    input   wire            tx_clk_resetn_in,
    input   wire            tx_phy_resetn_in,
    input   wire            tx_dp_resetn_in,
    output  wire            tx_clk_resetn_out,
    output  wire            tx_phy_resetn_out,
    output  wire            tx_dp_resetn_out,
    output  reg             tx_menable_out = 'd0,
    input   wire            tx_menable_in,

    // delay interface

    input   wire            sys_delay_clk,
    input   wire            sys_delay_resetn,

    // axi interface

    input   wire            axilite_clk,
    input   wire            axilite_resetn,
    input   wire            axilite_wrreq,
    input   wire  [ 15:0]   axilite_wraddr,
    input   wire  [ 31:0]   axilite_wrdata,
    output  wire            axilite_wrack,
    input   wire            axilite_rdreq,
    input   wire  [ 15:0]   axilite_rdaddr,
    output  wire  [ 31:0]   axilite_rddata,
    output  wire            axilite_rdack);

    // cmos & lvds data lines

    localparam              GPIO2_CMOS1 = (GPIO2_CMOS1_LVDS0 == 1) ? 1 : 2;

    // supported interface modes
 
    localparam    [  5:0]   CMOS_1L_PS_02X1 = {1'b1, 1'b0, 1'b0, 3'b000};
    localparam    [  5:0]   CMOS_1L_LS_02X1 = {1'b1, 1'b0, 1'b1, 3'b000};
    localparam    [  5:0]   CMOS_1L_PS_08X1 = {1'b1, 1'b0, 1'b0, 3'b001};
    localparam    [  5:0]   CMOS_1L_LS_08X1 = {1'b1, 1'b0, 1'b1, 3'b001};
    localparam    [  5:0]   CMOS_1L_PS_16X1 = {1'b1, 1'b0, 1'b0, 3'b010};
    localparam    [  5:0]   CMOS_1L_LS_16X1 = {1'b1, 1'b0, 1'b1, 3'b010};
    localparam    [  5:0]   LVDS_1L_PS_24X1 = {1'b0, 1'b0, 1'b0, 3'b011};
    localparam    [  5:0]   LVDS_1L_LS_24X1 = {1'b0, 1'b0, 1'b1, 3'b011};
    localparam    [  5:0]   LVDS_2L_PS_24X1 = {1'b0, 1'b1, 1'b0, 3'b011};
    localparam    [  5:0]   LVDS_2L_LS_24X1 = {1'b0, 1'b1, 1'b1, 3'b011};
    localparam    [  5:0]   CMOS_1L_PS_32X1 = {1'b1, 1'b0, 1'b0, 3'b100};
    localparam    [  5:0]   CMOS_1L_LS_32X1 = {1'b1, 1'b0, 1'b1, 3'b100};
    localparam    [  5:0]   LVDS_1L_PS_32X1 = {1'b0, 1'b0, 1'b0, 3'b100};
    localparam    [  5:0]   LVDS_1L_LS_32X1 = {1'b0, 1'b0, 1'b1, 3'b100};
    localparam    [  5:0]   LVDS_2L_PS_32X1 = {1'b0, 1'b1, 1'b0, 3'b100};
    localparam    [  5:0]   LVDS_2L_LS_32X1 = {1'b0, 1'b1, 1'b1, 3'b100};
    localparam    [  5:0]   CMOS_4L_PS_32X1 = {1'b1, 1'b1, 1'b0, 3'b100};
    localparam    [  5:0]   CMOS_4L_LS_32X1 = {1'b1, 1'b1, 1'b1, 3'b100};

    // internal registers
 
    reg                     tx_menable_fall = 'd0;
    reg                     tx_menable = 'd0;
    reg           [  1:0]   tx_bcnt_d = 'd0;
    reg           [  1:0]   tx_bcnt_2d = 'd0;
    reg                     tx_rdreq = 'd0;
    reg                     tx_rdtoggle = 'd0;
    reg                     tx_bvalid = 'd0;
    reg                     tx_rdvalid = 'd0;
    reg           [  1:0]   tx_bcnt = 'd0;
    reg           [  2:0]   tx_benable_cnt = 3'h1;
    reg                     tx_benable = 1'd0;
    reg           [  7:0]   tx_bstrobe = 'd0;
    reg           [  7:0]   tx_bdata_0 = 'd0;
    reg           [  7:0]   tx_bdata_1 = 'd0;
    reg           [  7:0]   tx_bdata_2 = 'd0;
    reg           [  7:0]   tx_bdata_3 = 'd0;
    reg           [  7:0]   tx_pstrobe = 'd0;
    reg           [  7:0]   tx_pdata_0 = 'd0;
    reg           [  7:0]   tx_pdata_1 = 'd0;
    reg           [  7:0]   tx_pdata_2 = 'd0;
    reg           [  7:0]   tx_pdata_3 = 'd0;
    reg                     tx_mcs_latency_enable = 'd0;
    reg           [ 15:0]   tx_mcs_latency_count = 'd0;
    reg           [  3:0]   tx_mcs_latency_incr = 'd0;
    reg                     axilite_delay_unlocked = 'd1;

    // internal signals
 
    wire                    tx_cmos1_lvds0;
    wire          [  5:0]   tx_mode;
    wire                    tx_ienable;
    wire                    tx_valid_mux;
    wire          [ 63:0]   tx_data_mux;
    wire          [  2:0]   tx_num_of_bits;
    wire                    tx_lstrb1_pstrb0;
    wire                    tx_mlane1_slane0;
    wire                    tx_sdr1_ddr0;
    wire                    tx_lsb1_msb0;
    wire                    tx_q1_i0;
    wire          [  4:0]   tx_buf_enable;
    wire                    tx_init_ack;
    wire          [  2:0]   tx_ss_ratio;
    wire                    tx_ss_mode;
    wire                    tx_master_enable;
    wire          [  3:0]   tx_data_sel;
    wire          [ 31:0]   tx_pattern;
    wire                    tx_pattern_pulse;
    wire                    tx_pattern_start;
    wire                    tx_pattern_done;
    wire          [ 31:0]   tx_pattern_latency;
    wire          [  2:0]   tx_mcs_delay;
    wire                    tx_ss_valid;
    wire          [ 63:0]   tx_ss_data;
    wire                    tx_lb_valid;
    wire          [ 63:0]   tx_lb_data;
    wire                    tx_dgen_valid;
    wire          [ 63:0]   tx_dgen_data;
    wire          [ 31:0]   tx_rdstrobe;
    wire          [ 31:0]   tx_rddata;
    wire          [  7:0]   tx_rddata_h;
    wire          [  3:0]   tx_data_p;
    wire          [  3:0]   tx_data_n;
    wire                    tx_hs_clk;
    wire                    tx_rst_clk;
    wire                    tx_clk_resetn;
    wire                    tx_serdes_resetn;
    wire                    tx_cdc_resetn;
    wire                    delay_locked;
    wire          [ 31:0]   axilite_cmos_lvds_id;
    wire          [  2:0]   axilite_num_of_bits;
    wire                    axilite_lstrb1_pstrb0;
    wire                    axilite_mlane1_slane0;
    wire                    axilite_sdr1_ddr0;
    wire                    axilite_lsb1_msb0;
    wire                    axilite_q1_i0;
    wire                    axilite_cmos1_lvds0;
    wire          [  5:0]   axilite_buf_enable;
    wire                    axilite_clk_sel;
    wire                    axilite_clk_msb;
    wire                    axilite_clk_enb;
    wire                    axilite_init_req;
    wire                    axilite_init_ack;
    wire                    axilite_lb_ovf;
    wire                    axilite_lb_unf;
    wire          [  2:0]   axilite_ss_ratio;
    wire                    axilite_ss_mode;
    wire                    axilite_master_enable;
    wire          [  3:0]   axilite_data_sel;
    wire          [ 31:0]   axilite_pattern;
    wire                    axilite_delay_locked;
    wire          [  4:0]   axilite_wrdelay_0;
    wire          [  4:0]   axilite_rddelay_0;
    wire          [  4:0]   axilite_wrdelay_1;
    wire          [  4:0]   axilite_rddelay_1;
    wire          [  4:0]   axilite_wrdelay_2;
    wire          [  4:0]   axilite_rddelay_2;
    wire          [  4:0]   axilite_wrdelay_3;
    wire          [  4:0]   axilite_rddelay_3;
    wire          [  4:0]   axilite_wrdelay_s;
    wire          [  4:0]   axilite_rddelay_s;
    wire          [  4:0]   axilite_wrdelay_c;
    wire          [  4:0]   axilite_rddelay_c;
    wire          [  4:0]   axilite_wrdelay_int_c;
    wire          [  4:0]   axilite_rddelay_int_c;
    wire          [  4:0]   axilite_wrdelay_ext_c;
    wire          [  4:0]   axilite_rddelay_ext_c;
    wire          [ 31:0]   axilite_clk_mon_count;
    wire          [ 15:0]   axilite_mcs_latency_count;
    wire          [  2:0]   axilite_mcs_delay;
    wire                    axilite_pattern_pulse;
    wire                    axilite_pattern_start;
    wire                    axilite_pattern_done;
    wire          [ 31:0]   axilite_pattern_latency;

    // configuration options

    assign axilite_cmos_lvds_id = (GPIO2_CMOS1_LVDS0 == 0) ? 32'h4c564453 :
        ((GPIO2_CMOS1_LVDS0 == 1) ? 32'h434d4f53 : 32'h4750494f);
    assign axilite_cmos1_lvds0 = (GPIO2_CMOS1_LVDS0 == 0) ? 1'b0 : 1'b1;
    assign tx_cmos1_lvds0 = (GPIO2_CMOS1_LVDS0 == 0) ? 1'b0 : 1'b1;

    assign tx_mode[5:5] = tx_cmos1_lvds0;
    assign tx_mode[4:4] = tx_mlane1_slane0;
    assign tx_mode[3:3] = tx_lstrb1_pstrb0;
    assign tx_mode[2:0] = tx_num_of_bits;

    // enable

    assign tx_ienable = (tx_master_enable == 1'd1) ? tx_menable : tx_enable;

    always @(negedge tx_resetn or posedge tx_clk) begin
        if (tx_resetn == 1'd0) begin
            tx_menable_out <= 1'd0;
        end else begin
            if (tx_sync == 1'd1) begin
                tx_menable_out <= tx_enable;
            end
        end
    end

    always @(negedge tx_resetn or negedge tx_clk) begin
        if (tx_resetn == 1'd0) begin
            tx_menable_fall <= 1'd0;
        end else begin
            tx_menable_fall <= tx_menable_in;
        end
    end

    always @(negedge tx_resetn or posedge tx_clk) begin
        if (tx_resetn == 1'd0) begin
            tx_menable <= 1'd0;
        end else begin
            tx_menable <= tx_menable_fall;
        end
    end

    // transmit data sources

    assign tx_valid_mux = (tx_data_sel == 4'd0) ? tx_ss_valid :
        ((tx_data_sel == 4'd1) ? tx_lb_valid : tx_dgen_valid);
    assign tx_data_mux = (tx_data_sel == 4'd0) ? tx_ss_data :
        ((tx_data_sel == 4'd1) ? tx_lb_data : tx_dgen_data);

    // transmit bit conversion - read

    always @(negedge tx_resetn or posedge tx_clk) begin
        if (tx_resetn == 1'b0) begin
            tx_bcnt_d <= 2'd0;
            tx_bcnt_2d <= 2'd0;
            tx_rdreq <= 1'd0;
            tx_rdtoggle <= 1'd0;
            tx_bvalid <= 1'd0;
            tx_rdvalid <= 1'd0;
            tx_bcnt <= 2'd0;
        end else begin
            tx_bcnt_d <= tx_bcnt;
            tx_bcnt_2d <= tx_bcnt_d;
            tx_rdreq <= tx_rdvalid & ~tx_rdtoggle;
            if (tx_rdvalid == 1'd1) begin
                tx_rdtoggle <= ~tx_rdtoggle;
            end
            if (tx_sdr1_ddr0 == 1'b1) begin
                tx_bvalid <= ~tx_bvalid;
            end else begin
                tx_bvalid <= 1'b1;
            end
            if (tx_bvalid == 1'b1) begin
                case (tx_mode)
                    CMOS_1L_LS_32X1, CMOS_1L_PS_32X1, LVDS_1L_LS_32X1, LVDS_1L_PS_32X1,
                    CMOS_1L_PS_16X1, CMOS_1L_LS_16X1, CMOS_1L_PS_08X1, CMOS_1L_LS_08X1: begin
                        tx_rdvalid <= ~(|tx_bcnt);
                        if (tx_bcnt >= 2'd3) begin
                            tx_bcnt <= 2'd0;
                        end else begin
                            tx_bcnt <= tx_bcnt + 1'b1;
                        end
                    end
                    LVDS_1L_LS_24X1, LVDS_1L_PS_24X1: begin
                        tx_rdvalid <= ~(|tx_bcnt);
                        if (tx_bcnt >= 2'd2) begin
                            tx_bcnt <= 2'd0;
                        end else begin
                            tx_bcnt <= tx_bcnt + 1'b1;
                        end
                    end
                    LVDS_2L_PS_24X1, LVDS_2L_LS_24X1: begin
                        tx_rdvalid <= ~tx_bcnt[1];
                        if (tx_bcnt >= 2'd2) begin
                            tx_bcnt <= 2'd0;
                        end else begin
                            tx_bcnt <= tx_bcnt + 1'b1;
                        end
                    end
                    LVDS_2L_PS_32X1, LVDS_2L_LS_32X1: begin
                        tx_rdvalid <= ~(|tx_bcnt);
                        if (tx_bcnt >= 2'd1) begin
                            tx_bcnt <= 2'd0;
                        end else begin
                            tx_bcnt <= tx_bcnt + 1'b1;
                        end
                    end
                    CMOS_1L_PS_02X1, CMOS_1L_LS_02X1, CMOS_4L_PS_32X1, CMOS_4L_LS_32X1: begin
                        tx_rdvalid <= 1'b1;
                        tx_bcnt <= 2'd0;
                    end
                    default: begin
                        tx_rdvalid <= 1'b0;
                        tx_bcnt <= 2'd0;
                    end
                endcase
            end else begin
                tx_rdvalid <= 1'b0;
                tx_bcnt <= tx_bcnt;
            end
        end
    end

    // transmit bit conversion - data

    always @(negedge tx_resetn or posedge tx_clk) begin
        if (tx_resetn == 1'b0) begin
            tx_benable_cnt <= 3'h1;
            tx_benable <= 1'd0;
        end else begin
            if (tx_benable_cnt != 3'd0) begin
                tx_benable_cnt <= tx_benable_cnt + 1'd1;
            end
            if (tx_benable_cnt == 3'd0) begin
                tx_benable <= 1'd1;
            end
        end
    end

    always @(negedge tx_resetn or posedge tx_clk) begin
        if (tx_resetn == 1'b0) begin
            tx_bstrobe <= 8'h00;
            tx_bdata_0 <= 8'h00;
            tx_bdata_1 <= 8'h00;
            tx_bdata_2 <= 8'h00;
            tx_bdata_3 <= 8'h00;
        end else begin
            if ((tx_benable == 1'b1) && (tx_bvalid == 1'b1)) begin
                case (tx_mode)
                    CMOS_1L_LS_32X1, CMOS_1L_PS_32X1, LVDS_1L_LS_32X1,
                    LVDS_1L_PS_32X1: begin
                        case (tx_bcnt_2d)
                            2'd0: begin
                                tx_bstrobe <= tx_rdstrobe[31:24];
                                tx_bdata_0 <= tx_rddata[31:24];
                            end
                            2'd1: begin
                                tx_bstrobe <= tx_rdstrobe[23:16];
                                tx_bdata_0 <= tx_rddata[23:16];
                            end
                            2'd2: begin
                                tx_bstrobe <= tx_rdstrobe[15:8];
                                tx_bdata_0 <= tx_rddata[15:8];
                            end
                            default: begin
                                tx_bstrobe <= tx_rdstrobe[7:0];
                                tx_bdata_0 <= tx_rddata[7:0];
                            end
                        endcase
                        tx_bdata_1 <= 8'h00;
                        tx_bdata_2 <= 8'h00;
                        tx_bdata_3 <= 8'h00;
                    end
                    CMOS_1L_PS_16X1, CMOS_1L_LS_16X1: begin
                        case (tx_bcnt_2d)
                            2'd0: begin
                                tx_bstrobe <= tx_rdstrobe[15:8];
                                tx_bdata_0 <= tx_rddata[15:8];
                            end
                            2'd1: begin
                                tx_bstrobe <= tx_rdstrobe[7:0];
                                tx_bdata_0 <= tx_rddata[7:0];
                            end
                            2'd2: begin
                                tx_bstrobe <= tx_rdstrobe[31:24];
                                tx_bdata_0 <= tx_rddata[31:24];
                            end
                            default: begin
                                tx_bstrobe <= tx_rdstrobe[23:16];
                                tx_bdata_0 <= tx_rddata[23:16];
                            end
                        endcase
                        tx_bdata_1 <= 8'h00;
                        tx_bdata_2 <= 8'h00;
                        tx_bdata_3 <= 8'h00;
                    end
                    CMOS_1L_PS_08X1, CMOS_1L_LS_08X1: begin
                        case (tx_bcnt_2d)
                            2'd0: begin
                                tx_bstrobe <= tx_rdstrobe[7:0];
                                tx_bdata_0 <= tx_rddata[7:0];
                            end
                            2'd1: begin
                                tx_bstrobe <= tx_rdstrobe[15:8];
                                tx_bdata_0 <= tx_rddata[15:8];
                            end
                            2'd2: begin
                                tx_bstrobe <= tx_rdstrobe[23:16];
                                tx_bdata_0 <= tx_rddata[23:16];
                            end
                            default: begin
                                tx_bstrobe <= tx_rdstrobe[31:24];
                                tx_bdata_0 <= tx_rddata[31:24];
                            end
                        endcase
                        tx_bdata_1 <= 8'h00;
                        tx_bdata_2 <= 8'h00;
                        tx_bdata_3 <= 8'h00;
                    end
                    LVDS_1L_LS_24X1, LVDS_1L_PS_24X1: begin
                        case (tx_bcnt_2d)
                            2'd0: begin
                                tx_bstrobe <= tx_rdstrobe[27:20];
                                tx_bdata_0 <= tx_rddata[27:20];
                            end
                            2'd1: begin
                                tx_bstrobe <= {tx_rdstrobe[19:16], tx_rdstrobe[11:8]};
                                tx_bdata_0 <= {tx_rddata[19:16], tx_rddata[11:8]};
                            end
                            default: begin
                                tx_bstrobe <= tx_rdstrobe[7:0];
                                tx_bdata_0 <= tx_rddata[7:0];
                            end
                        endcase
                        tx_bdata_1 <= 8'h00;
                        tx_bdata_2 <= 8'h00;
                        tx_bdata_3 <= 8'h00;
                    end
                    LVDS_2L_PS_24X1, LVDS_2L_LS_24X1: begin
                        case (tx_bcnt_2d)
                            2'd0: begin
                                tx_bstrobe <= tx_rdstrobe[27:20];
                                tx_bdata_0 <= tx_rddata[27:20];
                                tx_bdata_1 <= tx_rddata[11:4];
                            end
                            2'd1: begin
                                tx_bstrobe <= {tx_rdstrobe[19:16], tx_rdstrobe[27:24]};
                                tx_bdata_0 <= {tx_rddata_h[3:0], tx_rddata[27:24]};
                                tx_bdata_1 <= {tx_rddata_h[7:4], tx_rddata[11:8]};
                            end
                            default: begin
                                tx_bstrobe <= tx_rdstrobe[23:16];
                                tx_bdata_0 <= tx_rddata[23:16];
                                tx_bdata_1 <= tx_rddata[7:0];
                            end
                        endcase
                        tx_bdata_2 <= 8'h00;
                        tx_bdata_3 <= 8'h00;
                    end
                    LVDS_2L_PS_32X1, LVDS_2L_LS_32X1: begin
                        case (tx_bcnt_2d)
                            2'd0: begin
                                tx_bstrobe <= tx_rdstrobe[31:24];
                                tx_bdata_0 <= tx_rddata[31:24];
                                tx_bdata_1 <= tx_rddata[15:8];
                            end
                            default: begin
                                tx_bstrobe <= tx_rdstrobe[23:16];
                                tx_bdata_0 <= tx_rddata[23:16];
                                tx_bdata_1 <= tx_rddata[7:0];
                            end
                        endcase
                        tx_bdata_2 <= 8'h00;
                        tx_bdata_3 <= 8'h00;
                    end
                    CMOS_1L_PS_02X1, CMOS_1L_LS_02X1: begin
                        tx_bstrobe <= {tx_rdstrobe[1:0], tx_rdstrobe[9:8],
                            tx_rdstrobe[17:16], tx_rdstrobe[25:24]};
                        tx_bdata_0 <= {tx_rddata[1:0], tx_rddata[9:8],
                            tx_rddata[17:16], tx_rddata[25:24]};
                        tx_bdata_1 <= 8'h00;
                        tx_bdata_2 <= 8'h00;
                        tx_bdata_3 <= 8'h00;
                    end
                    CMOS_4L_PS_32X1, CMOS_4L_LS_32X1: begin
                        tx_bstrobe <= tx_rdstrobe[31:24];
                        tx_bdata_0 <= tx_rddata[23:16];
                        tx_bdata_1 <= tx_rddata[31:24];
                        tx_bdata_2 <= tx_rddata[ 7: 0];
                        tx_bdata_3 <= tx_rddata[15: 8];
                    end
                    default: begin
                        tx_bstrobe <= 8'h00;
                        tx_bdata_0 <= 8'h00;
                        tx_bdata_1 <= 8'h00;
                        tx_bdata_2 <= 8'h00;
                        tx_bdata_3 <= 8'h00;
                    end
                endcase
            end
        end
    end

    always @(negedge tx_resetn or posedge tx_clk) begin
        if (tx_resetn == 1'b0) begin
            tx_pstrobe <= 'd0;
            tx_pdata_0 <= 'd0;
            tx_pdata_1 <= 'd0;
            tx_pdata_2 <= 'd0;
            tx_pdata_3 <= 'd0;
        end else begin
            if (tx_sdr1_ddr0 == 1'b0) begin
                tx_pstrobe <= tx_bstrobe;
                tx_pdata_0 <= tx_bdata_0;
                tx_pdata_1 <= tx_bdata_1;
                tx_pdata_2 <= tx_bdata_2;
                tx_pdata_3 <= tx_bdata_3;
            end else if (tx_bvalid == 1'b0) begin
                tx_pstrobe[7:6] <= {2{tx_bstrobe[7]}};
                tx_pstrobe[5:4] <= {2{tx_bstrobe[6]}};
                tx_pstrobe[3:2] <= {2{tx_bstrobe[5]}};
                tx_pstrobe[1:0] <= {2{tx_bstrobe[4]}};
                tx_pdata_0[7:6] <= {2{tx_bdata_0[7]}};
                tx_pdata_0[5:4] <= {2{tx_bdata_0[6]}};
                tx_pdata_0[3:2] <= {2{tx_bdata_0[5]}};
                tx_pdata_0[1:0] <= {2{tx_bdata_0[4]}};
                tx_pdata_1[7:6] <= {2{tx_bdata_1[7]}};
                tx_pdata_1[5:4] <= {2{tx_bdata_1[6]}};
                tx_pdata_1[3:2] <= {2{tx_bdata_1[5]}};
                tx_pdata_1[1:0] <= {2{tx_bdata_1[4]}};
                tx_pdata_2[7:6] <= {2{tx_bdata_2[7]}};
                tx_pdata_2[5:4] <= {2{tx_bdata_2[6]}};
                tx_pdata_2[3:2] <= {2{tx_bdata_2[5]}};
                tx_pdata_2[1:0] <= {2{tx_bdata_2[4]}};
                tx_pdata_3[7:6] <= {2{tx_bdata_3[7]}};
                tx_pdata_3[5:4] <= {2{tx_bdata_3[6]}};
                tx_pdata_3[3:2] <= {2{tx_bdata_3[5]}};
                tx_pdata_3[1:0] <= {2{tx_bdata_3[4]}};
            end else begin
                tx_pstrobe[7:6] <= {2{tx_bstrobe[3]}};
                tx_pstrobe[5:4] <= {2{tx_bstrobe[2]}};
                tx_pstrobe[3:2] <= {2{tx_bstrobe[1]}};
                tx_pstrobe[1:0] <= {2{tx_bstrobe[0]}};
                tx_pdata_0[7:6] <= {2{tx_bdata_0[3]}};
                tx_pdata_0[5:4] <= {2{tx_bdata_0[2]}};
                tx_pdata_0[3:2] <= {2{tx_bdata_0[1]}};
                tx_pdata_0[1:0] <= {2{tx_bdata_0[0]}};
                tx_pdata_1[7:6] <= {2{tx_bdata_1[3]}};
                tx_pdata_1[5:4] <= {2{tx_bdata_1[2]}};
                tx_pdata_1[3:2] <= {2{tx_bdata_1[1]}};
                tx_pdata_1[1:0] <= {2{tx_bdata_1[0]}};
                tx_pdata_2[7:6] <= {2{tx_bdata_2[3]}};
                tx_pdata_2[5:4] <= {2{tx_bdata_2[2]}};
                tx_pdata_2[3:2] <= {2{tx_bdata_2[1]}};
                tx_pdata_2[1:0] <= {2{tx_bdata_2[0]}};
                tx_pdata_3[7:6] <= {2{tx_bdata_3[3]}};
                tx_pdata_3[5:4] <= {2{tx_bdata_3[2]}};
                tx_pdata_3[3:2] <= {2{tx_bdata_3[1]}};
                tx_pdata_3[1:0] <= {2{tx_bdata_3[0]}};
            end
        end
    end

    // mcs reset and its latency monitoring
 
    always @(negedge tx_resetn or posedge tx_clk) begin
        if (tx_resetn == 1'b0) begin
            tx_mcs_latency_enable <= 1'd1;
            tx_mcs_latency_count <= 16'd0;
            tx_mcs_latency_incr <= 4'd8;
        end else begin
            if ((tx_bvalid == 1'd1) && (tx_bstrobe != 8'd0)) begin
                tx_mcs_latency_enable <= 1'd0;
            end
            tx_mcs_latency_count <= tx_mcs_latency_count + tx_mcs_latency_incr;
            if (tx_mcs_latency_enable == 1'd0) begin
                tx_mcs_latency_incr <= 4'd0;
            end else if ((tx_bvalid == 1'd0) || (tx_bstrobe == 8'd0)) begin
                tx_mcs_latency_incr <= 4'd8;
            end else if (tx_bstrobe[7] == 1'd1) begin
                tx_mcs_latency_incr <= 4'd0;
            end else if (tx_bstrobe[6] == 1'd1) begin
                tx_mcs_latency_incr <= 4'd1;
            end else if (tx_bstrobe[5] == 1'd1) begin
                tx_mcs_latency_incr <= 4'd2;
            end else if (tx_bstrobe[4] == 1'd1) begin
                tx_mcs_latency_incr <= 4'd3;
            end else if (tx_bstrobe[3] == 1'd1) begin
                tx_mcs_latency_incr <= 4'd4;
            end else if (tx_bstrobe[2] == 1'd1) begin
                tx_mcs_latency_incr <= 4'd5;
            end else if (tx_bstrobe[1] == 1'd1) begin
                tx_mcs_latency_incr <= 4'd6;
            end else if (tx_bstrobe[0] == 1'd1) begin
                tx_mcs_latency_incr <= 4'd7;
            end
        end
    end

    // axilite delay unlocked
 
    always @(negedge axilite_resetn or posedge axilite_clk) begin
        if (axilite_resetn == 1'b0) begin
            axilite_delay_unlocked <= 1'd1;
        end else begin
            axilite_delay_unlocked <= ~axilite_delay_locked;
        end
    end

    // io buffers
 
    assign tx_cssi_data1_lssi_data0_p = (GPIO2_CMOS1_LVDS0 == 0) ? tx_data_p[0] : tx_data_p[1];
    assign tx_cssi_data0_lssi_data0_n = (GPIO2_CMOS1_LVDS0 == 0) ? tx_data_n[0] : tx_data_p[0];
    assign tx_cssi_data3_lssi_data1_p = (GPIO2_CMOS1_LVDS0 == 0) ? tx_data_p[1] : tx_data_p[3];
    assign tx_cssi_data2_lssi_data1_n = (GPIO2_CMOS1_LVDS0 == 0) ? tx_data_n[1] : tx_data_p[2];

    assign tx_gpio_in = tx_cssi_gpio_in;
    assign tx_cssi_gpio_out = tx_gpio_out;
    assign tx_cssi_gpio_enb = tx_gpio_enb;

    // instantiations

    axi_adrv9001_tx_ss i_tx_ss (
        .tx_clk                                 (tx_clk),
        .tx_resetn                              (tx_resetn),
        .tx_ss_enable                           (tx_ienable),
        .tx_ss_read                             (tx_rdreq),
        .tx_ss_valid                            (tx_ss_valid),
        .tx_ss_data                             (tx_ss_data),
        .tx_ss_ratio                            (tx_ss_ratio),
        .tx_num_of_bits                         (tx_num_of_bits),
        .tx_sync                                (tx_sync),
        .tx_read                                (tx_read),
        .tx_valid                               (tx_valid),
        .tx_data                                (tx_data));

    axi_adrv9001_tx_fifo i_tx_fifo (
        .rx_clk                                 (rx_clk),
        .rx_resetn                              (rx_resetn),
        .rx_valid                               (rx_valid),
        .rx_data                                (rx_data),
        .tx_clk                                 (tx_clk),
        .tx_resetn                              (tx_resetn),
        .tx_read                                (tx_read),
        .tx_valid                               (tx_lb_valid),
        .tx_data                                (tx_lb_data),
        .axilite_clk                            (axilite_clk),
        .axilite_resetn                         (axilite_resetn),
        .axilite_ovf                            (axilite_lb_ovf),
        .axilite_unf                            (axilite_lb_unf));

    axi_adrv9001_tx_dgen i_tx_dgen (
        .tx_clk                                 (tx_clk),
        .tx_resetn                              (tx_resetn),
        .tx_read                                (tx_read),
        .tx_valid                               (tx_dgen_valid),
        .tx_data                                (tx_dgen_data),
        .tx_lb_valid                            (tx_lb_valid),
        .tx_lb_data                             (tx_lb_data),
        .tx_pattern                             (tx_pattern),
        .tx_pattern_pulse                       (tx_pattern_pulse),
        .tx_pattern_start                       (tx_pattern_start),
        .tx_pattern_done                        (tx_pattern_done),
        .tx_pattern_latency                     (tx_pattern_latency),
        .tx_num_of_bits                         (tx_num_of_bits),
        .tx_data_sel                            (tx_data_sel));

    axi_adrv9001_tx_dmgr i_tx_dmgr (
        .tx_clk                                 (tx_clk),
        .tx_resetn                              (tx_resetn),
        .tx_valid                               (tx_valid_mux),
        .tx_data                                (tx_data_mux),
        .tx_q1_i0                               (tx_q1_i0),
        .tx_ss_mode                             (tx_ss_mode),
        .tx_ss_ratio                            (tx_ss_ratio),
        .tx_lsb1_msb0                           (tx_lsb1_msb0),
        .tx_cmos1_lvds0                         (tx_cmos1_lvds0),
        .tx_num_of_bits                         (tx_num_of_bits),
        .tx_lstrb1_pstrb0                       (tx_lstrb1_pstrb0),
        .tx_mlane1_slane0                       (tx_mlane1_slane0),
        .tx_rdvalid                             (tx_rdvalid),
        .tx_rdstrobe                            (tx_rdstrobe),
        .tx_rddata                              (tx_rddata),
        .tx_rddata_h                            (tx_rddata_h));

    clk_mon i_tx_clk_mon (
        .clk                                    (tx_clk),
        .axilite_resetn                         (axilite_resetn),
        .axilite_clk                            (axilite_clk),
        .axilite_clk_mon_count                  (axilite_clk_mon_count));

    axi_adrv9001_tx_if #(
        .GPIO2_CMOS1_LVDS0                      (GPIO2_CMOS1_LVDS0),
        .DEVICE_TYPE                            (DEVICE_TYPE),
        .IODELAY_GROUP                          (IODELAY_GROUP))
    i_tx_pdata_0 (
        .tx_resetn                              (tx_serdes_resetn),
        .tx_hs_clk                              (tx_hs_clk),
        .tx_clk                                 (tx_clk),
        .tx_pdata                               (tx_pdata_0),
        .tx_data_p                              (tx_data_p[0]),
        .tx_data_n                              (tx_data_n[0]),
        .tx_buf_enable                          (tx_buf_enable[1]),
        .tx_mcs_delay                           (tx_mcs_delay),
        .axilite_resetn                         (axilite_resetn),
        .axilite_clk                            (axilite_clk),
        .axilite_delay_wrdata                   (axilite_wrdelay_0),
        .axilite_delay_rddata                   (axilite_rddelay_0));

    axi_adrv9001_tx_if #(
        .GPIO2_CMOS1_LVDS0                      (GPIO2_CMOS1_LVDS0),
        .DEVICE_TYPE                            (DEVICE_TYPE),
        .IODELAY_GROUP                          (IODELAY_GROUP))
    i_tx_pdata_1 (
        .tx_resetn                              (tx_serdes_resetn),
        .tx_hs_clk                              (tx_hs_clk),
        .tx_clk                                 (tx_clk),
        .tx_pdata                               (tx_pdata_1),
        .tx_data_p                              (tx_data_p[1]),
        .tx_data_n                              (tx_data_n[1]),
        .tx_buf_enable                          (tx_buf_enable[2]),
        .tx_mcs_delay                           (tx_mcs_delay),
        .axilite_resetn                         (axilite_resetn),
        .axilite_clk                            (axilite_clk),
        .axilite_delay_wrdata                   (axilite_wrdelay_1),
        .axilite_delay_rddata                   (axilite_rddelay_1));

    axi_adrv9001_tx_if #(
        .GPIO2_CMOS1_LVDS0                      (GPIO2_CMOS1),
        .DEVICE_TYPE                            (DEVICE_TYPE),
        .IODELAY_GROUP                          (IODELAY_GROUP))
    i_tx_pdata_2 (
        .tx_resetn                              (tx_serdes_resetn),
        .tx_hs_clk                              (tx_hs_clk),
        .tx_clk                                 (tx_clk),
        .tx_pdata                               (tx_pdata_2),
        .tx_data_p                              (tx_data_p[2]),
        .tx_data_n                              (tx_data_n[2]),
        .tx_buf_enable                          (tx_buf_enable[3]),
        .tx_mcs_delay                           (tx_mcs_delay),
        .axilite_resetn                         (axilite_resetn),
        .axilite_clk                            (axilite_clk),
        .axilite_delay_wrdata                   (axilite_wrdelay_2),
        .axilite_delay_rddata                   (axilite_rddelay_2));

    axi_adrv9001_tx_if #(
        .GPIO2_CMOS1_LVDS0                      (GPIO2_CMOS1),
        .DEVICE_TYPE                            (DEVICE_TYPE),
        .IODELAY_GROUP                          (IODELAY_GROUP))
    i_tx_pdata_3 (
        .tx_resetn                              (tx_serdes_resetn),
        .tx_hs_clk                              (tx_hs_clk),
        .tx_clk                                 (tx_clk),
        .tx_pdata                               (tx_pdata_3),
        .tx_data_p                              (tx_data_p[3]),
        .tx_data_n                              (tx_data_n[3]),
        .tx_buf_enable                          (tx_buf_enable[4]),
        .tx_mcs_delay                           (tx_mcs_delay),
        .axilite_resetn                         (axilite_resetn),
        .axilite_clk                            (axilite_clk),
        .axilite_delay_wrdata                   (axilite_wrdelay_3),
        .axilite_delay_rddata                   (axilite_rddelay_3));

    axi_adrv9001_tx_if #(
        .GPIO2_CMOS1_LVDS0                      (GPIO2_CMOS1_LVDS0),
        .DEVICE_TYPE                            (DEVICE_TYPE),
        .IODELAY_GROUP                          (IODELAY_GROUP))
    i_tx_pstrobe (
        .tx_resetn                              (tx_serdes_resetn),
        .tx_hs_clk                              (tx_hs_clk),
        .tx_clk                                 (tx_clk),
        .tx_pdata                               (tx_pstrobe),
        .tx_data_p                              (tx_cssi_strobe_lssi_strobe_p),
        .tx_data_n                              (tx_cssi_nc_lssi_strobe_n),
        .tx_buf_enable                          (tx_buf_enable[0]),
        .tx_mcs_delay                           (tx_mcs_delay),
        .axilite_resetn                         (axilite_resetn),
        .axilite_clk                            (axilite_clk),
        .axilite_delay_wrdata                   (axilite_wrdelay_s),
        .axilite_delay_rddata                   (axilite_rddelay_s));

    axi_adrv9001_tx_clk #(
        .GPIO2_CMOS1_LVDS0                      (GPIO2_CMOS1_LVDS0),
        .REFCLK_DISABLE                         (REFCLK_DISABLE),
        .DEVICE_TYPE                            (DEVICE_TYPE),
        .MATCH_CLOCKING                         (MATCH_CLOCKING),
        .IODELAY_GROUP                          (IODELAY_GROUP))
    i_tx_pclock (
        .rx_buf_clk                             (rx_buf_clk),
        .tx_ref_clk_p                           (tx_cssi_refclk_lssi_refclk_p),
        .tx_ref_clk_n                           (tx_cssi_nc_lssi_refclk_n),
        .tx_hs_clk                              (tx_hs_clk),
        .tx_clk                                 (tx_clk),
        .tx_clock_p                             (tx_cssi_clk_lssi_clk_p),
        .tx_clock_n                             (tx_cssi_nc_lssi_clk_n),
        .tx_rst_clk                             (tx_rst_clk),
        .tx_clk_resetn                          (tx_clk_resetn),
        .tx_serdes_resetn                       (tx_serdes_resetn),
        .axilite_clk                            (axilite_clk),
        .axilite_resetn                         (axilite_resetn),
        .axilite_wrdelay_int_c                  (axilite_wrdelay_int_c),
        .axilite_rddelay_int_c                  (axilite_rddelay_int_c),
        .axilite_wrdelay_ext_c                  (axilite_wrdelay_ext_c),
        .axilite_rddelay_ext_c                  (axilite_rddelay_ext_c),
        .axilite_wrdelay_c                      (axilite_wrdelay_c),
        .axilite_rddelay_c                      (axilite_rddelay_c),
        .axilite_buf_enable                     (axilite_buf_enable[0]),
        .axilite_clk_enb                        (axilite_clk_enb),
        .axilite_clk_msb                        (axilite_clk_msb),
        .axilite_clk_sel                        (axilite_clk_sel));

    axi_adrv9001_rst_cntrl #(
        .RESET_PIPE_LENGTH                      (RESET_PIPE_LENGTH))
    i_tx_rst_cntrl (
        .axilite_init_req                       (axilite_init_req),
        .axilite_clk_enb                        (axilite_clk_enb),
        .axilite_master_enable                  (axilite_master_enable),
        .mcs_clk_resetn                         (mcs_clk_resetn),
        .mcs_phy_resetn                         (mcs_phy_resetn),
        .mcs_dp_resetn                          (mcs_dp_resetn),
        .clk_resetn_in                          (tx_clk_resetn_in),
        .phy_resetn_in                          (tx_phy_resetn_in),
        .dp_resetn_in                           (tx_dp_resetn_in),
        .clk_resetn_out                         (tx_clk_resetn_out),
        .phy_resetn_out                         (tx_phy_resetn_out),
        .dp_resetn_out                          (tx_dp_resetn_out),
        .rst_clk                                (tx_rst_clk),
        .clk_resetn                             (tx_clk_resetn),
        .clk                                    (tx_clk),
        .cdc_resetn                             (tx_cdc_resetn),
        .serdes_resetn                          (tx_serdes_resetn),
        .serdes_data_resetn                     (),
        .resetn                                 (tx_resetn),
        .init_ack                               (tx_init_ack));

    generate
    if (IODELAY_CTRL_ENABLE == 0) begin

    assign delay_locked = 1'b1;

    end else if (DEVICE_TYPE == "ULTRASCALE") begin

    (* IODELAY_GROUP = IODELAY_GROUP *)
    IDELAYCTRL #(.SIM_DEVICE("ULTRASCALE")) i_delay_ctrl (
        .RST                                    (~sys_delay_resetn),
        .REFCLK                                 (sys_delay_clk),
        .RDY                                    (delay_locked));

    end else begin

    (* IODELAY_GROUP = IODELAY_GROUP *)
    IDELAYCTRL i_delay_ctrl (
        .RST                                    (~sys_delay_resetn),
        .REFCLK                                 (sys_delay_clk),
        .RDY                                    (delay_locked));

    end
    endgenerate

    cdc_cntrl #(.DATA_WIDTH(59)) i_tx_cdc_cntrl (
        .src_resetn                             (axilite_resetn),
        .src_clk                                (axilite_clk),
        .src_data                               ({axilite_ss_ratio,
                                                    axilite_ss_mode,
                                                    axilite_master_enable,
                                                    axilite_data_sel,
                                                    axilite_pattern,
                                                    axilite_pattern_pulse,
                                                    axilite_pattern_start,
                                                    axilite_q1_i0,
                                                    axilite_lsb1_msb0,
                                                    axilite_sdr1_ddr0,
                                                    axilite_mlane1_slane0,
                                                    axilite_lstrb1_pstrb0,
                                                    axilite_num_of_bits,
                                                    axilite_mcs_delay,
                                                    axilite_buf_enable[5:1]}),
        .dest_resetn                            (tx_cdc_resetn),
        .dest_clk                               (tx_clk),
        .dest_data                              ({tx_ss_ratio,
                                                    tx_ss_mode,
                                                    tx_master_enable,
                                                    tx_data_sel,
                                                    tx_pattern,
                                                    tx_pattern_pulse,
                                                    tx_pattern_start,
                                                    tx_q1_i0,
                                                    tx_lsb1_msb0,
                                                    tx_sdr1_ddr0,
                                                    tx_mlane1_slane0,
                                                    tx_lstrb1_pstrb0,
                                                    tx_num_of_bits,
                                                    tx_mcs_delay,
                                                    tx_buf_enable}));

    cdc_cntrl #(.DATA_WIDTH(49)) i_axilite_cdc_cntrl (
        .src_resetn                             (tx_cdc_resetn),
        .src_clk                                (tx_clk),
        .src_data                               ({tx_mcs_latency_count,
                                                    tx_pattern_done,
                                                    tx_pattern_latency}),
        .dest_resetn                            (axilite_resetn),
        .dest_clk                               (axilite_clk),
        .dest_data                              ({axilite_mcs_latency_count,
                                                    axilite_pattern_done,
                                                    axilite_pattern_latency}));


    cdc_event #(.DATA_WIDTH(1)) i_axilite_cdc_event (
        .src_resetn                             (tx_cdc_resetn),
        .src_clk                                (tx_clk),
        .src_data                               (tx_init_ack),
        .dest_resetn                            (axilite_resetn),
        .dest_clk                               (axilite_clk),
        .dest_data                              (axilite_init_ack));

    cdc #(.DATA_WIDTH(1)) i_axilite_cdc (
        .src_data                               (delay_locked),
        .dest_resetn                            (axilite_resetn),
        .dest_clk                               (axilite_clk),
        .dest_data                              (axilite_delay_locked));

    axi_adrv9001_ssi_regs #(.ID(ID)) i_tx_regs (
        .axilite_cmos_lvds_id                   (axilite_cmos_lvds_id),
        .axilite_num_of_bits                    (axilite_num_of_bits),
        .axilite_lstrb1_pstrb0                  (axilite_lstrb1_pstrb0),
        .axilite_mlane1_slane0                  (axilite_mlane1_slane0),
        .axilite_sdr1_ddr0                      (axilite_sdr1_ddr0),
        .axilite_lsb1_msb0                      (axilite_lsb1_msb0),
        .axilite_q1_i0                          (axilite_q1_i0),
        .axilite_fall1_rise0                    (),
        .axilite_cmos1_lvds0                    (axilite_cmos1_lvds0),
        .axilite_clk_sel                        (axilite_clk_sel),
        .axilite_clk_msb                        (axilite_clk_msb),
        .axilite_clk_enb                        (axilite_clk_enb),
        .axilite_buf_enable                     (axilite_buf_enable),
        .axilite_init                           (axilite_init_req),
        .axilite_init_clr                       (axilite_init_ack),
        .axilite_ss_ratio                       (axilite_ss_ratio),
        .axilite_obs_ss_ratio                   (),
        .axilite_ss_mode                        (axilite_ss_mode),
        .axilite_data_sel                       (axilite_data_sel),
        .axilite_data_pat_0                     (axilite_pattern),
        .axilite_data_pat_1                     (),
        .axilite_data_0                         (),
        .axilite_data_1                         (),
        .axilite_frm_oos                        ('d0),
        .axilite_intr_frm_oos                   (),
        .axilite_frm_err                        ('d0),
        .axilite_intr_frm_err                   (),
        .axilite_mon_oos                        ('d0),
        .axilite_intr_mon_oos                   (),
        .axilite_mon_err                        ('d0),
        .axilite_intr_mon_err                   (),
        .axilite_lb_ovf                         (axilite_lb_ovf),
        .axilite_intr_lb_ovf                    (),
        .axilite_lb_unf                         (axilite_lb_unf),
        .axilite_intr_lb_unf                    (),
        .axilite_align_sel                      (),
        .axilite_delay_unlocked                 (axilite_delay_unlocked),
        .axilite_wrdelay_00                     (axilite_wrdelay_0),
        .axilite_rddelay_00                     (axilite_rddelay_0),
        .axilite_wrdelay_01                     (axilite_wrdelay_1),
        .axilite_rddelay_01                     (axilite_rddelay_1),
        .axilite_wrdelay_02                     (axilite_wrdelay_2),
        .axilite_rddelay_02                     (axilite_rddelay_2),
        .axilite_wrdelay_03                     (axilite_wrdelay_3),
        .axilite_rddelay_03                     (axilite_rddelay_3),
        .axilite_wrdelay_04                     (axilite_wrdelay_s),
        .axilite_rddelay_04                     (axilite_rddelay_s),
        .axilite_wrdelay_05                     (axilite_wrdelay_c),
        .axilite_rddelay_05                     (axilite_rddelay_c),
        .axilite_wrdelay_06                     (axilite_wrdelay_int_c),
        .axilite_rddelay_06                     (axilite_rddelay_int_c),
        .axilite_wrdelay_07                     (axilite_wrdelay_ext_c),
        .axilite_rddelay_07                     (axilite_rddelay_ext_c),
        .axilite_delay_sweep                    (),
        .axilite_timer                          (),
        .axilite_clk_mon_count                  (axilite_clk_mon_count),
        .axilite_mcs_latency_count              (axilite_mcs_latency_count),
        .axilite_mcs_latency_busy               (1'd0),
        .axilite_mcs_delay_count                (axilite_mcs_delay),
        .axilite_mcs_strobe                     (32'd0),
        .axilite_mcs_data                       (32'd0),
        .axilite_mcs_bstrobe                    (32'd0),
        .axilite_mcs_bdata                      (32'd0),
        .axilite_data_pat_pulse                 (axilite_pattern_pulse),
        .axilite_data_pat_start                 (axilite_pattern_start),
        .axilite_data_pat_start_clr             (axilite_pattern_done),
        .axilite_data_pat_latency               (axilite_pattern_latency),
        .axilite_master_enable                  (axilite_master_enable),
        .axilite_obs_master_enable              (),
        .axilite_intr_frm_oos_enable            (),
        .axilite_intr_frm_err_enable            (),
        .axilite_intr_mon_oos_enable            (),
        .axilite_intr_mon_err_enable            (),
        .axilite_intr_lb_ovf_enable             (),
        .axilite_intr_lb_unf_enable             (),
        .axilite_clk                            (axilite_clk),
        .axilite_resetn                         (axilite_resetn),
        .axilite_wrreq                          (axilite_wrreq),
        .axilite_wraddr                         (axilite_wraddr),
        .axilite_wrdata                         (axilite_wrdata),
        .axilite_wrack                          (axilite_wrack),
        .axilite_rdreq                          (axilite_rdreq),
        .axilite_rdaddr                         (axilite_rdaddr),
        .axilite_rddata                         (axilite_rddata),
        .axilite_rdack                          (axilite_rdack));

endmodule

// **********************************************************************************
// **********************************************************************************
