#!/usr/bin/perl
## ##################################################################################
## ##################################################################################
## ----------------------------------------------------------------------------------
## ################
## ##   ###########   Analog Devices Inc.
## ##      ########
## ##         #####   Copyright (c) 2019 Analog Devices Inc. All rights reserved.
## ##            ##   This file is the confidential and proprietary property of ADI.
## ##         #####   Possession or use of this file requires a written license.
## ##      ########   The licensing information may be found at: www.analog.com
## ##   ###########
## ################
## ----------------------------------------------------------------------------------
## Description:       AXI_ADRV9001 TDD registers
## ----------------------------------------------------------------------------------
## ##################################################################################
## ##################################################################################

use autodie;
require 'dooku.pl';

$addr_width = 16;
$addr_width_used = 12;

$no_of_instances = 1;
$instance_base_addr = 0x8000;
$instance_offset_addr = 0x0;

setdooku(@ARGV);
module(0x0, 'axi_adrv9001_tdd');

## ##################################################################################
## ##################################################################################

define('OFFSET', 0x8000, 'TDD address offset');

## ##################################################################################
## ##################################################################################
## Timing Reference. The following registers control the timing of all TDD related
## signals. Hardware maintains the following two counters.
##
## FRAME-PERIOD-COUNT:
##    1.  The counter starts if **BOTH** the following conditions are true.
##            A.  Software sets (write 0x1) to the START bit. (NOTE: the START bit
##                must be read as 0x0, otherwise see below).
##          AND
##            B.  A trigger condition set by the TRIGGER_MODE bits must occur.
##                Hardware supports four trigger inputs. This allows synchronization,
##                as an example the TDD frame may optionally be started with a PPS
##                event or device MCS event.
##    2.  The counter resets if **EITHER** of the following conditions are true.
##            A.  The count has reached FRAME_PERIOD value.
##          OR
##            B.  The count has reached FRAME_SWITCH_PERIOD value **AND**
##                FRAME-COUNT has reached the FRAME_SWITCH_NUMBER value **AND**
##                FRAME_SWITCH_ENABLE bit is set.
##                This is a premature termination of the counter.
##    3.  The counter restarts if
##            A.  The count has reached its reset point (#2 above) **AND**
##                FRAME-COUNT has reached the NUM_OF_FRAMES value **AND**
##                SEQUENCE_REPEAT bit is set **AND**
##                STOP bit is **NOT** set.
##                This is the graceful termination of the counter.
##
## FRAME-COUNT:
##    The frame count follows the FRAME-PERIOD-COUNT, with the exception that on
##    premature termination of the FRAME-PERIOD-COUNT (#2 above) it does **NOT**
##    restart or stop, instead it advances to the next frame. That is, the frame
##    is **SHORTENED** but seuqnece continues.
## ##################################################################################
## ##################################################################################

register(0x000, 'REG_FRAME_PERIOD');
field(0, 32, RW, 0x0, 'FRAME_PERIOD', 'The number of clock cycles in the TDD frame');

register(0x004, 'REG_NUM_OF_FRAMES');
field(0, 32, RW, 0x0, 'NUM_OF_FRAMES', 'The number of frames in the TDD sequence');

register(0x008, 'REG_FRAME_SWITCH_PERIOD');
field(0, 32, RW, 0x0, 'FRAME_SWITCH_PERIOD', 'The clock cycles value for premature
    termination of the current frame.');

register(0x00c, 'REG_FRAME_SWITCH_NUMBER');
field(0, 32, RW, 0x0, 'FRAME_SWITCH_NUMBER', 'The frame number for premature
    termination of the current clock count. This will be a shortened sequence.');

$field_base = 0;
$field_offset = 2;
$field_width = 2;
$field_type = "RW";
$field_default = 0x0;
$field_count = 4;
$field_index = 0;

register(0x010, 'REG_TRIGGERS');
for ($field_index = 0; $field_index < $field_count; $field_index++) {
    field_loop("TRIGGER_MODE_%d", "The core may optionally be controlled through
        the trigger inputs. There are five programmable trigger inputs, these bits
        control each one of the triggers independently. The data transfer is controlled
        as an AND function of all the trigger modes. In all modes, the duration of data
        transfer is controlled by the length.");
}

define('TRIGGER_DISABLED',      0x0, 'Trigger disabled');
define('TRIGGER_RISING_EDGE',   0x1, 'Time reference begins at rising edge of this input');
define('TRIGGER_FALLING_EDGE',  0x2, 'Time reference begins at falling edge of this input');
define('TRIGGER_EDGE',          0x3, 'Time reference begins at either edge of this input');

register(0x020, 'REG_SEQUENCE_CONTROL');
field(0, 1, RW1HC, 0x0, 'START', 'If set, indicates a request to start the TDD
    sequence, all sequence parameters above must be programmed first before setting
    this bit. Hardware clears this bit once the sequence is completed OR a stop
    request is initiated.');
field(1, 1, RW1HC, 0x0, 'STOP', 'If set, indicates a request to stop the current
    sequence, Hardware clears this bit once the sequence is stopped.');
field(2, 1, RW, 0x0, 'SEQUENCE_REPEAT', 'If set, the current sequence is repeated
    until a STOP request is received.');
field(3, 1, RW, 0x0, 'FRAME_SWITCH_ENABLE', 'If set, a sequence is shortened and the
    set frame is prematurely terminated.');

register(0x024, 'REG_SWRESET');
field(0, 1, RW1HC, 0x0, 'SWRESET', 'If set, the counters are reset. Hardware clears
    this bit once the internal reset is completed.');

register(0x028, 'REG_TIMER');
field(0, 32, RW, 0x0, 'TIMER', 'General purpose timer register, this is a count
    down timer, software may write desired interval and wait for it to reach zero.
    The count can be prematurely terminated, runs at AXILite clock (usually 100MHz).');

register(0x030, 'REG_CLK_MON');
field(0, 32, RO, 0x0, 'CLK_MON_COUNT', 'Clock monitor frequency count. If read 0x0,
    indicates a loss of clock. The format is 16.16 x 100MHz');

register(0x040, 'REG_GLOBAL_ENB');
field(0, 1, RW, 0x1, 'GLOBAL_ENB', 'If set allows global control of all enables
    if set to 0x0, all enables are asserted low, if set to 0x1, enables are asserted
    as per their individual setting.');

register(0x050, 'REG_CDC_STATE');
field(0, 1, RW1HC, 0x0, 'CDC_STATE', 'If set, cleared when CDC transfer is complete.');

## ##################################################################################
## ##################################################################################
## Enable Generation. The individual enables below are all controlled by a set of
## programmable registers as noted below.
##
##  The Enable signal is ASSERTED at a TDD clock cycle if:
##      1.  FRAME-PERIOD-COUNT has reached the PRIMARY_ASSERT value **AND**
##          FRAME-COUNT has reached the PRIMARY_FRAME_ASSERT value
##    OR
##      2.  FRAME-PERIOD-COUNT has reached the SECONDARY_ASSERT value **AND**
##          FRAME-COUNT has reached the SECONDARY_FRAME_ASSERT value
##
##  The Enable signal is DEASSERTED at a TDD clock cycle if:
##      1.  FRAME-PERIOD-COUNT has reached the PRIMARY_DEASSERT value **AND**
##          FRAME-COUNT has reached the PRIMARY_FRAME_DEASSERT value
##    OR
##      2.  FRAME-PERIOD-COUNT has reached the SECONDARY_DEASSERT value **AND**
##          FRAME-COUNT has reached the SECONDARY_FRAME_DEASSERT value
## ##################################################################################
## ##################################################################################

define('ENABLE_AUTO',       0x2, 'Enable modes');
define('ENABLE_HIGH',       0x1, 'Enable modes');
define('ENABLE_LOW',        0x0, 'Enable modes');

define('TRIG_START_INDEX',  0x00, 'Start index for triggers');
define('RX0_DMA_TRIG',      0x00, 'Enable identifiers');
define('TX0_DMA_TRIG',      0x01, 'Enable identifiers');
define('RX1_DMA_TRIG',      0x02, 'Enable identifiers');
define('TX1_DMA_TRIG',      0x03, 'Enable identifiers');
define('ORX0_DMA_TRIG',     0x04, 'Enable identifiers');
define('ORX1_DMA_TRIG',     0x05, 'Enable identifiers');
define('EXT0_TRIG',         0x06, 'Enable identifiers');
define('EXT1_TRIG',         0x07, 'Enable identifiers');
define('SPI_TRIG',          0x08, 'Enable identifiers');
define('GPIO_TRIG',         0x09, 'Enable identifiers');

define('DEV_START_INDEX',   0x0a, 'Start index for device signals');
define('RX0_DEVICE',        0x0a, 'Enable identifiers');
define('TX0_DEVICE',        0x0b, 'Enable identifiers');
define('RX1_DEVICE',        0x0c, 'Enable identifiers');
define('TX1_DEVICE',        0x0d, 'Enable identifiers');
define('ORX0_GPIO',         0x0e, 'Enable identifiers');
define('ORX1_GPIO',         0x0f, 'Enable identifiers');
define('TX0_GPIO',          0x10, 'Enable identifiers');
define('TX1_GPIO',          0x11, 'Enable identifiers');
define('IFRM0CTL0_GPIO',    0x12, 'Enable identifiers');
define('IFRM0CTL1_GPIO',    0x13, 'Enable identifiers');
define('IFRM1CTL0_GPIO',    0x14, 'Enable identifiers');
define('IFRM1CTL1_GPIO',    0x15, 'Enable identifiers');

define('DMA_START_INDEX',   0x16, 'Start index for dma signals');
define('RX0_DMA_DATA',      0x16, 'Enable identifiers');
define('TX0_DMA_DATA',      0x17, 'Enable identifiers');
define('RX1_DMA_DATA',      0x18, 'Enable identifiers');
define('TX1_DMA_DATA',      0x19, 'Enable identifiers');
define('ORX0_DMA_DATA',     0x1a, 'Enable identifiers');
define('ORX1_DMA_DATA',     0x1b, 'Enable identifiers');

$register_offset = 0x40;
$register_count = 0x1c;
$register_index = 0;

for ($register_index = 0; $register_index < $register_count; $register_index++) {

    $register_base = 0x100;
    register_loop("REG_PRIMARY_ASSERT_%02d");
    register_loop_field(0, 32, RW, 0x0, "PRIMARY_ASSERT_%02d", "Primary, clock count- high.");
    
    $register_base = 0x104;
    register_loop("REG_PRIMARY_DEASSERT_%02d");
    register_loop_field(0, 32, RW, 0x0, "PRIMARY_DEASSERT_%02d", "Primary, clock count- low.");
    
    $register_base = 0x108;
    register_loop("REG_SECONDARY_ASSERT_%02d");
    register_loop_field(0, 32, RW, 0x0, "SECONDARY_ASSERT_%02d", "Secondary, clock count- high.");
    
    $register_base = 0x10c;
    register_loop("REG_SECONDARY_DEASSERT_%02d");
    register_loop_field(0, 32, RW, 0x0, "SECONDARY_DEASSERT_%02d", "Secondary, clock count- low.");
    
    $register_base = 0x110;
    register_loop("REG_PRIMARY_FRAME_ASSERT_%02d");
    register_loop_field(0, 32, RW, 0x0, "PRIMARY_FRAME_ASSERT_%02d", "Primary, frame count- high.");
    
    $register_base = 0x114;
    register_loop("REG_PRIMARY_FRAME_DEASSERT_%02d");
    register_loop_field(0, 32, RW, 0x0, "PRIMARY_FRAME_DEASSERT_%02d", "Primary, frame count- low.");
    
    $register_base = 0x118;
    register_loop("REG_SECONDARY_FRAME_ASSERT_%02d");
    register_loop_field(0, 32, RW, 0x0, "SECONDARY_FRAME_ASSERT_%02d", "Secondary, frame count- high.");
    
    $register_base = 0x11c;
    register_loop("REG_SECONDARY_FRAME_DEASSERT_%02d");
    register_loop_field(0, 32, RW, 0x0, "SECONDARY_FRAME_DEASSERT_%02d", "Secondary, frame count- low.");
    
    $register_base = 0x120;
    register_loop("REG_MODE_%02d");
    register_loop_field(0,  2, RW, 0x0, "MODE_%02d", "Software or hardware mode.");

    $register_base = 0x124;
    register_loop("REG_SYNC_%02d");
    register_loop_field(0,  1, RW, 0x0, "SYNC_%02d", "Interface clock synchronous select.");
    
    $register_base = 0x128;
    register_loop("REG_CLK_RATIO_%02d");
    register_loop_field(0, 12, RO, 0x0, "CLK_RATIO_%02d", "Interface clock to TDD clock ratio.");
}

## ##################################################################################
## ##################################################################################
## NOTES:
##
##  1.  All counters start from 0x0 and upto and including the programmed values.
##        That is, program N-1 to all counters.
##  2.  If start bit is already set, setting it again will NOT restart the TDD
##        sequence, it is just ignored. Ideally if start bit is set, set STOP bit
##        the wait for both to clear and then proceed.
##  3.  If stop bit is already set, setting it again has no effect, if a reasonable
##        amount of time has passed and the bit doesn't self clear, reset is the
##        only option.
##  4.  There are no individual counter enable/disable bits. Instead, to disable the
##        enables, set the values to 0x0 (or same).
##  5.  It is possible to set DEASSERT values to be less than ASSERT values, this
##        will extend the enable over the frame boundary. However, be careful at the
##        sequence boundaries.
##  6.  Similar to #5 above, be careful about frame switch and premature termination.
##        Avoid (unless intentional) switching that masks assertion/de-assertion.
## ##################################################################################
## ##################################################################################

endregisters();
endmodule();
generate();

## ##################################################################################
## ##################################################################################
