// **********************************************************************************
// **********************************************************************************
// ----------------------------------------------------------------------------------
// ################
// ##   ###########   Analog Devices Inc.
// ##      ########
// ##         #####   Copyright (c) 2019 Analog Devices Inc. All rights reserved.
// ##            ##   This file is the confidential and proprietary property of ADI.
// ##         #####   Possession or use of this file requires a written license.
// ##      ########   The licensing information may be found at: www.analog.com
// ##   ###########
// ################
// ----------------------------------------------------------------------------------
// Description:       AXI_ADRV9001, receive interface single channel
// ----------------------------------------------------------------------------------
// **********************************************************************************
// **********************************************************************************

`timescale 1ps/1ps

module axi_adrv9001_rx_if #(

    parameter GPIO2_CMOS1_LVDS0 = 0,
    parameter PN_SWAP = 0,
    parameter DEVICE_TYPE = "ULTRASCALE",
    parameter IODELAY_GROUP = "IODELAY_GROUP") (

    // system interface

    input   wire            rx_resetn,
    input   wire            rx_hs_clk,
    input   wire            rx_clk,
    input   wire            rx_align,
    input   wire            rx_data_p,
    input   wire            rx_data_n,
    output  reg   [  7:0]   rx_pdata = 'd0,
    output  wire            rx_gpio_in_p,
    output  wire            rx_gpio_in_n,
    input   wire  [  2:0]   rx_mcs_delay,

    // axilite interface

    input   wire            axilite_resetn,
    input   wire            axilite_clk,
    input   wire  [  4:0]   axilite_delay_wrdata,
    output  wire  [  4:0]   axilite_delay_rddata);

    // internal registers
 
    reg                     axilite_delay_rdwr = 'd0;
    reg           [  4:0]   axilite_delay_wrdata_d = 'd0;
    reg                     rx_delay_enable = 'd0;
    reg                     rx_delay_rdwr = 'd0;
    reg           [  4:0]   rx_delay_wrdata_d = 'd0;
    reg           [  7:0]   rx_mdata_d = 8'd0;

    // internal signals
 
    wire          [  7:0]   rx_qdata;
    wire          [  7:0]   rx_mdata;
    wire                    rx_ddata;
    wire                    rx_sdata;
    wire                    rx_readn;
    wire          [  4:0]   rx_delay_wrdata;
    wire          [  4:0]   rx_delay_rddata;
    wire          [  3:0]   unused;

    // load delay value

    always @(negedge axilite_resetn or posedge axilite_clk) begin
        if (axilite_resetn == 1'b0) begin
            axilite_delay_wrdata_d <= 5'd0;
            axilite_delay_rdwr <= 1'b0;
        end else begin
            axilite_delay_wrdata_d <= axilite_delay_wrdata;
            if (axilite_delay_wrdata_d == axilite_delay_wrdata) begin
                axilite_delay_rdwr <= 1'b0;
            end else begin
                axilite_delay_rdwr <= 1'b1;
            end
        end
    end

    always @(negedge rx_resetn or posedge rx_clk) begin
        if (rx_resetn == 1'b0) begin
            rx_delay_wrdata_d <= 5'd0;
            rx_delay_enable <= 1'd0;
            rx_delay_rdwr <= 1'b0;
        end else begin
            rx_delay_wrdata_d <= rx_delay_wrdata;
            rx_delay_enable <= rx_delay_rdwr;
            if (rx_delay_wrdata_d == rx_delay_wrdata) begin
                rx_delay_rdwr <= 1'b0;
            end else begin
                rx_delay_rdwr <= 1'b1;
            end
        end
    end

    always @(negedge rx_resetn or posedge rx_clk) begin
        if (rx_resetn == 1'b0) begin
            rx_mdata_d <= 8'd0;
            rx_pdata <= 8'd0;
        end else begin
            rx_mdata_d <= rx_mdata;
            case (rx_mcs_delay)
                3'd7: rx_pdata <= {rx_mdata_d[6:0], rx_mdata[7:7]};
                3'd6: rx_pdata <= {rx_mdata_d[5:0], rx_mdata[7:6]};
                3'd5: rx_pdata <= {rx_mdata_d[4:0], rx_mdata[7:5]};
                3'd4: rx_pdata <= {rx_mdata_d[3:0], rx_mdata[7:4]};
                3'd3: rx_pdata <= {rx_mdata_d[2:0], rx_mdata[7:3]};
                3'd2: rx_pdata <= {rx_mdata_d[1:0], rx_mdata[7:2]};
                3'd1: rx_pdata <= {rx_mdata_d[0:0], rx_mdata[7:1]};
                default: rx_pdata <= rx_mdata;
            endcase
        end
    end

    // instantiations

    assign rx_mdata = ((GPIO2_CMOS1_LVDS0 == 0) && (PN_SWAP == 1)) ? ~rx_qdata : rx_qdata;

    generate
    if (GPIO2_CMOS1_LVDS0 == 2) begin

    assign rx_qdata = 8'd0;
    assign rx_gpio_in_p = 1'd0;
    assign rx_readn = 1'd1;

    end else if (DEVICE_TYPE == "ULTRASCALE") begin

    assign rx_gpio_in_p = 1'd0;

    ISERDESE3 #(
        .SIM_DEVICE             ("ULTRASCALE_PLUS"),
        .DATA_WIDTH             (8),
        .FIFO_ENABLE            ("FALSE"),
        .FIFO_SYNC_MODE         ("FALSE"),
        .IS_CLK_B_INVERTED      (1'b1),
        .IS_CLK_INVERTED        (1'b0),
        .IS_RST_INVERTED        (1'b0))
    i_serdes (
        .CLK                    (rx_hs_clk),
        .CLK_B                  (rx_hs_clk),
        .CLKDIV                 (rx_clk),
        .FIFO_RD_CLK            (rx_clk),
        .INTERNAL_DIVCLK        (),
        .RST                    (~rx_resetn),
        .FIFO_EMPTY             (rx_readn),
        .FIFO_RD_EN             (~rx_readn),
        .D                      (rx_ddata),
        .Q                      ({rx_qdata[0],
                                    rx_qdata[1],
                                    rx_qdata[2],
                                    rx_qdata[3],
                                    rx_qdata[4],
                                    rx_qdata[5],
                                    rx_qdata[6],
                                    rx_qdata[7]}));

    end else begin

    assign rx_readn = 1'd1;

    ISERDESE2 #(
        .DATA_RATE              ("DDR"),
        .DATA_WIDTH             (8),
        .INTERFACE_TYPE         ("NETWORKING"),
        .DYN_CLKDIV_INV_EN      ("FALSE"),
        .DYN_CLK_INV_EN         ("FALSE"),
        .NUM_CE                 (2),
        .OFB_USED               ("FALSE"),
        .IOBDELAY               ("BOTH"),
        .SERDES_MODE            ("MASTER"))
    i_serdes (
        .CLK                    (rx_hs_clk),
        .CLKB                   (~rx_hs_clk),
        .CLKDIV                 (rx_clk),
        .RST                    (~rx_resetn),
        .BITSLIP                (rx_align),
        .DDLY                   (rx_ddata),
        .Q1                     (rx_qdata[0]),
        .Q2                     (rx_qdata[1]),
        .Q3                     (rx_qdata[2]),
        .Q4                     (rx_qdata[3]),
        .Q5                     (rx_qdata[4]),
        .Q6                     (rx_qdata[5]),
        .Q7                     (rx_qdata[6]),
        .Q8                     (rx_qdata[7]),
        .CE1                    (1'b1),
        .CE2                    (1'b1),
        .CLKDIVP                (1'b0),
        .D                      (rx_sdata),
        .SHIFTIN1               (1'b0),
        .SHIFTIN2               (1'b0),
        .DYNCLKDIVSEL           (1'b0),
        .DYNCLKSEL              (1'b0),
        .OFB                    (1'b0),
        .OCLK                   (1'b0),
        .OCLKB                  (1'b0),
        .SHIFTOUT1              (),
        .SHIFTOUT2              (),
        .O                      (rx_gpio_in_p));

    end
    endgenerate

    // io delay

    generate
    if (GPIO2_CMOS1_LVDS0 == 2) begin

    assign rx_ddata = 1'd0;
    assign rx_delay_wrdata = 5'd0;
    assign rx_delay_rddata = 5'd0;
    assign axilite_delay_rddata = 5'd0;
    assign unused = 4'd0;

    end else if (DEVICE_TYPE == "ULTRASCALE") begin

    (* IODELAY_GROUP = IODELAY_GROUP *)
    IDELAYE3 # (
        .SIM_DEVICE             ("ULTRASCALE_PLUS"),
        .CASCADE                ("NONE"),
        .DELAY_FORMAT           ("COUNT"),
        .DELAY_TYPE             ("VAR_LOAD"),
        .DELAY_SRC              ("IDATAIN"),
        .DELAY_VALUE            (0),
        .IS_CLK_INVERTED        (1'b0),
        .IS_RST_INVERTED        (1'b0),
        .REFCLK_FREQUENCY       (300.0),
        .UPDATE_MODE            ("ASYNC"))
    i_delay (
        .CASC_IN                (1'd0),
        .CASC_RETURN            (1'd0),
        .CASC_OUT               (),
        .INC                    (1'd0),
        .DATAIN                 (1'd0),
        .EN_VTC                 (1'd0),
        .RST                    (1'd0),
        .CLK                    (rx_clk),
        .IDATAIN                (rx_sdata),
        .DATAOUT                (rx_ddata),
        .CE                     (rx_delay_enable),
        .LOAD                   (rx_delay_rdwr),
        .CNTVALUEIN             ({rx_delay_wrdata, 4'd0}),
        .CNTVALUEOUT            ({rx_delay_rddata, unused}));

    cdc #(.DATA_WIDTH(5)) i_rx_cdc (
        .src_data               (axilite_delay_wrdata),
        .dest_resetn            (rx_resetn),
        .dest_clk               (rx_clk),
        .dest_data              (rx_delay_wrdata));

    cdc #(.DATA_WIDTH(5)) i_axilite_cdc (
        .src_data               (rx_delay_rddata),
        .dest_resetn            (axilite_resetn),
        .dest_clk               (axilite_clk),
        .dest_data              (axilite_delay_rddata));

    end else begin

    assign rx_delay_wrdata = 5'd0;
    assign rx_delay_rddata = 5'd0;
    assign unused = 4'd0;

    (* IODELAY_GROUP = IODELAY_GROUP *)
    IDELAYE2 # (
        .CINVCTRL_SEL           ("FALSE"),
        .DELAY_SRC              ("IDATAIN"),
        .HIGH_PERFORMANCE_MODE  ("FALSE"),
        .IDELAY_TYPE            ("VAR_LOAD"),
        .IDELAY_VALUE           (0),
        .REFCLK_FREQUENCY       (200.0),
        .PIPE_SEL               ("FALSE"),
        .SIGNAL_PATTERN         ("DATA"))
    i_delay (
        .CE                     (1'b0),
        .INC                    (1'b0),
        .DATAIN                 (1'b0),
        .LDPIPEEN               (1'b0),
        .CINVCTRL               (1'b0),
        .REGRST                 (1'b0),
        .IDATAIN                (rx_sdata),
        .DATAOUT                (rx_ddata),
        .C                      (axilite_clk),
        .LD                     (axilite_delay_rdwr),
        .CNTVALUEIN             (axilite_delay_wrdata),
        .CNTVALUEOUT            (axilite_delay_rddata));

    end
    endgenerate

    // io buffers
 
    assign rx_gpio_in_n = (GPIO2_CMOS1_LVDS0 == 0) ? 1'd0 : rx_data_n;

    generate
    if (GPIO2_CMOS1_LVDS0 == 0) begin

    IBUFDS i_ibuf (
        .I                      (rx_data_p),
        .IB                     (rx_data_n),
        .O                      (rx_sdata));

    end else if (GPIO2_CMOS1_LVDS0 == 1) begin

    IBUF i_ibuf (
        .I                      (rx_data_p),
        .O                      (rx_sdata));

    end else begin

    assign rx_sdata = 1'd0;

    end
    endgenerate

endmodule

// **********************************************************************************
// **********************************************************************************
