// **********************************************************************************
// **********************************************************************************
// ----------------------------------------------------------------------------------
// ################
// ##   ###########   Analog Devices Inc.
// ##      ########
// ##         #####   Copyright (c) 2019 Analog Devices Inc. All rights reserved.
// ##            ##   This file is the confidential and proprietary property of ADI.
// ##         #####   Possession or use of this file requires a written license.
// ##      ########   The licensing information may be found at: www.analog.com
// ##   ###########
// ################
// ----------------------------------------------------------------------------------
// Description:       AXI_ADRV9001, receive data monitor
// ----------------------------------------------------------------------------------
// **********************************************************************************
// **********************************************************************************

`timescale 1ps/1ps

module axi_adrv9001_rx_dmon (

    // receive interface

    input   wire            rx_clk,
    input   wire            rx_resetn,
    input   wire            rx_valid,
    input   wire  [ 63:0]   rx_data,
    input   wire  [ 63:0]   rx_pattern,
    input   wire  [  3:0]   rx_data_sel,
    input   wire  [  2:0]   rx_num_of_bits,
    output  reg   [ 63:0]   rx_mon_data = 'd0,
    output  reg             rx_mon_oos = 'd0,
    output  reg             rx_mon_err = 'd0);

    // internal registers
 
    reg                     rx_valid_r = 'd0;
    reg           [ 63:0]   rx_data_r = 'd0;
    reg           [ 63:0]   rx_data_e = 'd0;
    reg           [ 63:0]   rx_ramp64_e = 'd0;
    reg           [ 63:0]   rx_nibb32_e = 'd0;
    reg           [ 63:0]   rx_ramp32_e = 'd0;
    reg           [ 63:0]   rx_pn1532_e = 'd0;
    reg           [ 63:0]   rx_pn0732_e = 'd0;
    reg           [ 63:0]   rx_ramp24_e = 'd0;
    reg           [ 63:0]   rx_ramp16_e = 'd0;
    reg           [ 63:0]   rx_ramp08_e = 'd0;
    reg           [ 63:0]   rx_ramp02_e = 'd0;
    reg           [  3:0]   rx_mon_cnt = 'd0;

    // internal signals

    wire          [ 63:0]   rx_patt64_e;
    wire          [ 63:0]   rx_ramp64_in;
    wire          [ 63:0]   rx_patt32_e;
    wire          [ 63:0]   rx_nibb32_in;
    wire          [ 63:0]   rx_ramp32_in;
    wire          [ 63:0]   rx_pn1532_in;
    wire          [ 63:0]   rx_pn1532_fn;
    wire          [ 63:0]   rx_pn0732_in;
    wire          [ 63:0]   rx_pn0732_fn;
    wire          [ 63:0]   rx_patt24_e;
    wire          [ 63:0]   rx_ramp24_in;
    wire          [ 63:0]   rx_patt16_e;
    wire          [ 63:0]   rx_ramp16_in;
    wire          [ 63:0]   rx_patt08_e;
    wire          [ 63:0]   rx_ramp08_in;
    wire          [ 63:0]   rx_patt02_e;
    wire          [ 63:0]   rx_ramp02_in;
    wire                    rx_match;
    wire                    rx_update;

    // pn7 function

    function [31:0] pn7;
        input [31:0] din;
        reg   [31:0] dout;
        begin
            dout[31] = din[20] ^ din[18] ^ din[19] ^ din[17];
            dout[30] = din[19] ^ din[17] ^ din[18] ^ din[16];
            dout[29] = din[18] ^ din[16] ^ din[17] ^ din[22] ^ din[21];
            dout[28] = din[17] ^ din[22] ^ din[16] ^ din[20];
            dout[27] = din[16] ^ din[22] ^ din[19];
            dout[26] = din[22] ^ din[18];
            dout[25] = din[21] ^ din[17];
            dout[24] = din[20] ^ din[16];
            dout[23] = din[19] ^ din[22] ^ din[21];
            dout[22] = din[18] ^ din[21] ^ din[20];
            dout[21] = din[17] ^ din[20] ^ din[19];
            dout[20] = din[16] ^ din[19] ^ din[18];
            dout[19] = din[22] ^ din[18] ^ din[21] ^ din[17];
            dout[18] = din[21] ^ din[17] ^ din[20] ^ din[16];
            dout[17] = din[20] ^ din[16] ^ din[19] ^ din[22] ^ din[21];
            dout[16] = din[19] ^ din[22] ^ din[18] ^ din[20];
            dout[15] = din[22] ^ din[21];
            dout[14] = din[21] ^ din[20];
            dout[13] = din[20] ^ din[19];
            dout[12] = din[19] ^ din[18];
            dout[11] = din[18] ^ din[17];
            dout[10] = din[17] ^ din[16];
            dout[ 9] = din[16] ^ din[22] ^ din[21];
            dout[ 8] = din[22] ^ din[20];
            dout[ 7] = din[21] ^ din[19];
            dout[ 6] = din[20] ^ din[18];
            dout[ 5] = din[19] ^ din[17];
            dout[ 4] = din[18] ^ din[16];
            dout[ 3] = din[17] ^ din[22] ^ din[21];
            dout[ 2] = din[16] ^ din[21] ^ din[20];
            dout[ 1] = din[22] ^ din[20] ^ din[21] ^ din[19];
            dout[ 0] = din[21] ^ din[19] ^ din[20] ^ din[18];
            pn7 = dout;
        end
    endfunction

    // pn15 function

    function [31:0] pn15;
        input [31:0] din;
        reg   [31:0] dout;
        begin
            dout[31] = din[29] ^ din[27];
            dout[30] = din[28] ^ din[26];
            dout[29] = din[27] ^ din[25];
            dout[28] = din[26] ^ din[24];
            dout[27] = din[25] ^ din[23];
            dout[26] = din[24] ^ din[22];
            dout[25] = din[23] ^ din[21];
            dout[24] = din[22] ^ din[20];
            dout[23] = din[21] ^ din[19];
            dout[22] = din[20] ^ din[18];
            dout[21] = din[19] ^ din[17];
            dout[20] = din[18] ^ din[16];
            dout[19] = din[17] ^ din[30] ^ din[29];
            dout[18] = din[16] ^ din[29] ^ din[28];
            dout[17] = din[30] ^ din[28] ^ din[29] ^ din[27];
            dout[16] = din[29] ^ din[27] ^ din[28] ^ din[26];
            dout[15] = din[30] ^ din[29];
            dout[14] = din[29] ^ din[28];
            dout[13] = din[28] ^ din[27];
            dout[12] = din[27] ^ din[26];
            dout[11] = din[26] ^ din[25];
            dout[10] = din[25] ^ din[24];
            dout[ 9] = din[24] ^ din[23];
            dout[ 8] = din[23] ^ din[22];
            dout[ 7] = din[22] ^ din[21];
            dout[ 6] = din[21] ^ din[20];
            dout[ 5] = din[20] ^ din[19];
            dout[ 4] = din[19] ^ din[18];
            dout[ 3] = din[18] ^ din[17];
            dout[ 2] = din[17] ^ din[16];
            dout[ 1] = din[16] ^ din[30] ^ din[29];
            dout[ 0] = din[30] ^ din[28];
            pn15 = dout;
        end
    endfunction

    // receive data sources
 
    always @(posedge rx_clk) begin
        if (rx_valid == 1'b1) begin
            rx_mon_data <= rx_data;
        end
    end

    always @(posedge rx_clk) begin
        rx_valid_r <= rx_valid;
        rx_data_r <= rx_data;
        case ({rx_num_of_bits, rx_data_sel})
            7'b1010011: rx_data_e <= rx_patt64_e;
            7'b1010101: rx_data_e <= rx_ramp64_e;
            7'b1000011: rx_data_e <= rx_patt32_e;
            7'b1000100: rx_data_e <= rx_nibb32_e;
            7'b1000101: rx_data_e <= rx_ramp32_e;
            7'b1000110: rx_data_e <= rx_pn1532_e;
            7'b1000111: rx_data_e <= rx_pn0732_e;
            7'b0110011: rx_data_e <= rx_patt24_e;
            7'b0110101: rx_data_e <= rx_ramp24_e;
            7'b0100011: rx_data_e <= rx_patt16_e;
            7'b0100101: rx_data_e <= rx_ramp16_e;
            7'b0010011: rx_data_e <= rx_patt08_e;
            7'b0010101: rx_data_e <= rx_ramp08_e;
            7'b0000011: rx_data_e <= rx_patt02_e;
            7'b0000101: rx_data_e <= rx_ramp02_e;
            default: rx_data_e <= {64{1'b1}};
        endcase
    end

    // 64bit (32I, 32Q) mode
 
    assign rx_patt64_e[63:0] = rx_pattern;

    assign rx_ramp64_in = (rx_mon_oos == 1'b0) ? rx_ramp64_e : rx_data;

    always @(posedge rx_clk) begin
        if (rx_valid == 1'b1) begin
            rx_ramp64_e[63:48] <= rx_ramp64_in[15:0] + 1'h1;
            rx_ramp64_e[47:32] <= rx_ramp64_in[15:0] + 1'h1;
            rx_ramp64_e[31:16] <= rx_ramp64_in[15:0] + 1'h1;
            rx_ramp64_e[15: 0] <= rx_ramp64_in[15:0] + 1'h1;
        end
    end

    // 32bit (16I, 16Q) mode
 
    assign rx_patt32_e[63:32] = rx_pattern[31:0];
    assign rx_patt32_e[31: 0] = rx_pattern[31:0];

    assign rx_nibb32_in = (rx_mon_oos == 1'b0) ? rx_nibb32_e : rx_data;

    always @(posedge rx_clk) begin
        if (rx_valid == 1'b1) begin
            rx_nibb32_e[63:60] <= rx_nibb32_in[3:0] + 2'h3;
            rx_nibb32_e[59:56] <= rx_nibb32_in[3:0] + 2'h3;
            rx_nibb32_e[55:52] <= rx_nibb32_in[3:0] + 2'h3;
            rx_nibb32_e[51:48] <= rx_nibb32_in[3:0] + 2'h3;
            rx_nibb32_e[47:44] <= rx_nibb32_in[3:0] + 2'h3;
            rx_nibb32_e[43:40] <= rx_nibb32_in[3:0] + 2'h3;
            rx_nibb32_e[39:36] <= rx_nibb32_in[3:0] + 2'h3;
            rx_nibb32_e[35:32] <= rx_nibb32_in[3:0] + 2'h3;
            rx_nibb32_e[31:28] <= rx_nibb32_in[3:0] + 2'h2;
            rx_nibb32_e[27:24] <= rx_nibb32_in[3:0] + 2'h2;
            rx_nibb32_e[23:20] <= rx_nibb32_in[3:0] + 2'h2;
            rx_nibb32_e[19:16] <= rx_nibb32_in[3:0] + 2'h2;
            rx_nibb32_e[15:12] <= rx_nibb32_in[3:0] + 2'h2;
            rx_nibb32_e[11: 8] <= rx_nibb32_in[3:0] + 2'h2;
            rx_nibb32_e[ 7: 4] <= rx_nibb32_in[3:0] + 2'h2;
            rx_nibb32_e[ 3: 0] <= rx_nibb32_in[3:0] + 2'h2;
        end
    end

    assign rx_ramp32_in = (rx_mon_oos == 1'b0) ? rx_ramp32_e : rx_data;

    always @(posedge rx_clk) begin
        if (rx_valid == 1'b1) begin
            rx_ramp32_e[63:48] <= rx_ramp32_in[15:0] + 2'h3;
            rx_ramp32_e[47:32] <= rx_ramp32_in[15:0] + 2'h3;
            rx_ramp32_e[31:16] <= rx_ramp32_in[15:0] + 2'h2;
            rx_ramp32_e[15: 0] <= rx_ramp32_in[15:0] + 2'h2;
        end
    end

    assign rx_pn1532_in = (rx_mon_oos == 1'b0) ? rx_pn1532_e :
        ((rx_data == 64'd0) ? ~rx_data : rx_data);
    assign rx_pn1532_fn = pn15({rx_pn1532_in[47:32], rx_pn1532_in[15:0]});

    always @(posedge rx_clk) begin
        if (rx_valid == 1'b1) begin
            rx_pn1532_e[63:48] <= rx_pn1532_fn[31:16];
            rx_pn1532_e[47:32] <= rx_pn1532_fn[31:16];
            rx_pn1532_e[31:16] <= rx_pn1532_fn[15: 0];
            rx_pn1532_e[15: 0] <= rx_pn1532_fn[15: 0];
        end
    end

    assign rx_pn0732_in = (rx_mon_oos == 1'b0) ? rx_pn0732_e :
        ((rx_data == 64'd0) ? ~rx_data : rx_data);
    assign rx_pn0732_fn = pn7({rx_pn0732_in[47:32], rx_pn0732_in[15:0]});

    always @(posedge rx_clk) begin
        if (rx_valid == 1'b1) begin
            rx_pn0732_e[63:48] <= rx_pn0732_fn[31:16];
            rx_pn0732_e[47:32] <= rx_pn0732_fn[31:16];
            rx_pn0732_e[31:16] <= rx_pn0732_fn[15: 0];
            rx_pn0732_e[15: 0] <= rx_pn0732_fn[15: 0];
        end
    end

    // 24bit (12I, 12Q) mode
 
    assign rx_patt24_e[63:48] = {4'd0, rx_pattern[27:16]};
    assign rx_patt24_e[47:32] = {4'd0, rx_pattern[11: 0]};
    assign rx_patt24_e[31:16] = {4'd0, rx_pattern[27:16]};
    assign rx_patt24_e[15: 0] = {4'd0, rx_pattern[11: 0]};

    assign rx_ramp24_in = (rx_mon_oos == 1'b0) ? rx_ramp24_e : rx_data;

    always @(posedge rx_clk) begin
        if (rx_valid == 1'b1) begin
            rx_ramp24_e[63:60] <= 4'd0;
            rx_ramp24_e[59:48] <= rx_ramp24_in[11:0] + 2'h3;
            rx_ramp24_e[47:44] <= 4'd0;
            rx_ramp24_e[43:32] <= rx_ramp24_in[11:0] + 2'h3;
            rx_ramp24_e[31:28] <= 4'd0;
            rx_ramp24_e[27:16] <= rx_ramp24_in[11:0] + 2'h2;
            rx_ramp24_e[15:12] <= 4'd0;
            rx_ramp24_e[11: 0] <= rx_ramp24_in[11:0] + 2'h2;
        end
    end

    // 16bit (16I) mode
 
    assign rx_patt16_e[63:48] = rx_pattern[15:0];
    assign rx_patt16_e[47:32] = rx_pattern[15:0];
    assign rx_patt16_e[31:16] = rx_pattern[15:0];
    assign rx_patt16_e[15: 0] = rx_pattern[15:0];

    assign rx_ramp16_in = (rx_mon_oos == 1'b0) ? rx_ramp16_e : rx_data;

    always @(posedge rx_clk) begin
        if (rx_valid == 1'b1) begin
            rx_ramp16_e[63:48] <= rx_ramp16_in[15:0] + 3'h7;
            rx_ramp16_e[47:32] <= rx_ramp16_in[15:0] + 3'h6;
            rx_ramp16_e[31:16] <= rx_ramp16_in[15:0] + 3'h5;
            rx_ramp16_e[15: 0] <= rx_ramp16_in[15:0] + 3'h4;
        end
    end

    // 8bit (8I) mode
 
    assign rx_patt08_e[63:56] = rx_pattern[7:0];
    assign rx_patt08_e[55:48] = rx_pattern[7:0];
    assign rx_patt08_e[47:40] = rx_pattern[7:0];
    assign rx_patt08_e[39:32] = rx_pattern[7:0];
    assign rx_patt08_e[31:24] = rx_pattern[7:0];
    assign rx_patt08_e[23:16] = rx_pattern[7:0];
    assign rx_patt08_e[15: 8] = rx_pattern[7:0];
    assign rx_patt08_e[ 7: 0] = rx_pattern[7:0];

    assign rx_ramp08_in = (rx_mon_oos == 1'b0) ? rx_ramp08_e : rx_data;

    always @(posedge rx_clk) begin
        if (rx_valid == 1'b1) begin
            rx_ramp08_e[63:56] <= rx_ramp08_in[7:0] + 4'hf;
            rx_ramp08_e[55:48] <= rx_ramp08_in[7:0] + 4'he;
            rx_ramp08_e[47:40] <= rx_ramp08_in[7:0] + 4'hd;
            rx_ramp08_e[39:32] <= rx_ramp08_in[7:0] + 4'hc;
            rx_ramp08_e[31:24] <= rx_ramp08_in[7:0] + 4'hb;
            rx_ramp08_e[23:16] <= rx_ramp08_in[7:0] + 4'ha;
            rx_ramp08_e[15: 8] <= rx_ramp08_in[7:0] + 4'h9;
            rx_ramp08_e[ 7: 0] <= rx_ramp08_in[7:0] + 4'h8;
        end
    end

    // 2bit (2I) mode
 
    assign rx_patt02_e[63:56] = {6'd0, rx_pattern[1:0]};
    assign rx_patt02_e[55:48] = {6'd0, rx_pattern[1:0]};
    assign rx_patt02_e[47:40] = {6'd0, rx_pattern[1:0]};
    assign rx_patt02_e[39:32] = {6'd0, rx_pattern[1:0]};
    assign rx_patt02_e[31:24] = {6'd0, rx_pattern[1:0]};
    assign rx_patt02_e[23:16] = {6'd0, rx_pattern[1:0]};
    assign rx_patt02_e[15: 8] = {6'd0, rx_pattern[1:0]};
    assign rx_patt02_e[ 7: 0] = {6'd0, rx_pattern[1:0]};

    assign rx_ramp02_in = (rx_mon_oos == 1'b0) ? rx_ramp02_e : rx_data;

    always @(posedge rx_clk) begin
        if (rx_valid == 1'b1) begin
            rx_ramp02_e[63:58] <= 6'd0;
            rx_ramp02_e[57:56] <= rx_ramp02_in[1:0] + 2'h3;
            rx_ramp02_e[55:50] <= 6'd0;
            rx_ramp02_e[49:48] <= rx_ramp02_in[1:0] + 2'h2;
            rx_ramp02_e[47:42] <= 6'd0;
            rx_ramp02_e[41:40] <= rx_ramp02_in[1:0] + 2'h1;
            rx_ramp02_e[39:34] <= 6'd0;
            rx_ramp02_e[33:32] <= rx_ramp02_in[1:0];
            rx_ramp02_e[31:26] <= 6'd0;
            rx_ramp02_e[25:24] <= rx_ramp02_in[1:0] + 2'h3;
            rx_ramp02_e[23:18] <= 6'd0;
            rx_ramp02_e[17:16] <= rx_ramp02_in[1:0] + 2'h2;
            rx_ramp02_e[15:10] <= 6'd0;
            rx_ramp02_e[ 9: 8] <= rx_ramp02_in[1:0] + 2'h1;
            rx_ramp02_e[ 7: 2] <= 6'd0;
            rx_ramp02_e[ 1: 0] <= rx_ramp02_in[1:0];
        end
    end

    // oos (16 good or bad data to go in or out of sync)

    assign rx_match = (rx_data_r == rx_data_e) ? 1'b1 : 1'b0;
    assign rx_update = ~(rx_mon_oos ^ rx_match);

    always @(negedge rx_resetn or posedge rx_clk) begin
        if (rx_resetn == 1'b0) begin
            rx_mon_err <= 1'd0;
            rx_mon_oos <= 1'd1;
            rx_mon_cnt <= 4'd0;
        end else begin
            if (rx_valid_r == 1'b1) begin
                rx_mon_err <= ~(rx_mon_oos | rx_match);
                if ((rx_update == 1'b1) && (rx_mon_cnt >= 4'hf)) begin
                    rx_mon_oos <= ~rx_mon_oos;
                end
                if (rx_update == 1'b1) begin
                    rx_mon_cnt <= rx_mon_cnt + 1'b1;
                end else begin
                    rx_mon_cnt <= 4'd0;
                end
            end
        end
    end

endmodule

// **********************************************************************************
// **********************************************************************************
