// **********************************************************************************
// **********************************************************************************
// ----------------------------------------------------------------------------------
// ################
// ##   ###########   Analog Devices Inc.
// ##      ########
// ##         #####   Copyright (c) 2019 Analog Devices Inc. All rights reserved.
// ##            ##   This file is the confidential and proprietary property of ADI.
// ##         #####   Possession or use of this file requires a written license.
// ##      ########   The licensing information may be found at: www.analog.com
// ##   ###########
// ################
// ----------------------------------------------------------------------------------
// Description:       AXI_ADRV9001, reset controller
// ----------------------------------------------------------------------------------
// **********************************************************************************
// **********************************************************************************

`timescale 1ps/1ps

module axi_adrv9001_rst_cntrl #(

    parameter RESET_PIPE_LENGTH = 8) (

    input   wire            axilite_init_req,
    input   wire            axilite_clk_enb,
    input   wire            axilite_master_enable,

    input   wire            mcs_clk_resetn,
    input   wire            mcs_phy_resetn,
    input   wire            mcs_dp_resetn,
    input   wire            clk_resetn_in,
    input   wire            phy_resetn_in,
    input   wire            dp_resetn_in,
    output  wire            clk_resetn_out,
    output  wire            phy_resetn_out,
    output  wire            dp_resetn_out,

    input   wire            rst_clk,
    output  wire            clk_resetn,

    input   wire            clk,
    output  wire            cdc_resetn,
    output  reg             serdes_resetn = 'd0,
    output  reg             serdes_data_resetn = 'd0,
    output  reg             resetn = 'd0,
    output  reg             init_ack = 'd0);

    // internal registers

    reg             [ 4:0]  cdc_resetn_count = 'd0;
    reg                     dp_resetn_in_n = 'd0;
    reg                     dp_resetn_out_n = 'd0;
    reg                     dp_resetn_in_p = 'd0;
    reg                     dp_resetn_out_p = 'd0;
    reg                     dp_resetn = 'd0;
    reg                     phy_resetn_in_n = 'd0;
    reg                     phy_resetn_out_n = 'd0;
    reg                     phy_resetn_in_p = 'd0;
    reg                     phy_resetn_out_p = 'd0;
    reg                     phy_resetn = 'd0;

    // internal signals

    wire                    init_req;
    wire                    master_enable;
    wire                    hs_master_enable;
    wire                    hs_clk_enb;
    wire                    hs_resetn_int;

    // clock, serdes and data path resets

    always @(negedge serdes_data_resetn or posedge clk) begin
        if (serdes_data_resetn == 1'b0) begin
            resetn <= 1'd0;
        end else begin
            resetn <= dp_resetn;
        end
    end

    always @(negedge clk) begin
        dp_resetn_in_n <= dp_resetn_in;
        dp_resetn_out_n <= dp_resetn_out;
    end

    always @(posedge clk) begin
        dp_resetn_in_p <= dp_resetn_in_n;
        dp_resetn_out_p <= dp_resetn_out_n;
    end

    always @(posedge clk) begin
        if (master_enable == 1'b1) begin
            dp_resetn <= dp_resetn_in_p;
        end else begin
            dp_resetn <= dp_resetn_out_p;
        end
    end

    always @(negedge serdes_resetn or posedge clk) begin
        if (serdes_resetn == 1'b0) begin
            serdes_data_resetn <= 1'd0;
        end else begin
            serdes_data_resetn <= phy_resetn;
        end
    end

    always @(negedge clk) begin
        phy_resetn_in_n <= phy_resetn_in;
        phy_resetn_out_n <= phy_resetn_out;
    end

    always @(posedge clk) begin
        phy_resetn_in_p <= phy_resetn_in_n;
        phy_resetn_out_p <= phy_resetn_out_n;
    end

    always @(posedge clk) begin
        if (master_enable == 1'b1) begin
            phy_resetn <= phy_resetn_in_p;
        end else begin
            phy_resetn <= phy_resetn_out_p;
        end
    end

    always @(negedge cdc_resetn or posedge clk) begin
        if (cdc_resetn == 1'b0) begin
            init_ack <= 1'd0;
            serdes_resetn <= 1'd0;
            cdc_resetn_count <= 5'd0;
        end else begin
            init_ack <= init_req;
            serdes_resetn <= ~init_req & cdc_resetn_count[4];
            if (cdc_resetn_count[4] == 1'b0) begin
                cdc_resetn_count <= cdc_resetn_count + 1'b1;
            end
        end
    end

    assign clk_resetn = (hs_master_enable == 1'b1) ? clk_resetn_in : clk_resetn_out;

    // instantiations

    cdc #(.DATA_WIDTH(5)) i_cdc (
        .src_data                               ({axilite_init_req,
                                                    axilite_master_enable,
                                                    mcs_clk_resetn,
                                                    mcs_phy_resetn,
                                                    mcs_dp_resetn}),
        .dest_resetn                            (1'b1),
        .dest_clk                               (clk),
        .dest_data                              ({init_req,
                                                    master_enable,
                                                    cdc_resetn,
                                                    phy_resetn_out,
                                                    dp_resetn_out}));

    cdc #(.DATA_WIDTH(3)) i_hs_cdc (
        .src_data                               ({axilite_master_enable,
                                                    axilite_clk_enb,
                                                    mcs_clk_resetn}),
        .dest_resetn                            (1'b1),
        .dest_clk                               (rst_clk),
        .dest_data                              ({hs_master_enable,
                                                    hs_clk_enb,
                                                    hs_resetn_int}));

    pipe #(
        .WIDTH                                  (1),
        .LENGTH                                 (RESET_PIPE_LENGTH))
    i_hs_pipe_out (
        .resetn                                 (hs_resetn_int),
        .clk                                    (rst_clk),
        .data_in                                (hs_clk_enb),
        .resetn_out                             (),
        .data_out                               (clk_resetn_out));

endmodule

// **********************************************************************************
// **********************************************************************************
