// **********************************************************************************
// **********************************************************************************
// ----------------------------------------------------------------------------------
// ################
// ##   ###########   Analog Devices Inc.
// ##      ########
// ##         #####   Copyright (c) 2019 Analog Devices Inc. All rights reserved.
// ##            ##   This file is the confidential and proprietary property of ADI.
// ##         #####   Possession or use of this file requires a written license.
// ##      ########   The licensing information may be found at: www.analog.com
// ##   ###########
// ################
// ----------------------------------------------------------------------------------
// Description:       AXI_ADRV9001, MCS Generation
// ----------------------------------------------------------------------------------
// **********************************************************************************
// **********************************************************************************

`timescale 1ps/1ps

module axi_adrv9001_mcs #(

    parameter MCS_OUT_CMOS1_LVDS0 = 0,
    parameter DEVICE_TYPE = "ULTRASCALE",
    parameter INSTANCE_ID = 0) (

    // system interface

    input   wire            dev_clk,
    input   wire            mcs_in_p,
    input   wire            mcs_in_n,
    output  wire            mcs_out_p,
    output  wire            mcs_out_n,
    output  reg             mcs_int_out = 1'd0,
    input   wire            mcs_int_in,
    output  reg             mcs_trig = 1'd0,
    output  reg             mcs_tx_clk_resetn = 1'd0,
    output  reg             mcs_tx_phy_resetn = 1'd0,
    output  reg             mcs_tx_dp_resetn = 1'd0,
    output  reg             mcs_rx_clk_resetn = 1'd0,
    output  reg             mcs_rx_phy_resetn = 1'd0,
    output  reg             mcs_rx_dp_resetn = 1'd0,
    
    // axilite interface

    input   wire            axilite_clk,
    input   wire            axilite_resetn,
    input   wire            axilite_mcs_request,
    output  wire            axilite_mcs_request_clr,
    input   wire  [  3:0]   axilite_mcs_width,
    input   wire  [  3:0]   axilite_mcs_num,
    input   wire            axilite_mcs_edge_fall1_rise0,
    input   wire  [ 31:0]   axilite_mcs_period,
    input   wire  [  1:0]   axilite_mcs_select,
    input   wire            axilite_mcs_trigger,
    output  wire            axilite_mcs_trigger_clr,
    input   wire  [ 15:0]   axilite_mcs_trigger_width,
    input   wire  [  1:0]   axilite_mcs_trigger_mode,
    input   wire            axilite_mcs_tx_clk_resetn,
    input   wire            axilite_mcs_tx_phy_resetn,
    input   wire            axilite_mcs_tx_dp_resetn,
    input   wire            axilite_mcs_rx_clk_resetn,
    input   wire            axilite_mcs_rx_phy_resetn,
    input   wire            axilite_mcs_rx_dp_resetn,
    input   wire            axilite_mcs_cdc_state,
    output  wire            axilite_mcs_cdc_state_clr,
    input   wire            axilite_mcs_swreset,
    output  wire  [ 31:0]   axilite_mcs_clk_mon_count);

    // internal registers
 
    reg                     mcs_seq = 'd0;
    reg                     mcs_request_d = 'd0;
    reg                     mcs_request_done = 'd0;
    reg                     mcs_status = 'd0;
    reg           [  3:0]   mcs_num_count = 'd0;
    reg           [ 31:0]   mcs_period_count = 'd0;
    reg                     mcs_trig_ack = 'd0;
    reg           [ 15:0]   mcs_trig_count = 'd0;
    reg                     mcs_out_d2 = 'd0;
    reg                     mcs_out_d1 = 'd0;

    // internal signals
 
    wire                    mcs_in;
    wire                    mcs_out;
    wire          [ 31:0]   mcs_period_width;
    wire                    mcs_request;
    wire          [  3:0]   mcs_width;
    wire          [  3:0]   mcs_num;
    wire                    mcs_edge_fall1_rise0;
    wire          [ 31:0]   mcs_period;
    wire          [  1:0]   mcs_select;
    wire                    mcs_trig_req;
    wire          [ 15:0]   mcs_trig_width;
    wire          [  1:0]   mcs_trig_mode;
    wire                    mcs_cdc_state;
    wire                    mcs_tx_clk_resetn_req;
    wire                    mcs_tx_phy_resetn_req;
    wire                    mcs_tx_dp_resetn_req;
    wire                    mcs_rx_clk_resetn_req;
    wire                    mcs_rx_phy_resetn_req;
    wire                    mcs_rx_dp_resetn_req;
    wire                    dev_resetn;

    // mcs based resets
 
    always @(negedge dev_resetn or posedge dev_clk) begin
        if (dev_resetn == 1'b0) begin
            mcs_tx_clk_resetn <= 1'd0;
            mcs_tx_phy_resetn <= 1'd0;
            mcs_tx_dp_resetn <= 1'd0;
            mcs_rx_clk_resetn <= 1'd0;
            mcs_rx_phy_resetn <= 1'd0;
            mcs_rx_dp_resetn <= 1'd0;
        end else begin
            mcs_tx_clk_resetn <= mcs_tx_clk_resetn_req;
            mcs_tx_phy_resetn <= mcs_tx_phy_resetn_req;
            mcs_tx_dp_resetn <= mcs_tx_dp_resetn_req;
            mcs_rx_clk_resetn <= mcs_rx_clk_resetn_req;
            mcs_rx_phy_resetn <= mcs_rx_phy_resetn_req;
            mcs_rx_dp_resetn <= mcs_rx_dp_resetn_req;
        end
    end

    // mcs
 
    assign mcs_period_width = {28'd0, mcs_width};

    always @(negedge dev_resetn or posedge dev_clk) begin
        if (dev_resetn == 1'b0) begin
            mcs_int_out <= 1'd0;
            mcs_seq <= 1'd0;
            mcs_request_d <= 1'd0;
            mcs_request_done <= 1'd0;
            mcs_status <= 1'd0;
            mcs_num_count <= 4'd0;
            mcs_period_count <= 32'd0;
        end else begin
            if (mcs_select == 2'd0) begin
                mcs_int_out <= mcs_seq;
            end else if (mcs_select == 2'd1) begin
                mcs_int_out <= mcs_in;
            end else begin
                mcs_int_out <= 1'd0;
            end
            if (mcs_period_count < mcs_period_width) begin
                mcs_seq <= mcs_status;
            end else begin
                mcs_seq <= 1'd0;
            end
            mcs_request_d <= mcs_request;
            if (mcs_status == 1'd1) begin
                if ((mcs_period_count >= mcs_period) && (mcs_num_count >= mcs_num)) begin
                    mcs_request_done <= 1'd1;
                    mcs_status <= 1'd0;
                end
                if (mcs_period_count >= mcs_period) begin
                    mcs_num_count <= mcs_num_count + 1'd1;
                end
                if (mcs_period_count >= mcs_period) begin
                    mcs_period_count <= 8'd0;
                end else begin
                    mcs_period_count <= mcs_period_count + 1'd1;
                end
            end else begin
                if ((mcs_request == 1'd1) && (mcs_request_d == 1'd0)) begin
                    mcs_request_done <= 1'd0;
                end
                mcs_status <= mcs_request & ~mcs_request_d;
                mcs_num_count <= 4'd0;
                mcs_period_count <= 32'd0;
            end
        end
    end

    // mcs trigger

    always @(negedge dev_resetn or posedge dev_clk) begin
        if (dev_resetn == 1'b0) begin
            mcs_trig_ack <= 1'd0;
            mcs_trig <= 1'd0;
            mcs_trig_count <= 16'd0;
        end else begin
            mcs_trig_ack <= mcs_trig_req;
            if ((mcs_trig_req == 1'd1) && (mcs_trig_ack == 1'd0)) begin
                if (mcs_trig_mode[1] == 1'b1) begin
                    mcs_trig <= ~mcs_trig;
                end else begin
                    mcs_trig <= mcs_trig_mode[0];
                end
            end else if (mcs_trig_count >= mcs_trig_width) begin
                if (mcs_trig_mode[1] == 1'b1) begin
                    mcs_trig <= mcs_trig;
                end else begin
                    mcs_trig <= ~mcs_trig_mode[0];
                end
            end
            if ((mcs_trig_req == 1'd1) && (mcs_trig_ack == 1'd0)) begin
                mcs_trig_count <= 16'd0;
            end else if (mcs_trig_count < mcs_trig_width) begin
                mcs_trig_count <= mcs_trig_count + 1'd1;
            end
        end
    end

    // mcs external

    always @(negedge dev_resetn or posedge dev_clk) begin
        if (dev_resetn == 1'b0) begin
            mcs_out_d2 <= 1'b0;
            mcs_out_d1 <= 1'b0;
        end else begin
            mcs_out_d2 <= mcs_int_in;
            mcs_out_d1 <= (mcs_edge_fall1_rise0 == 1'd1) ? mcs_out_d2 : mcs_int_in;
        end
    end

    // instantiations

    IBUFDS i_ibuf_mcs (
        .I                      (mcs_in_p),
        .IB                     (mcs_in_n),
        .O                      (mcs_in));

    generate
    if (MCS_OUT_CMOS1_LVDS0 == 1) begin

    OBUF i_obuf_mcs (
        .I                      (mcs_out),
        .O                      (mcs_out_p));

    assign mcs_out_n = 1'b0;

    end else begin

    OBUFDS i_obuf_mcs (
        .I                      (mcs_out),
        .O                      (mcs_out_p),
        .OB                     (mcs_out_n));

    end
    endgenerate

    generate
    if (DEVICE_TYPE == "ULTRASCALE") begin

    ODDRE1 #(
        .SIM_DEVICE             ("ULTRASCALE_PLUS"),
        .SRVAL                  (1'd0))
    i_oddr_mcs (
        .SR                     (1'b0),
        .C                      (dev_clk),
        .D1                     (mcs_out_d1),
        .D2                     (mcs_out_d2),
        .Q                      (mcs_out));

    end else begin

    ODDR #(
        .DDR_CLK_EDGE           ("SAME_EDGE"))
    i_oddr_mcs (
        .CE                     (1'b1),
        .R                      (1'b0),
        .S                      (1'b0),
        .C                      (dev_clk),
        .D1                     (mcs_out_d1),
        .D2                     (mcs_out_d2),
        .Q                      (mcs_out));

    end
    endgenerate

    clk_mon i_clk_mon (
        .clk                    (dev_clk),
        .axilite_resetn         (axilite_resetn),
        .axilite_clk            (axilite_clk),
        .axilite_clk_mon_count  (axilite_mcs_clk_mon_count));

    cdc_cntrl #(.DATA_WIDTH(70)) i_cdc_cntrl (
        .src_resetn             (axilite_resetn),
        .src_clk                (axilite_clk),
        .src_data               ({axilite_mcs_request,
                                    axilite_mcs_width,
                                    axilite_mcs_num,
                                    axilite_mcs_edge_fall1_rise0,
                                    axilite_mcs_period,
                                    axilite_mcs_select,
                                    axilite_mcs_trigger,
                                    axilite_mcs_trigger_width,
                                    axilite_mcs_trigger_mode,
                                    axilite_mcs_cdc_state,
                                    axilite_mcs_tx_clk_resetn,
                                    axilite_mcs_tx_phy_resetn,
                                    axilite_mcs_tx_dp_resetn,
                                    axilite_mcs_rx_clk_resetn,
                                    axilite_mcs_rx_phy_resetn,
                                    axilite_mcs_rx_dp_resetn}),
        .dest_resetn            (dev_resetn),
        .dest_clk               (dev_clk),
        .dest_data              ({mcs_request,
                                    mcs_width,
                                    mcs_num,
                                    mcs_edge_fall1_rise0,
                                    mcs_period,
                                    mcs_select,
                                    mcs_trig_req,
                                    mcs_trig_width,
                                    mcs_trig_mode,
                                    mcs_cdc_state,
                                    mcs_tx_clk_resetn_req,
                                    mcs_tx_phy_resetn_req,
                                    mcs_tx_dp_resetn_req,
                                    mcs_rx_clk_resetn_req,
                                    mcs_rx_phy_resetn_req,
                                    mcs_rx_dp_resetn_req}));

    cdc_cntrl #(.DATA_WIDTH(3)) i_cdc_status (
        .src_resetn             (dev_resetn),
        .src_clk                (dev_clk),
        .src_data               ({mcs_request_done,
                                    mcs_cdc_state,
                                    mcs_trig_ack}),
        .dest_resetn            (axilite_resetn),
        .dest_clk               (axilite_clk),
        .dest_data              ({axilite_mcs_request_clr,
                                    axilite_mcs_cdc_state_clr,
                                    axilite_mcs_trigger_clr}));

    cdc_resetp i_cdc_resetp_dev (
        .src_clk                (axilite_clk),
        .src_resetn             (~axilite_mcs_swreset),
        .dest_clk               (dev_clk),
        .dest_resetp            (dev_resetn));

endmodule

// **********************************************************************************
// **********************************************************************************
