// **********************************************************************************
// **********************************************************************************
// ----------------------------------------------------------------------------------
// ################
// ##   ###########   Analog Devices Inc.
// ##      ########
// ##         #####   Copyright (c) 2019 Analog Devices Inc. All rights reserved.
// ##            ##   This file is the confidential and proprietary property of ADI.
// ##         #####   Possession or use of this file requires a written license.
// ##      ########   The licensing information may be found at: www.analog.com
// ##   ###########
// ################
// ----------------------------------------------------------------------------------
// Description:       AXI_ADRV9001, GPIO
// ----------------------------------------------------------------------------------
// **********************************************************************************
// **********************************************************************************

`timescale 1ps/1ps

module axi_adrv9001_gpio #(

    parameter ID = 0,
    parameter SPI_DISABLE = 0) (

    // system interface

    inout   wire  [ 11:0]   dgpio,
    input   wire  [  7:0]   rx0_gpio_in,
    output  wire  [  7:0]   rx0_gpio_out,
    output  wire  [  7:0]   rx0_gpio_enb,
    input   wire  [  7:0]   rx1_gpio_in,
    output  wire  [  7:0]   rx1_gpio_out,
    output  wire  [  7:0]   rx1_gpio_enb,
    input   wire  [  9:0]   tx0_gpio_in,
    output  wire  [  9:0]   tx0_gpio_out,
    output  wire  [  9:0]   tx0_gpio_enb,
    input   wire  [  9:0]   tx1_gpio_in,
    output  wire  [  9:0]   tx1_gpio_out,
    output  wire  [  9:0]   tx1_gpio_enb,

    // gpio interface
 
    input   wire            gpio_clk,
    output  wire            gpio_resetn,
    output  reg             gpio_trig_out = 'd0,
    output  wire            gpio_gainindex_valid,
    output  reg   [  7:0]   gpio_gainindex = 'd0,

    // tdd enables

    input   wire            tdd_clk,
    input   wire            tdd_resetn,
    input   wire            tdd_tx0_dev_enable,
    input   wire            tdd_rx0_dev_enable,
    input   wire            tdd_tx1_dev_enable,
    input   wire            tdd_rx1_dev_enable,
    input   wire            tdd_orx0_gpio_enable,
    input   wire            tdd_orx1_gpio_enable,
    input   wire            tdd_tx0_gpio_enable,
    input   wire            tdd_tx1_gpio_enable,
    input   wire            tdd_ifrm0ctl0_gpio_enable,
    input   wire            tdd_ifrm0ctl1_gpio_enable,
    input   wire            tdd_ifrm1ctl0_gpio_enable,
    input   wire            tdd_ifrm1ctl1_gpio_enable,

    // trigger sources

    input   wire  [  3:0]   trig,
    input   wire            gpio_trig,

    // axi interface

    input   wire            axilite_clk,
    input   wire            axilite_resetn,
    input   wire            axilite_wrreq,
    input   wire  [15:0]    axilite_wraddr,
    input   wire  [31:0]    axilite_wrdata,
    output  wire            axilite_wrack,
    input   wire            axilite_rdreq,
    input   wire  [15:0]    axilite_rdaddr,
    output  wire  [31:0]    axilite_rddata,
    output  wire            axilite_rdack);

    // internal registers
 
    reg           [ 11:0]   tdd_dgpio_out = 'd0;
    reg           [  4:0]   axilite_trig_d = 'd0;
    reg           [  4:0]   axilite_dgpio_buf_trig = 'd0;
    reg                     axilite_dgpio_pio_buf_rd = 'd0;
    reg           [ 11:0]   axilite_dgpio_out = 'd0;

    // internal signals
 
    wire          [ 11:0]   dgpio_in;
    wire          [ 11:0]   dgpio_enb;
    wire          [ 11:0]   dgpio_out;
    wire          [ 11:0]   tdd_dgpio_enb;
    wire          [ 47:0]   tdd_dgpio_sel;
    wire          [ 11:0]   spi_dgpio_enb;
    wire          [ 11:0]   spi_dgpio_out;
    wire          [  3:0]   spi_ssn;
    wire                    spi_sclk_in;
    wire                    spi_sclk;
    wire                    spi_mosi;
    wire                    spi_miso;
    wire                    spi_enb;
    wire          [  4:0]   axilite_trig;
    wire          [ 11:0]   axilite_dgpio_enb;
    wire          [ 11:0]   axilite_dgpio_mode;
    wire          [ 11:0]   axilite_dgpio_pio_data;
    wire          [ 11:0]   axilite_dgpio_pio_reg_data;
    wire          [ 47:0]   axilite_dgpio_pio_sel;
    wire                    axilite_dgpio_pio_buf_enable;
    wire                    axilite_dgpio_pio_buf_wr;
    wire          [ 11:0]   axilite_dgpio_pio_buf_wrdata;
    wire          [ 11:0]   axilite_dgpio_pio_buf_rddata;
    wire          [ 11:0]   axilite_dgpio_pio_buf_mode;
    wire                    axilite_dgpio_pio_buf_full;
    wire                    axilite_dgpio_pio_buf_empty;
    wire          [  9:0]   axilite_dgpio_pio_buf_trig_mode;
    wire          [  7:0]   axilite_rx0_pio_data;
    wire          [  7:0]   axilite_rx1_pio_data;
    wire          [  9:0]   axilite_tx0_pio_data;
    wire          [  9:0]   axilite_tx1_pio_data;
    wire          [  2:0]   axilite_trig_src_group;
    wire          [  3:0]   axilite_trig_src_pin;
    wire          [  2:0]   axilite_gainindex_src_group;
    wire          [ 31:0]   axilite_gainindex_src_pin;
    wire          [ 31:0]   axilite_clk_mon_count;
    wire          [ 15:0]   axilite_sspi_ssn_sel;
    wire          [  3:0]   axilite_sspi_bidir_count;
    wire                    axilite_sspi_bidir_enable;
    wire                    axilite_sspi_miso_edge_sel;
    wire                    axilite_sspi_mosi_edge_sel;
    wire                    axilite_sspi_mosi_full;
    wire                    axilite_sspi_mosi_empty;
    wire                    axilite_sspi_miso_full;
    wire                    axilite_sspi_miso_empty;
    wire          [  3:0]   axilite_sspi_ss;
    wire                    axilite_sspi_mosi_rd;
    wire          [  7:0]   axilite_sspi_mosi_rddata;
    wire                    axilite_sspi_miso_wr;
    wire          [  7:0]   axilite_sspi_miso_wrdata;
    wire                    axilite_sspi_swreset;
    wire                    axilite_sspi_swreset_clr;
    wire                    axilite_sspi_enable;
    wire          [ 11:0]   gpio_trig_in;
    wire          [ 11:0]   gpio_gainindex_in;
    wire          [ 11:0]   gpio_dgpio_data;
    wire          [  7:0]   gpio_rx0_data;
    wire          [  7:0]   gpio_rx1_data;
    wire          [  9:0]   gpio_tx0_data;
    wire          [  9:0]   gpio_tx1_data;
    wire          [  2:0]   gpio_trig_src_group;
    wire          [  3:0]   gpio_trig_src_pin;
    wire          [  2:0]   gpio_gainindex_src_group;
    wire          [ 31:0]   gpio_gainindex_src_pin;

    // digital gpio

    genvar n;
    generate
    for (n = 0; n < 12; n = n + 1) begin: g_dgpio

    assign dgpio[n] = (dgpio_enb[n] == 1'b1) ? dgpio_out[n] : 1'bz;

    assign dgpio_in[n] = dgpio[n];
    assign dgpio_enb[n] = (axilite_dgpio_mode[n] == 1'b0) ? axilite_dgpio_enb[n] :
        (spi_dgpio_enb[n] | tdd_dgpio_enb[n]);
    assign dgpio_out[n] = (axilite_dgpio_mode[n] == 1'b0) ? axilite_dgpio_out[n] :
        (spi_dgpio_out[n] | tdd_dgpio_out[n]);

    assign tdd_dgpio_enb[n] = (tdd_dgpio_sel[((n*4)+3):(n*4)] >= 4'hc) ? 1'd0 : 1'd1;

    always @(negedge tdd_resetn or posedge tdd_clk) begin
        if (tdd_resetn == 1'd0) begin
            tdd_dgpio_out[n] <= 1'd0;
        end else begin
            case (tdd_dgpio_sel[((n*4)+3):(n*4)])
                4'b0000: tdd_dgpio_out[n] <= tdd_tx0_dev_enable;
                4'b0001: tdd_dgpio_out[n] <= tdd_rx0_dev_enable;
                4'b0010: tdd_dgpio_out[n] <= tdd_orx0_gpio_enable;
                4'b0011: tdd_dgpio_out[n] <= tdd_tx0_gpio_enable;
                4'b0100: tdd_dgpio_out[n] <= tdd_tx1_dev_enable;
                4'b0101: tdd_dgpio_out[n] <= tdd_rx1_dev_enable;
                4'b0110: tdd_dgpio_out[n] <= tdd_orx1_gpio_enable;
                4'b0111: tdd_dgpio_out[n] <= tdd_tx1_gpio_enable;
                4'b1000: tdd_dgpio_out[n] <= tdd_ifrm0ctl0_gpio_enable;
                4'b1001: tdd_dgpio_out[n] <= tdd_ifrm0ctl1_gpio_enable;
                4'b1010: tdd_dgpio_out[n] <= tdd_ifrm1ctl0_gpio_enable;
                4'b1011: tdd_dgpio_out[n] <= tdd_ifrm1ctl1_gpio_enable;
                default: tdd_dgpio_out[n] <= 1'd0;
            endcase
        end
    end

    end
    endgenerate

    assign spi_dgpio_enb = {1'd0, spi_enb, 10'd0};
    assign spi_dgpio_out = {1'd0, spi_miso, 10'd0};

    // trigger select

    assign gpio_trig_in = (gpio_trig_src_group == 3'd0) ? gpio_dgpio_data :
        ((gpio_trig_src_group == 3'd1) ? {4'd0, gpio_rx0_data} :
        ((gpio_trig_src_group == 3'd2) ? {4'd0, gpio_rx1_data} :
        ((gpio_trig_src_group == 3'd3) ? {2'd0, gpio_tx0_data} :
        ((gpio_trig_src_group == 3'd4) ? {2'd0, gpio_tx1_data} : 12'd0))));

    always @(negedge gpio_resetn or posedge gpio_clk) begin
        if (gpio_resetn == 1'd0) begin
            gpio_trig_out <= 1'd0;
        end else begin
            case (gpio_trig_src_pin)
                4'b0000: gpio_trig_out <= gpio_trig_in[ 0];
                4'b0001: gpio_trig_out <= gpio_trig_in[ 1];
                4'b0010: gpio_trig_out <= gpio_trig_in[ 2];
                4'b0011: gpio_trig_out <= gpio_trig_in[ 3];
                4'b0100: gpio_trig_out <= gpio_trig_in[ 4];
                4'b0101: gpio_trig_out <= gpio_trig_in[ 5];
                4'b0110: gpio_trig_out <= gpio_trig_in[ 6];
                4'b0111: gpio_trig_out <= gpio_trig_in[ 7];
                4'b1000: gpio_trig_out <= gpio_trig_in[ 8];
                4'b1001: gpio_trig_out <= gpio_trig_in[ 9];
                4'b1010: gpio_trig_out <= gpio_trig_in[10];
                4'b1011: gpio_trig_out <= gpio_trig_in[11];
                default: gpio_trig_out <= 1'd0;
            endcase
        end
    end

    // gain index

    assign gpio_gainindex_in = (gpio_gainindex_src_group == 3'd0) ? gpio_dgpio_data :
        ((gpio_gainindex_src_group == 3'd1) ? {4'd0, gpio_rx0_data} :
        ((gpio_gainindex_src_group == 3'd2) ? {4'd0, gpio_rx1_data} :
        ((gpio_gainindex_src_group == 3'd3) ? {2'd0, gpio_tx0_data} :
        ((gpio_gainindex_src_group == 3'd4) ? {2'd0, gpio_tx1_data} : 12'd0))));

    assign gpio_gainindex_valid = 1'd1;

    generate
    for (n = 0; n < 8; n = n + 1) begin: g_gainindex

    always @(negedge gpio_resetn or posedge gpio_clk) begin
        if (gpio_resetn == 1'd0) begin
            gpio_gainindex[n] <= 1'd0;
        end else begin
            case (gpio_gainindex_src_pin[((n*4)+3):(n*4)])
                4'b0000: gpio_gainindex[n] <= gpio_gainindex_in[ 0];
                4'b0001: gpio_gainindex[n] <= gpio_gainindex_in[ 1];
                4'b0010: gpio_gainindex[n] <= gpio_gainindex_in[ 2];
                4'b0011: gpio_gainindex[n] <= gpio_gainindex_in[ 3];
                4'b0100: gpio_gainindex[n] <= gpio_gainindex_in[ 4];
                4'b0101: gpio_gainindex[n] <= gpio_gainindex_in[ 5];
                4'b0110: gpio_gainindex[n] <= gpio_gainindex_in[ 6];
                4'b0111: gpio_gainindex[n] <= gpio_gainindex_in[ 7];
                4'b1000: gpio_gainindex[n] <= gpio_gainindex_in[ 8];
                4'b1001: gpio_gainindex[n] <= gpio_gainindex_in[ 9];
                4'b1010: gpio_gainindex[n] <= gpio_gainindex_in[10];
                4'b1011: gpio_gainindex[n] <= gpio_gainindex_in[11];
                default: gpio_gainindex[n] <= 1'd0;
            endcase
        end
    end

    end
    endgenerate

    // triggers & gpio fifo read
 
    generate
    for (n = 0; n < 5; n = n + 1) begin: g_trig

    always @(negedge axilite_resetn or posedge axilite_clk) begin
        if (axilite_resetn == 1'b0) begin
            axilite_trig_d[n] <= 1'd0;
            axilite_dgpio_buf_trig[n] <= 1'd0;
        end else begin
            axilite_trig_d[n] <= axilite_trig[n];
            case (axilite_dgpio_pio_buf_trig_mode[((n*2)+1):(n*2)])
                2'd3: axilite_dgpio_buf_trig[n] <= axilite_trig[n] ^ axilite_trig_d[n];
                2'd2: axilite_dgpio_buf_trig[n] <= ~axilite_trig[n] & axilite_trig_d[n];
                2'd1: axilite_dgpio_buf_trig[n] <= axilite_trig[n] & ~axilite_trig_d[n];
                default: axilite_dgpio_buf_trig[n] <= 1'd0;
            endcase
        end
    end

    end
    endgenerate

    always @(negedge axilite_resetn or posedge axilite_clk) begin
        if (axilite_resetn == 1'd0) begin
            axilite_dgpio_pio_buf_rd <= 1'd0;
        end else begin
            axilite_dgpio_pio_buf_rd <= | axilite_dgpio_buf_trig;
        end
    end

    generate
    for (n = 0; n < 12; n = n + 1) begin: g_dgpio_out

    always @(negedge axilite_resetn or posedge axilite_clk) begin
        if (axilite_resetn == 1'd0) begin
            axilite_dgpio_out[n] <= 1'd0;
        end else begin
            if (axilite_dgpio_pio_buf_mode[n] == 1'd0) begin
                axilite_dgpio_out[n] <= axilite_dgpio_pio_reg_data[n];
            end else if (axilite_dgpio_pio_buf_rd == 1'd1) begin
                axilite_dgpio_out[n] <= axilite_dgpio_pio_buf_rddata[n];
            end
        end
    end

    end
    endgenerate

    // spi

    generate
    if (SPI_DISABLE == 1) begin

    assign spi_ssn = 4'hf;
    assign spi_sclk_in = 1'd0;
    assign spi_mosi = 1'd0;

    end else begin

    for (n = 0; n < 4; n = n + 1) begin: g_spi_csn

    assign spi_ssn[n] = (axilite_sspi_ssn_sel[((n*4)+3):(n*4)] == 4'h0) ? dgpio_in[0] :
        ((axilite_sspi_ssn_sel[((n*4)+3):(n*4)] == 4'h1) ? dgpio_in[1] :
        ((axilite_sspi_ssn_sel[((n*4)+3):(n*4)] == 4'h2) ? dgpio_in[2] :
        ((axilite_sspi_ssn_sel[((n*4)+3):(n*4)] == 4'h3) ? dgpio_in[3] :
        ((axilite_sspi_ssn_sel[((n*4)+3):(n*4)] == 4'h4) ? dgpio_in[4] :
        ((axilite_sspi_ssn_sel[((n*4)+3):(n*4)] == 4'h5) ? dgpio_in[5] :
        ((axilite_sspi_ssn_sel[((n*4)+3):(n*4)] == 4'h6) ? dgpio_in[6] :
        ((axilite_sspi_ssn_sel[((n*4)+3):(n*4)] == 4'h7) ? dgpio_in[7] :
        ((axilite_sspi_ssn_sel[((n*4)+3):(n*4)] == 4'h8) ? dgpio_in[8] :
        ((axilite_sspi_ssn_sel[((n*4)+3):(n*4)] == 4'h9) ? dgpio_in[9] :
        ((axilite_sspi_ssn_sel[((n*4)+3):(n*4)] == 4'ha) ? dgpio_in[10] :
        ((axilite_sspi_ssn_sel[((n*4)+3):(n*4)] == 4'hb) ? dgpio_in[11] :
        1'd1)))))))))));

    end

    assign spi_sclk_in = dgpio_in[9];
    assign spi_mosi = dgpio_in[11];

    end
    endgenerate

    // instantiations

    cdc #(.DATA_WIDTH(5)) i_cdc_trig (
        .src_data                               ({gpio_trig, trig}),
        .dest_resetn                            (axilite_resetn),
        .dest_clk                               (axilite_clk),
        .dest_data                              (axilite_trig));

    cdc #(.DATA_WIDTH(48)) i_cdc_gpio_data (
        .src_data                               ({dgpio_in,
                                                    rx0_gpio_in,
                                                    rx1_gpio_in,
                                                    tx0_gpio_in,
                                                    tx1_gpio_in}),
        .dest_resetn                            (gpio_resetn),
        .dest_clk                               (gpio_clk),
        .dest_data                              ({gpio_dgpio_data,
                                                    gpio_rx0_data,
                                                    gpio_rx1_data,
                                                    gpio_tx0_data,
                                                    gpio_tx1_data}));

    cdc_cntrl #(.DATA_WIDTH(42)) i_cdc_cntrl_gpio (
        .src_resetn                             (axilite_resetn),
        .src_clk                                (axilite_clk),
        .src_data                               ({axilite_trig_src_group,
                                                    axilite_trig_src_pin,
                                                    axilite_gainindex_src_group,
                                                    axilite_gainindex_src_pin}),
        .dest_resetn                            (gpio_resetn),
        .dest_clk                               (gpio_clk),
        .dest_data                              ({gpio_trig_src_group,
                                                    gpio_trig_src_pin,
                                                    gpio_gainindex_src_group,
                                                    gpio_gainindex_src_pin}));

    cdc #(.DATA_WIDTH(48)) i_cdc_axilite_data (
        .src_data                               ({dgpio_in,
                                                    rx0_gpio_in,
                                                    rx1_gpio_in,
                                                    tx0_gpio_in,
                                                    tx1_gpio_in}),
        .dest_resetn                            (axilite_resetn),
        .dest_clk                               (axilite_clk),
        .dest_data                              ({axilite_dgpio_pio_data,
                                                    axilite_rx0_pio_data,
                                                    axilite_rx1_pio_data,
                                                    axilite_tx0_pio_data,
                                                    axilite_tx1_pio_data}));

    cdc_cntrl #(.DATA_WIDTH(48)) i_cdc_cntrl_tdd (
        .src_resetn                             (axilite_resetn),
        .src_clk                                (axilite_clk),
        .src_data                               (axilite_dgpio_pio_sel),
        .dest_resetn                            (tdd_resetn),
        .dest_clk                               (tdd_clk),
        .dest_data                              (tdd_dgpio_sel));

    cdc_resetp i_cdc_resetp_gpio (
        .src_clk                                (axilite_clk),
        .src_resetn                             (axilite_resetn),
        .dest_clk                               (gpio_clk),
        .dest_resetp                            (gpio_resetn));

    clk_mon i_gpio_clk_mon (
        .clk                                    (gpio_clk),
        .axilite_resetn                         (axilite_resetn),
        .axilite_clk                            (axilite_clk),
        .axilite_clk_mon_count                  (axilite_clk_mon_count));

    mem_buf #(
        .ADDRESS_WIDTH                          (8),
        .DATA_WIDTH                             (12))
    i_buf (
        .wr_clk                                 (axilite_clk),
        .wr_resetn                              (axilite_resetn),
        .wr_enable                              (axilite_dgpio_pio_buf_enable),
        .wr_valid                               (axilite_dgpio_pio_buf_wr),
        .wr_data                                (axilite_dgpio_pio_buf_wrdata),
        .wr_empty                               (axilite_dgpio_pio_buf_empty),
        .wr_full                                (axilite_dgpio_pio_buf_full),
        .rd_clk                                 (axilite_clk),
        .rd_resetn                              (axilite_resetn),
        .rd_enable                              (axilite_dgpio_pio_buf_enable),
        .rd_read                                (axilite_dgpio_pio_buf_rd),
        .rd_valid                               (),
        .rd_data                                (axilite_dgpio_pio_buf_rddata));

    generate
    if (SPI_DISABLE == 1) begin

    assign spi_sclk = 1'd0;
    assign spi_miso = 1'd0;
    assign spi_enb = 1'd0;
    assign axilite_sspi_swreset_clr = 1'd0;
    assign axilite_sspi_ss = 4'hf;
    assign axilite_sspi_mosi_rddata = 8'd0;
    assign axilite_sspi_mosi_full = 1'd0;
    assign axilite_sspi_mosi_empty = 1'd1;
    assign axilite_sspi_miso_full = 1'd0;
    assign axilite_sspi_miso_empty = 1'd1;

    end else begin

    BUFH i_sspi_clk (
        .I                                      (spi_sclk_in),
        .O                                      (spi_sclk));
        
    axi_adrv9001_gpio_sspi i_sspi (
        .spi_ssn                                (spi_ssn),
        .spi_sclk                               (spi_sclk),
        .spi_mosi                               (spi_mosi),
        .spi_miso                               (spi_miso),
        .spi_enb                                (spi_enb),
        .axilite_clk                            (axilite_clk),
        .axilite_resetn                         (axilite_resetn),
        .axilite_swreset                        (axilite_sspi_swreset),
        .axilite_swreset_clr                    (axilite_sspi_swreset_clr),
        .axilite_ss                             (axilite_sspi_ss),
        .axilite_enable                         (axilite_sspi_enable),
        .axilite_bidir_enb                      (axilite_sspi_bidir_enable),
        .axilite_bidir_count                    (axilite_sspi_bidir_count),
        .axilite_mosi_edge_sel                  (axilite_sspi_mosi_edge_sel),
        .axilite_miso_edge_sel                  (axilite_sspi_miso_edge_sel),
        .axilite_mosi_rd                        (axilite_sspi_mosi_rd),
        .axilite_mosi_rddata                    (axilite_sspi_mosi_rddata),
        .axilite_mosi_full                      (axilite_sspi_mosi_full),
        .axilite_mosi_empty                     (axilite_sspi_mosi_empty),
        .axilite_miso_wr                        (axilite_sspi_miso_wr),
        .axilite_miso_wrdata                    (axilite_sspi_miso_wrdata),
        .axilite_miso_full                      (axilite_sspi_miso_full),
        .axilite_miso_empty                     (axilite_sspi_miso_empty));

    end
    endgenerate

    axi_adrv9001_gpio_regs #(.ID(ID)) i_regs (
        .axilite_timer                          (),
        .axilite_dgpio_pio_data                 (axilite_dgpio_pio_data),
        .axilite_dgpio_pio_data_out             (axilite_dgpio_pio_reg_data),
        .axilite_dgpio_pio_data_out_enb         (axilite_dgpio_enb),
        .axilite_dgpio_pio_data_mode            (axilite_dgpio_mode),
        .axilite_dgpio_pio_sel_00               (axilite_dgpio_pio_sel[(( 0*4)+3):( 0*4)]),
        .axilite_dgpio_pio_sel_01               (axilite_dgpio_pio_sel[(( 1*4)+3):( 1*4)]),
        .axilite_dgpio_pio_sel_02               (axilite_dgpio_pio_sel[(( 2*4)+3):( 2*4)]),
        .axilite_dgpio_pio_sel_03               (axilite_dgpio_pio_sel[(( 3*4)+3):( 3*4)]),
        .axilite_dgpio_pio_sel_04               (axilite_dgpio_pio_sel[(( 4*4)+3):( 4*4)]),
        .axilite_dgpio_pio_sel_05               (axilite_dgpio_pio_sel[(( 5*4)+3):( 5*4)]),
        .axilite_dgpio_pio_sel_06               (axilite_dgpio_pio_sel[(( 6*4)+3):( 6*4)]),
        .axilite_dgpio_pio_sel_07               (axilite_dgpio_pio_sel[(( 7*4)+3):( 7*4)]),
        .axilite_dgpio_pio_sel_08               (axilite_dgpio_pio_sel[(( 8*4)+3):( 8*4)]),
        .axilite_dgpio_pio_sel_09               (axilite_dgpio_pio_sel[(( 9*4)+3):( 9*4)]),
        .axilite_dgpio_pio_sel_10               (axilite_dgpio_pio_sel[((10*4)+3):(10*4)]),
        .axilite_dgpio_pio_sel_11               (axilite_dgpio_pio_sel[((11*4)+3):(11*4)]),
        .axilite_dgpio_pio_buf_enable           (axilite_dgpio_pio_buf_enable),
        .axilite_dgpio_pio_buf_data_wr          (axilite_dgpio_pio_buf_wr),
        .axilite_dgpio_pio_buf_data_data        (axilite_dgpio_pio_buf_wrdata),
        .axilite_dgpio_pio_buf_mode             (axilite_dgpio_pio_buf_mode),
        .axilite_dgpio_pio_buf_full             (axilite_dgpio_pio_buf_full),
        .axilite_dgpio_pio_buf_empty            (axilite_dgpio_pio_buf_empty),
        .axilite_dgpio_pio_buf_trigger_mode_0   (axilite_dgpio_pio_buf_trig_mode[1:0]),
        .axilite_dgpio_pio_buf_trigger_mode_1   (axilite_dgpio_pio_buf_trig_mode[3:2]),
        .axilite_dgpio_pio_buf_trigger_mode_2   (axilite_dgpio_pio_buf_trig_mode[5:4]),
        .axilite_dgpio_pio_buf_trigger_mode_3   (axilite_dgpio_pio_buf_trig_mode[7:6]),
        .axilite_dgpio_pio_buf_trigger_mode_4   (axilite_dgpio_pio_buf_trig_mode[9:8]),
        .axilite_rx0_pio_data                   (axilite_rx0_pio_data),
        .axilite_rx0_pio_data_out               (rx0_gpio_out),
        .axilite_rx0_pio_data_out_enb           (rx0_gpio_enb),
        .axilite_rx1_pio_data                   (axilite_rx1_pio_data),
        .axilite_rx1_pio_data_out               (rx1_gpio_out),
        .axilite_rx1_pio_data_out_enb           (rx1_gpio_enb),
        .axilite_tx0_pio_data                   (axilite_tx0_pio_data),
        .axilite_tx0_pio_data_out               (tx0_gpio_out),
        .axilite_tx0_pio_data_out_enb           (tx0_gpio_enb),
        .axilite_tx1_pio_data                   (axilite_tx1_pio_data),
        .axilite_tx1_pio_data_out               (tx1_gpio_out),
        .axilite_tx1_pio_data_out_enb           (tx1_gpio_enb),
        .axilite_trig_src_group                 (axilite_trig_src_group),
        .axilite_trig_src_pin                   (axilite_trig_src_pin),
        .axilite_gainindex_src_group            (axilite_gainindex_src_group),
        .axilite_gainindex_00_src_pin           (axilite_gainindex_src_pin[((0*4)+3):(0*4)]),
        .axilite_gainindex_01_src_pin           (axilite_gainindex_src_pin[((1*4)+3):(1*4)]),
        .axilite_gainindex_02_src_pin           (axilite_gainindex_src_pin[((2*4)+3):(2*4)]),
        .axilite_gainindex_03_src_pin           (axilite_gainindex_src_pin[((3*4)+3):(3*4)]),
        .axilite_gainindex_04_src_pin           (axilite_gainindex_src_pin[((4*4)+3):(4*4)]),
        .axilite_gainindex_05_src_pin           (axilite_gainindex_src_pin[((5*4)+3):(5*4)]),
        .axilite_gainindex_06_src_pin           (axilite_gainindex_src_pin[((6*4)+3):(6*4)]),
        .axilite_gainindex_07_src_pin           (axilite_gainindex_src_pin[((7*4)+3):(7*4)]),
        .axilite_clk_mon_count                  (axilite_clk_mon_count),
        .axilite_sspi_ssn0_sel                  (axilite_sspi_ssn_sel[3:0]),
        .axilite_sspi_ssn1_sel                  (axilite_sspi_ssn_sel[7:4]),
        .axilite_sspi_ssn2_sel                  (axilite_sspi_ssn_sel[11:8]),
        .axilite_sspi_ssn3_sel                  (axilite_sspi_ssn_sel[15:12]),
        .axilite_sspi_bidir_count               (axilite_sspi_bidir_count),
        .axilite_sspi_bidir_enable              (axilite_sspi_bidir_enable),
        .axilite_sspi_miso_edge_sel             (axilite_sspi_miso_edge_sel),
        .axilite_sspi_mosi_edge_sel             (axilite_sspi_mosi_edge_sel),
        .axilite_sspi_mosi_full                 (axilite_sspi_mosi_full),
        .axilite_sspi_mosi_empty                (axilite_sspi_mosi_empty),
        .axilite_sspi_miso_full                 (axilite_sspi_miso_full),
        .axilite_sspi_miso_empty                (axilite_sspi_miso_empty),
        .axilite_sspi_ss                        (axilite_sspi_ss),
        .axilite_intr_sspi_ss                   (),
        .axilite_sspi_mosi_data_rd              (axilite_sspi_mosi_rd),
        .axilite_sspi_mosi_data_data            (axilite_sspi_mosi_rddata),
        .axilite_sspi_miso_data_wr              (axilite_sspi_miso_wr),
        .axilite_sspi_miso_data_data            (axilite_sspi_miso_wrdata),
        .axilite_sspi_swreset                   (axilite_sspi_swreset),
        .axilite_sspi_swreset_clr               (axilite_sspi_swreset_clr),
        .axilite_sspi_enable                    (axilite_sspi_enable),
        .axilite_clk                            (axilite_clk),
        .axilite_resetn                         (axilite_resetn),
        .axilite_wrreq                          (axilite_wrreq),
        .axilite_wraddr                         (axilite_wraddr),
        .axilite_wrdata                         (axilite_wrdata),
        .axilite_wrack                          (axilite_wrack),
        .axilite_rdreq                          (axilite_rdreq),
        .axilite_rdaddr                         (axilite_rdaddr),
        .axilite_rddata                         (axilite_rddata),
        .axilite_rdack                          (axilite_rdack));

endmodule

// **********************************************************************************
// **********************************************************************************
