# AXI ADRV9001

## Contents

* [Introduction](#introduction)
* [Functional Description](#functional-description)
  * [IO Standard](#io-standard)
  * [Clocks and Resets](#clocking)
  * [Clock Monitors](#clock-monitors)
  * [SSI Interface](#ssi-interface)
    * [Transmit SSI](#transmit-ssi)
    * [Receive SSI](#receive-ssi)
  * [Device and DMA Enables](#enables)
  * [Interfaces](#interfaces)
* [Parameters](#parameters)
* [Ports](#ports)
  * [External System Ports](#external-system-ports)
  * [External ADRV9001 Ports](#external-adrv9001-ports)
  * [Internal Master/Slave Ports](#internal-master-slave-ports)
  * [Internal Data Ports](#internal-data-ports)
  * [Internal Trigger/GPIO Ports](#internal-trigger-gpio-ports)
  * [Internal AXILite Ports](#internal-axilite-ports)
* [Register Map](#register-map)
* [System Integration](#system-integration)

## <a name="introduction"></a>Introduction

This IP core is complementary to the ADRV9001 device, it interfaces one-to-one physically to the ADRV9001 device. All interface standards such as LVDS or CMOS; protocols related to STROBE such as pulse or level; and conversions from external serial based on number of lines to internal parallel bus; are encapsulated inside the core. The user or internal FPGA logic is presented with a simple AXIS like interface.

## <a name="functional-description"></a>Functional Description

A functional block diagram of the IP core is shown below. In all the modes supported in both LVDS and CMOS IO standards, the FPGA SERDES modules are used. The core internally runs at a quarter of the interface clock. At 61.44 MSPS 2 lane LVDS, the interface runs at 491.52 MHz and internally at 122.88 MHz.

![AXI\_ADRV9001 Block Diagram](docs/axi_adrv9001.png)

### <a name="io-standard"></a>IO Standard

The core supports both LVDS and CMOS IO standards, but this is a configuration option. See the [parameters](#parameters) section for more details. The internal logic of the core remains identical in both LVDS and CMOS mode, however not all configurations are meaningful as the device can support only a limited number of options.

### <a name="clocking"></a>Clocks and Resets

The core contains multiple clock domains, the AXILite interface runs at the processor bus clock (usually 100MHz) and the IO delay controllers runs at a system generated 200MHz (ZC706) or 300MHz (ZCU102) clock. This clock is most commonly sourced directly from the processor domain (HPS, NIOS, Microblaze, or Zynq).

Apart from the above mentioned system clocks, the core generates a device clock (same as ADRV9001) for MCS functions, a TDD clock (defaults to 200MHz) for various TDD functions, and a GPIO clock (defaults to 200MHz) for GPIO handling. All these three clocks are generated using the reference clock on board. This clock can either be sourced from the SMA on board or driven by ADRV9001 itself (usually a divided device clock).

The SSI interfaces uses local clocking for conversion between serial and parallel data streams.

All reset signals (software or hardware) are generated, and used internally.

### <a name="clock-monitors"></a>Clock Monitors

All variable clocks of the interface core are monitored and reported to the software. The driver call returns the frequency of the respective clock in Hz. This is assuming an AXILite CPU clock frequency of 100 MHz. The following are considered variable clocks.

* The on-board reference clock input (device clock), default 38.4MHz.
* The reference clock generated by ADRV9001, (device clock out from ADRV9001), default 19.2MHz.
* The generated device clock (this clock drives MCS and other top level signals), default 38.4MHz.
* The generated TDD clock (this clock drives all TDD enables), default 200MHz.
* The generated GPIO clock (this clock drives all GPIO signals), default is 200MHz.
* The four SSI internal clocks (these clocks run at a quarter of interface rate).

The clock monitors are capable of monitoring frequencies from a minimum of 1.526kHz to a maximum of 6.554THz. In most practical systems, the maximum is limitted by device (usually 300MHz). Since it monitors clocks down to 1.526kHz, it takes about 1.3ms to report correct frequency.

#### <a name="system-interface"></a>System Interface

##### MCS

The interface core uses the device clock for all MCS logic. The MCS signals are generated as a programmable sequence of pulses with programmable period, and width. The signal may also be sourced externally either from a dedicated source or from another interface core instance. All the programmable features of the MCS logic can be found in the register map. A pseudo code for generation of mcs signal is given below. Please refer to the register map for details.

```C

  function generate_mcs (

    in  int mcs_period_us;
    in  int mcs_high_time_us;
    in  int mcs_num_of_pulses;

  ) {

    axi_adrv9001_regs.mcs_width = (int) (mcs_high_time_us * 38.4) + 1;
    axi_adrv9001_regs.mcs_period = (int) (mcs_period_us * 38.4) + 1;
    axi_adrv9001_regs.mcs_num = mcs_num_of_pulses;
    axi_adrv9001_regs.mcs_request = 1;

    while (axi_adrv9001_regs.mcs_request == 1);
  }

```

#### <a name="ssi-interface"></a>SSI Interface

The ssi interface encapsulates the oddities of ADRV9001's digital interface, and presents a simple sample interface, along with a valid marker. The structural break down of data flow within the SSI interface is given below. This is mainly in the receive direction, the transmit follows the same concepts in reverse direction.

##### I/O type (CMOS/LVDS)

At the physical level, the interface core differentiates ONLY the IO buffers as CMOS or LVDS. That is, beyond the IO buffers the logic is the same, and data path do not differentiate between the buffer types.

##### Sampling edges (SDR/DDR)

After the IO buffers, the interface cores samples the signals at either edges of the data, rising and/or falling. Thus it supports SDR and DDR modes in native way of sampling. In SDR mode, user may choose either the rising edge or falling edge. In DDR mode, data is sampled at both edges. In the transmit direction the same concept applies in reverse direction.

##### Data path controls (LANES, WIDTHS, etc.)

After the sampling state, data is then delineated or transmitted based on the number of lanes, the sample width and other modes. This is where the interface core forces some restrictions to be compatible with ADRV9001.


#### <a name="transmit-ssi"></a>Transmit SSI

In the transmit direction, the core asserts the read signal to indicate a sample request. The upstream module must turn around this read signal as a valid signal along with the data. The latency of read to valid turn around is irrelevant, however it must remain constant. In TDD mode, the enables must be set such that this latency is compensated.

Inside the core this data is sourced along with internal test signals (RAMP, PRBS) and optionally a receive data loopback. The data is then passed to the SERDES module.

The SERDES module uses a 8:1 conversion in DDR mode and is run at a quarter rate of the interface clock. The transmit clock may optionally be sourced by the receive clock.

The core uses two instances of this module for the two channels.

#### <a name="receive-ssi"></a>Receive SSI

The receive SSI consists of a SERDES module, it converts the serial data to byte wide parallel data. The internal clock runs at a quarter of the interface clock. The samples are delineated using the strobe signal. The core monitors the strobe signal based on the mode of operation. If the strobe signal behavior differs from expected, the core asserts OOS (out of sync). It takes 4 correct alignment of the strobe signals to clear the OOS status.

The received samples may optionally be monitored for a ramp or PRBS.

The data is then passed to either the receive channel or the observation receive channel as directed by the TDD enable signals.

At the downstream interface, the valid signal is asserted high to qualify data. The core do not support any back pressure.

The core uses two instances of this module for the two channels.

### <a name="enables"></a>Device and DMA Enables

The transfer of samples between an internal core (such as DMA) and SSI modules are controlled by enable signals generated by the TDD modules. The core drives individual dedicated signals to the device enable pins. It also generates DMA enables for the internal interface. All these enables are generated at TDD clock (defaults 200MHz) and can be synchronized by the same means available to other cores.

An enable is a simple signal whose assertion and deassertion times are governed by a TDD frame (free running counters with programmable parameters).

A TDD frame has two layers, a base period counter and a secondary frame counters.

The enable signal has two independent floating windows (primary and secondary) within the TDD frame. The assertion count sets the enable signal to high, and deassertion count sets it to low. The primary and secondary windows may overlap, slide over or reverse itself.

Since the TDD enables are generated at TDD clock and passes clock domain crossing to SSI clock, their behavior changes slightly.

There are two ways one can control the behavior of the enable signals at SSI clock.

In normal mode (ENABLE-SYNC == 0), the TDD duration is maintained when clock domains are crossed. All enables are forced to follow the same duration, however due to the inherent uncertainty with CDC, the number of samples actually transferred during the enable time varies. That is, the number of samples are rounded down to the nearest duration. Thus some enable signals may carry one more or one less sample than others.

In synchronous mode (ENABLE-SYNC == 1), the TDD duration is discarded, instead the number of samples transferred during the enable signal is maintained across enable signals. So all enable duration carries the same number of samples and the enable duration is lengthened or shortened to accomadate the same number of samples.

The timing diagram below illustrates this concept.

![AXI\_ADRV9001 Enable Sync](docs/axi_adrv9001_tdd_enable_sync.png)

If so equipped, the dma and tdd enable controls provide the following use cases (note that the names may not exactly with what you see in an application):

---
| TDD Enable Sync   | DMA Start Sync | DMA Cyclic Sync | Use case description |
| ----------------- | -------------- | --------------- | -------------------- |
| 0x0 | 0x0 | 0x0 | Unknown |
| 0x1 | 0x0 | 0x0 | Constant Samples per Enable. |
| 0x1 | 0x1 | 0x0 | Constant Samples per Enable, aligned start of frame with enable. |
| 0x1 | 0x1 | 0x1 | Constant Samples per Enable, aligned start of frame and cyclic frame with enable. |
|||||
---

This however, also depends on the enable duration and the number of samples per frame. The TDD clock signals always carry the uncertainty associated with crossing over to ssi clock domains. This means, if you were to look at the enable signals in absolute time their edges will jitter (in analog terms, phase noise) and subsequently drift (wander) over time.

In order to best utilize the synchronization mechanism, software must program the counters based on sampling frequency, TDD/SSI clock ratio and the internal bus width.

```C

  function tdd_and_dma_sync (

      in  int   frame_period_us;          /* desired frame duration in us */
      in  int   number_of_frames;         /* number of frames to transmit and receive in total */
      in  int   sampling_freq_hz;         /* sampling frequency in Hz */
      in  int   sample_width_bits;        /* sample width in bits, 2, 8, 16, 24, 32 or 64 */
      in  int   number_of_lanes;          /* number of lanes used, 1, 2 or 4 */
      in  int   data_rate_ddr2_sdr1;      /* data rate, sdr (1) or ddr (2) */

      out int   samples_per_frame;        /* number of samples per frame for synchronous mode */
      out int   samples_per_dma_buffer;   /* total number of samples that must be in dma buffer */
      out int   tdd_count_per_frame;      /* tdd count to be set per enable */

  ) {  

      int samples_per_ssi_clk;
      int tdd_count_per_ssi_clk;

      samples_per_ssi_clk = (sample_width_bits < 8) ? 8 : sample_width_bits;
      samples_per_ssi_clk = 64/samples_per_ssi_clk;

      samples_per_frame = frame_period_us * sampling_freq_hz;
      samples_per_frame = (samples_per_frame/samples_per_ssi_clk);
      samples_per_frame = samples_per_frame * samples_per_ssi_clk;

      samples_per_dma_buffer = samples_per_frame * number_of_frames;

      tdd_count_per_ssi_clk = (sampling_freq_hz*sample_wdith_bits)/(number_of_lanes*data_rate_ddr2_sdr1);
      tdd_count_per_ssi_clk = tdd_count_per_ssi_clk/(4*1000);
      tdd_count_per_ssi_clk = (200*1000*1000)/tdd_count_per_ssi_clk;

      tdd_count_per_frame = sampling_freq_hz/1000;
      tdd_count_per_frame = (200*1000*1000)/tdd_count_per_frame;
      tdd_count_per_frame = tdd_count_per_frame * samples_per_frame;
      tdd_count_per_frame = tdd_count_per_frame + tdd_count_per_ssi_clk;
      tdd_count_per_frame = (tdd_count_per_frame + 500)/1000;
  }

```

### <a name="interfaces"></a>Interfaces

#### AXILite Interface (AXILite)

This must be connected to an AXILite master (usually CPU through an interconnect). The core requires an addressable space of 64K bytes.

#### Device Interface (DIO)

This must be exported as an external interface and must be connected to the ADRV9001 device on board. There must be one instance for each of the ADRV9001 devices on board.

#### System Interface (SYS)

This must be exported as an external interface and must be connected to the resources on board. This may be shared across all instances.

#### Transmit Sample Interfaces (TX0, TX1)

These internal interfaces provide data samples with the corresponding clock and reset signals. They may be connected to a compatible master inside the FPGA.

#### Receive Sample Interfaces (RX0, RX1)

These internal interfaces provide data samples with the corresponding clock and reset signals. They may be connected to a compatible slave inside the FPGA.

#### Observation Receive Sample Interfaces (ORX0, ORX1)

These internal interfaces provide data samples with the corresponding clock and reset signals. They may be connected to a compatible slave inside the FPGA. The ORX interfaces share the same clock and reset signals as the receive path.

#### GPIO Interface (Digital, SSI)

The core supports true GPIO on the 12 DGPIO pins of the device. The SSI GPIO has limitted use in certain configurations. In order to support true GPIO, the SSI interface must be reconfigured to do so. The data path elements (SERDES) and clocking do NOT support GPIO.

If so configured, the following is supported:

##### Digital GPIO (12 pins)

These pins support two modes, software and hardware. In hardware mode, any of the TDD enables may be sourced as output. In software mode, the pin may be configured as either input or output.

##### SSI GPIO (10 TX0 pins, 10 TX1 pins, 8 RX0 pins and 8 RX1 pins)

These pins support only software mode.

#### GPIO Gain Interface (GPIO\_GAIN)

These internal interfaces provide gain data with the corresponding clock and reset signals. They may be connected to a compatible slave inside the FPGA. The gain index is a collection of 7 GPIO ports. Any GPIO port may be mapped to any one of the gain index data bits. The core outputs an 8 bit gain index with the MSB set to 0x0. The gain index is a free running data stream, capture may be controlled at the DMA cores using any one of the trigger sources mentioned below.

#### Trigger Interface (TRIG)

These signals provide trigger sources for TDD, MCS and other system level synchronization of multiple instances and the ADRV9001 devices. These may be sourced externally or internally. They may also be optionally used to trigger external peripherals such as DMA. The following table summarizes the trigger sources.

---
| Port         | Trigger Source    | Description |
| ------------ | ----------------- | ----------- |
| trig\_out[0] | External-Carrier  | This is trigger input 0. |
| trig\_out[1] | External-Carrier  | This is trigger input 1. |
| trig\_out[2] | External-ADRV9001 | This is internal MCS (either sourced by ADRV9001 or this core). |
| trig\_out[3] | Internal-ADRV9001 | This can be sourced from any of the 12 digital GPIO pins from the device. |
||||
---

All trigger sources independently support the following modes.

---
| Mode | Description |
| ---- | ----------- |
| 0x0  | Disabled, all events on this signal are ignored. |
| 0x1  | Edge triggered, the function (e.g. TDD, DMA, and such) starts at either edge of the signal. |
| 0x2  | Rising edge triggered, function starts at the next zero to one transition. |
| 0x3  | Falling edge triggered, function starts at the next one to zero transition. |
||||
---

#### User GPIO Interface (USR\_GPIO)

These signals provide general purpose control/monitor functionalities within the interface core as desired by software and downstream modules.

## <a name="parameters"></a>Parameters

---
| Parameter                        | Type    | Default | Description |
| -------------------------------- | ------- | ------- | ----------- |
| AXI\_ADRV9001\_INSTANCE\_ID      | Integer | 0       | The instance ID of this core. If system has more than one instances (MIMO) assign unique values to identify the cores in software. Also make sure to assign one of them as the master instance (0x0). |
| AXI\_ADRV9001\_DEVICE\_TYPE\_ID  | String  | 7SERIES | Also supports ULTRASCALE for Zynq MP SoC (e.g. ZCU102). |
| AXI\_ADRV9001\_MCS\_OUT\_CMOS1\_LVDS0  | Integer | 0  | The IO configuration of MCS output signal, for CMOS set it to 0x1. |
| AXI\_ADRV9001\_RX0\_GPIO2\_CMOS1\_LVDS0 | Integer | 0       | The IO configuration of receive 0 instance, for GPIO set to 0x2, CMOS mode set to 0x1, LVDS set to 0x0. |
| AXI\_ADRV9001\_TX0\_GPIO2\_CMOS1\_LVDS0 | Integer | 0       | The IO configuration of transmit 0 instance, for GPIO set to 0x2, CMOS mode set to 0x1, LVDS set to 0x0. |
| AXI\_ADRV9001\_RX1\_GPIO2\_CMOS1\_LVDS0 | Integer | 0       | The IO configuration of receive 1 instance, for GPIO set to 0x2, CMOS mode set to 0x1, LVDS set to 0x0. |
| AXI\_ADRV9001\_TX1\_GPIO2\_CMOS1\_LVDS0 | Integer | 0       | The IO configuration of transmit 1 instance, for GPIO set to 0x2, CMOS mode set to 0x1, LVDS set to 0x0. |
| AXI\_ADRV9001\_RX0\_IODELAY\_CTRL\_ENABLE  | Integer  | 1 | If set, IODELAY controller for RX0 is instantiated. |
| AXI\_ADRV9001\_RX1\_IODELAY\_CTRL\_ENABLE  | Integer  | 0 | If set, IODELAY controller for RX1 is instantiated. |
| AXI\_ADRV9001\_TX0\_IODELAY\_CTRL\_ENABLE  | Integer  | 0 | If set, IODELAY controller for TX0 is instantiated. |
| AXI\_ADRV9001\_TX1\_IODELAY\_CTRL\_ENABLE  | Integer  | 0 | If set, IODELAY controller for TX1 is instantiated. |
| AXI\_ADRV9001\_RX0\_IODELAY\_GROUP  | String  | AXI\_ADRV9001\_IODELAY | The placement identifier for the RX0 IODELAY group. |
| AXI\_ADRV9001\_RX1\_IODELAY\_GROUP  | String  | AXI\_ADRV9001\_IODELAY | The placement identifier for the RX1 IODELAY group. |
| AXI\_ADRV9001\_TX0\_IODELAY\_GROUP  | String  | AXI\_ADRV9001\_IODELAY | The placement identifier for the TX0 IODELAY group. |
| AXI\_ADRV9001\_TX1\_IODELAY\_GROUP  | String  | AXI\_ADRV9001\_IODELAY | The placement identifier for the TX1 IODELAY group. |
|||||
---

## <a name="ports"></a>Ports

### <a name="external-system-ports"></a>External System Ports

These ports may optionally be connected internally if not available from the hardware.

---
| Port               | Direction/Size  | Description |
| ------------------ | --------------- | ----------- |
| sys\_delay\_clk    | Input           | The delay controller clock (200MHz/300MHz) clock. |
| sys\_ref\_clk\_p   | Input           | The reference clock input (usually on board clock generator). |
| sys\_ref\_clk\_n   | Input           | The reference clock input (usually on board clock generator). |
| sys\_mcs\_in\_p    | Input           | The MCS input, LVDS true (registered on the internal device clock). |
| sys\_mcs\_in\_n    | Input           | The MCS input, LVDS complementary (registered on the internal device clock). |
| sys\_mcs\_out\_p   | Output          | The MCS output, LVDS true (registered on the internal device clock). |
| sys\_mcs\_out\_n   | Output          | The MCS output, LVDS complementary (registered on the internal device clock). |
||||
---

### <a name="external-adrv9001-ports"></a>External ADRV9001 Ports

These ports must be connected to the ADRV9001 device. The data path IO are multi purpose ports, and depend on the IO configuration (GPIO, CMOS or LVDS). In GPIO mode, use only the cssi gpio pins and discard the other pins. In CMOS or LVDS mode, discard the cssi gpio pins. Also in CMOS mode, some of the pins can be configured as GPIO with direction as dictated by the data path. That is, the receive data pins becomes general purpose inputs, but can not be set as general purpose outputs.

---
| Port                                        | Direction/Size  | Description |
| ------------------------------------------- | --------------- | ----------- |
| dio\_rx0\_cssi\_clk\_lssi\_clk\_p           | Input           | Receive 0 clock (CMOS), clock true (LVDS). |            
| dio\_rx0\_cssi\_nc\_lssi\_clk\_n            | Input           | Receive 0 no-connect (CMOS), clock complementary (LVDS). |
| dio\_rx0\_cssi\_strobe\_lssi\_strobe\_p     | Input           | Receive 0 strobe (CMOS), strobe true (LVDS). |           
| dio\_rx0\_cssi\_nc\_lssi\_strobe\_n         | Input           | Receive 0 no-connect (CMOS), strobe complementary (LVDS). |
| dio\_rx0\_cssi\_data1\_lssi\_data0\_p       | Input           | Receive 0 data line 1 (CMOS), data line 0 true (LVDS). |
| dio\_rx0\_cssi\_data0\_lssi\_data0\_n       | Input           | Receive 0 data line 0 (CMOS), data line 0 complementary (LVDS). |
| dio\_rx0\_cssi\_data3\_lssi\_data1\_p       | Input           | Receive 0 data line 3 (CMOS), data line 1 true (LVDS). |
| dio\_rx0\_cssi\_data2\_lssi\_data1\_n       | Input           | Receive 0 data line 2 (CMOS), data line 1 complementary (LVDS). |
| dio\_rx0\_cssi\_gpio\_in                    | Input[8]        | Receive 0 GPIO inputs (not applicable in data mode). |
| dio\_rx0\_cssi\_gpio\_out                   | Output[8]       | Receive 0 GPIO outputs (not applicable in data mode). |
| dio\_rx0\_cssi\_gpio\_enb                   | Output[8]       | Receive 0 GPIO output enables (not applicable in data mode). |
| dio\_rx1\_cssi\_clk\_lssi\_clk\_p           | Input           | Receive 1 clock (CMOS), clock true (LVDS). |            
| dio\_rx1\_cssi\_nc\_lssi\_clk\_n            | Input           | Receive 1 no-connect (CMOS), clock complementary (LVDS). |
| dio\_rx1\_cssi\_strobe\_lssi\_strobe\_p     | Input           | Receive 1 strobe (CMOS), strobe true (LVDS). |           
| dio\_rx1\_cssi\_nc\_lssi\_strobe\_n         | Input           | Receive 1 no-connect (CMOS), strobe complementary (LVDS). |
| dio\_rx1\_cssi\_data1\_lssi\_data0\_p       | Input           | Receive 1 data line 1 (CMOS), data line 0 true (LVDS). |
| dio\_rx1\_cssi\_data0\_lssi\_data0\_n       | Input           | Receive 1 data line 0 (CMOS), data line 0 complementary (LVDS). |
| dio\_rx1\_cssi\_data3\_lssi\_data1\_p       | Input           | Receive 1 data line 3 (CMOS), data line 1 true (LVDS). |
| dio\_rx1\_cssi\_data2\_lssi\_data1\_n       | Input           | Receive 1 data line 2 (CMOS), data line 1 complementary (LVDS). |
| dio\_rx1\_cssi\_gpio\_in                    | Input[8]        | Receive 0 GPIO inputs (not applicable in data mode). |
| dio\_rx1\_cssi\_gpio\_out                   | Output[8]       | Receive 0 GPIO outputs (not applicable in data mode). |
| dio\_rx1\_cssi\_gpio\_enb                   | Output[8]       | Receive 0 GPIO output enables (not applicable in data mode). |
| dio\_tx0\_cssi\_refclk\_lssi\_refclk\_p     | Input           | Transmit 0 reference clock (CMOS), reference clock true (LVDS). |
| dio\_tx0\_cssi\_nc\_lssi\_refclk\_n         | Input           | Transmit 0 no-connect (CMOS), reference clock complementary (LVDS). |
| dio\_tx0\_cssi\_clk\_lssi\_clk\_p           | Output          | Transmit 0 clock (CMOS), clock true (LVDS). |            
| dio\_tx0\_cssi\_nc\_lssi\_clk\_n            | Output          | Transmit 0 no-connect (CMOS), clock complementary (LVDS). |
| dio\_tx0\_cssi\_strobe\_lssi\_strobe\_p     | Output          | Transmit 0 strobe (CMOS), strobe true (LVDS). |           
| dio\_tx0\_cssi\_nc\_lssi\_strobe\_n         | Output          | Transmit 0 no-connect (CMOS), strobe complementary (LVDS). |
| dio\_tx0\_cssi\_data1\_lssi\_data0\_p       | Output          | Transmit 0 data line 1 (CMOS), data line 0 true (LVDS). |
| dio\_tx0\_cssi\_data0\_lssi\_data0\_n       | Output          | Transmit 0 data line 0 (CMOS), data line 0 complementary (LVDS). |
| dio\_tx0\_cssi\_data3\_lssi\_data1\_p       | Output          | Transmit 0 data line 3 (CMOS), data line 1 true (LVDS). |
| dio\_tx0\_cssi\_data2\_lssi\_data1\_n       | Output          | Transmit 0 data line 2 (CMOS), data line 1 complementary (LVDS). |
| dio\_tx0\_cssi\_gpio\_in                    | Input[10]       | Transmit 0 GPIO inputs (not applicable in data mode). |
| dio\_tx0\_cssi\_gpio\_out                   | Output[10]      | Transmit 0 GPIO outputs (not applicable in data mode). |
| dio\_tx0\_cssi\_gpio\_enb                   | Output[10]      | Transmit 0 GPIO output enables (not applicable in data mode). |
| dio\_tx1\_cssi\_refclk\_lssi\_refclk\_p     | Input           | Transmit 1 reference clock (CMOS), reference clock true (LVDS). |
| dio\_tx1\_cssi\_nc\_lssi\_refclk\_n         | Input           | Transmit 1 no-connect (CMOS), reference clock complementary (LVDS). |
| dio\_tx1\_cssi\_clk\_lssi\_clk\_p           | Output          | Transmit 1 clock (CMOS), clock true (LVDS). |            
| dio\_tx1\_cssi\_nc\_lssi\_clk\_n            | Output          | Transmit 1 no-connect (CMOS), clock complementary (LVDS). |
| dio\_tx1\_cssi\_strobe\_lssi\_strobe\_p     | Output          | Transmit 1 strobe (CMOS), strobe true (LVDS). |           
| dio\_tx1\_cssi\_nc\_lssi\_strobe\_n         | Output          | Transmit 1 no-connect (CMOS), strobe complementary (LVDS). |
| dio\_tx1\_cssi\_data1\_lssi\_data0\_p       | Output          | Transmit 1 data line 1 (CMOS), data line 0 true (LVDS). |
| dio\_tx1\_cssi\_data0\_lssi\_data0\_n       | Output          | Transmit 1 data line 0 (CMOS), data line 0 complementary (LVDS). |
| dio\_tx1\_cssi\_data3\_lssi\_data1\_p       | Output          | Transmit 1 data line 3 (CMOS), data line 1 true (LVDS). |
| dio\_tx1\_cssi\_data2\_lssi\_data1\_n       | Output          | Transmit 1 data line 2 (CMOS), data line 1 complementary (LVDS). |
| dio\_tx1\_cssi\_gpio\_in                    | Input[10]       | Transmit 1 GPIO inputs (not applicable in data mode). |
| dio\_tx1\_cssi\_gpio\_out                   | Output[10]      | Transmit 1 GPIO outputs (not applicable in data mode). |
| dio\_tx1\_cssi\_gpio\_enb                   | Output[10]      | Transmit 1 GPIO output enables (not applicable in data mode). |
| dio\_rx0\_enable                            | Output          | Receive 0 Enable. | 
| dio\_rx1\_enable                            | Output          | Receive 1 Enable. |
| dio\_tx0\_enable                            | Output          | Transmit 0 Enable. | 
| dio\_tx1\_enable                            | Output          | Transmit 1 Enable. |
| dio\_dgpio                                  | InOut[12]       | Digital GPIO. |
| dio\_mode                                   | Output          | Mode. |
| dio\_resetb                                 | Output          | Reset (active low). |
| dio\_gp\_int                                | Input           | Interrupt. |
| dio\_dev\_clk                               | Input           | The reference clock from device. |
||||
---

### <a name="internal-master-slave-ports"></a>Internal Master/Slave Ports

These ports must be connected as outlined below for multiple instances in a master-slave configuration.

---
| Port                  | Direction/Size  | Description |
| --------------------- | --------------- | ----------- |
| gpio\_out\_clk        | Output          | Master GPIO clock source (to self and slave cores).|
| gpio\_clk             | Input           | Master and slave GPIO clock input (must be connected to a master gpio\_out\_clk).|
| dev\_out\_clk         | Output          | Master device clock source (to self and slave cores).|
| mcs\_out\_int         | Output[2]       | Master MCS source (to self and slave cores).|
| dev\_clk              | Input           | Master and slave device clock input (must be connected to a master dev\_out\_clk).|
| mcs\_int              | Input[2]        | Master and slave MCS input (must be connected to a master mcs\_out\_int).|
| tdd\_out\_clk         | Output          | Master TDD clock source (to self and slave cores).|
| tdd\_out\_resetn      | Output          | Master TDD reset source (to self and slave cores).|
| tdd\_out\_genb        | Output          | Master TDD global enable source (to self and slave cores).|
| tdd\_out\_state       | Output          | Master TDD state source (to self and slave cores).|
| tdd\_out\_last        | Output          | Master TDD last frame source (to self and slave cores).|
| tdd\_out\_clk\_cnt    | Output[32]      | Master TDD clock count source (to self and slave cores).|
| tdd\_out\_frm\_cnt    | Output[32]      | Master TDD frame count source (to self and slave cores).|
| tdd\_clk              | Input           | Master and slave TDD clock input (must be connected to a master tdd\_out\_clk).|
| tdd\_resetn           | Input           | Master and slave TDD reset input (must be connected to a master tdd\_out\_resetn).|
| tdd\_genb             | Input           | Master and slave TDD global enable input (must be connected to a master tdd\_out\_resetn).|
| tdd\_state            | Input           | Master and slave TDD state input (must be connected to a master tdd\_out\_state).|
| tdd\_last             | Input           | Master and slave TDD last frame input (must be connected to a master tdd\_out\_state).|
| tdd\_clk\_cnt         | Input[32]       | Master and slave TDD clock count input (must be connected to a master tdd\_out\_clk\_cnt).|
| tdd\_frm\_cnt         | Input[32]       | Master and slave TDD frame count input (must be connected to a master tdd\_out\_frm\_cnt).|
| tx\_menb\_out         | Output          | Master and slave transmit enable (to self and slave cores). |
| rx\_menb\_out         | Output          | Master and slave receive enable (to self and slave cores). |
| orx\_menb\_out        | Output          | Master and slave observation receive enable (to self and slave cores). |
| tx\_menb\_in          | Input           | Master and slave transmit enable input (must be connected to a master tx\_menb\_out). |
| rx\_menb\_in          | Input           | Master and slave receive enable input (must be connected to a master rx\_menb\_out). |
| orx\_menb\_in         | Input           | Master and slave observation receive enable input (must be connected to a master orx\_menb\_out). |

||||
---

### <a name="internal-data-ports"></a>Internal Data Ports

These ports may be connected to the corresponding DMA modules or other DSP cores.

---
| Port                  | Direction/Size  | Description |
| --------------------- | --------------- | ----------- |
| rx0\_resetn           | Output          | Receive 0 reset (active low). |
| rx0\_clk              | Output          | Receive 0 clock. | 
| rx0\_senb             | Output          | Receive 0 sample enable. |
| rx0\_strig            | Output          | Receive 0 sample enable trigger. |
| rx0\_svld             | Output          | Receive 0 sample valid. |
| rx0\_sd               | Output[64]      | Receive 0 digital samples, see format below. |
| orx0\_senb            | Output          | Observation receive 0 sample enable. |
| orx0\_strig           | Output          | Observation receive 0 sample enable trigger. |
| orx0\_svld            | Output          | Observation receive 0 sample valid. |
||||
| rx1\_resetn           | Output          | Receive 1 reset (active low). |
| rx1\_clk              | Output          | Receive 1 clock. |
| rx1\_senb             | Output          | Receive 1 sample enable. |
| rx1\_strig            | Output          | Receive 1 sample enable trigger. |
| rx1\_svld             | Output          | Receive 1 sample valid. |
| rx1\_sd               | Output[64]      | Receive 1 digital samples, see format below. |
| orx1\_senb            | Output          | Observation receive 1 sample enable. |
| orx1\_strig           | Output          | Observation receive 1 sample enable trigger. |
| orx1\_svld            | Output          | Observation receive 1 sample valid. |
| orx1\_sd              | Output[64]      | Observation receive 1 digital samples, see format below. |
||||
| tx0\_resetn           | Output          | Transmit 0 reset (active low). |
| tx0\_clk              | Output          | Transmit 0 clock. |
| tx0\_senb             | Output          | Transmit 0 sample enable. |
| tx0\_strig            | Output          | Transmit 0 sample enable trigger. |
| tx0\_srd              | Output          | Transmit 0 sample read. |
| tx0\_sav              | Input           | Transmit 0 sample available. |
| tx0\_svld             | Input           | Transmit 0 sample valid. |
| tx0\_sd               | Input[64]       | Transmit 0 digital samples, see format belw. |
||||
| tx1\_resetn           | Output          | Transmit 1 reset (active low). |
| tx1\_clk              | Output          | Transmit 1 clock. |
| tx1\_senb             | Output          | Transmit 1 sample enable. |
| tx1\_strig            | Output          | Transmit 1 sample enable trigger. |
| tx1\_srd              | Output          | Transmit 1 sample read. |
| tx1\_sav              | Input           | Transmit 1 sample available. |
| tx1\_svld             | Input           | Transmit 1 sample valid. |
| tx1\_sd               | Input[64]       | Transmit 1 digital samples, see format belw. |
||||
| gpio\_resetn          | Output          | GPIO reset (active low). |
| gpio\_gain\_senb      | Output          | GPIO gain enable. |
| gpio\_gain\_strig     | Output          | GPIO gain enable trigger. |
| gpio\_gain\_svld      | Output          | GPIO gain valid. |
| gpio\_gain\_sd        | Output[8]       | GPIO gain data. |
||||

### Data Format (Transmit do not support 64bit sample width).

---
| Sample Width          | Data Format     | Sample Order    |
| --------------------- | --------------- | --------------- |
| 2                     | sd[57:56]       | Sample (N + 7). |
|                       | sd[49:48]       | Sample (N + 6). |
|                       | sd[41:40]       | Sample (N + 5). |
|                       | sd[33:32]       | Sample (N + 4). |
|                       | sd[25:24]       | Sample (N + 3). |
|                       | sd[17:16]       | Sample (N + 2). |
|                       | sd[ 9: 8]       | Sample (N + 1). |
|                       | sd[ 1: 0]       | Sample (N + 0). |
||||
| 8                     | sd[63:56]       | Sample (N + 7). |
|                       | sd[55:48]       | Sample (N + 6). |
|                       | sd[47:40]       | Sample (N + 5). |
|                       | sd[39:32]       | Sample (N + 4). |
|                       | sd[31:24]       | Sample (N + 3). |
|                       | sd[23:16]       | Sample (N + 2). |
|                       | sd[15: 8]       | Sample (N + 1). |
|                       | sd[ 7: 0]       | Sample (N + 0). |
||||
| 16                    | sd[63:48]       | Sample (N + 3). |
|                       | sd[47:32]       | Sample (N + 2). |
|                       | sd[31:16]       | Sample (N + 1). |
|                       | sd[15: 0]       | Sample (N + 0). |
||||
| 24                    | sd[59:48]       | Sample-I (N + 1). |
|                       | sd[43:32]       | Sample-Q (N + 1). |
|                       | sd[27:16]       | Sample-I (N + 0). |
|                       | sd[11: 0]       | Sample-Q (N + 0). |
||||
| 32                    | sd[63:48]       | Sample-I (N + 1). |
|                       | sd[47:32]       | Sample-Q (N + 1). |
|                       | sd[31:16]       | Sample-I (N + 0). |
|                       | sd[15: 0]       | Sample-Q (N + 0). |
||||
| 64                    | sd[63:32]       | Sample-I (N). |
|                       | sd[31: 0]       | Sample-Q (N). |
||||
---

### <a name="internal-trigger-gpio-ports"></a>Internal Trigger/GPIO Ports

---
| Port                  | Direction/Size  | Description |
| --------------------- | --------------- | ----------- |
| trig                  | InOut[2]        | This input is taken as an asynchronous trigger source or destination. |
| trig\_out             | Output[4]       | This output is a collection of trigger sources for external peripherals and the internal timing references such as TDD. |
| usr\_gpio\_in         | Input[32]       | This input is passed to the software for indirect control of downstream instances. |
| usr\_gpio\_out        | Output[32]      | This output is passed down from software for indirect control of downstream instances. |
||||
---

### <a name="internal-axilite-ports"></a>Internal AXILite Ports

---
| Port                  | Direction/Size  | Description |
| --------------------- | --------------- | ----------- |
| axilite\_clk          | Input           | Processor clock. |
| axilite\_resetn       | Input           | Processor reset (active low). |
| axilite\_awvalid      | Input           | Write address channel, valid. |
| axilite\_awaddr       | Input[16]       | Write address channel, address. |
| axilite\_awready      | Output          | Write address channel, ready. |
| axilite\_wvalid       | Input           | Write channel, valid. |
| axilite\_wdata        | Input[32]       | Write channel, data. | 
| axilite\_wready       | Output          | Write channel, ready. |
| axilite\_bvalid       | Output          | Write response channel, valid. |
| axilite\_bresp        | Output[2]       | Write response channel, data (response). |
| axilite\_bready       | Input           | Write response channel, ready. |
| axilite\_arvalid      | Input           | Read address channel, valid. |
| axilite\_araddr       | Input[16]       | Read address channel, address. |
| axilite\_arready      | Output          | Read address channel, ready. |
| axilite\_rvalid       | Output          | Read data channel, valid. |
| axilite\_rresp        | Output[2]       | Read data channel, response. |
| axilite\_rdata        | Output[32]      | Read data channel, data. |
| axilite\_rready       | Input           | Read data channel, ready. |
| interrupt             | Output          | Processor interrupt (active high). |
||||
---

## <a name="register-map"></a>Register Map

---
| Address offset                             | Instance                                     | Description                                             |
| ------------------------------------------ | -------------------------------------------- | ------------------------------------------------------- |
| [0x0000](hdl/axi_adrv9001_regs.md)         | [Main](hdl/axi_adrv9001_regs.md)             | Top level, primary IO controls (reset, mode, mcs etc.). |
| [0x1000](hdl/axi_adrv9001_gpio_regs.md)    | [GPIO](hdl/axi_adrv9001_gpio_regs.md)        | GPIO control (including DGPIO and SSI).                 |
| [0x4000](hdl/axi_adrv9001_rx_regs.md)      | [RECEIVE-0](hdl/axi_adrv9001_rx_regs.md)     | Receive 0 data path only (CMOS/LVDS).                   |
| [0x5000](hdl/axi_adrv9001_rx_regs.md)      | [RECEIVE-1](hdl/axi_adrv9001_rx_regs.md)     | Receive 1 data path only (CMOS/LVDS).                   |
| [0x6000](hdl/axi_adrv9001_tx_regs.md)      | [TRANSMIT-0](hdl/axi_adrv9001_tx_regs.md)    | Transmit 0 data path only (CMOS/LVDS).                  |
| [0x7000](hdl/axi_adrv9001_tx_regs.md)      | [TRANSMIT-1](hdl/axi_adrv9001_tx_regs.md)    | Transmit 1 data path only (CMOS/LVDS).                  |
| [0x8000](hdl/axi_adrv9001_tdd_regs.md)     | [TDD](hdl/axi_adrv9001_tdd_regs.md)          | TDD frame timing and device enable timings.             |
| [0x9000](hdl/axi_adrv9001_tdd_dma_regs.md) | [TDD-DMA](hdl/axi_adrv9001_tdd_dma_regs.md)  | TDD DMA enable timings.                                 |
||||
---

## <a name="system-integration"></a>System integration

A typical system level port connections is shown below.

![AXI\_ADRV9001 System Integration](docs/axi_adrv9001_system.png)

A typical use case in Xilinx Vivado IPI board design.

![AXI\_ADRV9001 Vivado IPI](docs/axi_adrv9001_vivado.png)


