## ##################################################################################
## ##################################################################################
## ----------------------------------------------------------------------------------
## ################
## ##   ###########   Analog Devices Inc.
## ##      ########
## ##         #####   Copyright (c) 2019 Analog Devices Inc. All rights reserved.
## ##            ##   This file is the confidential and proprietary property of ADI.
## ##         #####   Possession or use of this file requires a written license.
## ##      ########   The licensing information may be found at: www.analog.com
## ##   ###########
## ################
## ----------------------------------------------------------------------------------
## Description:       Vivado TCL include
## ----------------------------------------------------------------------------------
## ##################################################################################
## ##################################################################################
## General

variable m_root
set m_root .

if [info exists ::env(M_ROOT)] {
    set m_root $::env(M_ROOT)
}

## ##################################################################################
## ##################################################################################
## IP Packaging

proc add_ip_cores {file_list file_group} {

    set m_file_group [ipx::get_file_groups -filter "name =~ *${file_group}*"]
    foreach m_file $file_list {
        set m_core [get_ipdefs -all *${m_file}* -filter {upgrade_versions == ""}]
        ipx::add_subcore $m_core $m_file_group
    }
}

proc add_ip_files {file_list file_group} {

    set m_file_group [ipx::get_file_groups -filter "name =~ *${file_group}*"]
    foreach m_file $file_list {
        ipx::add_file $m_file $m_file_group
    }
}

proc pkg_ip_core {ip_name} {

    ipx::package_project -root_dir . -vendor analog.com \
        -library user -taxonomy /Analog_Devices
    set_property name $ip_name [ipx::current_core]
    set_property vendor_display_name {Analog Devices} [ipx::current_core]
    set_property company_url {http://www.analog.com} [ipx::current_core]
    set_property auto_family_support_level level_2 [ipx::current_core]
    ipx::save_core
}

proc add_interface_ports {name dir width {type none}} {

    ipx::add_bus_abstraction_port $name [ipx::current_busabs]
    set if_param_list {}
    lappend if_param_list master_presence required
    lappend if_param_list slave_presence required
    if {$width > 0} {
        lappend if_param_list master_width $width
        lappend if_param_list slave_width $width
    }
    lappend if_param_list master_direction $dir
    if {$dir eq "in"} {
        lappend if_param_list slave_direction out
    } elseif {$dir eq "out"} {
        lappend if_param_list slave_direction in
    } else {
        lappend if_param_list slave_direction $dir
    }
    if {$type ne "none"} {
        lappend if_param_list is_${type} true
    }
    set_property -dict $if_param_list [ipx::get_bus_abstraction_ports $name \
        -of_objects [ipx::current_busabs]]
}

proc pkg_interface {if_name} {

    ipx::create_abstraction_definition analog.com interface ${if_name}_rtl 1.0
    ipx::create_bus_definition analog.com interface ${if_name} 1.0
    set_property xml_file_name interfaces/${if_name}_rtl.xml [ipx::current_busabs]
    set_property xml_file_name interfaces/${if_name}.xml [ipx::current_busdef]
    set_property bus_type_vlnv analog.com:interface:${if_name}:1.0 [ipx::current_busabs]
    ipx::save_abstraction_definition [ipx::current_busabs]
    ipx::save_bus_definition [ipx::current_busdef]
}

## ##################################################################################
## ##################################################################################
## Ignored warnings and such

proc msg_settings {mode} {

    if {$mode eq "none"} {
        set_msg_config -id {Vivado 12-1462} -new_severity INFO
        set_msg_config -id {Common 17-741} -new_severity INFO
        set_msg_config -id {BD 41-238} -new_severity INFO
        set_msg_config -id {IP_Flow 19-3507} -new_severity INFO
        set_msg_config -id {IP_Flow 19-3656} -new_severity INFO
        set_msg_config -id {IP_Flow 19-2999} -new_severity INFO
        set_msg_config -id {IP_Flow 19-1654} -new_severity INFO
        set_msg_config -id {IP_Flow 19-4623} -new_severity INFO
        set_msg_config -id {IP_Flow 19-3833} -new_severity INFO
        set_msg_config -id {IP_Flow 19-1971} -new_severity INFO
        set_msg_config -id {IP_Flow 19-459} -new_severity INFO
        set_msg_config -id {IP_Flow 19-731} -new_severity INFO
        set_msg_config -id {Timing 38-282} -new_severity ERROR
    }
}

proc gen_system_id {s} {

    set s [format "%-32s" $s]
    set m "0x"
    for {set x 0} {$x < 8} {incr x} {
        set w [string range $s [expr ($x*4)] [expr (($x*4)+3)]]
        for {set y 3} {$y >= 0} {incr y -1} {
            set c [string range $w $y $y]
            scan $c "%c" b
            set b [format "%02x" $b]
            set m $m$b
        }
    }
    return $m
}

proc gen_build_id {} {

    set s [clock seconds]
    set s [clock format $s -format "%Y-%m-%d-%H-%M"]
    set s [gen_system_id $s]
    return $s
}

proc gen_build_version {} {

    set s [clock seconds]
    set s [clock format $s -format "0x%m%d%Y00%H%M%S"]
    return $s
}

## ##################################################################################
## ##################################################################################
## Version check

set TOOL_VERSION "2024.2"
set v [version -short]

if {[string compare $TOOL_VERSION $v] != 0} {
    return -code error [format "ERROR: vivado version ($v) mismatch, required ($TOOL_VERSION)!"]
}

## ##################################################################################
## ##################################################################################
## Peripherals

variable addrmap_list
set addrmap_list {}

variable axilite_index
variable axilite_addr
variable axilite_mask
variable axilite_cpu_intf
set axilite_index 0
set axilite_addr 0xa0000000
set axilite_mask 0x0fffffff

variable addrmap_offset
variable addrmap_addr
set addrmap_offset 0x0
set addrmap_addr 0x0

variable aximm_mem_intf
variable aximm_mem_index
set aximm_mem_intf {}
set aximm_mem_index 0

variable aximm_index
set aximm_index 0

variable intr_index
set intr_index -1

## ##################################################################################
## ##################################################################################

proc generate_address_map {ifile} {

    global addrmap_list

    set ip_define [file rootname [file tail $ifile]]
    set ip_define [string toupper $ip_define]
    set ip_file [open $ifile w]

    fconfigure $ip_file -translation lf

    puts $ip_file "// **********************************************************************************"
    puts $ip_file "// **********************************************************************************"
    puts $ip_file "// ----------------------------------------------------------------------------------"
    puts $ip_file "// ################"
    puts $ip_file "// ##   ###########   Analog Devices Inc."
    puts $ip_file "// ##      ########"
    puts $ip_file "// ##         #####   Copyright (c) 2019 Analog Devices Inc. All rights reserved."
    puts $ip_file "// ##            ##   This file is the confidential and proprietary property of ADI."
    puts $ip_file "// ##         #####   Possession or use of this file requires a written license."
    puts $ip_file "// ##      ########   The licensing information may be found at: www.analog.com"
    puts $ip_file "// ##   ###########"
    puts $ip_file "// ################"
    puts $ip_file "// ----------------------------------------------------------------------------------"
    puts $ip_file "// Description:       Auto generated, local modifications are discouraged."
    puts $ip_file "// ----------------------------------------------------------------------------------"
    puts $ip_file "// **********************************************************************************"
    puts $ip_file "// **********************************************************************************"
    puts $ip_file ""
    puts $ip_file "#ifndef ${ip_define}_H"
    puts $ip_file "#define ${ip_define}_H"
    puts $ip_file ""

    puts $ip_file "#ifdef ${ip_define}_BASE"
    puts $ip_file ""
    foreach {ip_name ip_addr ip_offset} $addrmap_list {
        puts $ip_file "#define ${ip_name}_ID $ip_addr"
    }
    puts $ip_file ""
    puts $ip_file "#endif"
    puts $ip_file ""

    puts $ip_file "#ifdef ${ip_define}_OFFSET"
    puts $ip_file ""
    foreach {ip_name ip_addr ip_offset} $addrmap_list {
        puts $ip_file "#define ${ip_name}_ID $ip_offset"
    }
    puts $ip_file ""
    puts $ip_file "#endif"
    puts $ip_file ""

    puts $ip_file "#endif"
    puts $ip_file ""

    puts $ip_file "// **********************************************************************************"
    puts $ip_file "// **********************************************************************************"
    puts $ip_file ""

    close $ip_file
}

proc add_addr_seg {addr m_intf p_seg p_name} {

    set m_space [get_bd_addr_spaces -of_objects [get_bd_intf_pins $m_intf]]
    set m_segment [get_bd_addr_segs $p_seg]
    set m_range [get_property range $m_segment]
    set m_mname [lindex [split $m_intf /] 0]

    if {$m_range < 0x1000} {
        set m_range 0x1000
    }

    create_bd_addr_seg -offset $addr -range $m_range $m_space $m_segment seg_${m_mname}_${p_name}
}

proc add_addr_map {addr m_intf p_intf} {

    set m_space [get_bd_addr_spaces -of_objects [get_bd_intf_pins $m_intf]]
    set m_segment [get_bd_addr_segs -of_objects [get_bd_intf_pins $p_intf]]
    set m_range [get_property range $m_segment]
    set m_mname [lindex [split $m_intf /] 0]
    set m_pname [regsub / $p_intf _]

    if {$m_range < 0x1000} {
        set m_range 0x1000
    }

    create_bd_addr_seg -offset $addr -range $m_range $m_space $m_segment seg_${m_mname}_${m_pname}
}

proc add_axilite_cpu {ip_intf} {

    global axilite_cpu_intf

    set ip_param_list {}
    set ip_vlnv [get_ipdefs -all -filter {name =~ smartconnect && upgrade_versions == ""}]
    create_bd_cell -type ip -vlnv $ip_vlnv sys_axilite_xc
    lappend ip_param_list config.num_si 1
    lappend ip_param_list config.num_mi 1
    set_property -dict $ip_param_list [get_bd_cells sys_axilite_xc]
    connect_bd_net -net axilite_resetn [get_bd_pins sys_axilite_xc/aresetn]
    connect_bd_net -net axilite_clk [get_bd_pins sys_axilite_xc/aclk]
    connect_bd_intf_net [get_bd_intf_pins $ip_intf] [get_bd_intf_pins sys_axilite_xc/s00_axi]
    set axilite_cpu_intf $ip_intf
}

proc add_axilite_peripheral {ip_intf} {

    global axilite_index
    global axilite_addr
    global axilite_mask
    global axilite_cpu_intf
    global addrmap_offset
    global addrmap_addr

    set m_addr [expr $axilite_addr + ($axilite_index * 0x10000)]
    set m_addr [format "0x%08x" $m_addr]
    set m_index $axilite_index
    set m_intc sys_axilite_xc
    set addrmap_offset [expr ($axilite_index * 0x10000)]
    set addrmap_addr [expr $axilite_addr + $addrmap_offset]
    set addrmap_offset [expr ($addrmap_addr & $axilite_mask)]

    if {$axilite_index == 15} {

        set ip_param_list {}
        set ip_vlnv [get_ipdefs -all -filter {name =~ smartconnect && upgrade_versions == ""}]
        create_bd_cell -type ip -vlnv $ip_vlnv sys_axilite_xc_1
        lappend ip_param_list config.num_si 1
        lappend ip_param_list config.num_mi 1
        set_property -dict $ip_param_list [get_bd_cells sys_axilite_xc_1]
        connect_bd_net -net axilite_resetn [get_bd_pins sys_axilite_xc_1/aresetn]
        connect_bd_net -net axilite_clk [get_bd_pins sys_axilite_xc_1/aclk]

        set_property config.num_mi 16 [get_bd_cells sys_axilite_xc]
        connect_bd_intf_net [get_bd_intf_pins sys_axilite_xc/m15_axi] [get_bd_intf_pins sys_axilite_xc_1/s00_axi]
    }

    if {$axilite_index >= 15} {

        set m_index [expr $axilite_index - 15]
        set m_intc sys_axilite_xc_1
    }

    set_property config.num_mi [expr $m_index + 1] [get_bd_cells ${m_intc}]
    set m_intf m${m_index}_axi
    if {$m_index < 10} {
        set m_intf m0${m_index}_axi
    }

    connect_bd_intf_net [get_bd_intf_pins ${m_intc}/${m_intf}] [get_bd_intf_pins $ip_intf]
    add_addr_map $m_addr $axilite_cpu_intf $ip_intf

    set axilite_index [expr $axilite_index + 1]
}

proc add_cpu_addrmap {name {offset 0x0}} {

    global addrmap_offset
    global addrmap_addr
    global addrmap_list

    lappend addrmap_list [string toupper $name]
    lappend addrmap_list [format "0x%08x" [expr $addrmap_addr + $offset]]
    lappend addrmap_list [format "0x%08x" [expr $addrmap_offset + $offset]]
}

proc append_cpu_addrmap {addr offset name} {

    global addrmap_list

    lappend addrmap_list [string toupper $name]
    lappend addrmap_list [format "0x%08x" $addr]
    lappend addrmap_list [format "0x%08x" $offset]
}

proc add_aximm_memory {ip_addr ip_intf {mem_intf ""}} {

    global aximm_mem_intf
    global aximm_mem_index
    global addrmap_offset
    global addrmap_addr

    set addrmap_offset 0x0
    set addrmap_addr $ip_addr

    if {$aximm_mem_index == 0} {

        set ip_param_list {}
        set ip_vlnv [get_ipdefs -all -filter {name =~ smartconnect && upgrade_versions == ""}]
        create_bd_cell -type ip -vlnv $ip_vlnv sys_aximm_xc
        lappend ip_param_list config.num_si 1
        lappend ip_param_list config.num_mi 1
        set_property -dict $ip_param_list [get_bd_cells sys_aximm_xc]
        connect_bd_net -net aximm_clk [get_bd_pins sys_aximm_xc/aclk]
        connect_bd_net -net aximm_resetn [get_bd_pins sys_aximm_xc/aresetn]
    }

    set_property config.num_mi [expr $aximm_mem_index + 1] [get_bd_cells sys_aximm_xc]
    set m_intf m${aximm_mem_index}_axi
    if {$aximm_mem_index < 10} {
        set m_intf m0${aximm_mem_index}_axi
    }

    connect_bd_intf_net [get_bd_intf_pins sys_aximm_xc/${m_intf}] [get_bd_intf_pins $ip_intf]

    set m_intf $ip_intf
    if {$mem_intf ne ""} {
        set m_intf $mem_intf
    }

    lappend aximm_mem_intf [list $ip_addr $m_intf]
    set aximm_mem_index [expr $aximm_mem_index + 1]
}

proc add_aximm_peripheral {ip_intf} {

    global aximm_index
    global aximm_mem_intf
    global aximm_mem_index

    set m_name [lindex [split $ip_intf /] 0]
    set m_index $aximm_index
    set m_intc sys_aximm_xc

    if {$aximm_index == 15} {

        set ip_param_list {}
        set ip_vlnv [get_ipdefs -all -filter {name =~ smartconnect && upgrade_versions == ""}]
        create_bd_cell -type ip -vlnv $ip_vlnv sys_aximm_xc_1
        lappend ip_param_list config.num_si 1
        lappend ip_param_list config.num_mi 1
        set_property -dict $ip_param_list [get_bd_cells sys_aximm_xc_1]
        connect_bd_net -net aximm_clk [get_bd_pins sys_aximm_xc_1/aclk]
        connect_bd_net -net aximm_resetn [get_bd_pins sys_aximm_xc_1/aresetn]

        set_property config.num_si 16 [get_bd_cells sys_aximm_xc]
        connect_bd_intf_net [get_bd_intf_pins sys_aximm_xc_1/m00_axi] [get_bd_intf_pins sys_aximm_xc/s15_axi]
    }

    if {$aximm_index >= 15} {

        set m_index [expr $aximm_index - 15]
        set m_intc sys_aximm_xc_1
    }

    set_property config.num_si [expr $m_index + 1] [get_bd_cells ${m_intc}]
    set m_intf s${m_index}_axi
    if {$m_index < 10} {
        set m_intf s0${m_index}_axi
    }

    connect_bd_intf_net [get_bd_intf_pins $ip_intf] [get_bd_intf_pins ${m_intc}/${m_intf}]

    for {set m 0} {$m < $aximm_mem_index} {incr m} {
        add_addr_map [lindex $aximm_mem_intf $m 0] $ip_intf [lindex $aximm_mem_intf $m 1]
    }

    set aximm_index [expr $aximm_index + 1]
}

proc add_interrupt {ip_intr} {

    global intr_index

    set intr_index [expr $intr_index + 1]
    set_property config.num_ports [expr $intr_index + 1] [get_bd_cells sys_intrs]
    set_property config.intr_width [expr $intr_index + 1] [get_bd_cells axi_sysid]
    connect_bd_net [get_bd_pins $ip_intr] [get_bd_pins sys_intrs/in${intr_index}]
}

## ##################################################################################
## ##################################################################################

