#####################################################################################################
#   Before running, ensure that 'c_src' folder is in same level as this script and that c_src       #
#       has been compiled with a makefile that outputs obj files and uses -fstack-usage in CFLAGS   #
#   Valgrind should be installed if memory analysis on executable is desired, can exit otherwise    #
#   Compatible with Python 3.7+                                                                     #
#####################################################################################################

import os
import subprocess
from operator import itemgetter

def buffer(list_to_buff):
    max_lengths = []
    for i in range(len(list_to_buff[0])):
        max_lengths.append(max([len(item[i]) for item in list_to_buff]))
    return max_lengths

cwd = os.path.dirname(os.path.realpath(__file__))

def get_parsed_data(dir_name):
    d = {}

    d['object_file_paths'] = []
    d['parsed_data'] = []
    obj_file_found = False
    su_file_found = False
    for (dirpath, dirnames, filenames) in os.walk(os.path.join(cwd, dir_name)):
        for f in filenames:
            filename, file_extension = os.path.splitext(f)
            if file_extension == ".o":
                obj_file_found = True
                d['object_file_paths'].append(os.path.abspath(os.path.join(dirpath, f)))
            elif file_extension == ".su":
                su_file_found = True
                with open(os.path.abspath(os.path.join(dirpath, f)), "r") as file:
                    data = file.read().split()
                    d['parsed_data'].extend([data[i:i+3] for i in range(0, len(data), 3)])

    if not obj_file_found or not su_file_found:
        input(dir_name + "/ contains no obj files or no su files, make sure to compile" + dir_name + "with makefile. Closing program...")
        exit()

    if not os.path.isdir(os.path.join(cwd, dir_name)):
        input("\'" + dir_name + "\' folder not found in current directory. Closing program...")
        exit()

    return d

def calculate_memory(executable_filename, d):
    d['su_buffs'] = buffer(d['parsed_data'])
    d['parsed_data'] = [(func_name, int(num_of_bytes), static_dynamic) for (func_name, num_of_bytes, static_dynamic) in d['parsed_data']]
    d['total_bytes'] = sum([item[1] for item in d['parsed_data']])
    d['parsed_data'].sort(key=itemgetter(1), reverse=True)

    obj_file_sizes_raw = subprocess.check_output(["size", "-t"] + d['object_file_paths']).decode('utf-8').split('\n')
    obj_file_sizes = obj_file_sizes_raw[1:-2]
    parsed_obj_sizes = [line.split() for line in obj_file_sizes]
    parsed_obj_sizes = [(text, data, bss, int(dec), hexa, filename) for (text, data, bss, dec, hexa, filename) in parsed_obj_sizes]
    sortter = zip(parsed_obj_sizes, obj_file_sizes)
    _, obj_file_sizes = zip(*sorted(sortter, key=lambda x: x[0][3], reverse=True))
    d['sorted_obj_data'] = [obj_file_sizes_raw[0]] + list(obj_file_sizes) + [obj_file_sizes_raw[-2]]
    d['sorted_obj_data'] = "\n".join(d['sorted_obj_data'])

    executable_func_sizes = subprocess.check_output(["nm", "-CSr", "--size-sort", "-t", "d", os.path.join(cwd, executable_filename)]).decode('utf-8').split("\n")[:-1]
    executable_func_sizes = [item.split() for item in executable_func_sizes]
    d['executable_total'] = sum([int(item[1]) for item in executable_func_sizes])

    return d

def save_memory_check(executable_filename, output_filename, d, run_valgrind):
    with open(output_filename, "w") as file:
        print('Saving summary results: ' + output_filename)
        seperator = "-" * 150 + "\n"
        file.write("Size of Object Files:\n" + seperator)
        file.write(d['sorted_obj_data'] + "\n" + seperator * 2)
        file.write("\n\nStack Usage of Individual Functions:\n")
        for (func_name, num_of_bytes, static_dynamic) in d['parsed_data']:
            file.write("%s %s %s\n" % (func_name.ljust(d['su_buffs'][0]), str(num_of_bytes).rjust(d['su_buffs'][1]), static_dynamic.rjust(d['su_buffs'][2])))
        file.write("-" * (sum(d['su_buffs']) + 3))
        file.write("\nTotal Number of Bytes: %s\n" % (str(d['total_bytes'])))
        file.write(seperator * 2)
        file.write("Size of Executable:\n")
        file.write(subprocess.check_output(["size", os.path.join(cwd, executable_filename)]).decode('utf-8'))
        file.write(seperator * 2)
        file.write("Size of Used Functions in Executable:\n" + seperator)
        file.write(subprocess.check_output(["nm", "-CSr", "--size-sort", "-t", "d", os.path.join(cwd, executable_filename)]).decode('utf-8'))
        file.write("\nTotal Number of Bytes: %s\n" % (str(d['executable_total'])))

        if run_valgrind:
            try:
                file.write(subprocess.check_output(["valgrind", "--leak-check=full", os.path.join(cwd, executable_filename)], stderr=subprocess.STDOUT).decode('utf-8'))
            except subprocess.CalledProcessError as e:
                print("ERROR: It looks like Valgrind found some issues with the build.  Please investigate further.")

        print('Done.')

if __name__ == '__main__':
    # Temporarily disabled.
    #k = input("Do you want a memory report for:\n  1) Executable only\n  2) Executable and c_src\n[1]> ")
    k = "1"

    v = input("Do you want a run Valgrind (requires hardware connected)?\n  1) Yes\n  2) No\n[1]> ")
    run_valgrind = True
    if v == "2":
        run_valgrind = False

    print("\nList of folders in current working directory:")
    for folder in next(os.walk('.'))[1]:
        print("\t" + folder)

    valid_dir = False
    while not valid_dir:
        executable_filename = input("Please input the path to executable [" + os.getcwd() + "/test1/example]> " + cwd + os.sep).strip()
        if executable_filename == "":
            executable_filename = "test1/example"
        executable_filename = os.path.join(cwd, executable_filename)
        if not os.path.isfile(executable_filename):
            print("Not a valid path\n")
            continue
        valid_dir = True

    output_filename = os.path.join(cwd, "memory_profile__executable.txt")
    d = get_parsed_data(os.path.split(os.path.split(executable_filename)[0])[1])
    d = calculate_memory(executable_filename, d)
    save_memory_check(executable_filename, output_filename, d, run_valgrind)

    if k == '2':
        output_filename = os.path.join(cwd, "memory_profile_total.txt")
        d = get_parsed_data("c_src")

        save_memory_check(executable_filename, output_filename, d, run_valgrind)
