/**
* \file
* \brief linux_uio implementation of the FPGA9001 HAL
* 
* Copyright 2020-2025 Analog Devices Inc.
* Released under the ADRV9001 API license, for more information
* see the "LICENSE.txt" file in this zip file.
*/

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <sys/mman.h>
#include <string.h>
#include <dirent.h>

#include "adi_fpga9001_hal_linux_uio.h"
#include "adi_linux_uio_logging.h"
#include "fpga9001_utilities.h"

#define ADI_FPGA9001_UIO1_SIZE_ZC706  0x40000000 // 1GB
#define ADI_FPGA9001_UIO1_SIZE_ZCU102 0x20000000 // 512 MB

static const adi_hal_LogCfg_t fpga9001LogCfg = { 
    .interfaceEnabled = 1,
    .logfd = NULL,
    .logFileName = "fpga9001Log.txt",
    .currentLineNumber = 0
};

static const adi_fpga9001_hal_BbicCfg_t bbicCfg = { 
    .coreRegsCfg = { 
        .drvName = "/dev/uio0",
        .mapSize = 0x1000000    /* 16MB */
    },
    .ramCfg = { 
        .drvName = "/dev/uio1"
    }
};

static void getLatestVersion(char *binBaseName, char *versionStr);

adi_fpga9001_hal_linux_uio_Cfg_t *linux_uio_fpga9001_open()
{
    int32_t halError = 0;
    adi_fpga9001_Platform_e platform = ADI_FPGA9001_PLATFORM_UNSUPPORTED;
    adi_fpga9001_hal_linux_uio_Cfg_t *halCfg = (adi_fpga9001_hal_linux_uio_Cfg_t *)calloc(1, sizeof(adi_fpga9001_hal_linux_uio_Cfg_t));
    if (NULL == halCfg)
    {
        return NULL;
    }
    
    /* Copy default values */
    halCfg->logCfg = fpga9001LogCfg;
    halCfg->bbicCfg = bbicCfg;
    
    /* Open log file */
    if (0 != linux_uio_LogFileOpen(&halCfg->logCfg))
    {
        free(halCfg);
        return NULL;
    }
    
    /* Open UIO drivers */

    halCfg->bbicCfg.coreRegsCfg.fd = open(halCfg->bbicCfg.coreRegsCfg.drvName, (O_RDWR | O_SYNC));
    if (halCfg->bbicCfg.coreRegsCfg.fd < 0)
    {
        free(halCfg);
        return NULL;
    }

    halCfg->bbicCfg.coreRegsCfg.fptr = mmap(NULL, halCfg->bbicCfg.coreRegsCfg.mapSize,
        (PROT_READ | PROT_WRITE), MAP_SHARED, halCfg->bbicCfg.coreRegsCfg.fd, 0);
    if (halCfg->bbicCfg.coreRegsCfg.fptr == MAP_FAILED)
    {
        perror("Unable to memory map register space (uio-0)");
        close(halCfg->bbicCfg.coreRegsCfg.fd);
        free(halCfg);
        return NULL;
    }
    
    halError = linux_uio_adi_fpga9001_hal_CurrentPlatform_Get(halCfg, &platform);
    if (0 != halError)
    {
        return NULL;
    }

    switch (platform)
    {
    case ADI_FPGA9001_PLATFORM_ZC706:
        halCfg->bbicCfg.ramCfg.mapSize = ADI_FPGA9001_UIO1_SIZE_ZC706;
        break;
    case ADI_FPGA9001_PLATFORM_ZCU102:
        halCfg->bbicCfg.ramCfg.mapSize = ADI_FPGA9001_UIO1_SIZE_ZCU102;
        break;
    default:
        printf("Unrecognized platform\n");
        free(halCfg);
        return NULL;
    }
    
    halCfg->bbicCfg.ramCfg.fd = open(halCfg->bbicCfg.ramCfg.drvName, (O_RDWR | O_SYNC));
    if (halCfg->bbicCfg.ramCfg.fd < 0)
    {
        free(halCfg);
        return NULL;
    }
    
    halCfg->bbicCfg.ramCfg.fptr = mmap(NULL, halCfg->bbicCfg.ramCfg.mapSize,
        (PROT_READ | PROT_WRITE), MAP_SHARED, halCfg->bbicCfg.ramCfg.fd, 0);
    if (halCfg->bbicCfg.ramCfg.fptr == MAP_FAILED)
    {
        perror("Unable to memory map RAM space (uio-1)");
        close(halCfg->bbicCfg.ramCfg.fd);
        free(halCfg);
        return NULL;
    }
    
    return halCfg;
}

int32_t linux_uio_fpga9001_close(adi_fpga9001_hal_linux_uio_Cfg_t *devHalCfg)
{
    int32_t halError = 0;
    
    if (NULL == devHalCfg)
    {
        return -2;
    }
        
    /* Close log file */
    if (NULL != devHalCfg->logCfg.logfd)
    {
        halError = linux_uio_LogFileClose(&devHalCfg->logCfg);
        ADI_ERROR_RETURN(halError);
    }
    
    /* Close UIO drivers */
    if (-1 != devHalCfg->bbicCfg.coreRegsCfg.fd)
    {
        munmap(devHalCfg->bbicCfg.coreRegsCfg.fptr, devHalCfg->bbicCfg.coreRegsCfg.mapSize);
        halError = close(devHalCfg->bbicCfg.coreRegsCfg.fd);
        ADI_ERROR_RETURN(halError);
        
        devHalCfg->bbicCfg.coreRegsCfg.fd = -1;
    }
    
    if (-1 != devHalCfg->bbicCfg.ramCfg.fd)
    {
        munmap(devHalCfg->bbicCfg.ramCfg.fptr, devHalCfg->bbicCfg.ramCfg.mapSize);
        halError = close(devHalCfg->bbicCfg.ramCfg.fd);
        ADI_ERROR_RETURN(halError);
        
        devHalCfg->bbicCfg.ramCfg.fd = -1;
    }
    
    free(devHalCfg);
    return halError;
}

/* intentionally, NO null checks whatsoever-
 * first of all, program should not reach here if open and map have failed.
 * second, and most importantly, uio is opened as a single register space.
 * that is a crash or unexpected results, if you read the wrong hardware.
 * any null checking is meaningless, because they are only in software. */

int32_t linux_uio_adi_fpga9001_hal_register_read(void *devHalCfg, const uint32_t addr, uint32_t *data)
{
    adi_fpga9001_hal_linux_uio_Cfg_t *halCfg = NULL;

    halCfg = (adi_fpga9001_hal_linux_uio_Cfg_t *)devHalCfg;
    *data = *((volatile uint32_t*)(halCfg->bbicCfg.coreRegsCfg.fptr + addr));
    return(0);
}

int32_t linux_uio_adi_fpga9001_hal_register_write(void *devHalCfg, const uint32_t addr, uint32_t data)
{
    adi_fpga9001_hal_linux_uio_Cfg_t *halCfg = NULL;

    halCfg = (adi_fpga9001_hal_linux_uio_Cfg_t *)devHalCfg;
    *((volatile uint32_t *)(halCfg->bbicCfg.coreRegsCfg.fptr + addr)) = data;
    return(0);
}

int32_t linux_uio_adi_fpga9001_hal_ram_read(void *devHalCfg, const uint32_t ramAddress, uint32_t data[], uint32_t length)
{
    adi_fpga9001_hal_linux_uio_Cfg_t *halCfg = NULL;

    halCfg = (adi_fpga9001_hal_linux_uio_Cfg_t *)devHalCfg;
    memcpy(data, (void *)((uintptr_t)halCfg->bbicCfg.ramCfg.fptr + ramAddress), length * sizeof(uint32_t));
    return(0);
}

int32_t linux_uio_adi_fpga9001_hal_ram_write(void *devHalCfg, const uint32_t ramAddress, uint32_t data[], uint32_t length)
{
    adi_fpga9001_hal_linux_uio_Cfg_t *halCfg = NULL;

    halCfg = (adi_fpga9001_hal_linux_uio_Cfg_t *)devHalCfg;
    memcpy((void *)((uintptr_t)halCfg->bbicCfg.ramCfg.fptr + ramAddress), data, length * sizeof(uint32_t));
    return(0);
}

int32_t linux_uio_adi_fpga9001_hal_SsiType_Set(void *devHalCfg, adi_fpga9001_SsiType_e ssiType)
{
    int32_t halError = 0;
    char cmdBuf[256] = { 0 };
    char versionBuf[256] = { 0 };
    char binBasenameBuf[256] = { 0 };
    adi_fpga9001_Platform_e platform = ADI_FPGA9001_PLATFORM_UNSUPPORTED;

    halError = linux_uio_adi_fpga9001_hal_CurrentPlatform_Get((adi_fpga9001_hal_linux_uio_Cfg_t *)devHalCfg, &platform);
    if (0 != halError)
    {
        return halError;
    }

	sprintf(binBasenameBuf,
		"%s_%s_%s_",
		(ssiType == ADI_FPGA9001_SSI_TYPE_CMOS_COMMON_CLOCK || ssiType == ADI_FPGA9001_SSI_TYPE_LVDS_COMMON_CLOCK) ? "adrv9001cc" : "adrv9001",
		platform == ADI_FPGA9001_PLATFORM_ZC706 ? "zc706" : "zcu102",
		((ssiType == ADI_FPGA9001_SSI_TYPE_CMOS) || (ssiType == ADI_FPGA9001_SSI_TYPE_CMOS_COMMON_CLOCK)) ? "mcc" :
		            ((ssiType == ADI_FPGA9001_SSI_TYPE_LVDS) || (ssiType == ADI_FPGA9001_SSI_TYPE_LVDS_COMMON_CLOCK)) ? "mll" :
		            (ssiType == ADI_FPGA9001_SSI_TYPE_CH1CMOS_CH2LVDS) ? "mcl" : "mlc");

    system("pkill -9 mon_pb");
    
    system("echo 0 > /sys/class/fpga_manager/fpga0/flags");
    system("mkdir -p /lib/firmware");
    getLatestVersion(binBasenameBuf, versionBuf);
    sprintf(cmdBuf, "cp /home/analog/adrv9001_server/resources/Adi.Adrv9001.FpgaBinaries/%s%s.bin /lib/firmware/fpga.bin", binBasenameBuf, versionBuf);
    system(cmdBuf);
    system("echo fpga.bin > /sys/class/fpga_manager/fpga0/firmware");
    
    system("/home/analog/platform/mon_pb &");
    
    return (0);
}

static bool startsWith(const char *pre, const char *str)
{
    size_t lenpre = strlen(pre),
           lenstr = strlen(str);
    return lenstr < lenpre ? false : strncmp(pre, str, lenpre) == 0;
}

static void getVersion(const char *str, int version[])
{
    char *toFree = strdup(str);
    char *p;
    int i = 0;

    // tokenize from the char after the last '_' using '.' separator
    p = strtok(strrchr(toFree, '_') + 1, ".");
    while (p != NULL)
    {
        if (i < 3)
        {
            version[i] = atoi(p);
        }
        p = strtok(NULL, ".");
        i++;
    }

    free(toFree);
}

static bool versionLt(int l[], int r[])
{
    if (l[0] < r[0]) {
        return true;
    }
    else if (l[0] > r[0]) {
        return false;
    }
    else {
        if (l[1] < r[1]) {
            return true;
        }
        else if (l[1] > r[1]) {
            return false;
        }
        else {
            return l[2] < r[2];
        }
    }
}

static void getLatestVersion(char *binBaseName, char *versionStr)
{
    DIR *d;
    struct dirent *dir;
    int latest[] = { 0, 0, 0 };
    int version[] = { 0, 0, 0 };
    d = opendir("/home/analog/adrv9001_server/resources/Adi.Adrv9001.FpgaBinaries");
    if (d) {
        while ((dir = readdir(d)) != NULL) {
            if ((dir->d_type == DT_REG) &&
                (startsWith(binBaseName, dir->d_name)))
            {
                printf("%s\n", dir->d_name);
                getVersion(dir->d_name, version);
                if (versionLt(latest, version))
                {
                    memcpy(latest, version, 12);
                }
            }
        }
        closedir(d);
    }

    sprintf(versionStr, "%d.%d.%d", latest[0], latest[1], latest[2]);
}



int32_t linux_uio_adi_fpga9001_hal_CurrentPlatform_Get(void *halCfg, adi_fpga9001_Platform_e *platform)
{
	uint32_t buff32[AXI_SYSID_TOP_SYS_INFO_0_SIZE] = { 0 };
    adi_fpga9001_Device_t *device = (adi_fpga9001_Device_t *)calloc(1, sizeof(adi_fpga9001_Device_t));
    device->common.devHalInfo = halCfg;
    
	axi_sysid_sys_info_0_get(device, AXI_SYSID_ID, buff32, AXI_SYSID_TOP_SYS_INFO_0_SIZE);
    free(device);
    
    if (strstr((char *)buff32, "ZCU102") != NULL)
    {
        *platform = ADI_FPGA9001_PLATFORM_ZCU102;
    }
    else
    {
        *platform = ADI_FPGA9001_PLATFORM_ZC706;
    }
    
    return 0;
}

int32_t linux_uio_adi_fpga9001_hal_DmaRelativeAddress_Get(void *devHalCfg, adi_fpga9001_hal_Dma_e dma, uint32_t *addr)
{
    int32_t halError = 0;
    adi_fpga9001_Platform_e platform = ADI_FPGA9001_PLATFORM_UNSUPPORTED;
    halError = linux_uio_adi_fpga9001_hal_CurrentPlatform_Get((adi_fpga9001_hal_linux_uio_Cfg_t *)devHalCfg, &platform);
    if (0 != halError)
    {
        return halError;
    }
    
    switch (dma)
    {
    case ADI_FPGA9001_HAL_DMA_RX1:
        *addr = 0x00000000;
        break;
    case ADI_FPGA9001_HAL_DMA_RX2:
        *addr = 0x08000000;
        break;
    case ADI_FPGA9001_HAL_DMA_TX1:
        *addr = 0x10000000;
        break;
    case ADI_FPGA9001_HAL_DMA_TX2:
        *addr = 0x18000000;
        break;
    case ADI_FPGA9001_HAL_DMA_ORX1:
        *addr = 0x20000000;
        break;
    case ADI_FPGA9001_HAL_DMA_ORX2:
        *addr = 0x28000000;
        break;
    case ADI_FPGA9001_HAL_DMA_SPI:
        *addr = 0x30000000;
        break;
    default:
        return -2;
    }
    
    /* The ZCU102 has 1/2 the memory of the ZC706 */
    if (platform == ADI_FPGA9001_PLATFORM_ZCU102)
    {
        *addr = *addr >> 1;
    }
    
    return 0;
}
