/* 
* \file
*
* Copyright 2015 - 2018 Analog Devices Inc.
* Released under the ADRV9001 API license, for more information.
* see the "LICENSE.txt" file in this zip file.
*/

#include <string.h>
#include <stdarg.h>
#include <time.h>

#include "adi_linux_uio_logging.h"
#include "adi_linux_uio_types.h"

/**
* \brief Opens a logFile. If the file is already open it will be closed and reopened.
*
* This function opens the file for writing and saves the resulting file 
* descriptor to the devHalCfg structure.
*
* \param devHalCfg Pointer to device instance specific platform settings
* \param filename The user provided name of the file to open.
*
* \retval 0 Function completed successfully, no action required
* \retval -2 The function has been called with a null pointer
* \retval -5 If the function failed to open or write to the specified filename
*/  
int32_t linux_uio_LogFileOpen(adi_hal_LogCfg_t *logCfg)
{
    int32_t halError = 0;
    int32_t result = 0;
    struct tm tm = { 0 };
    time_t t = { 0 };

    if (logCfg == NULL)
    {
        halError = (int32_t)-2;
        return halError;
    }
    
    //if logfile already open, close it
    if(logCfg->logfd != 0)
    {
        halError = linux_uio_LogFileClose(logCfg);
    }

    /* open a new log file */
    logCfg->logfd = fopen(logCfg->logFileName, "w+");
    if (logCfg->logfd == NULL)
    {
        return (int32_t)-5;
    }
    
    t = time(NULL);
   
    tm = *localtime(&t);

    result = fprintf(logCfg->logfd, "API log file \nStart date %04d-%02d-%02d %02d:%02d:%02d \n\n\n", tm.tm_year + 1900, tm.tm_mon + 1, tm.tm_mday, tm.tm_hour, tm.tm_min, tm.tm_sec);
    if (result < 0)
    {
        return -5;
    }

    result = fflush(logCfg->logfd);
    if (result < 0)
    {
        halError = -5;
    }

    return halError;
}

/**
* \brief Gracefully closes the log file(s).
*
* \param devHalCfg Pointer to device instance specific platform settings
*
* \retval 0 Function completed successfully, no action required
* \retval -2 The function has been called with a null pointer
* \retval -5 Error while flushing or closing the log file.
*/
int32_t linux_uio_LogFileClose(adi_hal_LogCfg_t *logCfg)
{
    int32_t halError = (int32_t)0;
    int32_t result = 0;
    
    if (logCfg == NULL)
    {
        halError = (int32_t)-2;
        return halError;
    }
        
    if (logCfg->logfd != 0)
    {
        result = fflush(logCfg->logfd);
        if (result < 0)
        {
            halError = -5;
        }
        
        result = fclose(logCfg->logfd);
        if (result < 0)
        {
            halError = -5;
        }
        
        logCfg->logfd = NULL;
    }
    else
    {
        /* Log file is not open, can not close the file - no error */
        halError = 0;
    }

    return halError;
}

/**
* \brief Writes a message to the currently open logFile specified in the 
*        adi_hal_LogCfg_t of the devHalCfg structure passed
* 
* Uses the vfprintf functionality to allow the user to supply the format and
* the number of aguments that will be logged.
*
* \param devHalCfg Pointer to device instance specific platform settings
* \param logLevel the log level to be written into
* \param comment the string to include in the line added to the log.
* \param argp variable argument list to be printed
*
* \retval 0 Function completed successfully, no action required
* \retval -2 The function has been called with a null pointer
* \retval -5 If the function failed to flush to write 
*/
int32_t linux_uio_LogWrite(void *devHalCfg, uint32_t logLevel, const char *comment, va_list argp)
{
    int32_t halError = (int32_t)0;
    int32_t result = 0;
    adi_hal_Cfg_t *halCfg = NULL;
    time_t t = time(NULL);
    struct tm tm = { 0 };
    FILE *logfd = NULL;
    char logMessage[ADI_HAL_MAX_LOG_LINE] = { 0 };
    const char *logLevelChar = NULL;
    logMessage[0] = 0;
    
    if (devHalCfg == NULL)
    {
        halError = (int32_t)-2;
        return halError;
    }

    halCfg = (adi_hal_Cfg_t *)devHalCfg;
    
    if ((halCfg->logCfg.interfaceEnabled == 0) || (logLevel == ADI_LOGLEVEL_NONE))
    {
        /* If logging disabled, exit gracefully */
        halError = (int32_t)0;
        return halError;
    }

    /* Auto Open of log file is currently disabled */
    if (halCfg->logCfg.logfd == 0)
    {
        /* Log file is not open */
        halError = (int32_t)-5;
        return halError;
    }
    else
    {
        logfd = (FILE *)halCfg->logCfg.logfd;

        if (halCfg->logCfg.currentLineNumber >= LINUX_UIO_LOG_MAX_NUM_LINES)
        {
            /* seek back to beginning of file */
            /* fseek(logfd, 2, SEEK_SET); */ 
            rewind(logfd);
            halCfg->logCfg.currentLineNumber = 0;
        }
        
        switch (logLevel)
        {
        case ADI_LOGLEVEL_TRACE:
            logLevelChar = "TRACE:";
            break;
        case ADI_LOGLEVEL_DEBUG:
            logLevelChar = "DEBUG:";
            break;
        case ADI_LOGLEVEL_INFO:
            logLevelChar = "INFO:";
            break;
        case ADI_LOGLEVEL_WARN:
            logLevelChar = "WARN:";
            break;
        case ADI_LOGLEVEL_ERROR:
            logLevelChar = "ERROR:";
            break;
        case ADI_LOGLEVEL_FATAL:
            logLevelChar = "FATAL";
            break;
        default:
            return 0;
        }
        
        tm = *localtime(&t);
        /* Print timestamp */
        result = snprintf(logMessage, ADI_HAL_MAX_LOG_LINE, "%04d-%02d-%02d %02d:%02d:%02d: %s", tm.tm_year + 1900, tm.tm_mon + 1, tm.tm_mday, tm.tm_hour, tm.tm_min, tm.tm_sec, logLevelChar);
        if (result < 0)
        {
            halError = (int32_t)-5;
            return halError;
        }

        result = vsnprintf(logMessage + strlen(logMessage), ADI_HAL_MAX_LOG_LINE, comment, argp) ;
        if (result < 0)
        {
            halError = (int32_t)-5;
            return halError;
        }

        result = fprintf(logfd, "%s\n", logMessage);
        if (result < 0)
        {
            halError = (int32_t)-5;
            return halError;
        }

        result = fflush(logfd);
        if (result < 0)
        {
            halError = (int32_t)-5;
            return halError;
        }

        halCfg->logCfg.currentLineNumber += 1;
    }

    return halError;
}

