/**
 * \file
 * \brief linux_uio implementation of the ADRV9001 HAL
 * 
 * Copyright 2020-2025 Analog Devices Inc.
 * Released under the ADRV9001 API license, for more information.
 * see the "LICENSE.txt" file in this zip file.
 */

#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <linux/spi/spidev.h>

#include "adi_adrv9001_hal_linux_uio.h"
#include "adi_linux_uio_logging.h"

#include "axi_adrv9001.h"
#include "axi_mspi.h"
#include "adi_fpga9001_mcs.h"
#include "adi_fpga9001_ssi.h"
#include "adrv9001_zcu102.h"

static const adi_hal_LogCfg_t adrv9001LogCfg = { 
    .interfaceEnabled = 1,
    .logfd = NULL,
    .logFileName = "adrv9001Log.txt",
    .currentLineNumber = 0
};

static const adi_adrv9001_hal_linux_uio_spiCfg_t spiCfg = { 
    .chipSelectIndex = 0,
    .CPHA = 0,
    .CPOL = 0,
    .enSpiStreaming = 0,
    .autoIncAddrUp = 1,
    .fourWireMode = 1,
    .spiClkFreq_Hz = 25000000,
};

adi_adrv9001_hal_linux_uio_Cfg_t *linux_uio_adrv9001_cfg_create()
{
    adi_adrv9001_hal_linux_uio_Cfg_t *halCfg = (adi_adrv9001_hal_linux_uio_Cfg_t *)calloc(1, sizeof(adi_adrv9001_hal_linux_uio_Cfg_t));
 
    /* Copy default values */
    halCfg->logCfg = adrv9001LogCfg;
    halCfg->spiCfg = spiCfg;
    
    return halCfg;
}

int32_t linux_uio_adi_adrv9001_hal_open(void *devHalCfg)
{
    int32_t halError = 0;
    adi_adrv9001_hal_linux_uio_Cfg_t *halCfg = NULL;
    struct axi_mspi_params spi_params;
    
    if (NULL == devHalCfg)
    {
        return -2;
    }
    
    halCfg = (adi_adrv9001_hal_linux_uio_Cfg_t *)devHalCfg;
    
    /* Open log file */
    halError = linux_uio_LogFileOpen(&halCfg->logCfg);
    ADI_ERROR_RETURN(halError);
    
    /* spi device settings */

    spi_params.clk_period = 0;
    spi_params.clk_init = (halCfg->spiCfg.CPOL == 1) ? 1 : 0;
    spi_params.bidir_enable = 0;
    spi_params.bidir_offset = 0;
    spi_params.miso_sample_edge = (halCfg->spiCfg.CPHA == 1) ? 1 : 0;

    if ((halCfg->spiCfg.spiClkFreq_Hz >= 195000) && (halCfg->spiCfg.spiClkFreq_Hz <= 50000000))
    {
        spi_params.clk_period = (50000000/halCfg->spiCfg.spiClkFreq_Hz)-1;
    }

    if (halCfg->spiCfg.fourWireMode == 0)
    {
        spi_params.bidir_enable = 1;
        spi_params.bidir_offset = 1;
    }

    axi_mspi_config(halCfg->fpga9001, AXI_MSPI_ID, &spi_params);
    axi_mspi_fifo_config(halCfg->fpga9001, AXI_MSPI_ID);
    
    return halError;
}

int32_t linux_uio_adi_adrv9001_hal_close(void *devHalCfg)
{
    int32_t halError = 0;
    adi_adrv9001_hal_linux_uio_Cfg_t *halCfg = NULL;
    
    if (NULL == devHalCfg)
    {
        return -2;
    }
    
    halCfg = (adi_adrv9001_hal_linux_uio_Cfg_t *)devHalCfg;
    
    /* Close log file */
    if (NULL != halCfg->logCfg.logfd)
    {
        halError = linux_uio_LogFileClose(&halCfg->logCfg);
        ADI_ERROR_RETURN(halError);
    }
    
    return halError;
}

int32_t linux_uio_adi_adrv9001_hal_spi_write(void *devHalCfg, const uint8_t txData[], uint32_t numTxBytes)
{
    adi_adrv9001_hal_linux_uio_Cfg_t *halCfg = NULL;
    
    if (NULL == devHalCfg)
    {
        return -2;
    }
    
    halCfg = (adi_adrv9001_hal_linux_uio_Cfg_t *)devHalCfg;
    axi_mspi_slave_select_set(halCfg->fpga9001, AXI_MSPI_ID, (0x1 << halCfg->spiCfg.chipSelectIndex));
    axi_mspi_fifo_xfer(halCfg->fpga9001, AXI_MSPI_ID, &txData[0], 0, numTxBytes);
    // TODO: further testing of mosi_xfer necessary
    // axi_mspi_fifo_mosi_xfer(halCfg->fpga9001, AXI_MSPI_ID, &txData[0], numTxBytes);
    axi_mspi_slave_select_set(halCfg->fpga9001, AXI_MSPI_ID, 0x0);
    return (0);
}

int32_t linux_uio_adi_adrv9001_hal_spi_read(void *devHalCfg, const uint8_t txData[], uint8_t rxData[], uint32_t numTxRxBytes)
{
    adi_adrv9001_hal_linux_uio_Cfg_t *halCfg = NULL;
    
    if (NULL == devHalCfg)
    {
        return -2;
    }
    
    halCfg = (adi_adrv9001_hal_linux_uio_Cfg_t *)devHalCfg;
    axi_mspi_slave_select_set(halCfg->fpga9001, AXI_MSPI_ID, (0x1 << halCfg->spiCfg.chipSelectIndex));
    /* streaming to ahb will fail (arm failed to boot) without this, why? */
    axi_mspi_top_timer_delay_us(halCfg->fpga9001, AXI_MSPI_ID, 1);
    axi_mspi_fifo_xfer(halCfg->fpga9001, AXI_MSPI_ID, &txData[0], &rxData[0], numTxRxBytes);
    axi_mspi_slave_select_set(halCfg->fpga9001, AXI_MSPI_ID, 0x0);
    return(0);
}

int32_t linux_uio_adi_adrv9001_hal_resetbPin_set(void *devHalCfg, uint8_t pinLevel)
{
    adi_adrv9001_hal_linux_uio_Cfg_t *halCfg = NULL;
    if (NULL == devHalCfg)
    {
        return -2;
    }
    
    halCfg = (adi_adrv9001_hal_linux_uio_Cfg_t *)devHalCfg;
    if (NULL == halCfg->fpga9001)
    {
        return -2;
    }
    
    axi_adrv9001_top_resetb_set(halCfg->fpga9001, AXI_ADRV9001_ID, AXI_ADRV9001_TOP_OFFSET, pinLevel);
    
    return 0;
}


int32_t linux_uio_adi_adrv9001_hal_image_page_get(void *devHalCfg, const char *imagePath, uint32_t pageIndex, uint32_t pageSize, uint8_t *rdBuff)
{
	static const size_t BIN_ELEMENT_SIZE = 1;
	FILE *imageFilePointer = NULL;
	uint32_t fileSize = 0;

	adi_adrv9001_hal_linux_uio_Cfg_t *halCfg = NULL;
	if (NULL == devHalCfg)
	{
		return -2;
	}

	/*Open binary file*/
	if ((imageFilePointer = fopen(imagePath, "rb")) == NULL)
	{
		printf("Cannot open binary file '%s'\n", imagePath);
		ADI_ERROR_RETURN(3);
	}

	/*Check File Pointer is NULL*/
	if (imageFilePointer == NULL)
	{
		printf("Invalid binary image path encountered while attempting to load binary image");
		ADI_ERROR_RETURN(3);
	}

	/*Determine file size*/
	if (fseek(imageFilePointer, 0, SEEK_END) < 0)
	{
	    printf("Unable to move file descriptor to the end of the binary image file");
		ADI_ERROR_RETURN(3);
	}
	fileSize = ftell(imageFilePointer);

	/*Check that binary file is not empty*/
	if (fileSize == 0)
	{
		printf("Empty binary image file encountered while attempting to load binary image");
		ADI_ERROR_RETURN(3);
	}

	/*Check that size of the file is a multiple of 4*/
	if ((fileSize % 4) != 0)
	{
		printf("Binary image file is expected to be a multiple of 4");
		ADI_ERROR_RETURN(3);
	}

	/*Move file pointer to the beginning of current page of the file*/
	if (fseek(imageFilePointer, (pageIndex * pageSize), SEEK_SET) < 0)
	{
		printf("Unable to move file descriptor to the beginning of current page of binary image file");
		ADI_ERROR_RETURN(3);
	}

	/*Read binary file*/
	if (fread(&rdBuff[0], BIN_ELEMENT_SIZE, pageSize, imageFilePointer) != pageSize)
	{
		printf("Fatal error while reading binary file. Possible memory shortage");
		ADI_ERROR_RETURN(3);
	}

	/*Close binary file*/
	if (fclose(imageFilePointer) < 0)
	{
		printf("Fatal error while trying to close binary file. Possible memory shortage while flushing / other I/O errors.");
		ADI_ERROR_RETURN(3);
	}
 
	return 0;
}




