/**
* \file
* \brief linux_uio implementation of the ADM1293 HAL
* 
* The linux_uio implementation simply uses the Linux i2c-dev userspace adapter.
* See the kernel documentation (https://www.kernel.org/doc/Documentation/i2c/dev-interface) for an introduction.
*
* Copyright 2020 Analog Devices Inc.
* Released under the ADRV9001 API license, for more information.
* see the "LICENSE.txt" file in this zip file.
*/

#include <unistd.h>
#include <stdlib.h>
#include <fcntl.h>
#include <stdio.h>
#include <sys/ioctl.h>
#include <linux/i2c.h>
#include <linux/i2c-dev.h>

#include "adi_adm1293_hal_linux_uio.h"
#include "adi_linux_uio_logging.h"
#include "adi_adm1293_types.h"

static const adi_hal_LogCfg_t adm1293LogCfg = { 
    .interfaceEnabled = 1,
    .logfd = NULL,
    .logFileName = "adm1293Log.txt",
    .currentLineNumber = 0
};

static const adi_adm1293_hal_linux_uio_i2cCfg_t i2cCfg = { 
    .fd = -1,
    .drvName = "/dev/i2c-7"
};

adi_adm1293_hal_linux_uio_Cfg_t *linux_uio_adm1293_open(uint8_t deviceIndex)
{
    adi_adm1293_hal_linux_uio_Cfg_t *halCfg = (adi_adm1293_hal_linux_uio_Cfg_t *)calloc(1, sizeof(adi_adm1293_hal_linux_uio_Cfg_t));
    if (NULL == halCfg)
    {
        return NULL;
    }
    
    /* Copy default values */
    halCfg->logCfg = adm1293LogCfg;
    halCfg->i2cCfg = i2cCfg;
    
    switch (deviceIndex)
    {
    case 0:
        halCfg->i2cCfg.slaveAddress = ADI_ADRV9001CE01_POWERMONITOR_VDD_1P0;
        break;
    case 1:
        halCfg->i2cCfg.slaveAddress = ADI_ADRV9001CE01_POWERMONITOR_VDDA_1P0;
        break;
    case 2:
        halCfg->i2cCfg.slaveAddress = ADI_ADRV9001CE01_POWERMONITOR_VDD_1P8;
        break;
    case 3:
        halCfg->i2cCfg.slaveAddress = ADI_ADRV9001CE01_POWERMONITOR_VDDA_1P8;
        break;
    case 4:
        halCfg->i2cCfg.slaveAddress = ADI_ADRV9001CE01_POWERMONITOR_VDDA_1P3;
        break;
    default:
        free(halCfg);
        return NULL;
    }
    
    /* Open log file */
    if (0 != linux_uio_LogFileOpen(&halCfg->logCfg))
    {
        free(halCfg);
        return NULL;
    }
    
    /* Open i2c file */
    halCfg->i2cCfg.fd = open(halCfg->i2cCfg.drvName, O_RDWR);
    if (halCfg->i2cCfg.fd < 0)
    {
        free(halCfg);
        return NULL;
    }
    
    return halCfg;
}

int32_t linux_uio_adm1293_close(adi_adm1293_hal_linux_uio_Cfg_t *devHalCfg)
{
    int32_t halError = 0;
    
    if (NULL == devHalCfg)
    {
        return -2;
    }
        
    /* Close log file */
    if (NULL != devHalCfg->logCfg.logfd)
    {
        halError = linux_uio_LogFileClose(&devHalCfg->logCfg);
        ADI_ERROR_RETURN(halError);
    }
    
    /* Close i2c file */
    if (-1 != devHalCfg->i2cCfg.fd)
    {
        halError = close(devHalCfg->i2cCfg.fd);
        ADI_ERROR_RETURN(halError);
        
        devHalCfg->i2cCfg.fd = -1;
    }
    
    free(devHalCfg);
    return halError;
}

int32_t linux_uio_adi_adm1293_hal_i2c_write(void *devHalCfg, const uint8_t wrData[], uint32_t numWrBytes)
{
    int32_t halError = 0;
    int retVal = 0;
    adi_adm1293_hal_linux_uio_Cfg_t *halCfg = NULL;
    
    if (NULL == devHalCfg)
    {
        return -2;
    }
    
    /* Cast to implementation-specific HAL object */
    halCfg = (adi_adm1293_hal_linux_uio_Cfg_t *)devHalCfg;
    
    if ((numWrBytes > 0) &&
        (wrData != NULL))
    {
        if (halCfg->i2cCfg.fd > 0)
        {
            /* Configure the slave address */
            if (ioctl(halCfg->i2cCfg.fd, I2C_SLAVE, halCfg ->i2cCfg.slaveAddress) < 0)
            {
                return -1;
            }
            
            /* Write the data */
            retVal = write(halCfg->i2cCfg.fd, &wrData[0], numWrBytes);
            if (retVal != (int)numWrBytes)
            {
                perror("I2C : Failed to Write to device");
                return -3;
            }
        }
        else
        {
            perror("I2C : Failed to Write to device, file descriptor invalid");
            return -4;
        }
    }
    
    return halError;
}

int32_t linux_uio_adi_adm1293_hal_i2c_read(void *devHalCfg, 
                                       const uint8_t wrData[], 
                                       uint32_t numWrBytes, 
                                       uint8_t rdData[], 
                                       uint32_t numRdBytes)
{
    int32_t halError = 0;
    adi_adm1293_hal_linux_uio_Cfg_t *halCfg = NULL;
    
    struct i2c_msg msgs[2] = {
    {
        .addr = 0,
        .flags = 0,
        .len = numWrBytes,
        .buf = (uint8_t *)wrData
    },
    {
        .addr = 0,
        .flags = I2C_M_RD,
        .len = numRdBytes,
        .buf = rdData
    }
    };
    struct i2c_rdwr_ioctl_data msgset = {
        .msgs = msgs,
        .nmsgs = 2
    };
    
    if (NULL == devHalCfg)
    {
        return -2;
    }
    
    /* Cast to implementation-specific HAL object */
    halCfg = (adi_adm1293_hal_linux_uio_Cfg_t *)devHalCfg;
    
    msgs[0].addr = halCfg->i2cCfg.slaveAddress;
    msgs[1].addr = halCfg->i2cCfg.slaveAddress;
    
    if (halCfg->i2cCfg.fd > 0)
    {
        /* Write Register address */
        if ((numWrBytes > 0) && 
            (wrData != NULL))
        {
            if ((numRdBytes > 0) &&
                (rdData != NULL))
            {
                /* Duplex transfer */
                ioctl(halCfg->i2cCfg.fd, I2C_RDWR, &msgset);
                rdData = msgset.msgs[1].buf;
            }
        }
        else
        {
            printf("Invalid parameter rdData. Can't be NULL");
            return -1;
        }
    }
    else
    {
        printf("I2C : Failed to Write to device, file descriptor invalid\n");
        return -3;
    }
    
    return halError;
}
