/**
 * \file
 * \brief Contains top level TDD fpga9001 related functions
 *
 * FPGA9001 API Version: $ADI_FPGA9001_API_VERSION$
 */

/**
 * Copyright 2019-2025 Analog Devices Inc.
 * Released under the FPGA9001 API license, for more information
 * see the "LICENSE.txt" file in this zip file.
 */

#include "fpga9001_utilities.h"
#include "adi_fpga9001_tdd.h"

/* TDD address lookup table */

static const uint32_t adi_fpga9001_TddSelectLUT[] =
{
    AXI_ADRV9001_TDD_RX0_DEVICE,
    AXI_ADRV9001_TDD_TX0_DEVICE,
    AXI_ADRV9001_TDD_RX1_DEVICE,
    AXI_ADRV9001_TDD_TX1_DEVICE,
    AXI_ADRV9001_TDD_ORX0_GPIO,
    AXI_ADRV9001_TDD_ORX1_GPIO,
    AXI_ADRV9001_TDD_RX0_DMA_DATA,
    AXI_ADRV9001_TDD_TX0_DMA_DATA,
    AXI_ADRV9001_TDD_RX1_DMA_DATA,
    AXI_ADRV9001_TDD_TX1_DMA_DATA,
    AXI_ADRV9001_TDD_ORX0_DMA_DATA,
    AXI_ADRV9001_TDD_ORX1_DMA_DATA,
    AXI_ADRV9001_TDD_TX0_GPIO,
    AXI_ADRV9001_TDD_TX1_GPIO,
    AXI_ADRV9001_TDD_IFRM0CTL0_GPIO,
    AXI_ADRV9001_TDD_IFRM0CTL1_GPIO,
    AXI_ADRV9001_TDD_IFRM1CTL0_GPIO,
    AXI_ADRV9001_TDD_IFRM1CTL1_GPIO,
    AXI_ADRV9001_TDD_EXT0_TRIG,
    AXI_ADRV9001_TDD_EXT1_TRIG,
    AXI_ADRV9001_TDD_RX0_DMA_TRIG,
    AXI_ADRV9001_TDD_TX0_DMA_TRIG,
    AXI_ADRV9001_TDD_RX1_DMA_TRIG,
    AXI_ADRV9001_TDD_TX1_DMA_TRIG,
    AXI_ADRV9001_TDD_ORX0_DMA_TRIG,
    AXI_ADRV9001_TDD_ORX1_DMA_TRIG,
    AXI_ADRV9001_TDD_SPI_TRIG,
    AXI_ADRV9001_TDD_GPIO_TRIG,
};

static __maybe_unused int32_t adi_fpga9001_Tdd_Enable_Set_Validate(adi_fpga9001_Device_t *fpga9001,
                                                    adi_fpga9001_TddSelect_e tddSelect,
                                                    adi_fpga9001_TddMode_e tddMode)
{
    ADI_RANGE_CHECK(fpga9001, tddSelect, 0, ADI_FPGA9001_TDDSELECT_MAX_COUNT);
    ADI_RANGE_CHECK(fpga9001, tddMode, ADI_FPGA9001_TDD_ENABLE_LOW, ADI_FPGA9001_TDD_ENABLE_AUTO);
    ADI_API_RETURN(fpga9001);
}

/* Set TDD enable as requested by select */
int32_t adi_fpga9001_Tdd_Enable_Set(adi_fpga9001_Device_t *fpga9001,
                                    adi_fpga9001_TddSelect_e tddSelect,
                                    adi_fpga9001_TddMode_e tddMode)
{
    ADI_PERFORM_VALIDATION(adi_fpga9001_Tdd_Enable_Set_Validate, fpga9001, tddSelect, tddMode);
    axi_adrv9001_tdd_mode_set((void *)fpga9001, AXI_ADRV9001_ID, AXI_ADRV9001_TDD_OFFSET, adi_fpga9001_TddSelectLUT[tddSelect], tddMode);
    ADI_API_RETURN(fpga9001);
}

static __maybe_unused int32_t adi_fpga9001_Tdd_Enable_Get_Validate(adi_fpga9001_Device_t *fpga9001,
                                                    adi_fpga9001_TddSelect_e tddSelect,
                                                    adi_fpga9001_TddMode_e *mode)
{
    ADI_NULL_PTR_RETURN(&fpga9001->common, mode);
    ADI_RANGE_CHECK(fpga9001, tddSelect, 0, ADI_FPGA9001_TDDSELECT_MAX_COUNT);
    ADI_API_RETURN(fpga9001);
}

/* Return TDD enable as requested by select */
int32_t adi_fpga9001_Tdd_Enable_Get(adi_fpga9001_Device_t *fpga9001,
                                    adi_fpga9001_TddSelect_e tddSelect,
                                    adi_fpga9001_TddMode_e *mode)
{
    ADI_PERFORM_VALIDATION(adi_fpga9001_Tdd_Enable_Get_Validate, fpga9001, tddSelect, mode);
    axi_adrv9001_tdd_mode_get((void *)fpga9001, AXI_ADRV9001_ID, AXI_ADRV9001_TDD_OFFSET, adi_fpga9001_TddSelectLUT[tddSelect], mode);
    ADI_API_RETURN(fpga9001);
}

/* Configure all TDD enables */
int32_t adi_fpga9001_Tdd_Configure(adi_fpga9001_Device_t *fpga9001,
                                   adi_fpga9001_TddConfig_t *tddConfig)
{
    int32_t i = 0;
    uint32_t tdd_mode = 0;
    uint32_t tdd_ratio = 0;
    uint32_t tdd_clk_freq = 0;
    axi_adrv9001_tdd_frame_params_t tdd_frame = { 0 };
    axi_adrv9001_tdd_enable_params_t tdd_enable = { 0 };

    /* get count from micro seconds, if so desired */
    axi_adrv9001_tdd_clk_mon_count_freq_hz((void *)fpga9001, AXI_ADRV9001_ID, AXI_ADRV9001_TDD_OFFSET, &tdd_clk_freq);
    tdd_clk_freq = tdd_clk_freq/1000000;

    for (i = 0; i < ADI_FPGA9001_TDDSELECT_MAX_COUNT; i++)
    {
        tdd_ratio = (tddConfig->enables[i].inMicroSeconds) ? tdd_clk_freq : 1;
        tdd_enable.primary_assert = tddConfig->enables[i].primaryAssert * tdd_ratio;
        tdd_enable.primary_deassert = tddConfig->enables[i].primaryDeassert * tdd_ratio;
        tdd_enable.secondary_assert = tddConfig->enables[i].secondaryAssert * tdd_ratio;
        tdd_enable.secondary_deassert = tddConfig->enables[i].secondaryDeassert * tdd_ratio;
        tdd_enable.primary_frame_assert = tddConfig->enables[i].primaryFrameAssert;
        tdd_enable.primary_frame_deassert = tddConfig->enables[i].primaryFrameDeassert;
        tdd_enable.secondary_frame_assert = tddConfig->enables[i].secondaryFrameAssert;
        tdd_enable.secondary_frame_deassert = tddConfig->enables[i].secondaryFrameDeassert;
        tdd_mode = (tddConfig->enables[i].enable) ? AXI_ADRV9001_TDD_ENABLE_AUTO : AXI_ADRV9001_TDD_ENABLE_LOW;
        axi_adrv9001_tdd_enable_set((void *)fpga9001, AXI_ADRV9001_ID, adi_fpga9001_TddSelectLUT[i], &tdd_enable);
        axi_adrv9001_tdd_mode_set((void *)fpga9001, AXI_ADRV9001_ID, AXI_ADRV9001_TDD_OFFSET, adi_fpga9001_TddSelectLUT[i], tdd_mode);
        axi_adrv9001_tdd_sync_set(fpga9001, AXI_ADRV9001_ID, AXI_ADRV9001_TDD_OFFSET, adi_fpga9001_TddSelectLUT[i], tddConfig->enables[i].enableSyncToSample);
    }

    tdd_ratio = (tddConfig->frameTiming.inMicroSeconds) ? tdd_clk_freq : 1;
    tdd_frame.period = tddConfig->frameTiming.framePeriod * tdd_ratio;
    tdd_frame.num_of_frames = tddConfig->frameTiming.numberFrames;
    tdd_frame.continous = tddConfig->frameTiming.repeatFrameSequence;
    tdd_frame.frame_switch_period = tddConfig->frameTiming.frameSwitchTime * tdd_ratio;
    tdd_frame.frame_switch_num = tddConfig->frameTiming.frameSwitchNum;
    tdd_frame.frame_switch_enable = tddConfig->frameTiming.frameSwitchEnable;
    if (tddConfig->frameTiming.frameStartTrig != ADI_FPGA9001_TDD_TRIGGER_IMMEDIATE) {
        tdd_frame.triggers[(int) tddConfig->frameTiming.frameStartTrig] = AXI_ADRV9001_TDD_TRIGGER_RISING_EDGE;
    }
    axi_adrv9001_tdd_frame_set((void *)fpga9001, AXI_ADRV9001_ID, &tdd_frame);
    
    ADI_API_RETURN(fpga9001);
}

/* Read all current TDD configurations */
int32_t adi_fpga9001_Tdd_Inspect(adi_fpga9001_Device_t *fpga9001,
                                 adi_fpga9001_TddConfig_t *tddConfig)
{
    int32_t i = 0;
    uint32_t tdd_mode = 0;
    uint32_t tdd_ratio = 0;
    uint32_t tdd_clk_freq = 0;
    axi_adrv9001_tdd_frame_params_t tdd_frame = { 0 };
    axi_adrv9001_tdd_enable_params_t tdd_enable = { 0 };

    /* get count from micro seconds, if so desired */
    axi_adrv9001_tdd_clk_mon_count_freq_hz((void *)fpga9001, AXI_ADRV9001_ID, AXI_ADRV9001_TDD_OFFSET, &tdd_clk_freq);
    tdd_clk_freq = (tdd_clk_freq == 0) ? 1 : tdd_clk_freq/1000000;

    for (i = 0; i < ADI_FPGA9001_TDDSELECT_MAX_COUNT; i++)
    {
        axi_adrv9001_tdd_enable_get((void *)fpga9001, AXI_ADRV9001_ID, adi_fpga9001_TddSelectLUT[i], &tdd_enable);
        tdd_ratio = (tddConfig->enables[i].inMicroSeconds) ? tdd_clk_freq : 1;
        tddConfig->enables[i].primaryAssert = tdd_enable.primary_assert/tdd_ratio;
        tddConfig->enables[i].primaryDeassert = tdd_enable.primary_deassert/tdd_ratio;
        tddConfig->enables[i].secondaryAssert = tdd_enable.secondary_assert/tdd_ratio;
        tddConfig->enables[i].secondaryDeassert = tdd_enable.secondary_deassert/tdd_ratio;
        tddConfig->enables[i].primaryFrameAssert = tdd_enable.primary_frame_assert;
        tddConfig->enables[i].primaryFrameDeassert = tdd_enable.primary_frame_deassert;
        tddConfig->enables[i].secondaryFrameAssert = tdd_enable.secondary_frame_assert;
        tddConfig->enables[i].secondaryFrameDeassert = tdd_enable.secondary_frame_deassert;
        axi_adrv9001_tdd_mode_get((void *)fpga9001, AXI_ADRV9001_ID, AXI_ADRV9001_TDD_OFFSET, adi_fpga9001_TddSelectLUT[i], &tdd_mode);
        tddConfig->enables[i].enable = (tdd_mode == AXI_ADRV9001_TDD_ENABLE_AUTO) ? true : false;
    }

    axi_adrv9001_tdd_frame_get((void *)fpga9001, AXI_ADRV9001_ID, &tdd_frame);
    tdd_ratio = (tddConfig->frameTiming.inMicroSeconds) ? tdd_clk_freq : 1;
    tddConfig->frameTiming.framePeriod = tdd_frame.period / tdd_ratio;
    tddConfig->frameTiming.numberFrames = tdd_frame.num_of_frames;
    tddConfig->frameTiming.repeatFrameSequence = tdd_frame.continous;
    tddConfig->frameTiming.frameSwitchTime = tdd_frame.frame_switch_period / tdd_ratio;
    tddConfig->frameTiming.frameSwitchNum = tdd_frame.frame_switch_num;
    tddConfig->frameTiming.frameSwitchEnable = tdd_frame.frame_switch_enable;
    for (i = 0; i < 4; i++) {
        if (tdd_frame.triggers[i] == AXI_ADRV9001_TDD_TRIGGER_DISABLED) continue;
        tddConfig->frameTiming.frameStartTrig = i;
        break;
    }

    ADI_API_RETURN(fpga9001);
}

int32_t adi_fpga9001_Tdd_Start(adi_fpga9001_Device_t *fpga9001)
{
    ADI_NULL_DEVICE_PTR_RETURN(fpga9001);
    
    axi_adrv9001_tdd_start((void *)fpga9001, AXI_ADRV9001_ID);
    
    ADI_API_RETURN(fpga9001);
}

static __maybe_unused int32_t adi_fpga9001_Tdd_StartBit_Inspect(adi_fpga9001_Device_t *fpga9001)
{
    uint32_t timeout_count = (uint32_t) 1e9;
    uint32_t tddStatus = 0xFF;

    /* timer works on 10ns (100MHz), count == timeout_us * 100 */

    axi_adrv9001_tdd_timer_set(fpga9001, AXI_ADRV9001_ID, AXI_ADRV9001_TDD_OFFSET, timeout_count);

    while (axi_adrv9001_tdd_timer_get(fpga9001, AXI_ADRV9001_ID, AXI_ADRV9001_TDD_OFFSET, &timeout_count) == 0) {
        axi_adrv9001_tdd_status((void *)fpga9001, AXI_ADRV9001_ID, &tddStatus);
        if ((tddStatus == 0) || (timeout_count == 0)) break;
    }

    if (tddStatus != 0)
    {
        ADI_ERROR_REPORT(&fpga9001->common,
            ADI_COMMON_ERRSRC_API,
            ADI_COMMON_ERR_API_FAIL,
            ADI_COMMON_ACT_ERR_API_NOT_IMPLEMENTED,
            tddStatus,
            "Unable to stop TDD. Function adi_fpga9001_Tdd_Stop() has failed");
        ADI_API_RETURN(fpga9001);
    }

    ADI_API_RETURN(fpga9001);
}

int32_t adi_fpga9001_Tdd_Stop(adi_fpga9001_Device_t *fpga9001)
{
    uint32_t tddStatus = 0;

    ADI_NULL_DEVICE_PTR_RETURN(fpga9001);

    axi_adrv9001_tdd_status(fpga9001, AXI_ADRV9001_ID, &tddStatus);

    if (tddStatus == 1)
    {
        axi_adrv9001_tdd_stop(fpga9001, AXI_ADRV9001_ID);
        ADI_EXPECT(adi_fpga9001_Tdd_StartBit_Inspect, fpga9001);
    }

    ADI_API_RETURN(fpga9001);
}
