/**
 * \file
 * \brief Functions for FPGA9001 Synchronous Serial Interface (SSI) configuration
 *
 * FPGA9001 API Version: $ADI_FPGA9001_API_VERSION$
 */

/**
 * Copyright 2015-2025 Analog Devices Inc.
 * Released under the FPGA9001 API license, for more information
 * see the "LICENSE.txt" file in this zip file.
 */

#include "adi_adrv9001_user.h"
#include "adi_common_error.h"
#include "adi_common_macros.h"
#include "adi_common_types.h"
#include "adrv9001_reg_addr_macros.h"
#include "adi_fpga9001_ssi.h"
#include "adi_fpga9001_hal.h"
#include "axi_adrv9001.h"
#include "axi_adrv9001_top_regs.h"
#include "fpga9001_utilities.h"

static __maybe_unused int32_t populateSsiParam(adi_fpga9001_Device_t *fpga9001,
                                               adi_common_Port_e port,
                                               uint32_t interfaceRate_Hz,
                                               uint32_t sampleRate_Hz,
                                               adi_fpga9001_SsiConfig_t *ssiConfig,
                                               axi_adrv9001_ssi_params_t *param)
{
    bool ssi_ddr = false;
    uint32_t ss_ratio = 0;
    param->gpio = 0;
    param->tx_clk_msb = 0;
    param->tx_buf_enable = 0x3f;
    param->data_rate = 0;
    param->bit_order = 0;
    param->iq_order = 0;
    param->edge = 0;
    param->ss_ratio = 0;
    param->tx_ss_mode = 0;
    param->orx_ss_ratio = 0;
    
    param->tx_clk_sel = 0;
    if (ssiConfig->txRefClockPin == 0)
    {
        param->tx_clk_sel = 1;
    }

    if (!(0 == interfaceRate_Hz) && !(0 == sampleRate_Hz) && (port == ADI_TX))
    {
        if (sampleRate_Hz <= interfaceRate_Hz)
        {
                int y = interfaceRate_Hz / sampleRate_Hz;
                while (y >>= 1) ++ss_ratio;
        }
        param->ss_ratio = ss_ratio;
        param->tx_ss_mode = 1;
    }
    
    ssi_ddr = ssiConfig->ddrEn;
    param->data_rate = (ssi_ddr == true) ? AXI_ADRV9001_SSI_DATA_RATE_DDR : AXI_ADRV9001_SSI_DATA_RATE_SDR;
    param->mode = ((uint32_t)ssiConfig->ssiType) & 0x1;
    param->num_of_lanes = (ssiConfig->numLaneSel) ? AXI_ADRV9001_SSI_MULTIPLE_LANES : AXI_ADRV9001_SSI_SINGLE_LANE;
    param->strb_type = (ssiConfig->strobeType) ? AXI_ADRV9001_SSI_STRB_TYPE_LEVEL : AXI_ADRV9001_SSI_STRB_TYPE_PULSE;
    
    switch (ssiConfig->ssiDataFormatSel)
    {
    case ADI_FPGA9001_SSI_FORMAT_2_BIT_SYMBOL_DATA:
        param->num_of_bits = AXI_ADRV9001_SSI_NUM_OF_BITS_02;
        break;
    case ADI_FPGA9001_SSI_FORMAT_8_BIT_SYMBOL_DATA:
        param->num_of_bits = AXI_ADRV9001_SSI_NUM_OF_BITS_08;
        break;
    case ADI_FPGA9001_SSI_FORMAT_16_BIT_SYMBOL_DATA:
        param->num_of_bits = AXI_ADRV9001_SSI_NUM_OF_BITS_16;
        break;
    case ADI_FPGA9001_SSI_FORMAT_12_BIT_I_Q_DATA:
        param->num_of_bits = AXI_ADRV9001_SSI_NUM_OF_BITS_24;
        break;
    case ADI_FPGA9001_SSI_FORMAT_16_BIT_I_Q_DATA:
    case ADI_FPGA9001_SSI_FORMAT_15_BIT_I_Q_DATA_1_BIT_GAIN_CHANGE:
        param->num_of_bits = AXI_ADRV9001_SSI_NUM_OF_BITS_32;
        break;
    case ADI_FPGA9001_SSI_FORMAT_22_BIT_I_Q_DATA_1_BIT_GAIN_CHANGE_8_BIT_GAIN_INDEX:
        param->num_of_bits = AXI_ADRV9001_SSI_NUM_OF_BITS_64;
        break;
    default:
        ADI_ERROR_REPORT(&fpga9001->common,
                         ADI_COMMON_ERRSRC_API,
                         ADI_COMMON_ERR_API_FAIL,
                         ADI_COMMON_ACT_ERR_CHECK_PARAM,
                         0,
                         "unrecognized SSI data format type");
        ADI_API_RETURN(fpga9001);
    };

    ADI_API_RETURN(fpga9001);
}

int32_t adi_fpga9001_ssi_Calibrate(adi_fpga9001_Device_t *fpga9001,
                                   adi_common_Port_e port,
                                   adi_common_ChannelNumber_e channel,
                                   uint32_t *delay,
                                   uint32_t *sweep)
{
    uint32_t ssiId = 0;

    ssiId = fpga9001_SsiIdGet(fpga9001, port, channel);
    *sweep = (uint32_t) -1;
    if (axi_adrv9001_ssi_delay_calibrate((void *)fpga9001, AXI_ADRV9001_ID, ssiId, 0, delay) != 0)
    {
            ADI_ERROR_REPORT(&fpga9001->common,
                ADI_COMMON_ERRSRC_API,
                ADI_COMMON_ERR_API_FAIL,
                ADI_COMMON_ACT_ERR_API_NOT_IMPLEMENTED,
                0,
                "function axi_adrv9001_ssi_calibrate has failed");
            ADI_API_RETURN(fpga9001);
    }
    axi_adrv9001_ssi_delay_sweep_get((void *)fpga9001, AXI_ADRV9001_ID, ssiId, sweep);
    ADI_API_RETURN(fpga9001);
}

static __maybe_unused int32_t adi_fpga9001_ssi_TxClockPolarity_Set_Validate(adi_fpga9001_Device_t *fpga9001,
                                                             adi_common_ChannelNumber_e channel)
{
    ADI_RANGE_CHECK(fpga9001, channel, ADI_CHANNEL_1, ADI_CHANNEL_2);
    ADI_API_RETURN(fpga9001);
}

int32_t adi_fpga9001_ssi_TxClockPolarity_Set(adi_fpga9001_Device_t *fpga9001,
                                             adi_common_ChannelNumber_e channel,
                                             bool inverted)
{
    uint32_t ssiId = 0;
    uint32_t tx_clk_msb = 0;
    ADI_PERFORM_VALIDATION(adi_fpga9001_ssi_TxClockPolarity_Set_Validate, fpga9001, channel);
    
    ssiId = fpga9001_SsiIdGet(fpga9001, ADI_TX, channel);
    if (ssiId == (uint32_t) - 1) return (-1);

    tx_clk_msb = (inverted) ? 1 : 0;
    axi_adrv9001_ssi_clk_msb_set(fpga9001, AXI_ADRV9001_ID, ssiId, tx_clk_msb);

    ADI_API_RETURN(fpga9001);
}

static __maybe_unused int32_t adi_fpga9001_ssi_TxClockPolarity_Get_Validate(adi_fpga9001_Device_t *fpga9001,
                                                                            adi_common_ChannelNumber_e channel,
                                                                            bool *inverted)
{
    ADI_RANGE_CHECK(fpga9001, channel, ADI_CHANNEL_1, ADI_CHANNEL_2);
    ADI_NULL_PTR_RETURN(&fpga9001->common, inverted);
    ADI_API_RETURN(fpga9001);
}

int32_t adi_fpga9001_ssi_TxClockPolarity_Get(adi_fpga9001_Device_t *fpga9001,
                                             adi_common_ChannelNumber_e channel,
                                             bool *inverted)
{
    uint32_t ssiId = 0;
    uint32_t data = 0;
    ADI_PERFORM_VALIDATION(adi_fpga9001_ssi_TxClockPolarity_Get_Validate, fpga9001, channel, inverted);
    
    ssiId = fpga9001_SsiIdGet(fpga9001, ADI_TX, channel);
    if (ssiId == (uint32_t) - 1) return (-1);

    axi_adrv9001_ssi_clk_msb_get(fpga9001, AXI_ADRV9001_ID, ssiId, &data);
    *inverted = (data == 0) ? false : true;

    ADI_API_RETURN(fpga9001);
}

static __maybe_unused int32_t adi_fpga9001_ssi_Configure_Validate(adi_fpga9001_Device_t *fpga9001,
                                                   adi_common_Port_e port,
                                                   adi_common_ChannelNumber_e channel,
                                                   adi_fpga9001_SsiConfig_t *ssiCfg)
{
    ADI_RANGE_CHECK(fpga9001, port, ADI_RX, ADI_TX);
    ADI_RANGE_CHECK(fpga9001, channel, ADI_CHANNEL_1, ADI_CHANNEL_2);
    ADI_NULL_PTR_RETURN(&fpga9001->common, ssiCfg);
    ADI_RANGE_CHECK(fpga9001, ssiCfg->ssiType, ADI_FPGA9001_SSI_TYPE_DISABLE, ADI_FPGA9001_SSI_TYPE_LVDS);
    ADI_RANGE_CHECK(fpga9001, ssiCfg->ssiDataFormatSel, ADI_FPGA9001_SSI_FORMAT_2_BIT_SYMBOL_DATA, ADI_FPGA9001_SSI_FORMAT_22_BIT_I_Q_DATA_1_BIT_GAIN_CHANGE_8_BIT_GAIN_INDEX);
    ADI_RANGE_CHECK(fpga9001, ssiCfg->numLaneSel, ADI_FPGA9001_SSI_1_LANE, ADI_FPGA9001_SSI_4_LANE);
    ADI_RANGE_CHECK(fpga9001, ssiCfg->strobeType, ADI_FPGA9001_SSI_SHORT_STROBE, ADI_FPGA9001_SSI_LONG_STROBE);
    if ((ssiCfg->ssiType == ADI_FPGA9001_SSI_TYPE_CMOS) &&
        (ssiCfg->ssiDataFormatSel == ADI_FPGA9001_SSI_FORMAT_12_BIT_I_Q_DATA))
    {
        ADI_ERROR_REPORT(&fpga9001->common,
            ADI_COMMON_ERRSRC_API,
            ADI_COMMON_ERR_API_FAIL,
            ADI_COMMON_ACT_ERR_API_NOT_IMPLEMENTED,
            0,
            "Unsupported interface mode. 12 bit I/Q is not supported in CMOS.");
        ADI_API_RETURN(fpga9001);
    }
    if ((ssiCfg->ssiType == ADI_FPGA9001_SSI_TYPE_CMOS) &&
        (ssiCfg->numLaneSel == ADI_FPGA9001_SSI_2_LANE))
    {
        ADI_ERROR_REPORT(&fpga9001->common,
            ADI_COMMON_ERRSRC_API,
            ADI_COMMON_ERR_API_FAIL,
            ADI_COMMON_ACT_ERR_API_NOT_IMPLEMENTED,
            0,
            "Unsupported interface mode. Only 1 lane and 4 lanes are supported in CMOS.");
        ADI_API_RETURN(fpga9001);
    }
    if ((ssiCfg->ssiType == ADI_FPGA9001_SSI_TYPE_LVDS) &&
        (ssiCfg->numLaneSel == ADI_FPGA9001_SSI_4_LANE))
    {
        ADI_ERROR_REPORT(&fpga9001->common,
            ADI_COMMON_ERRSRC_API,
            ADI_COMMON_ERR_API_FAIL,
            ADI_COMMON_ACT_ERR_API_NOT_IMPLEMENTED,
            0,
            "Unsupported interface mode. Only 1 lane and 2 lanes are supported in LVDS.");
        ADI_API_RETURN(fpga9001);
    }
    if (ssiCfg->ssiType == ADI_FPGA9001_SSI_TYPE_LVDS)
    {
        if ((ssiCfg->ssiDataFormatSel == ADI_FPGA9001_SSI_FORMAT_2_BIT_SYMBOL_DATA) ||
            (ssiCfg->ssiDataFormatSel == ADI_FPGA9001_SSI_FORMAT_8_BIT_SYMBOL_DATA) ||
            (ssiCfg->ssiDataFormatSel == ADI_FPGA9001_SSI_FORMAT_16_BIT_SYMBOL_DATA))
        {
            ADI_ERROR_REPORT(&fpga9001->common,
                ADI_COMMON_ERRSRC_API,
                ADI_COMMON_ERR_API_FAIL,
                ADI_COMMON_ACT_ERR_API_NOT_IMPLEMENTED,
                0,
                "Unsupported interface mode. Only 12 bit and 16 bit I/Q data are supported in LVDS.");
            ADI_API_RETURN(fpga9001);
        }
    }
    ADI_API_RETURN(fpga9001);
}

int32_t adi_fpga9001_ssi_Configure(adi_fpga9001_Device_t *fpga9001,
                                   adi_common_Port_e port,
                                   adi_common_ChannelNumber_e channel,
                                   uint32_t interfaceRate_Hz,
                                   uint32_t sampleRate_Hz,
                                   adi_fpga9001_SsiConfig_t *ssiConfig)
{
    uint32_t ssiId = 0;
    axi_adrv9001_ssi_params_t param = { 0 };

    ADI_PERFORM_VALIDATION(adi_fpga9001_ssi_Configure_Validate, fpga9001, port, channel, ssiConfig);

    /* No action taken if ADI_FPGA9001_SSI_TYPE_DISABLE */
    if((ADI_FPGA9001_SSI_TYPE_CMOS == ssiConfig->ssiType) ||
       (ADI_FPGA9001_SSI_TYPE_LVDS == ssiConfig->ssiType))
    {
        ssiId = fpga9001_SsiIdGet(fpga9001, port, channel);
        if (ssiId == (uint32_t) - 1) return (-1);

        ADI_EXPECT(populateSsiParam, fpga9001, port, interfaceRate_Hz, sampleRate_Hz, ssiConfig, &param);
        if (axi_adrv9001_ssi_config((void *)fpga9001, AXI_ADRV9001_ID, ssiId, &param) != 0)
        {
            ADI_ERROR_REPORT(&fpga9001->common,
                ADI_COMMON_ERRSRC_API,
                ADI_COMMON_ERR_API_FAIL,
                ADI_COMMON_ACT_ERR_API_NOT_IMPLEMENTED,
                0,
                "function axi_adrv9001_ssi_config has failed");
            ADI_API_RETURN(fpga9001);
        }
    }

    ADI_API_RETURN(fpga9001);
}

static __maybe_unused int32_t adi_fpga9001_ssi_Inspect_Validate(adi_fpga9001_Device_t *fpga9001,
                                                  adi_common_Port_e port,
                                                  adi_common_ChannelNumber_e channel,
                                                  adi_fpga9001_SsiConfig_t *ssiCfg)
{
    ADI_RANGE_CHECK(fpga9001, port, ADI_RX, ADI_ELB);
    ADI_RANGE_CHECK(fpga9001, channel, ADI_CHANNEL_1, ADI_CHANNEL_2);
    ADI_NULL_PTR_RETURN(&fpga9001->common, ssiCfg);
    ADI_API_RETURN(fpga9001);
}

static __maybe_unused int32_t readSsiControlData(adi_fpga9001_Device_t *fpga9001,
                                  adi_common_Port_e port,
                                  adi_common_ChannelNumber_e channel,
                                  uint32_t *read_data)
{
    uint32_t ssiId = 0;
    adi_common_Port_e port_hack = ADI_RX;

    /* hack to keep some tests to happy. */

    port_hack = port;

    if ((port == ADI_ORX) || (port == ADI_ILB) || (port == ADI_ELB)) {
        port_hack = ADI_RX;
    }

    ssiId = fpga9001_SsiIdGet(fpga9001, port_hack, channel);
    if (ssiId == (uint32_t) - 1) return (-1);

    axi_adrv9001_ssi_reg_control_get((void *)fpga9001, AXI_ADRV9001_ID, ssiId, read_data);

    ADI_API_RETURN(fpga9001);
}
int32_t adi_fpga9001_ssi_Inspect(adi_fpga9001_Device_t *fpga9001,
                                 adi_common_Port_e port,
                                 adi_common_ChannelNumber_e channel,
                                 adi_fpga9001_SsiConfig_t *ssiCfg)
{
    uint32_t ssiId = 0;
    uint32_t value = 0;
    
    ADI_PERFORM_VALIDATION(adi_fpga9001_ssi_Inspect_Validate, fpga9001, port, channel, ssiCfg);

    ssiId = fpga9001_SsiIdGet(fpga9001, port, channel);

    axi_adrv9001_ssi_cmos1_lvds0_get(fpga9001, AXI_ADRV9001_ID, ssiId, &value);
    ssiCfg->ssiType = (value == AXI_ADRV9001_SSI_MODE_CMOS) ? ADI_FPGA9001_SSI_TYPE_CMOS : ADI_FPGA9001_SSI_TYPE_LVDS;

    axi_adrv9001_ssi_num_of_bits_get(fpga9001, AXI_ADRV9001_ID, ssiId, &value);
    ssiCfg->ssiDataFormatSel = (adi_fpga9001_SsiFormat_e)value;

    axi_adrv9001_ssi_lstrb1_pstrb0_get(fpga9001, AXI_ADRV9001_ID, ssiId, &value);
    ssiCfg->strobeType = (adi_fpga9001_SsiStrobeType_e)value;

    axi_adrv9001_ssi_sdr1_ddr0_get(fpga9001, AXI_ADRV9001_ID, ssiId, &value);
    ssiCfg->ddrEn = (value == 0) ? true : false;

    axi_adrv9001_ssi_mlane1_slane0_get(fpga9001, AXI_ADRV9001_ID, ssiId, &value);
    if (value)
    {
        if (ADI_FPGA9001_SSI_TYPE_CMOS == ssiCfg->ssiType)
        {
            ssiCfg->numLaneSel = ADI_FPGA9001_SSI_4_LANE;
        }
        else
        {
            ssiCfg->numLaneSel = ADI_FPGA9001_SSI_2_LANE;
        }
    }
    else
    {
        ssiCfg->numLaneSel = ADI_FPGA9001_SSI_1_LANE;
    }

    ADI_API_RETURN(fpga9001);
}

static __maybe_unused int32_t adi_fpga9001_ssi_BytesPerSample_GetValidate(adi_fpga9001_Device_t *fpga9001, uint8_t *bytesPerSample)
{
    ADI_NULL_PTR_RETURN(&fpga9001->common, bytesPerSample);
    ADI_API_RETURN(fpga9001);
}

int32_t adi_fpga9001_ssi_BytesPerSample_Get(adi_fpga9001_Device_t *fpga9001,
                                            adi_common_Port_e port,
                                            adi_common_ChannelNumber_e channel,
                                            uint8_t *bytesPerSample)
{
    uint32_t ssiId = 0;
    uint32_t value = 0;
    adi_fpga9001_SsiFormat_e format = ADI_FPGA9001_SSI_FORMAT_2_BIT_SYMBOL_DATA;

    ADI_PERFORM_VALIDATION(adi_fpga9001_ssi_BytesPerSample_GetValidate, fpga9001, bytesPerSample);

    ssiId = fpga9001_SsiIdGet(fpga9001, port, channel);
    axi_adrv9001_ssi_num_of_bits_get(fpga9001, AXI_ADRV9001_ID, ssiId, &value);
    format = (adi_fpga9001_SsiFormat_e)value;

    switch (format)
    {
    case ADI_FPGA9001_SSI_FORMAT_2_BIT_SYMBOL_DATA:     /* Falls through */
    case ADI_FPGA9001_SSI_FORMAT_8_BIT_SYMBOL_DATA:
        *bytesPerSample = 1;
        break;
    case ADI_FPGA9001_SSI_FORMAT_16_BIT_SYMBOL_DATA:
        *bytesPerSample = 2;
        break;
    case ADI_FPGA9001_SSI_FORMAT_12_BIT_I_Q_DATA:
        *bytesPerSample = 3;
        break;
    case ADI_FPGA9001_SSI_FORMAT_16_BIT_I_Q_DATA:
    case ADI_FPGA9001_SSI_FORMAT_15_BIT_I_Q_DATA_1_BIT_GAIN_CHANGE:
        *bytesPerSample = 4;
        break;
    case ADI_FPGA9001_SSI_FORMAT_22_BIT_I_Q_DATA_1_BIT_GAIN_CHANGE_8_BIT_GAIN_INDEX:
        *bytesPerSample = 8;
        break;
    default:
        ADI_SHOULD_NOT_EXECUTE(fpga9001);
    }

    ADI_API_RETURN(fpga9001);
}

int32_t adi_fpga9001_ssi_Reset(adi_fpga9001_Device_t *fpga9001)
{
    axi_adrv9001_ssi_init_set(fpga9001, AXI_ADRV9001_ID, fpga9001_SsiIdGet(fpga9001, ADI_RX, ADI_CHANNEL_1), 0x1);
    axi_adrv9001_ssi_init_set(fpga9001, AXI_ADRV9001_ID, fpga9001_SsiIdGet(fpga9001, ADI_RX, ADI_CHANNEL_2), 0x1);
    axi_adrv9001_ssi_init_set(fpga9001, AXI_ADRV9001_ID, fpga9001_SsiIdGet(fpga9001, ADI_TX, ADI_CHANNEL_1), 0x1);
    axi_adrv9001_ssi_init_set(fpga9001, AXI_ADRV9001_ID, fpga9001_SsiIdGet(fpga9001, ADI_TX, ADI_CHANNEL_2), 0x1);
    ADI_API_RETURN(fpga9001);
}

static uint32_t fpga9001_SsiDataSelGet(adi_fpga9001_Device_t *fpga9001,
                                       adi_fpga9001_SsiTestModeData_e sel)
{
    if (sel == ADI_FPGA9001_SSI_TESTMODE_DATA_NORMAL) return(AXI_ADRV9001_SSI_DATA_SEL_DMA);
    if (sel == ADI_FPGA9001_SSI_TESTMODE_DATA_FIXED_PATTERN) return(AXI_ADRV9001_SSI_DATA_SEL_PATTERN);
    if (sel == ADI_FPGA9001_SSI_TESTMODE_DATA_RAMP_NIBBLE) return(AXI_ADRV9001_SSI_DATA_SEL_NIBBLE_RAMP);
    if (sel == ADI_FPGA9001_SSI_TESTMODE_DATA_RAMP_16_BIT) return(AXI_ADRV9001_SSI_DATA_SEL_RAMP);
    if (sel == ADI_FPGA9001_SSI_TESTMODE_DATA_PRBS15) return(AXI_ADRV9001_SSI_DATA_SEL_PRBS15);
    if (sel == ADI_FPGA9001_SSI_TESTMODE_DATA_PRBS7) return(AXI_ADRV9001_SSI_DATA_SEL_PRBS7);

    ADI_ERROR_REPORT(&fpga9001->common, ADI_COMMON_ERRSRC_API, 0, 0, sel, "Invalid SSI data select.");
    return((uint32_t) -1);
}

static __maybe_unused int32_t adi_fpga9001_ssi_Rx_TestMode_Configure_Validate(adi_fpga9001_Device_t *fpga9001,
                                                               adi_common_ChannelNumber_e channel,
                                                               adi_fpga9001_RxSsiTestModeCfg_t *ssiTestModeConfig)
{
    ADI_NULL_PTR_RETURN(&fpga9001->common, ssiTestModeConfig);
    ADI_RANGE_CHECK(fpga9001, channel, ADI_CHANNEL_1, ADI_CHANNEL_2);
    ADI_RANGE_CHECK(fpga9001, ssiTestModeConfig->testData, ADI_FPGA9001_SSI_TESTMODE_DATA_NORMAL, ADI_FPGA9001_SSI_TESTMODE_DATA_PRBS7);
    ADI_API_RETURN(fpga9001);
}

int32_t adi_fpga9001_ssi_Rx_TestMode_Configure(adi_fpga9001_Device_t *fpga9001,
                                               adi_common_ChannelNumber_e channel,
                                               adi_fpga9001_RxSsiTestModeCfg_t *ssiTestModeConfig)
{
    uint32_t ssiId;
    uint64_t pattern;

    ADI_PERFORM_VALIDATION(adi_fpga9001_ssi_Rx_TestMode_Configure_Validate, fpga9001, channel, ssiTestModeConfig);

    ssiId = fpga9001_SsiIdGet(fpga9001, ADI_RX, channel);
    if (ssiId == (uint32_t) -1) return(-1);

    pattern = (uint64_t) ssiTestModeConfig->fixedDataPatternToCheck;
    pattern = (pattern << 32) | ssiTestModeConfig->fixedDataPatternToCheck;

    axi_adrv9001_ssi_data_pattern_set((void *)fpga9001, AXI_ADRV9001_ID, ssiId, pattern);
    axi_adrv9001_ssi_data_sel_set((void *)fpga9001, AXI_ADRV9001_ID, ssiId, fpga9001_SsiDataSelGet(fpga9001, ssiTestModeConfig->testData));
    axi_adrv9001_ssi_timer_delay_us((void *)fpga9001, AXI_ADRV9001_ID, ssiId, 1);
    axi_adrv9001_ssi_reg_status_set((void *)fpga9001, AXI_ADRV9001_ID, ssiId, (uint32_t) -1);
    ADI_API_RETURN(fpga9001);
}

static __maybe_unused int32_t adi_fpga9001_ssi_Rx_TestMode_Inspect_Validate(adi_fpga9001_Device_t *fpga9001,
                                                             adi_common_ChannelNumber_e channel,
                                                             bool *dataError)
{
    ADI_NULL_PTR_RETURN(&fpga9001->common, dataError);
    ADI_RANGE_CHECK(fpga9001, channel, ADI_CHANNEL_1, ADI_CHANNEL_2);
    ADI_API_RETURN(fpga9001);
}

int32_t adi_fpga9001_ssi_Rx_TestMode_Inspect(adi_fpga9001_Device_t *fpga9001,
                                             adi_common_ChannelNumber_e channel,
                                             bool *dataError)
{
    uint32_t ssiId;
    uint32_t ssi_status = -1;

    ADI_PERFORM_VALIDATION(adi_fpga9001_ssi_Rx_TestMode_Inspect_Validate, fpga9001, channel, dataError);

    ssiId = fpga9001_SsiIdGet(fpga9001, ADI_RX, channel);
    if (ssiId == (uint32_t) -1) return(-1);

    axi_adrv9001_ssi_reg_status_get((void *)fpga9001, AXI_ADRV9001_ID, ssiId, &ssi_status);
    *dataError = (ssi_status == 0) ? false : true;
    ADI_API_RETURN(fpga9001);
}

static __maybe_unused int32_t adi_fpga9001_ssi_Tx_TestMode_Configure_Validate(adi_fpga9001_Device_t *fpga9001,
                                                               adi_common_ChannelNumber_e channel,
                                                               adi_fpga9001_TxSsiTestModeCfg_t *ssiTestModeConfig)
{
    ADI_NULL_PTR_RETURN(&fpga9001->common, ssiTestModeConfig);
    ADI_RANGE_CHECK(fpga9001, channel, ADI_CHANNEL_1, ADI_CHANNEL_2);
    ADI_API_RETURN(fpga9001);
}

int32_t adi_fpga9001_ssi_Tx_TestMode_Configure(adi_fpga9001_Device_t *fpga9001,
                                               adi_common_ChannelNumber_e channel,
                                               adi_fpga9001_TxSsiTestModeCfg_t *ssiTestModeConfig)
{
    uint32_t ssiId;
    uint64_t pattern;

    ADI_PERFORM_VALIDATION(adi_fpga9001_ssi_Tx_TestMode_Configure_Validate, fpga9001, channel, ssiTestModeConfig);

    ssiId = fpga9001_SsiIdGet(fpga9001, ADI_TX, channel);
    if (ssiId == (uint32_t) -1) return(-1);

    pattern = (uint64_t) ssiTestModeConfig->fixedDataPatternToTransmit;

    axi_adrv9001_ssi_data_pattern_set((void *)fpga9001, AXI_ADRV9001_ID, ssiId, pattern);
    axi_adrv9001_ssi_data_sel_set((void *)fpga9001, AXI_ADRV9001_ID, ssiId, fpga9001_SsiDataSelGet(fpga9001, ssiTestModeConfig->testData));
    axi_adrv9001_ssi_timer_delay_us((void *)fpga9001, AXI_ADRV9001_ID, ssiId, 1);
    ADI_API_RETURN(fpga9001);
}

static __maybe_unused int32_t adi_fpga9001_ssi_Delay_Configure_Validate(adi_fpga9001_Device_t *fpga9001,
                                                         adi_fpga9001_SsiType_e ssiType,
                                                         adi_fpga9001_SsiCalibrationCfg_t *ssiCalibration)
{
    static const uint8_t MAX_DELAY = 31;

	ADI_RANGE_CHECK(fpga9001, ssiType, ADI_FPGA9001_SSI_TYPE_CMOS, ADI_FPGA9001_SSI_TYPE_LVDS_COMMON_CLOCK);
    ADI_NULL_PTR_RETURN(&fpga9001->common, ssiCalibration);

    ADI_RANGE_CHECK(fpga9001, ssiCalibration->rxStrobeDelay[0], 0, MAX_DELAY);
    ADI_RANGE_CHECK(fpga9001, ssiCalibration->rxStrobeDelay[1], 0, MAX_DELAY);
    ADI_RANGE_CHECK(fpga9001, ssiCalibration->rxIDataDelay[0], 0, MAX_DELAY);
    ADI_RANGE_CHECK(fpga9001, ssiCalibration->rxIDataDelay[1], 0, MAX_DELAY);
    ADI_RANGE_CHECK(fpga9001, ssiCalibration->rxQDataDelay[0], 0, MAX_DELAY);
    ADI_RANGE_CHECK(fpga9001, ssiCalibration->rxQDataDelay[1], 0, MAX_DELAY);
    
    ADI_RANGE_CHECK(fpga9001, ssiCalibration->txClkDelay[0], 0, MAX_DELAY);
    ADI_RANGE_CHECK(fpga9001, ssiCalibration->txClkDelay[1], 0, MAX_DELAY);
    ADI_RANGE_CHECK(fpga9001, ssiCalibration->txStrobeDelay[0], 0, MAX_DELAY);
    ADI_RANGE_CHECK(fpga9001, ssiCalibration->txStrobeDelay[1], 0, MAX_DELAY);
    ADI_RANGE_CHECK(fpga9001, ssiCalibration->txIDataDelay[0], 0, MAX_DELAY);
    ADI_RANGE_CHECK(fpga9001, ssiCalibration->txIDataDelay[1], 0, MAX_DELAY);
    ADI_RANGE_CHECK(fpga9001, ssiCalibration->txQDataDelay[0], 0, MAX_DELAY);
    ADI_RANGE_CHECK(fpga9001, ssiCalibration->txQDataDelay[1], 0, MAX_DELAY);

    ADI_API_RETURN(fpga9001);
}

/* we have a very badly thought out structure here, parameters must be grouped within interface */
/* if we grouped them by interface, all the things below could have been just a single function */

static void fpga9001_SsiDelaySet_1(adi_fpga9001_Device_t *fpga9001,
                                      adi_common_Port_e port,
                                      adi_common_ChannelNumber_e channel,
                                      uint32_t *data)
{
    int32_t i = 0;
    uint32_t ssiId = 0;

    ssiId = fpga9001_SsiIdGet(fpga9001, port, channel);
    for (i = 0; i < 8; i++) {
        axi_adrv9001_ssi_wrdelay_set(fpga9001, AXI_ADRV9001_ID, ssiId, i, *(data + i));
    }
}

static void fpga9001_SsiDelaySet(adi_fpga9001_Device_t *fpga9001,
                                 adi_fpga9001_SsiType_e ssiType,
                                 adi_fpga9001_SsiCalibrationCfg_t *ssiCalibration)
{
    uint32_t ssiDelay[8];

    ssiDelay[0] = ssiCalibration->rxIDataDelay[0];
	if (ssiType == ADI_FPGA9001_SSI_TYPE_LVDS || ssiType == ADI_FPGA9001_SSI_TYPE_CH1LVDS_CH2CMOS || ssiType == ADI_FPGA9001_SSI_TYPE_LVDS_COMMON_CLOCK)
        ssiDelay[1] = ssiCalibration->rxQDataDelay[0];
    else
        ssiDelay[1] = ssiCalibration->rxIDataDelay[0];
    ssiDelay[2] = ssiCalibration->rxQDataDelay[0];
    ssiDelay[3] = ssiCalibration->rxQDataDelay[0];
    ssiDelay[4] = ssiCalibration->rxStrobeDelay[0];
    ssiDelay[5] = 0;
    ssiDelay[6] = 0;
    ssiDelay[7] = 0;
    fpga9001_SsiDelaySet_1(fpga9001, ADI_RX, ADI_CHANNEL_1, &ssiDelay[0]);

    ssiDelay[0] = ssiCalibration->rxIDataDelay[1];
	if (ssiType == ADI_FPGA9001_SSI_TYPE_LVDS || ssiType == ADI_FPGA9001_SSI_TYPE_CH1CMOS_CH2LVDS || ssiType == ADI_FPGA9001_SSI_TYPE_LVDS_COMMON_CLOCK)
        ssiDelay[1] = ssiCalibration->rxQDataDelay[1];
    else
        ssiDelay[1] = ssiCalibration->rxIDataDelay[1];
    ssiDelay[2] = ssiCalibration->rxQDataDelay[1];
    ssiDelay[3] = ssiCalibration->rxQDataDelay[1];
    ssiDelay[4] = ssiCalibration->rxStrobeDelay[1];
    ssiDelay[5] = 0;
    ssiDelay[6] = 0;
    ssiDelay[7] = 0;
    fpga9001_SsiDelaySet_1(fpga9001, ADI_RX, ADI_CHANNEL_2, &ssiDelay[0]);

    ssiDelay[0] = ssiCalibration->txIDataDelay[0];
	if (ssiType == ADI_FPGA9001_SSI_TYPE_LVDS || ssiType == ADI_FPGA9001_SSI_TYPE_CH1LVDS_CH2CMOS || ssiType == ADI_FPGA9001_SSI_TYPE_LVDS_COMMON_CLOCK)
        ssiDelay[1] = ssiCalibration->txQDataDelay[0];
    else
        ssiDelay[1] = ssiCalibration->txIDataDelay[0];
    ssiDelay[2] = ssiCalibration->txQDataDelay[0];
    ssiDelay[3] = ssiCalibration->txQDataDelay[0];
    ssiDelay[4] = ssiCalibration->txStrobeDelay[0];
    ssiDelay[5] = ssiCalibration->txClkDelay[0];
    ssiDelay[6] = 0;
    ssiDelay[7] = ssiCalibration->txClkDelay[0];
    fpga9001_SsiDelaySet_1(fpga9001, ADI_TX, ADI_CHANNEL_1, &ssiDelay[0]);

    ssiDelay[0] = ssiCalibration->txIDataDelay[1];
	if (ssiType == ADI_FPGA9001_SSI_TYPE_LVDS || ssiType == ADI_FPGA9001_SSI_TYPE_CH1CMOS_CH2LVDS || ssiType == ADI_FPGA9001_SSI_TYPE_LVDS_COMMON_CLOCK)
        ssiDelay[1] = ssiCalibration->txQDataDelay[1];
    else
        ssiDelay[1] = ssiCalibration->txIDataDelay[1];
    ssiDelay[2] = ssiCalibration->txQDataDelay[1];
    ssiDelay[3] = ssiCalibration->txQDataDelay[1];
    ssiDelay[4] = ssiCalibration->txStrobeDelay[1];
    ssiDelay[5] = ssiCalibration->txClkDelay[1];
    ssiDelay[6] = 0;
    ssiDelay[7] = ssiCalibration->txClkDelay[1];
    fpga9001_SsiDelaySet_1(fpga9001, ADI_TX, ADI_CHANNEL_2, &ssiDelay[0]);
}

int32_t adi_fpga9001_ssi_Delay_Configure(adi_fpga9001_Device_t *fpga9001,
                                         adi_fpga9001_SsiType_e ssiType,
                                         adi_fpga9001_SsiCalibrationCfg_t *ssiCalibration)
{
    ADI_PERFORM_VALIDATION(adi_fpga9001_ssi_Delay_Configure_Validate, fpga9001, ssiType, ssiCalibration);
    fpga9001_SsiDelaySet(fpga9001, ssiType, ssiCalibration);
    ADI_EXPECT(adi_fpga9001_ssi_TxClockPolarity_Set, fpga9001, ADI_CHANNEL_1, ssiCalibration->txClkInverted[0]);
    ADI_EXPECT(adi_fpga9001_ssi_TxClockPolarity_Set, fpga9001, ADI_CHANNEL_2, ssiCalibration->txClkInverted[1]);
    ADI_API_RETURN(fpga9001);
}

static void fpga9001_SsiDelayGet_1(adi_fpga9001_Device_t *fpga9001,
                                      adi_common_Port_e port,
                                      adi_common_ChannelNumber_e channel,
                                      uint32_t *data)
{
    int32_t i;
    uint32_t ssiId = 0;

    /* sw can read delays in on e of the two ways :
     *
     * raw delay - is the delay as output by the controller, for this
     * there must be a valid clock.
     *
     * register delay - is the register that sets the delay value.
     *
     * it is meaningless to read the register value, it will always match,
     * but the test seems to want to read this delay!!
     *
     */

    ssiId = fpga9001_SsiIdGet(fpga9001, port, channel);
    for (i = 0; i < 6; i++) {
        axi_adrv9001_ssi_wrdelay_get(fpga9001, AXI_ADRV9001_ID, ssiId, i, (data + i));
    }
}

static void fpga9001_SsiDelayGet(adi_fpga9001_Device_t *fpga9001,
                                 adi_fpga9001_SsiType_e ssiType,
                                 adi_fpga9001_SsiCalibrationCfg_t *ssiCalibration)
{
    uint32_t ssiDelay[6];

    fpga9001_SsiDelayGet_1(fpga9001, ADI_RX, ADI_CHANNEL_1, &ssiDelay[0]);
    ssiCalibration->rxIDataDelay[0] = ssiDelay[0];
    if (ssiType == ADI_FPGA9001_SSI_TYPE_LVDS) 
        ssiCalibration->rxQDataDelay[0] = ssiDelay[1];
    else
        ssiCalibration->rxQDataDelay[0] = ssiDelay[2];
    ssiCalibration->rxStrobeDelay[0] = ssiDelay[4];

    fpga9001_SsiDelayGet_1(fpga9001, ADI_RX, ADI_CHANNEL_2, &ssiDelay[0]);
    ssiCalibration->rxIDataDelay[1] = ssiDelay[0];
    if (ssiType == ADI_FPGA9001_SSI_TYPE_LVDS) 
        ssiCalibration->rxQDataDelay[1] = ssiDelay[1];
    else
        ssiCalibration->rxQDataDelay[1] = ssiDelay[2];
    ssiCalibration->rxStrobeDelay[1] = ssiDelay[4];

    fpga9001_SsiDelayGet_1(fpga9001, ADI_TX, ADI_CHANNEL_1, &ssiDelay[0]);
    ssiCalibration->txIDataDelay[0] = ssiDelay[0];
    if (ssiType == ADI_FPGA9001_SSI_TYPE_LVDS) 
        ssiCalibration->txQDataDelay[0] = ssiDelay[1];
    else
        ssiCalibration->txQDataDelay[0] = ssiDelay[2];
    ssiCalibration->txStrobeDelay[0] = ssiDelay[4];
    ssiCalibration->txClkDelay[0] = ssiDelay[5];

    fpga9001_SsiDelayGet_1(fpga9001, ADI_TX, ADI_CHANNEL_2, &ssiDelay[0]);
    ssiCalibration->txIDataDelay[1] = ssiDelay[0];
    if (ssiType == ADI_FPGA9001_SSI_TYPE_LVDS) 
        ssiCalibration->txQDataDelay[1] = ssiDelay[1];
    else
        ssiCalibration->txQDataDelay[1] = ssiDelay[2];
    ssiCalibration->txStrobeDelay[1] = ssiDelay[4];
    ssiCalibration->txClkDelay[1] = ssiDelay[5];
}

static __maybe_unused int32_t adi_fpga9001_ssi_Delay_Inspect_Validate(adi_fpga9001_Device_t *fpga9001,
                                                       adi_fpga9001_SsiType_e ssiType,
                                                       adi_fpga9001_SsiCalibrationCfg_t *ssiCalibration)
{
    ADI_RANGE_CHECK(fpga9001, ssiType, ADI_FPGA9001_SSI_TYPE_CMOS, ADI_FPGA9001_SSI_TYPE_LVDS);
    ADI_NULL_PTR_RETURN(&fpga9001->common, ssiCalibration);
    ADI_API_RETURN(fpga9001);
}

int32_t adi_fpga9001_ssi_Delay_Inspect(adi_fpga9001_Device_t *fpga9001,
                                       adi_fpga9001_SsiType_e ssiType,
                                       adi_fpga9001_SsiCalibrationCfg_t *ssiCalibration)
{
    ADI_PERFORM_VALIDATION(adi_fpga9001_ssi_Delay_Inspect_Validate, fpga9001, ssiType, ssiCalibration);
    fpga9001_SsiDelayGet(fpga9001, ssiType, ssiCalibration);
    ADI_EXPECT(adi_fpga9001_ssi_TxClockPolarity_Get, fpga9001, ADI_CHANNEL_1, &ssiCalibration->txClkInverted[0]);
    ADI_EXPECT(adi_fpga9001_ssi_TxClockPolarity_Get, fpga9001, ADI_CHANNEL_2, &ssiCalibration->txClkInverted[1]);
    ADI_API_RETURN(fpga9001);
}

static __maybe_unused int32_t adi_fpga9001_ssi_Delay_Calibrate_Validate(adi_fpga9001_Device_t *fpga9001,
                                                                        adi_common_ChannelNumber_e channel,
                                                                        uint32_t *ssiDelay)
{
    ADI_RANGE_CHECK(fpga9001, channel, ADI_CHANNEL_1, ADI_CHANNEL_2);
    ADI_NULL_PTR_RETURN(fpga9001, ssiDelay);
    ADI_API_RETURN(fpga9001);
}

int32_t adi_fpga9001_ssi_Delay_Calibrate(adi_fpga9001_Device_t *fpga9001,
                                         adi_common_ChannelNumber_e channel,
                                         uint32_t *ssiDelay)
{
    uint32_t ssiId = 0;
    
    ADI_PERFORM_VALIDATION(adi_fpga9001_ssi_Delay_Calibrate_Validate, fpga9001, channel, ssiDelay);
 
    ssiId = fpga9001_SsiIdGet(fpga9001, ADI_RX, channel);
    
    ADI_EXPECT(axi_adrv9001_ssi_delay_calibrate, fpga9001, AXI_ADRV9001_ID, ssiId, 1, ssiDelay);
    
    ADI_API_RETURN(fpga9001);
}

int32_t adi_fpga9001_ssi_TxSs_Ratio_Set_Validate(adi_fpga9001_Device_t *fpga9001,
                                                 adi_common_ChannelNumber_e channel,
                                                 uint32_t interfaceRate_Hz,
                                                 uint32_t sampleRate_Hz)
{
    ADI_RANGE_CHECK(fpga9001, channel, ADI_CHANNEL_1, ADI_CHANNEL_2);
    
    if (0 == interfaceRate_Hz)
    {
        ADI_ERROR_REPORT(&fpga9001->common,
            ADI_COMMON_ERRSRC_API,
            ADI_COMMON_ERR_API_FAIL,
            ADI_COMMON_ACT_ERR_CHECK_PARAM,
            interfaceRate_Hz,
            "interfaceRate_Hz cannot be 0");
        ADI_API_RETURN(fpga9001);   
    }
    
    if (0 == sampleRate_Hz)
    {
        ADI_ERROR_REPORT(&fpga9001->common,
            ADI_COMMON_ERRSRC_API,
            ADI_COMMON_ERR_API_FAIL,
            ADI_COMMON_ACT_ERR_CHECK_PARAM,
            sampleRate_Hz,
            "sampleRate_Hz cannot be 0");
        ADI_API_RETURN(fpga9001);   
    }
    
    ADI_API_RETURN(fpga9001);
}

int32_t adi_fpga9001_ssi_TxSs_Ratio_Set(adi_fpga9001_Device_t *fpga9001,
                                        adi_common_ChannelNumber_e channel,
                                        uint32_t interfaceRate_Hz,
                                        uint32_t sampleRate_Hz)
{
    uint32_t ssiId = 0;
    uint32_t ss_ratio = 0;

    ADI_PERFORM_VALIDATION(adi_fpga9001_ssi_TxSs_Ratio_Set_Validate, fpga9001, channel, interfaceRate_Hz, sampleRate_Hz);

    if (sampleRate_Hz <= interfaceRate_Hz)
    {
        int y = interfaceRate_Hz / sampleRate_Hz;
        while (y >>= 1) ++ss_ratio;
    }
    
    ssiId = fpga9001_SsiIdGet(fpga9001, ADI_TX, channel);
    axi_adrv9001_ssi_ss_ratio_set(fpga9001, AXI_ADRV9001_ID, ssiId, ss_ratio);
    ADI_API_RETURN(fpga9001);
}

int32_t adi_fpga9001_ssi_TxSs_Mode_Set(adi_fpga9001_Device_t *fpga9001,
                                       adi_common_ChannelNumber_e channel,
                                       uint32_t data)
{
    uint32_t ssiId = 0;

    ADI_RANGE_CHECK(fpga9001, channel, ADI_CHANNEL_1, ADI_CHANNEL_2);

    ssiId = fpga9001_SsiIdGet(fpga9001, ADI_TX, channel);
    axi_adrv9001_ssi_ss_mode_set(fpga9001, AXI_ADRV9001_ID, ssiId, data);
    ADI_API_RETURN(fpga9001);
}

int32_t adi_fpga9001_ssi_RxSs_Ratio_Set_Validate(adi_fpga9001_Device_t *fpga9001,
                                                 adi_common_ChannelNumber_e channel,
                                                 uint32_t interfaceRate_Hz,
                                                 uint32_t sampleRate_Hz)
{
    ADI_RANGE_CHECK(fpga9001, channel, ADI_CHANNEL_1, ADI_CHANNEL_2);
    
    if (0 == interfaceRate_Hz)
    {
        ADI_ERROR_REPORT(&fpga9001->common,
            ADI_COMMON_ERRSRC_API,
            ADI_COMMON_ERR_API_FAIL,
            ADI_COMMON_ACT_ERR_CHECK_PARAM,
            interfaceRate_Hz,
            "interfaceRate_Hz cannot be 0");
        ADI_API_RETURN(fpga9001);   
    }
    
    if (0 == sampleRate_Hz)
    {
        ADI_ERROR_REPORT(&fpga9001->common,
            ADI_COMMON_ERRSRC_API,
            ADI_COMMON_ERR_API_FAIL,
            ADI_COMMON_ACT_ERR_CHECK_PARAM,
            sampleRate_Hz,
            "sampleRate_Hz cannot be 0");
        ADI_API_RETURN(fpga9001);   
    }
    
    ADI_API_RETURN(fpga9001);
}

int32_t adi_fpga9001_ssi_RxSs_Ratio_Set(adi_fpga9001_Device_t *fpga9001,
                                        adi_common_ChannelNumber_e channel,
                                        uint32_t interfaceRate_Hz,
                                        uint32_t sampleRate_Hz)
{
    uint32_t ssiId = 0;
    uint32_t ss_ratio = 0;

    ADI_PERFORM_VALIDATION(adi_fpga9001_ssi_RxSs_Ratio_Set_Validate, fpga9001, channel, interfaceRate_Hz, sampleRate_Hz);

    ssiId = fpga9001_SsiIdGet(fpga9001, ADI_RX, channel);
    axi_adrv9001_ssi_ss_ratio_set(fpga9001, AXI_ADRV9001_ID, ssiId, ss_ratio);
    ADI_API_RETURN(fpga9001);
}

int32_t adi_fpga9001_ssi_OrxSs_Ratio_Set_Validate(adi_fpga9001_Device_t *fpga9001,
                                                  adi_common_ChannelNumber_e channel,
                                                  uint32_t interfaceRate_Hz,
                                                  uint32_t sampleRate_Hz)
{
    ADI_RANGE_CHECK(fpga9001, channel, ADI_CHANNEL_1, ADI_CHANNEL_2);
    
    if (0 == interfaceRate_Hz)
    {
        ADI_ERROR_REPORT(&fpga9001->common,
            ADI_COMMON_ERRSRC_API,
            ADI_COMMON_ERR_API_FAIL,
            ADI_COMMON_ACT_ERR_CHECK_PARAM,
            interfaceRate_Hz,
            "interfaceRate_Hz cannot be 0");
        ADI_API_RETURN(fpga9001);   
    }
    
    if (0 == sampleRate_Hz)
    {
        ADI_ERROR_REPORT(&fpga9001->common,
            ADI_COMMON_ERRSRC_API,
            ADI_COMMON_ERR_API_FAIL,
            ADI_COMMON_ACT_ERR_CHECK_PARAM,
            sampleRate_Hz,
            "sampleRate_Hz cannot be 0");
        ADI_API_RETURN(fpga9001);   
    }
    
    ADI_API_RETURN(fpga9001);
}

int32_t adi_fpga9001_ssi_OrxSs_Ratio_Set(adi_fpga9001_Device_t *fpga9001,
                                         adi_common_ChannelNumber_e channel,
                                         uint32_t interfaceRate_Hz,
                                         uint32_t sampleRate_Hz)
{
    uint32_t ssiId = 0;
    uint32_t orx_ss_ratio = 0;
    
    ADI_PERFORM_VALIDATION(adi_fpga9001_ssi_OrxSs_Ratio_Set_Validate, fpga9001, channel, interfaceRate_Hz, sampleRate_Hz);

    ssiId = fpga9001_SsiIdGet(fpga9001, ADI_RX, channel);
    axi_adrv9001_ssi_obs_ss_ratio_set(fpga9001, AXI_ADRV9001_ID, ssiId, orx_ss_ratio);
    ADI_API_RETURN(fpga9001);
}
