/**
* \file
* \brief FPGA9001 Multi-Chip Synchronization (MCS) functions
*
* ADRV9001 API Version: $ADI_ADRV9001_API_VERSION$
*/

/**
* Copyright 2019-2025 Analog Devices Inc.
* Released under the ADRV9001 API license, for more information
* see the "LICENSE.txt" file in this zip file.
*/
#include "adi_adrv9001_user.h"
#include "adi_common_types.h"
#include "adi_fpga9001_mcs.h"
#include "adi_fpga9001_datachain.h"
#include "axi_adrv9001.h"
#include "adrv9001_zcu102.h"
#include "fpga9001_utilities.h"

static __maybe_unused int32_t adi_fpga9001_Mcs_Configure_Validate(adi_fpga9001_Device_t *fpga9001, 
                                                   adi_fpga9001_McsCfg_t *mcsCfg)
{
    ADI_RANGE_CHECK(fpga9001, mcsCfg->numberOfPulses, 1, 16);
    ADI_RANGE_CHECK(fpga9001, mcsCfg->mcsPulseWidth, 1, 16);
    ADI_RANGE_CHECK(fpga9001, mcsCfg->edge, ADI_FPGA9001_MCS_EDGE_RISING, ADI_FPGA9001_MCS_EDGE_FALLING);
    ADI_API_RETURN(fpga9001);
}

int32_t adi_fpga9001_Mcs_Configure(adi_fpga9001_Device_t *fpga9001, 
                                   adi_fpga9001_McsCfg_t *mcsCfg)
{
    uint32_t status = -1;
    uint32_t ssi_id_rx2 = 0;
    ssi_id_rx2 = fpga9001_SsiIdGet(fpga9001, ADI_RX, ADI_CHANNEL_2);
    axi_adrv9001_mcs_params_t mcs_params = { 0 };
    ADI_PERFORM_VALIDATION(adi_fpga9001_Mcs_Configure_Validate, fpga9001, mcsCfg);

    axi_adrv9001_mcs_status(fpga9001, AXI_ADRV9001_ID, &status);
    if (status != 0) 
    {
        ADI_ERROR_REPORT(&fpga9001->common, ADI_COMMON_ERRSRC_API, ADI_COMMON_ERR_API_FAIL,
        ADI_COMMON_ACT_ERR_RESET_FEATURE, 0, "previous MCS sequence pending.");
        ADI_ERROR_RETURN(fpga9001->common.error.newAction);
    }

    mcs_params.period = mcsCfg->mcsPeriod - 1;
    mcs_params.width  = mcsCfg->mcsPulseWidth - 1;
    mcs_params.count  = mcsCfg->numberOfPulses - 1;
    mcs_params.edge = (uint32_t)mcsCfg->edge;
    mcs_params.select = AXI_ADRV9001_TOP_MCS_SEQUENCE;

    axi_adrv9001_mcs_config((void *)fpga9001, AXI_ADRV9001_ID, &mcs_params);

    ADI_API_RETURN(fpga9001);
}

int32_t adi_fpga9001_Mcs_TxSsiConfigure(adi_fpga9001_Device_t *fpga9001)
{
    axi_adrv9001_top_reg_mcs_tx_resetn_set(fpga9001, AXI_ADRV9001_ID, AXI_ADRV9001_TOP_OFFSET, 0x0);
    axi_adrv9001_top_mcs_timer_delay_us(fpga9001, AXI_ADRV9001_ID, AXI_ADRV9001_TOP_OFFSET, 100);

    axi_adrv9001_top_mcs_tx_clk_resetn_set(fpga9001, AXI_ADRV9001_ID, AXI_ADRV9001_TOP_OFFSET, 0x1);
    axi_adrv9001_top_mcs_timer_delay_us(fpga9001, AXI_ADRV9001_ID, AXI_ADRV9001_TOP_OFFSET, 100);
    axi_adrv9001_top_mcs_tx_phy_resetn_set(fpga9001, AXI_ADRV9001_ID, AXI_ADRV9001_TOP_OFFSET, 0x1);
    axi_adrv9001_top_mcs_timer_delay_us(fpga9001, AXI_ADRV9001_ID, AXI_ADRV9001_TOP_OFFSET, 100);
    axi_adrv9001_top_mcs_tx_dp_resetn_set(fpga9001, AXI_ADRV9001_ID, AXI_ADRV9001_TOP_OFFSET, 0x1);
    axi_adrv9001_top_mcs_timer_delay_us(fpga9001, AXI_ADRV9001_ID, AXI_ADRV9001_TOP_OFFSET, 100);

    ADI_API_RETURN(fpga9001);
}

int32_t adi_fpga9001_Mcs_RxSsiConfigure(adi_fpga9001_Device_t *fpga9001)
{
    int i;
    uint32_t diff;
    uint32_t ssiId;
    uint32_t timeout;
    uint32_t busy[2];
    uint32_t strobe[2];
    uint32_t latency[2];

    axi_adrv9001_top_reg_mcs_rx_resetn_set(fpga9001, AXI_ADRV9001_ID, AXI_ADRV9001_TOP_OFFSET, 0x0);
    axi_adrv9001_top_mcs_timer_delay_us(fpga9001, AXI_ADRV9001_ID, AXI_ADRV9001_TOP_OFFSET, 100);

    axi_adrv9001_top_mcs_rx_clk_resetn_set(fpga9001, AXI_ADRV9001_ID, AXI_ADRV9001_TOP_OFFSET, 0x1);
    axi_adrv9001_top_mcs_timer_delay_us(fpga9001, AXI_ADRV9001_ID, AXI_ADRV9001_TOP_OFFSET, 100);
    axi_adrv9001_top_mcs_rx_phy_resetn_set(fpga9001, AXI_ADRV9001_ID, AXI_ADRV9001_TOP_OFFSET, 0x1);
    axi_adrv9001_top_mcs_timer_delay_us(fpga9001, AXI_ADRV9001_ID, AXI_ADRV9001_TOP_OFFSET, 100);

    busy[0] = 1;
    busy[1] = 1;

    timeout = (uint32_t) -1;

    axi_adrv9001_top_mcs_timer_set(fpga9001, AXI_ADRV9001_ID, AXI_ADRV9001_TOP_OFFSET, timeout);

    while ((timeout > 0) && ((busy[0] == 1) || (busy[1] == 1))) {

        axi_adrv9001_top_mcs_timer_get(fpga9001, AXI_ADRV9001_ID, AXI_ADRV9001_TOP_OFFSET, &timeout);
        for (i = 0; i < 2; i++) {
            ssiId = fpga9001_SsiIdGet(fpga9001, ADI_RX, ((i == 0) ? ADI_CHANNEL_1 : ADI_CHANNEL_2));
            axi_adrv9001_ssi_mcs_latency_busy_get(fpga9001, AXI_ADRV9001_ID, ssiId, &busy[i]);
        }
    }

    if ((busy[0] == 1) || (busy[1] == 1)) {

        ADI_ERROR_REPORT(&fpga9001->common, ADI_COMMON_ERRSRC_API, ADI_COMMON_ERR_API_FAIL,
            ADI_COMMON_ACT_ERR_RESET_FEATURE, 0, "likely failure, no valid strobes after mcs.");
        ADI_ERROR_RETURN(fpga9001->common.error.newAction);
    }

    axi_adrv9001_top_mcs_timer_delay_us(fpga9001, AXI_ADRV9001_ID, AXI_ADRV9001_TOP_OFFSET, 100);

    for (i = 0; i < 2; i++) {
        ssiId = fpga9001_SsiIdGet(fpga9001, ADI_RX, ((i == 0) ? ADI_CHANNEL_1 : ADI_CHANNEL_2));
        axi_adrv9001_ssi_mcs_latency_count_get(fpga9001, AXI_ADRV9001_ID, ssiId, &latency[i]);
        axi_adrv9001_ssi_mcs_strobe_get(fpga9001, AXI_ADRV9001_ID, ssiId, &strobe[i]);
        strobe[i] = strobe[i] & 0xff;
    }

    diff = (latency[0] > latency[1]) ? (latency[0] - latency[1]) : (latency[1] - latency[0]);

    if ((diff > 8) && (strobe[0] < 0x10) && (strobe[1] > 0x0f)) {
        ssiId = fpga9001_SsiIdGet(fpga9001, ADI_RX, ADI_CHANNEL_1);
        axi_adrv9001_ssi_mcs_delay_count_set(fpga9001, AXI_ADRV9001_ID, ssiId, 0x2);
    }

    if ((diff > 8) && (strobe[0] > 0x0f) && (strobe[1] < 0x10)) {
        ssiId = fpga9001_SsiIdGet(fpga9001, ADI_RX, ADI_CHANNEL_2);
        axi_adrv9001_ssi_mcs_delay_count_set(fpga9001, AXI_ADRV9001_ID, ssiId, 0x2);
    }

    axi_adrv9001_top_mcs_timer_delay_us(fpga9001, AXI_ADRV9001_ID, AXI_ADRV9001_TOP_OFFSET, 100);
    axi_adrv9001_top_mcs_rx_dp_resetn_set(fpga9001, AXI_ADRV9001_ID, AXI_ADRV9001_TOP_OFFSET, 0x1);
    axi_adrv9001_top_mcs_timer_delay_us(fpga9001, AXI_ADRV9001_ID, AXI_ADRV9001_TOP_OFFSET, 100);

    ADI_API_RETURN(fpga9001);
}

int32_t adi_fpga9001_Mcs_Start(adi_fpga9001_Device_t *fpga9001)
{
    uint32_t status = -1;
    uint32_t timeout_us = -1;

    ADI_NULL_DEVICE_PTR_RETURN(fpga9001);
    axi_adrv9001_mcs_start(fpga9001, AXI_ADRV9001_ID);
    axi_adrv9001_top_timer_set(fpga9001, AXI_ADRV9001_ID, AXI_ADRV9001_TOP_OFFSET, timeout_us);
    while (status != 0)
    {
        axi_adrv9001_mcs_status(fpga9001, AXI_ADRV9001_ID, &status);
        axi_adrv9001_top_timer_get(fpga9001, AXI_ADRV9001_ID, AXI_ADRV9001_TOP_OFFSET, &timeout_us);
        if (timeout_us == 0)
        {
            ADI_ERROR_REPORT(&fpga9001->common, ADI_COMMON_ERRSRC_API, ADI_COMMON_ERR_API_FAIL,
                ADI_COMMON_ACT_ERR_RESET_FEATURE, 0, "MCS sequence incomplete, timed-out.");
            ADI_ERROR_RETURN(fpga9001->common.error.newAction);
        }
    }
    ADI_API_RETURN(fpga9001);
}

int32_t adi_fpga9001_Mcs_RxMcsToStrobeSampleLatency_Get (adi_fpga9001_Device_t *fpga9001, adi_common_ChannelNumber_e channel, uint32_t *latency)
{
    ADI_NULL_DEVICE_PTR_RETURN(fpga9001);
    ADI_NULL_PTR_RETURN(&fpga9001->common, latency);
    uint32_t ssiId = fpga9001_SsiIdGet(fpga9001, ADI_RX, channel);
    axi_adrv9001_ssi_mcs_latency_busy_get(fpga9001, AXI_ADRV9001_ID, ssiId, latency);
    if (*latency == 0) {
        axi_adrv9001_ssi_mcs_latency_count_get(fpga9001, AXI_ADRV9001_ID, ssiId, latency);
    } else {
        *latency = 0;
    }
    
    ADI_API_RETURN(fpga9001);
}

int32_t adi_fpga9001_Mcs_Done_Get (adi_fpga9001_Device_t *fpga9001, bool *mcsDone)
{
    uint32_t status = -1;

    ADI_NULL_DEVICE_PTR_RETURN(fpga9001);
    ADI_NULL_PTR_RETURN(&fpga9001->common, mcsDone);
    axi_adrv9001_mcs_status(fpga9001, AXI_ADRV9001_ID, &status);
    *mcsDone = (status == 0);

    ADI_API_RETURN(fpga9001);
}

int32_t adi_fpga9001_Mcs_RxMcsStatus_Get(adi_fpga9001_Device_t *fpga9001, 
                                         adi_fpga9001_McsStatus_t *mcsStatus)
{
    int i;
    uint32_t busy;
    uint32_t ssiId;

    ADI_NULL_DEVICE_PTR_RETURN(fpga9001);
    ADI_NULL_PTR_RETURN(&fpga9001->common, mcsStatus);

    for (i = 0; i < 2; i++) {

        ssiId = fpga9001_SsiIdGet(fpga9001, ADI_RX, ((i == 0) ? ADI_CHANNEL_1 : ADI_CHANNEL_2));

        axi_adrv9001_ssi_mcs_latency_busy_get(fpga9001, AXI_ADRV9001_ID, ssiId, &busy);

        if (busy == 1) {

            ADI_ERROR_REPORT(&fpga9001->common, ADI_COMMON_ERRSRC_API, ADI_COMMON_ERR_API_FAIL,
                ADI_COMMON_ACT_ERR_RESET_FEATURE, 0, "likely failure, no valid strobes after mcs.");
            ADI_ERROR_RETURN(fpga9001->common.error.newAction);
        }

        axi_adrv9001_ssi_mcs_latency_count_get(fpga9001, AXI_ADRV9001_ID, ssiId, &mcsStatus->latency[i]);
        axi_adrv9001_ssi_align_sel_get(fpga9001, AXI_ADRV9001_ID, ssiId, &mcsStatus->align_sel[i]);
        axi_adrv9001_ssi_mcs_strobe_get(fpga9001, AXI_ADRV9001_ID, ssiId, &mcsStatus->strobe[i]);
        axi_adrv9001_ssi_mcs_data_get(fpga9001, AXI_ADRV9001_ID, ssiId, &mcsStatus->data[i]);
        axi_adrv9001_ssi_mcs_bstrobe_get(fpga9001, AXI_ADRV9001_ID, ssiId, &mcsStatus->bstrobe[i]);
        axi_adrv9001_ssi_mcs_bdata_get(fpga9001, AXI_ADRV9001_ID, ssiId, &mcsStatus->bdata[i]);
        axi_adrv9001_ssi_mcs_delay_count_get(fpga9001, AXI_ADRV9001_ID, ssiId, &mcsStatus->delay_sel[i]);
    }

    ADI_API_RETURN(fpga9001);
}

