/**
* \file
* \brief Contains ADI Transceiver Hardware Abstraction functions
*        Analog Devices maintains and provides updates to this code layer.
*        The end user should not modify this file or any code in this directory.
*/

/**
* Copyright 2020 Analog Devices Inc.
* Released under the ADRV9001 API license, for more information
* see the "LICENSE.txt" file in this zip file.
*/

#include "adi_fpga9001_hal.h"
#include "adi_fpga9001_hal_wrapper.h"

int32_t adi_fpga9001_hal_wrapper_Register_Read(adi_fpga9001_Device_t *fpga9001, const uint32_t addr, uint32_t *data)
{
    int32_t halError = 0;
    
    ADI_NULL_DEVICE_PTR_RETURN(fpga9001);
    ADI_NULL_PTR_RETURN(fpga9001, data);
    
    halError = adi_fpga9001_hal_Register_Read(fpga9001->common.devHalInfo, addr, data);
    ADI_ERROR_REPORT(fpga9001, 2, halError, ADI_COMMON_ACT_ERR_RESET_FEATURE, NULL, "FPGA register read failed");
    
    ADI_API_RETURN(fpga9001);
}

int32_t adi_fpga9001_hal_wrapper_Register_Write(adi_fpga9001_Device_t *fpga9001, uint32_t addr, uint32_t data)
{
    int32_t halError = 0;
    
    ADI_NULL_DEVICE_PTR_RETURN(fpga9001);
    
    halError = adi_fpga9001_hal_Register_Write(fpga9001->common.devHalInfo, addr, data);
    ADI_ERROR_REPORT(fpga9001, 2, halError, ADI_COMMON_ACT_ERR_RESET_FEATURE, NULL, "FPGA register write failed");
    
    ADI_API_RETURN(fpga9001);
}

int32_t adi_fpga9001_hal_wrapper_Ram_Read(adi_fpga9001_Device_t *fpga9001, 
                                          const uint32_t ramAddress,
                                          uint32_t data[],
                                          uint32_t length)
{
    int32_t halError = 0;
    
    ADI_NULL_DEVICE_PTR_RETURN(fpga9001);
    ADI_NULL_PTR_RETURN(fpga9001, data);
    
    halError = adi_fpga9001_hal_Ram_Read(fpga9001->common.devHalInfo, ramAddress, data, length);
    ADI_ERROR_REPORT(fpga9001, 2, halError, ADI_COMMON_ACT_ERR_RESET_FEATURE, NULL, "FPGA RAM read failed");
    
    ADI_API_RETURN(fpga9001);
}

int32_t adi_fpga9001_hal_wrapper_Ram_Write(adi_fpga9001_Device_t *fpga9001, 
                                           const uint32_t ramAddress,
                                           uint32_t data[],
                                           uint32_t length)
{
    int32_t halError = 0;
    
    ADI_NULL_DEVICE_PTR_RETURN(fpga9001);
    ADI_NULL_PTR_RETURN(fpga9001, data);
    
    halError = adi_fpga9001_hal_Ram_Write(fpga9001->common.devHalInfo, ramAddress, data, length);
    ADI_ERROR_REPORT(fpga9001, 2, halError, ADI_COMMON_ACT_ERR_RESET_FEATURE, NULL, "FPGA RAM write failed");
    
    ADI_API_RETURN(fpga9001);
}

int32_t adi_fpga9001_hal_wrapper_SsiType_Set(adi_fpga9001_Device_t *fpga9001, adi_fpga9001_SsiType_e ssiType)
{
    int32_t halError = 0;
    
    ADI_NULL_DEVICE_PTR_RETURN(fpga9001);
	ADI_RANGE_CHECK(fpga9001, ssiType, ADI_FPGA9001_SSI_TYPE_CMOS, ADI_FPGA9001_SSI_TYPE_LVDS_COMMON_CLOCK);
    
    halError = adi_fpga9001_hal_SsiType_Set(fpga9001->common.devHalInfo, ssiType);
    ADI_ERROR_REPORT(fpga9001, 2, halError, ADI_COMMON_ACT_ERR_RESET_FEATURE, NULL, "Failed to set FPGA SSI type");
    
    ADI_API_RETURN(fpga9001);
}

int32_t adi_fpga9001_hal_wrapper_DmaRelativeAddress_Get(adi_fpga9001_Device_t *fpga9001,
                                                        adi_fpga9001_hal_Dma_e dma,
                                                        uint32_t *addr)
{
    int32_t halError = 0;
    
    ADI_NULL_DEVICE_PTR_RETURN(fpga9001);
    ADI_RANGE_CHECK(fpga9001, dma, ADI_FPGA9001_HAL_DMA_RX1, ADI_FPGA9001_HAL_DMA_SPI);
    ADI_NULL_PTR_RETURN(fpga9001, addr);
    
    halError = adi_fpga9001_hal_DmaRelativeAddress_Get(fpga9001->common.devHalInfo, dma, addr);
    ADI_ERROR_REPORT(fpga9001, 2, halError, ADI_COMMON_ACT_ERR_RESET_FEATURE, NULL, "Failed to get DMA relative address");
    
    ADI_API_RETURN(fpga9001);
}

int32_t adi_fpga9001_hal_wrapper_CurrentPlatform_Get(adi_fpga9001_Device_t *fpga9001, adi_fpga9001_Platform_e *platform)
{
    int32_t halError = 0;
    ADI_NULL_DEVICE_PTR_RETURN(fpga9001);
    halError = adi_fpga9001_hal_CurrentPlatform_Get(fpga9001->common.devHalInfo, platform);
    ADI_ERROR_REPORT(fpga9001, 2, halError, ADI_COMMON_ACT_ERR_RESET_FEATURE, NULL, "Failed to get FPGA platform type");
    ADI_API_RETURN(fpga9001);
}

