/**
 * \file:
 * \brief Functions to configure and control the FPGA9001 GPIO pins
 *
 * FPGA9001 API Version: $ADI_FPGA9001_API_VERSION$
 */

/**
 * Copyright 2018-2025 Analog Devices Inc.
 * Released under the FPGA9001 API license, for more information
 * see the "LICENSE.txt" file in this zip file.
 */

#include "adi_fpga9001_gpio.h"
#include "adi_fpga9001.h"
#include "fpga9001_utilities.h"

static __maybe_unused int32_t adi_fpga9001_gpio_OperationMode_Set_Validate(adi_fpga9001_Device_t *fpga9001, adi_fpga9001_OperationModes_e operationMode)
{
    ADI_RANGE_CHECK(fpga9001, operationMode, ADI_FPGA9001_OPERATION_MODE_NORMAL, ADI_FPGA9001_OPERATION_MODE_JTAG);

    ADI_API_RETURN(fpga9001);
}

int32_t adi_fpga9001_gpio_OperationMode_Set(adi_fpga9001_Device_t *fpga9001, adi_fpga9001_OperationModes_e operationMode)
{
    ADI_PERFORM_VALIDATION(adi_fpga9001_gpio_OperationMode_Set_Validate, fpga9001, operationMode);
    axi_adrv9001_top_mode_set((void *)fpga9001, AXI_ADRV9001_ID, AXI_ADRV9001_TOP_OFFSET, (uint32_t) operationMode);
    ADI_API_RETURN(fpga9001);
}

static __maybe_unused int32_t adi_fpga9001_gpio_OperationMode_Get_Validate(adi_fpga9001_Device_t *fpga9001, adi_fpga9001_OperationModes_e *operationMode)
{
    ADI_NULL_PTR_RETURN(&fpga9001->common, operationMode);

    ADI_API_RETURN(fpga9001);
}

int32_t adi_fpga9001_gpio_OperationMode_Get(adi_fpga9001_Device_t *fpga9001, adi_fpga9001_OperationModes_e *operationMode)
{
    uint32_t mode = 0;
    ADI_PERFORM_VALIDATION(adi_fpga9001_gpio_OperationMode_Get_Validate, fpga9001, operationMode);
    axi_adrv9001_top_mode_get((void *)fpga9001, AXI_ADRV9001_ID, AXI_ADRV9001_TOP_OFFSET, &mode);
    *operationMode = (mode == 0) ? ADI_FPGA9001_OPERATION_MODE_NORMAL : ADI_FPGA9001_OPERATION_MODE_JTAG;
    ADI_API_RETURN(fpga9001);
}

static __maybe_unused int32_t adi_fpga9001_gpio_ReadValidate(adi_fpga9001_Device_t *fpga9001, adi_fpga9001_GpioGroup_e group, uint16_t *rdData)
{
    ADI_NULL_PTR_RETURN(&fpga9001->common, rdData);
    ADI_RANGE_CHECK(fpga9001, group, ADI_FPGA9001_GPIO_GROUP_DGPIO, ADI_FPGA9001_GPIO_GROUP_TX1);

    ADI_API_RETURN(fpga9001);
}

int32_t adi_fpga9001_gpio_Read(adi_fpga9001_Device_t *fpga9001, adi_fpga9001_GpioGroup_e group, uint16_t *rdData)
{
    uint32_t data = 0;
    uint32_t gpioGroup = 0;

    ADI_PERFORM_VALIDATION(adi_fpga9001_gpio_ReadValidate, fpga9001, group , rdData);
    
    axi_adrv9001_gpio_group_data((void *)fpga9001, AXI_ADRV9001_ID, (uint32_t) group, &data);
    *rdData = (uint16_t) data;
    ADI_API_RETURN(fpga9001);
}

static __maybe_unused int32_t adi_fpga9001_gpio_HopIndex_GainIndex_Configure_Validate(adi_fpga9001_Device_t *fpga9001,
                                                                       adi_fpga9001_DmaTrigger_e trigger,
                                                                       uint32_t index[],
                                                                       uint32_t indexLength,
                                                                       adi_fpga9001_GpioPin_e gpioPin[],
                                                                       uint32_t gpioPinLength)
{
    uint8_t i = 0;
    static const uint8_t HOP_INDEX_VALUE_MAX = 64;
    static const uint8_t GPIO_PIN_MIN = 1;
    static const uint8_t GPIO_PIN_MAX = 10;
    ADI_RANGE_CHECK(fpga9001, trigger, ADI_FPGA9001_DMA_TRIGGER_SMA_1, ADI_FPGA9001_DMA_TRIGGER_IMMEDIATE);
    ADI_RANGE_CHECK(fpga9001, indexLength, 1, HOP_INDEX_VALUE_MAX);
    ADI_ENTRY_PTR_ARRAY_EXPECT(fpga9001, index, indexLength);
    ADI_RANGE_CHECK(fpga9001, gpioPinLength, GPIO_PIN_MIN, GPIO_PIN_MAX);
    ADI_ENTRY_PTR_ARRAY_EXPECT(fpga9001, gpioPin, gpioPinLength);
    for (i = 0; i < gpioPinLength; i++)
    {
        ADI_RANGE_CHECK(fpga9001, gpioPin[i], ADI_FPGA9001_GPIO_00, ADI_FPGA9001_GPIO_11);
    }
    ADI_API_RETURN(fpga9001);
}

int32_t adi_fpga9001_gpio_HopIndex_GainIndex_Configure(adi_fpga9001_Device_t *fpga9001,
                                                       adi_fpga9001_DmaTrigger_e trigger,
                                                       uint32_t index[],
                                                       uint32_t indexLength,
                                                       adi_fpga9001_GpioPin_e gpioPin[],
                                                       uint32_t gpioPinLength)
{
    uint8_t i = 0;
    uint32_t regData = 0;
    uint32_t triggerMode = 0;

    ADI_PERFORM_VALIDATION(adi_fpga9001_gpio_HopIndex_GainIndex_Configure_Validate, fpga9001, trigger, index, indexLength, gpioPin, gpioPinLength);

    axi_adrv9001_gpio_dgpio_pio_buf_enable_set((void *)fpga9001, AXI_ADRV9001_ID, AXI_ADRV9001_GPIO_OFFSET, 0);
    axi_adrv9001_gpio_dgpio_pio_buf_enable_set((void *)fpga9001, AXI_ADRV9001_ID, AXI_ADRV9001_GPIO_OFFSET, 1);

    for (i = 0; i < indexLength; i++)
    {
        axi_adrv9001_gpio_dgpio_pio_buf_data_set((void *)fpga9001, AXI_ADRV9001_ID, AXI_ADRV9001_GPIO_OFFSET, index[i]);
    }

    axi_adrv9001_gpio_dgpio_pio_buf_mode_get((void *)fpga9001, AXI_ADRV9001_ID, AXI_ADRV9001_GPIO_OFFSET, &regData);
    for (i = 0; i < gpioPinLength; i++)
    {
        axi_adrv9001_gpio_pin_source_set((void *)fpga9001, AXI_ADRV9001_ID, gpioPin[i], AXI_ADRV9001_GPIO_MODE_SW_BUF);
        axi_adrv9001_gpio_pin_mode_set((void *)fpga9001, AXI_ADRV9001_ID, gpioPin[i], AXI_ADRV9001_GPIO_MODE_SW_BUF);
        axi_adrv9001_gpio_pin_direction_set((void *)fpga9001, AXI_ADRV9001_ID, AXI_ADRV9001_GPIO_IO_GROUP_DGPIO, gpioPin[i], 0x1);
        axi_adrv9001_gpio_pin_set((void *)fpga9001, AXI_ADRV9001_ID, AXI_ADRV9001_GPIO_IO_GROUP_DGPIO, gpioPin[i], 0);
        regData = regData | (0x1 << gpioPin[i]);
    }
    axi_adrv9001_gpio_dgpio_pio_buf_mode_set((void *)fpga9001, AXI_ADRV9001_ID, AXI_ADRV9001_GPIO_OFFSET, regData);

    axi_adrv9001_gpio_reg_dgpio_pio_buf_triggers_set(fpga9001, AXI_ADRV9001_ID, AXI_ADRV9001_GPIO_OFFSET, 0x0);
    if (trigger != ADI_FPGA9001_DMA_TRIGGER_IMMEDIATE) {
        axi_adrv9001_gpio_dgpio_pio_buf_trigger_mode_set(fpga9001, AXI_ADRV9001_ID, AXI_ADRV9001_GPIO_OFFSET,
            (uint32_t) trigger, AXI_ADRV9001_GPIO_BUF_TRIGGER_RISING_EDGE);
    }
    
    ADI_API_RETURN(fpga9001);
}
static __maybe_unused int32_t adi_fpga9001_gpio_pin_Configure_Validate(adi_fpga9001_Device_t *fpga9001,
                                                        adi_fpga9001_GpioPinCfg_t *gpioPinConfig)
{
    ADI_NULL_PTR_RETURN(&fpga9001->common, gpioPinConfig);

    ADI_RANGE_CHECK(fpga9001, gpioPinConfig->mode, ADI_FPGA9001_GPIO_MODE_SOFTWARE, ADI_FPGA9001_GPIO_MODE_HARDWARE);
    ADI_RANGE_CHECK(fpga9001, gpioPinConfig->direction, ADI_FPGA9001_GPIO_DIRECTION_INPUT, ADI_FPGA9001_GPIO_DIRECTION_OUTPUT);
    ADI_RANGE_CHECK(fpga9001, gpioPinConfig->dataOut, ADI_FPGA9001_GPIO_DATAOUT_CLEAR, ADI_FPGA9001_GPIO_DATAOUT_SET);
    ADI_RANGE_CHECK(fpga9001, gpioPinConfig->pin, ADI_FPGA9001_GPIO_00, ADI_FPGA9001_GPIO_11);
    ADI_RANGE_CHECK(fpga9001, gpioPinConfig->hwSourceMode, ADI_FPGA9001_GPIOHWSOURCEMODE_SELECT_TDD_TX1_ENABLE, ADI_FPGA9001_GPIOHWSOURCEMODE_SELECT_TDD_GENERAL_PURPOSE_6);
    ADI_API_RETURN(fpga9001);
}

int32_t adi_fpga9001_gpio_pin_Configure(adi_fpga9001_Device_t *fpga9001,
                                        adi_fpga9001_GpioPinCfg_t *gpioPinConfig)
{
    uint32_t gpio_pin = (uint32_t)gpioPinConfig->pin;
    
    ADI_PERFORM_VALIDATION(adi_fpga9001_gpio_pin_Configure_Validate, fpga9001, gpioPinConfig);

	axi_adrv9001_gpio_pin_mode_set((void *)fpga9001, AXI_ADRV9001_ID, gpio_pin, (uint32_t)gpioPinConfig->mode);
	axi_adrv9001_gpio_pin_direction_set((void *)fpga9001, AXI_ADRV9001_ID, AXI_ADRV9001_GPIO_IO_GROUP_DGPIO, gpio_pin, (uint32_t)gpioPinConfig->direction);
    axi_adrv9001_gpio_pin_source_set((void *)fpga9001, AXI_ADRV9001_ID, gpio_pin, (uint32_t)gpioPinConfig->hwSourceMode);
    axi_adrv9001_gpio_pin_set((void *)fpga9001, AXI_ADRV9001_ID, AXI_ADRV9001_GPIO_IO_GROUP_DGPIO, gpio_pin, (uint32_t)gpioPinConfig->dataOut);

    ADI_API_RETURN(fpga9001);
}

static __maybe_unused int32_t adi_fpga9001_gpio_pin_Inspect_Validate(adi_fpga9001_Device_t *fpga9001, 
                                                      adi_fpga9001_GpioPinCfg_t *gpioPinConfig)
{
    ADI_NULL_PTR_RETURN(&fpga9001->common, gpioPinConfig);
    ADI_RANGE_CHECK(fpga9001, gpioPinConfig->pin, ADI_FPGA9001_GPIO_00, ADI_FPGA9001_GPIO_11);
    ADI_API_RETURN(fpga9001);
}

int32_t adi_fpga9001_gpio_pin_Inspect(adi_fpga9001_Device_t *fpga9001, 
                                      adi_fpga9001_GpioPinCfg_t *gpioPinConfig)
{
    uint32_t gpio_pin = (uint32_t)gpioPinConfig->pin;
    uint32_t gpio_mode;
    uint32_t gpio_direction;
    uint32_t gpio_data;
    uint32_t gpio_source;

    ADI_PERFORM_VALIDATION(adi_fpga9001_gpio_pin_Inspect_Validate, fpga9001, gpioPinConfig);

    axi_adrv9001_gpio_pin_mode_get((void *)fpga9001, AXI_ADRV9001_ID, gpio_pin, &gpio_mode);
    axi_adrv9001_gpio_pin_direction_get((void *)fpga9001, AXI_ADRV9001_ID, AXI_ADRV9001_GPIO_IO_GROUP_DGPIO, gpio_pin, &gpio_direction);
    axi_adrv9001_gpio_pin_data((void *)fpga9001, AXI_ADRV9001_ID, AXI_ADRV9001_GPIO_IO_GROUP_DGPIO, gpio_pin, &gpio_data);
    axi_adrv9001_gpio_pin_source_get((void *)fpga9001, AXI_ADRV9001_ID, gpio_pin, &gpio_source);

    gpioPinConfig->mode = (adi_fpga9001_GpioMode_e)gpio_mode;
    gpioPinConfig->direction = (adi_fpga9001_GpioDirection_e)gpio_direction;
    gpioPinConfig->dataOut = (adi_fpga9001_GpioDataOut_e)gpio_data;
    gpioPinConfig->hwSourceMode = (adi_fpga9001_GpioHWSourceModes_e)gpio_source;

    ADI_API_RETURN(fpga9001);
}

static __maybe_unused uint8_t adi_fpga9001_gpio_pin_SourceMode_Configure_Validate(adi_fpga9001_Device_t *fpga9001,
                                                                   adi_fpga9001_GpioPinCfg_t gpioPinConfig[],
                                                                   uint32_t length)
{
    uint8_t i = 0;

    ADI_NULL_PTR_RETURN(&fpga9001->common, gpioPinConfig);

    ADI_RANGE_CHECK(fpga9001, length, 1, ADI_FPGA9001_NUM_GPIO_PINS);

    for (i = 0; i < length; i++)
    {
        ADI_EXPECT(adi_fpga9001_gpio_pin_Configure_Validate, fpga9001, (gpioPinConfig + i));
    }

    ADI_API_RETURN(fpga9001);
}

int32_t adi_fpga9001_gpio_pin_SourceMode_Configure(adi_fpga9001_Device_t *fpga9001, adi_fpga9001_GpioPinCfg_t gpioPinConfig[], uint32_t length)
{
    uint8_t i = 0;

    ADI_PERFORM_VALIDATION(adi_fpga9001_gpio_pin_SourceMode_Configure_Validate, fpga9001, gpioPinConfig, length);

    for (i = 0; i < length; i++)
    {
        ADI_EXPECT(adi_fpga9001_gpio_pin_Configure, fpga9001, (gpioPinConfig + i));
    }

    ADI_API_RETURN(fpga9001);
}

static __maybe_unused int32_t adi_fpga9001_gpio_pin_SourceMode_Inspect_Validate(adi_fpga9001_Device_t *fpga9001, adi_fpga9001_GpioPinCfg_t *gpioPinConfig)
{
    ADI_NULL_PTR_RETURN(&fpga9001->common, gpioPinConfig);
    ADI_API_RETURN(fpga9001);
}

int32_t adi_fpga9001_gpio_pin_SourceMode_Inspect(adi_fpga9001_Device_t *fpga9001,
                                                 adi_fpga9001_GpioPinCfg_t *gpioPinConfig)
{
    uint8_t i = 0;

    ADI_PERFORM_VALIDATION(adi_fpga9001_gpio_pin_SourceMode_Inspect_Validate, fpga9001, gpioPinConfig);

    for (i = 0; i < ADI_FPGA9001_NUM_GPIO_PINS; i++)
    {
        ADI_EXPECT(adi_fpga9001_gpio_pin_Inspect, fpga9001, (gpioPinConfig + i));
    }

    ADI_API_RETURN(fpga9001);
}

static __maybe_unused int32_t adi_fpga9001_gpio_GainIndex_Configure_Validate(adi_fpga9001_Device_t *fpga9001,
                                                              adi_fpga9001_GpioGainIndexCfg_t *gainIndexConfig)
{
    uint8_t i = 0;

    ADI_NULL_DEVICE_PTR_RETURN(fpga9001);
    ADI_NULL_PTR_RETURN(&fpga9001->common, gainIndexConfig);
    ADI_RANGE_CHECK(fpga9001, gainIndexConfig->group, ADI_FPGA9001_GPIO_GROUP_DGPIO, ADI_FPGA9001_GPIO_GROUP_TX1);

    for (i = 0; i < 8; i++)
    {
        if (ADI_FPGA9001_GPIO_GROUP_DGPIO == gainIndexConfig->group)
        {
            ADI_RANGE_CHECK(fpga9001, gainIndexConfig->readGainIndex[i], ADI_FPGA9001_GPIO_00, ADI_FPGA9001_GPIO_11);
        }
        else if ((ADI_FPGA9001_GPIO_GROUP_RX0 == gainIndexConfig->group) || (ADI_FPGA9001_GPIO_GROUP_RX1 == gainIndexConfig->group))
        {
            ADI_RANGE_CHECK(fpga9001, gainIndexConfig->readGainIndex[i], ADI_FPGA9001_GPIO_00, ADI_FPGA9001_GPIO_07);
        }
        else
        {
            ADI_RANGE_CHECK(fpga9001, gainIndexConfig->readGainIndex[i], ADI_FPGA9001_GPIO_00, ADI_FPGA9001_GPIO_09);
        }
    }

    ADI_API_RETURN(fpga9001);
}

int32_t adi_fpga9001_gpio_GainIndex_Configure(adi_fpga9001_Device_t *fpga9001,
                                              adi_fpga9001_GpioGainIndexCfg_t *gainIndexConfig)
{
    uint32_t pins[8];
    int32_t i;

    ADI_PERFORM_VALIDATION(adi_fpga9001_gpio_GainIndex_Configure_Validate, fpga9001, gainIndexConfig);

    for (i = 0; i < 8; i++)
    {
        pins[i] = (uint32_t)gainIndexConfig->readGainIndex[i];
    }

    axi_adrv9001_gpio_gainindex_sel((void *)fpga9001, AXI_ADRV9001_ID, (uint32_t) gainIndexConfig->group, &pins[0], 8);
    axi_adrv9001_gpio_trig_sel((void *)fpga9001, AXI_ADRV9001_ID, (uint32_t) gainIndexConfig->group, pins[7]);
    ADI_API_RETURN(fpga9001);
}

static __maybe_unused int32_t adi_fpga9001_gpio_GainIndex_Get_Validate(adi_fpga9001_Device_t *fpga9001,
                                                        uint32_t *gainIndex)
{
    ADI_NULL_DEVICE_PTR_RETURN(fpga9001);
    ADI_NULL_PTR_RETURN(&fpga9001->common, gainIndex);
    ADI_API_RETURN(fpga9001);
}

int32_t adi_fpga9001_gpio_GainIndex_Get(adi_fpga9001_Device_t *fpga9001,
                                        uint32_t *gainIndex)
{
    uint32_t gpio_data;
    uint32_t value = 0;

    ADI_PERFORM_VALIDATION(adi_fpga9001_gpio_GainIndex_Get_Validate, fpga9001, gainIndex);

    gpio_data = 0;
    axi_adrv9001_gpio_group_data((void *)fpga9001, AXI_ADRV9001_ID, AXI_ADRV9001_GPIO_IO_GROUP_TXSSI1, &value);
    gpio_data = (gpio_data << 8) | value;
    axi_adrv9001_gpio_group_data((void *)fpga9001, AXI_ADRV9001_ID, AXI_ADRV9001_GPIO_IO_GROUP_TXSSI0, &value);
    gpio_data = (gpio_data << 8) | value;
    axi_adrv9001_gpio_group_data((void *)fpga9001, AXI_ADRV9001_ID, AXI_ADRV9001_GPIO_IO_GROUP_RXSSI1, &value);
    gpio_data = (gpio_data << 8) | value;
    axi_adrv9001_gpio_group_data((void *)fpga9001, AXI_ADRV9001_ID, AXI_ADRV9001_GPIO_IO_GROUP_RXSSI0, &value);
    gpio_data = (gpio_data << 8) | value;

    *gainIndex = gpio_data;
    ADI_API_RETURN(fpga9001);
}

static __maybe_unused int32_t adi_fpga9001_gpio_TriggerSource_Configure_Validate(adi_fpga9001_Device_t *fpga9001,
                                                                  adi_fpga9001_GpioTriggerCfg_t *triggerCfg)
{
    ADI_NULL_DEVICE_PTR_RETURN(fpga9001);
    ADI_NULL_PTR_RETURN(&fpga9001->common, triggerCfg);
    ADI_RANGE_CHECK(fpga9001, triggerCfg->group, ADI_FPGA9001_GPIO_GROUP_DGPIO, ADI_FPGA9001_GPIO_GROUP_TX1);

    if (ADI_FPGA9001_GPIO_GROUP_DGPIO == triggerCfg->group)
    {
        ADI_RANGE_CHECK(fpga9001, triggerCfg->triggerSource, ADI_FPGA9001_GPIO_00, ADI_FPGA9001_GPIO_11);
    }
    else if ((ADI_FPGA9001_GPIO_GROUP_RX0 == triggerCfg->group) || (ADI_FPGA9001_GPIO_GROUP_RX1 == triggerCfg->group))
    {
        ADI_RANGE_CHECK(fpga9001, triggerCfg->triggerSource, ADI_FPGA9001_GPIO_00, ADI_FPGA9001_GPIO_07);
    }
    else
    {
        ADI_RANGE_CHECK(fpga9001, triggerCfg->triggerSource, ADI_FPGA9001_GPIO_00, ADI_FPGA9001_GPIO_09);
    }

    ADI_API_RETURN(fpga9001);
}

int32_t adi_fpga9001_gpio_TriggerSource_Configure(adi_fpga9001_Device_t *fpga9001,
                                                  adi_fpga9001_GpioTriggerCfg_t *triggerCfg)
{

    ADI_PERFORM_VALIDATION(adi_fpga9001_gpio_TriggerSource_Configure_Validate, fpga9001, triggerCfg);

    axi_adrv9001_gpio_trig_sel((void *)fpga9001, AXI_ADRV9001_ID, (uint32_t) triggerCfg->group, (uint32_t)(triggerCfg->triggerSource));
    ADI_API_RETURN(fpga9001);
}

static __maybe_unused int32_t adi_fpga9001_gpio_smapin_Validate(adi_fpga9001_Device_t *fpga9001,
                                                 adi_fpga9001_GpioSmaCfg_t *smaConfig)
{
    ADI_NULL_PTR_RETURN(&fpga9001->common, smaConfig);
    ADI_API_RETURN(fpga9001);
}

int32_t adi_fpga9001_gpio_smapin_Configure(adi_fpga9001_Device_t *fpga9001,
                                           adi_fpga9001_GpioSmaCfg_t *smaConfig)
{
    ADI_PERFORM_VALIDATION(adi_fpga9001_gpio_smapin_Validate, fpga9001, smaConfig);

    axi_adrv9001_top_trig0_ext_oe_set(fpga9001, AXI_ADRV9001_ID, AXI_ADRV9001_TOP_OFFSET, (uint32_t) smaConfig->smaPin1);
    axi_adrv9001_top_trig1_ext_oe_set(fpga9001, AXI_ADRV9001_ID, AXI_ADRV9001_TOP_OFFSET, (uint32_t) smaConfig->smaPin2);

    ADI_API_RETURN(fpga9001);
}

int32_t adi_fpga9001_gpio_smapin_Inspect(adi_fpga9001_Device_t *fpga9001,
                                         adi_fpga9001_GpioSmaCfg_t *smaConfig)
{
    uint32_t pinConfig = 0;
    ADI_PERFORM_VALIDATION(adi_fpga9001_gpio_smapin_Validate, fpga9001, smaConfig);

    axi_adrv9001_top_trig0_ext_oe_get(fpga9001, AXI_ADRV9001_ID, AXI_ADRV9001_TOP_OFFSET, &pinConfig);
    smaConfig->smaPin1  = (pinConfig == 1) ? true : false;
    axi_adrv9001_top_trig1_ext_oe_get(fpga9001, AXI_ADRV9001_ID, AXI_ADRV9001_TOP_OFFSET, &pinConfig);
    smaConfig->smaPin2  = (pinConfig == 1) ? true : false;

    ADI_API_RETURN(fpga9001);
}
