/** 
* \file 
* \brief Contains Frequency Hopping functions implementation 
* 
* ADRV9001 API Version: $ADI_FPGA9001_API_VERSION$ 
*/ 
 
/** 
* Copyright 2020 Analog Devices Inc. 
* Released under the FPGA9001 API license, for more information 
* see the "LICENSE.txt" file in this zip file. 
*/ 
 
#include "adi_fpga9001_fh.h" 
#include "adi_fpga9001.h" 
#include "fpga9001_utilities.h" 
#include "adi_fpga9001_hal_wrapper.h" 
 
#include "adi_adrv9001_spi.h" 
#include "adrv9001_arm_macros.h" 
#include "adrv9001_init.h" 
#include "adrv9001_reg_addr_macros.h" 
 
static __maybe_unused int32_t adi_fpga9001_fh_Generate_SpiDma_Trigger_Validate(adi_fpga9001_Device_t *fpga9001,
                                                                               adi_fpga9001_TddConfig_t *tddConfig,
                                                                               adi_fpga9001_FhPerDynamicLoad_e numberHopsPerDynamicLoad,
                                                                               uint32_t bytesPerFrame,
                                                                               adi_fpga9001_TddEnableTiming_t *spiTriggerConfig,
                                                                               uint32_t *numberFrames)
{
    ADI_NULL_PTR_RETURN(&fpga9001->common, tddConfig);
    ADI_NULL_PTR_RETURN(&fpga9001->common, spiTriggerConfig);
    ADI_NULL_PTR_RETURN(&fpga9001->common, numberFrames);
    ADI_RANGE_CHECK(fpga9001, numberHopsPerDynamicLoad, ADI_FPGA9001_FH_HOP_PER_DYNAMIC_LOAD_ONE, ADI_FPGA9001_FH_HOP_PER_DYNAMIC_LOAD_EIGHT);
    if (bytesPerFrame <= 0)
    {
        ADI_ERROR_REPORT(&fpga9001->common,
            ADI_COMMON_ERRSRC_API,
            ADI_COMMON_ERR_API_FAIL,
            ADI_COMMON_ACT_ERR_CHECK_PARAM,
            bytesPerTable,
            "bytesPerFrame must be a positive integer.");
        ADI_ERROR_RETURN(fpga9001->common.error.newAction);
    }
    ADI_API_RETURN(fpga9001);
}

int32_t adi_fpga9001_fh_Generate_SpiDma_Trigger(adi_fpga9001_Device_t *fpga9001,
                                                adi_fpga9001_TddConfig_t *tddConfig,
                                                adi_fpga9001_FhPerDynamicLoad_e numberHopsPerDynamicLoad,
                                                uint32_t bytesPerFrame,
                                                adi_fpga9001_TddEnableTiming_t *spiTriggerConfig,
                                                uint32_t *numberFrames)
{
    //uint32_t spiLatency = 0;
    //static uint32_t TDD_SPI_CLOCK_RATIO = 16; /* 200 MHz / 12.5 MHz = 16 */
    //static uint32_t ADDITIONAL_DELAY = 3000; /* TODO: Can be removed later; For testing and tuning */

    ADI_PERFORM_VALIDATION(adi_fpga9001_fh_Generate_SpiDma_Trigger_Validate, fpga9001, tddConfig, numberHopsPerDynamicLoad, bytesPerFrame, spiTriggerConfig, numberFrames);

    //spiLatency = (TDD_SPI_CLOCK_RATIO * 8) * bytesPerFrame;
    //spiLatency += ADDITIONAL_DELAY;
    
    spiTriggerConfig->enable = true;
    spiTriggerConfig->inMicroSeconds = false;
    spiTriggerConfig->primaryAssert = tddConfig->enables[ADI_FPGA9001_TDDSELECT_GENERAL_PURPOSE_6].primaryAssert;// - spiLatency;
    spiTriggerConfig->primaryDeassert = spiTriggerConfig->primaryAssert + 200;
    spiTriggerConfig->secondaryAssert = 0;
    spiTriggerConfig->secondaryDeassert = 0;

    switch (numberHopsPerDynamicLoad)
    {
    case ADI_FPGA9001_FH_HOP_PER_DYNAMIC_LOAD_ONE:
        *numberFrames = 0;
        spiTriggerConfig->secondaryAssert = tddConfig->enables[ADI_FPGA9001_TDDSELECT_GENERAL_PURPOSE_6].primaryDeassert;// - spiLatency;
        spiTriggerConfig->secondaryDeassert = spiTriggerConfig->secondaryAssert + 200;
        spiTriggerConfig->primaryFrameAssert = 0;
        spiTriggerConfig->primaryFrameDeassert = 0;
        spiTriggerConfig->secondaryFrameAssert = 0;
        spiTriggerConfig->secondaryFrameDeassert = 0;
        break;
    case ADI_FPGA9001_FH_HOP_PER_DYNAMIC_LOAD_FOUR:
        *numberFrames = 3;
	    spiTriggerConfig->primaryFrameAssert = 0;
	    spiTriggerConfig->primaryFrameDeassert = 0;
	    spiTriggerConfig->secondaryFrameAssert = 2;
	    spiTriggerConfig->secondaryFrameDeassert = 2;
        break;
    case ADI_FPGA9001_FH_HOP_PER_DYNAMIC_LOAD_EIGHT:
        *numberFrames = 7;
	    spiTriggerConfig->primaryFrameAssert = 0;
	    spiTriggerConfig->primaryFrameDeassert = 0;
	    spiTriggerConfig->secondaryFrameAssert = 4;
	    spiTriggerConfig->secondaryFrameDeassert = 4;
        break;
    }
    
    ADI_API_RETURN(fpga9001);
}
