/**
 * \file
 *
 * \brief Functions to configure the FPGA9001 Data Chains
 *
 * FPGA9001 API Version: $ADI_FPGA9001_API_VERSION$
 */

/**
 * Copyright 2015-2025 Analog Devices Inc.
 * Released under the FPGA9001 API license, for more information
 * see the "LICENSE.txt" file in this zip file.
 */

#include <stdlib.h>

#include "adi_adrv9001_user.h"
#include "adi_fpga9001_datachain.h"
#include "adi_fpga9001_error.h"
#include "adi_fpga9001_hal_wrapper.h"
#include "adi_fpga9001_ssi.h"
#include "fpga9001_utilities.h"
#include "axi_mspi.h" 
#include "axi_dma.h" 
#include "axi_adrv9001.h" 
#include "axi_common.h"
int32_t adi_fpga9001_DataChain_Dma_Inspect(adi_fpga9001_Device_t *fpga9001,
                                           adi_common_Port_e port,
                                           adi_common_ChannelNumber_e channel,
                                           adi_fpga9001_DmaCfg_t *dmaCfg)
{
    uint32_t dmaId = 0;
    uint32_t value = 0;

    ADI_NULL_DEVICE_PTR_RETURN(fpga9001);
    ADI_RANGE_CHECK(fpga9001, port, ADI_RX, ADI_ELB);
    ADI_RANGE_CHECK(fpga9001, channel, ADI_CHANNEL_1, ADI_CHANNEL_2);
    ADI_NULL_PTR_RETURN(&fpga9001->common, dmaCfg);

    dmaId = fpga9001_DMAIdGet(fpga9001, port, channel);
    axi_dma_top_length_get(fpga9001, dmaId, AXI_DMA_TOP_OFFSET, &dmaCfg->length);
    axi_dma_top_cyclic_get(fpga9001, dmaId, AXI_DMA_TOP_OFFSET, &value);
    dmaCfg->continuous = (value == 1) ? true : false;
    axi_dma_top_reg_flags_get(fpga9001, dmaId, AXI_DMA_TOP_OFFSET, &value);
    dmaCfg->dmaEnableSync = (value == 0x3) ? true : false;
    ADI_API_RETURN(fpga9001);
}

int32_t adi_fpga9001_DataChain_DmaStatus_Get(adi_fpga9001_Device_t *fpga9001,
                                             adi_common_Port_e port,
                                             adi_common_ChannelNumber_e channel,
                                             adi_fpga9001_DmaStatus_t *status)
{
    uint32_t dmaId;
    uint32_t value = 0;

    ADI_NULL_DEVICE_PTR_RETURN(fpga9001);
    ADI_RANGE_CHECK(fpga9001, port, ADI_RX, ADI_ELB);
    ADI_RANGE_CHECK(fpga9001, channel, ADI_CHANNEL_1, ADI_CHANNEL_2);
    ADI_NULL_PTR_RETURN(&fpga9001->common, status);

    dmaId = fpga9001_DMAIdGet(fpga9001, port, channel);
    axi_dma_top_ovf_unf_get((void *)fpga9001, dmaId, AXI_DMA_TOP_OFFSET, &value);
    status->overUnderflow = (value == 1) ? true : false;
    axi_dma_top_busy_get((void *)fpga9001, dmaId, AXI_DMA_TOP_OFFSET, &value);
    status->active = (value == 1) ? true : false;
    status->complete = ~status->active;
    status->haltComplete = ~status->active;
    ADI_API_RETURN(fpga9001);
}

static __maybe_unused int32_t adi_fpga9001_DataChain_PerformTransfer_Validate(adi_fpga9001_Device_t *fpga9001,
                                                                              adi_common_Port_e port,
                                                                              adi_common_ChannelNumber_e channel,
                                                                              adi_fpga9001_DmaCfg_t *dmaCfg)
{
    ADI_NULL_DEVICE_PTR_RETURN(fpga9001);
    ADI_RANGE_CHECK(fpga9001, port, ADI_RX, ADI_ELB);
    ADI_RANGE_CHECK(fpga9001, channel, ADI_CHANNEL_1, ADI_CHANNEL_2);
    ADI_RANGE_CHECK(fpga9001, dmaCfg->trigger, ADI_FPGA9001_DMA_TRIGGER_SMA_1, ADI_FPGA9001_DMA_TRIGGER_IMMEDIATE);
    ADI_RANGE_CHECK(fpga9001, dmaCfg->captureType, ADI_FPGA9001_CAPTURE_TYPE_DISABLE, ADI_FPGA9001_CAPTURE_TYPE_RX_GAIN);
    if ((dmaCfg->captureType == ADI_FPGA9001_CAPTURE_TYPE_RX_GAIN) && (ADI_RX != port))
    {
        ADI_ERROR_REPORT(&fpga9001->common,
            ADI_COMMON_ERRSRC_API,
            ADI_COMMON_ERR_INV_PARAM,
            ADI_COMMON_ACT_ERR_CHECK_PARAM,
            port,
            "Invalid port selected. Gain index capture can be performed only for Rx channels.");

    }

    ADI_API_RETURN(fpga9001);
}

static __maybe_unused int32_t adi_fpga9001_DataChain_ConfigureTransfer(adi_fpga9001_Device_t *fpga9001,
                                                        adi_common_Port_e port,
                                                        adi_common_ChannelNumber_e channel,
                                                        uint32_t dmaId,
                                                        adi_fpga9001_DmaCfg_t *dmaCfg)
{
    uint32_t addr;
    axi_dma_params_t dmaParams = { 0 };

	if (dmaCfg->trigger == ADI_FPGA9001_DMA_TRIGGER_SYNC)
	{
		dmaParams.triggers[(int) dmaCfg->trigger] = AXI_DMA_TOP_TRIGGER_EDGE;
	}
	else if (dmaCfg->trigger != ADI_FPGA9001_DMA_TRIGGER_IMMEDIATE)
	{
		dmaParams.triggers[(int) dmaCfg->trigger] = AXI_DMA_TOP_TRIGGER_RISING_EDGE;
	}
	else {
		dmaParams.triggers[(int) dmaCfg->trigger] = 0;
	}

    if (ADI_FPGA9001_CAPTURE_TYPE_RX_GAIN == dmaCfg->captureType)
    {
        if (ADI_CHANNEL_1 == channel)
        {
            ADI_EXPECT(adi_fpga9001_hal_wrapper_DmaRelativeAddress_Get, 
                       fpga9001,
                       ADI_FPGA9001_HAL_DMA_RX2,
                       &addr);
        }
        else
        {
            ADI_EXPECT(adi_fpga9001_hal_wrapper_DmaRelativeAddress_Get, 
                       fpga9001,
                       ADI_FPGA9001_HAL_DMA_RX1,
                       &addr);
        }
    }
    else
    {
        ADI_EXPECT(adi_fpga9001_hal_wrapper_DmaRelativeAddress_Get, 
                   fpga9001,
                   fpga9001_Dma_Get(fpga9001, port, channel),
                   &addr);
    }

    if (ADI_TX == port)
    {
        axi_dma_top_length_get((void *)fpga9001, dmaId, AXI_DMA_TOP_OFFSET, &dmaParams.length);
    }
    else
    {
        dmaParams.length = dmaCfg->length;
    }
    /* '(port == ADI_TX) ? 1 : 0;' moved to adi_fpga9001_DataChain_PerformSynchronousTransfer and 
     * adi_fpga9001_DataChain_PerformTransfer.
     */
    dmaParams.cyclic = (uint8_t)dmaCfg->continuous;
    dmaParams.qthreshold = 0x2;
    dmaParams.start_address = (uint64_t) addr;

    dmaParams.cyclic_sync = (dmaCfg->dmaEnableSync) ? 1 : 0;
    dmaParams.start_sync = (dmaCfg->dmaEnableSync) ? 1 : 0;

    ADI_EXPECT(axi_dma_config, fpga9001, dmaId, AXI_DMA_TOP_OFFSET, &dmaParams);
    ADI_EXPECT(axi_dma_begin, fpga9001, dmaId, AXI_DMA_TOP_OFFSET);

    ADI_API_RETURN(fpga9001);
}

/* most likely there will be a need for a dmaId version of this function in the future */
/* lazy approach here, ideally you want enumerated ids pushed upstream */

int32_t adi_fpga9001_DataChain_Dma_Configure(adi_fpga9001_Device_t *fpga9001,
                                             adi_common_Port_e port,
                                             adi_common_ChannelNumber_e channel,
                                             adi_fpga9001_DmaCfg_t *dmaCfg)
{
    uint32_t dmaId;

    dmaId = fpga9001_DMAIdGet(fpga9001, port, channel);
    return(adi_fpga9001_DataChain_ConfigureTransfer(fpga9001, port, channel, dmaId, dmaCfg));
}

static __maybe_unused int32_t adi_fpga9001_DataChain_CaptureWait(adi_fpga9001_Device_t *fpga9001,
                                                  adi_common_Port_e port,
                                                  adi_common_ChannelNumber_e channel,
                                                  uint32_t dmaId,
                                                  adi_fpga9001_DmaCfg_t *dmaCfg)
{
    uint32_t timeout_us = 0;

    timeout_us = KILO_TO_BASE_UNIT(dmaCfg->timeout_ms);
    ADI_EXPECT(axi_dma_finish, fpga9001, dmaId, AXI_DMA_TOP_OFFSET, timeout_us);
    ADI_API_RETURN(fpga9001); 
}  

int32_t adi_fpga9001_DataChain_Dma_Master_Set(adi_fpga9001_Device_t *fpga9001,
                                             adi_common_Port_e port,
                                             bool enableMasterToEnable,
                                             bool enableMasterToTrigger)
{
    uint32_t ssiId0 = fpga9001_SsiIdGet(fpga9001, port, ADI_CHANNEL_1);
    uint32_t ssiId1 = fpga9001_SsiIdGet(fpga9001, port, ADI_CHANNEL_2);
    uint32_t dmaId0 = fpga9001_DMAIdGet(fpga9001, port, ADI_CHANNEL_1);
    uint32_t dmaId1 = fpga9001_DMAIdGet(fpga9001, port, ADI_CHANNEL_2);
    uint32_t dataEnable = (enableMasterToEnable == true) ? 0x1: 
                                                0x0;
    uint32_t dataTrigger = (enableMasterToTrigger == true) ? 0x1: 
                                                0x0;
    /* Enablenedness of Channel1 DMA_ENABLE as master for Channel2 */
    axi_adrv9001_ssi_master_enable_set(fpga9001, AXI_ADRV9001_ID, ssiId0, dataEnable);
    axi_adrv9001_ssi_master_enable_set(fpga9001, AXI_ADRV9001_ID, ssiId1, dataEnable);

    /* Enablenedness of Channel1 DMA_TRIGGER as master for Channel2 */
    axi_dma_top_master_trigger_set(fpga9001, dmaId0, AXI_DMA_TOP_OFFSET, dataTrigger);
    axi_dma_top_master_trigger_set(fpga9001, dmaId1, AXI_DMA_TOP_OFFSET, dataTrigger);

    ADI_API_RETURN(fpga9001);
}

int32_t adi_fpga9001_DataChain_Dma_Master_Get(adi_fpga9001_Device_t *fpga9001,
                                             adi_common_Port_e port,
                                             bool *enableMasterToEnable,
                                             bool *enableMasterToTrigger)
{
    uint32_t ssiId0 = fpga9001_SsiIdGet(fpga9001, port, ADI_CHANNEL_1);
    uint32_t ssiId1 = fpga9001_SsiIdGet(fpga9001, port, ADI_CHANNEL_2);
    uint32_t dmaId0 = fpga9001_DMAIdGet(fpga9001, port, ADI_CHANNEL_1);
    uint32_t dmaId1 = fpga9001_DMAIdGet(fpga9001, port, ADI_CHANNEL_2);
    uint32_t dataEnable = 0;
    uint32_t dataTrigger = 0;
    /* Enablenedness of Channel1 DMA_ENABLE as master for Channel2 */
    axi_adrv9001_ssi_master_enable_get(fpga9001, AXI_ADRV9001_ID, ssiId0, &dataEnable);
    if (dataEnable == 1)
    {
        axi_adrv9001_ssi_master_enable_get(fpga9001, AXI_ADRV9001_ID, ssiId1, &dataEnable);
    }

    /* Enablenedness of Channel1 DMA_TRIGGER as master for Channel2 */
    axi_dma_top_master_trigger_get(fpga9001, dmaId0, AXI_DMA_TOP_OFFSET, &dataTrigger);
    if (dataTrigger == 1)
    {
        axi_dma_top_master_trigger_get(fpga9001, dmaId1, AXI_DMA_TOP_OFFSET, &dataTrigger);
    }
        
    *enableMasterToEnable = (dataEnable == 0) ? false :
                                                    true;
    *enableMasterToTrigger = (dataTrigger == 0) ? false :
                                                    true;
        
    ADI_API_RETURN(fpga9001);
}

/* same as the dma-configure above, but waits, no dma-id version. */

int32_t adi_fpga9001_DataChain_Dma_Wait(adi_fpga9001_Device_t *fpga9001,
                                        adi_common_Port_e port,
                                        adi_common_ChannelNumber_e channel,
                                        adi_fpga9001_DmaCfg_t *dmaCfg)
{
    uint32_t dmaId;

    dmaId = fpga9001_DMAIdGet(fpga9001, port, channel);
    return(adi_fpga9001_DataChain_CaptureWait(fpga9001, port, channel, dmaId, dmaCfg));
}

static __maybe_unused int32_t adi_fpga9001_DataChain_Sync(adi_fpga9001_Device_t *fpga9001)
{

    /* Issue an MCS pulse to align two DMA channels */

    axi_adrv9001_top_mcs_trigger_set(fpga9001, AXI_ADRV9001_ID, AXI_ADRV9001_TOP_OFFSET, 0x1);
    axi_adrv9001_top_mcs_cdc_state_sync(fpga9001, AXI_ADRV9001_ID, AXI_ADRV9001_TOP_OFFSET);

    ADI_API_RETURN(fpga9001);
}
 
/* A synchronous transfer won't prevent the selection of any of the four trigger sources.
 * Unlike adi_fpga9001_DataChain_PerformTransfer, if the trigger is set to immediate
 * then FMC.SMA, rising edge, will be selected as the trigger source.
 */
int32_t adi_fpga9001_DataChain_PerformSynchronousTransfer(adi_fpga9001_Device_t *fpga9001,
                                                          adi_common_Port_e port,
                                                          adi_fpga9001_DmaCfg_t *dmaCfg)
{
    uint32_t dmaIdChannel1;
    uint32_t dmaIdChannel2;
    /* ADI_CHANNEL_1 is used for validation function. For synchronousTransfer we just want to validate port
     * and dmaCfg, since they are provided by the user. No need to verify the channel number.
     */
    ADI_PERFORM_VALIDATION(adi_fpga9001_DataChain_PerformTransfer_Validate, fpga9001, port, ADI_CHANNEL_1, dmaCfg);
    /* Verify that the capture type is not ADI_FPGA9001_CAPTURE_TYPE_RX_GAIN */
    if (ADI_FPGA9001_CAPTURE_TYPE_RX_GAIN == dmaCfg->captureType)
    {
        ADI_ERROR_REPORT(&fpga9001->common,
            ADI_COMMON_ERRSRC_API,
            ADI_COMMON_ERR_API_FAIL,
            ADI_COMMON_ACT_ERR_CHECK_PARAM,
            dmaCfg->captureType,
            "Invalid capture type for synchronous transfer");
        ADI_ERROR_RETURN(fpga9001->common.error.newAction);
    }
    dmaIdChannel1 = fpga9001_DMAIdGet(fpga9001, port, ADI_CHANNEL_1);
    dmaIdChannel2 = fpga9001_DMAIdGet(fpga9001, port, ADI_CHANNEL_2);

    /* Set trigger to SYNC (MCS signal as source) if the user has selected IMMEDIATE */
    if (ADI_FPGA9001_DMA_TRIGGER_IMMEDIATE == dmaCfg->trigger)
    {
        dmaCfg->trigger = ADI_FPGA9001_DMA_TRIGGER_SYNC;
    }

    /* If the user has set the trigger mode to immediate, then */                                                     
    /* Setup DMA for channel 1 and 2 for specified port */
    ADI_EXPECT(adi_fpga9001_DataChain_ConfigureTransfer, fpga9001, port, ADI_CHANNEL_1, dmaIdChannel1, dmaCfg);
    ADI_EXPECT(adi_fpga9001_DataChain_ConfigureTransfer, fpga9001, port, ADI_CHANNEL_2, dmaIdChannel2, dmaCfg);

    /* Synchronize capture with MCS pulse */
    ADI_EXPECT(adi_fpga9001_DataChain_Sync, fpga9001);

    /* Poll status of DMA if the transfer type is not cyclic.
       Potentially only need to call captureWait once, however both are called in case there is a
       time out error with one of the DMAs.
     */
    if (false == dmaCfg->continuous)
    {
        ADI_EXPECT(adi_fpga9001_DataChain_CaptureWait, fpga9001, port, ADI_CHANNEL_1, dmaIdChannel1, dmaCfg);
        ADI_EXPECT(adi_fpga9001_DataChain_CaptureWait, fpga9001, port, ADI_CHANNEL_2, dmaIdChannel2, dmaCfg);
    }

    ADI_API_RETURN(fpga9001);
}

/* trigger select here is re-interpreted as follows. */
/* hardware supports four trigger sources. */
/* 1. CARRIER.SMA.0 (In ZC706, this is J67). */
/* 2. CARRIER.SMA.1 (In ZC706, this is J68). */
/* 3. FMC.SMA (In ADRV9001, this is MCS (this can also be sourced by fpga9001)). */
/* 4. fpga9001.GPIO (In ADRV9001, this is any one of the digital GPIO.) */
/* Each source can be either disabled (if all are disabled, it is immediate) */
/* Individually rising edge OR falling edge (combination is allowed). */
/* Immediate Trigger                : all trigger sources are disabled. */
/* ADI_FPGA9001_TDD_TRIGGER_SMA_1   : CARRIER.SMA.0 is selected, rising edge. */
/* ADI_FPGA9001_TDD_TRIGGER_SMA_2   : CARRIER.SMA.1 is selected, rising edge. */
/* ADI_FPGA9001_TDD_TRIGGER_MCS     : FMC.SMA is selected, rising edge */
/* ADI_FPGA9001_TDD_TRIGGER_GPIO    : fpga9001.GPIO is selected, rising edge. */
/* Also, setting a trigger source while DMA is in progress will result in UNKNOWN  behavior. */
/* It is certain that data integrity is lost, as data is disabled until trigger occurs. */
int32_t adi_fpga9001_DataChain_PerformTransfer(adi_fpga9001_Device_t *fpga9001,
                                               adi_common_Port_e port,
                                               adi_common_ChannelNumber_e channel,
                                               adi_fpga9001_DmaCfg_t *dmaCfg)
{
    uint32_t dmaId = 0;
    ADI_PERFORM_VALIDATION(adi_fpga9001_DataChain_PerformTransfer_Validate, fpga9001, port, channel, dmaCfg);
    /* Determine DMA Id */
    dmaId = (ADI_FPGA9001_CAPTURE_TYPE_RX_GAIN == dmaCfg->captureType) ? AXI_DMA_GAIN_ID :
                                                                         fpga9001_DMAIdGet(fpga9001, port, channel);
    /* Configure DMA */
    ADI_EXPECT(adi_fpga9001_DataChain_ConfigureTransfer, fpga9001, port, channel, dmaId, dmaCfg);
    /* Poll status of DMA if the transfer type is not cyclic */
    if (false == dmaCfg->continuous)
    {
        ADI_EXPECT(adi_fpga9001_DataChain_CaptureWait, fpga9001, port, channel, dmaId, dmaCfg);
    }

    ADI_API_RETURN(fpga9001);
}

int32_t adi_fpga9001_DataChain_PerformTransfer_Start(adi_fpga9001_Device_t *fpga9001,
                                                     adi_common_Port_e port,
                                                     adi_common_ChannelNumber_e channel,
                                                     adi_fpga9001_DmaCfg_t *dmaCfg)
{
    uint32_t dmaId = 0;
    ADI_PERFORM_VALIDATION(adi_fpga9001_DataChain_PerformTransfer_Validate, fpga9001, port, channel, dmaCfg);
    /* Determine DMA Id */
    dmaId = (ADI_FPGA9001_CAPTURE_TYPE_RX_GAIN == dmaCfg->captureType) ? AXI_DMA_GAIN_ID :
                                                                         fpga9001_DMAIdGet(fpga9001, port, channel);
    /* Configure DMA */
    ADI_EXPECT(adi_fpga9001_DataChain_ConfigureTransfer, fpga9001, port, channel, dmaId, dmaCfg);

    ADI_API_RETURN(fpga9001);
}

int32_t adi_fpga9001_DataChain_PerformTransfer_Finish(adi_fpga9001_Device_t *fpga9001,
                                                      adi_common_Port_e port,
                                                      adi_common_ChannelNumber_e channel,
                                                      adi_fpga9001_DmaCfg_t *dmaCfg)
{
    uint32_t dmaId = 0;
    ADI_PERFORM_VALIDATION(adi_fpga9001_DataChain_PerformTransfer_Validate, fpga9001, port, channel, dmaCfg);
    /* Determine DMA Id */
    dmaId = (ADI_FPGA9001_CAPTURE_TYPE_RX_GAIN == dmaCfg->captureType) ? AXI_DMA_GAIN_ID :
                                                                         fpga9001_DMAIdGet(fpga9001, port, channel);

    /* Poll status of DMA if the transfer type is not cyclic */
    if (false == dmaCfg->continuous)
    {
        ADI_EXPECT(adi_fpga9001_DataChain_CaptureWait, fpga9001, port, channel, dmaId, dmaCfg);
    }

    ADI_API_RETURN(fpga9001);
}

static __maybe_unused int32_t adi_fpga9001_DataChain_HopTable_Transfer_Validate(adi_fpga9001_Device_t *fpga9001,
                                                                  adi_fpga9001_DmaCfg_t *dmaCfg)
{
    ADI_NULL_DEVICE_PTR_RETURN(fpga9001);
    ADI_RANGE_CHECK(fpga9001, dmaCfg->trigger, ADI_FPGA9001_DMA_TRIGGER_SMA_1, ADI_FPGA9001_DMA_TRIGGER_IMMEDIATE);
    ADI_API_RETURN(fpga9001);
}

int32_t adi_fpga9001_DataChain_HopTable_Transfer_Start(adi_fpga9001_Device_t *fpga9001,
                                                       adi_fpga9001_DmaCfg_t *dmaCfg)
{
    uint32_t dmaId = 0;
    uint32_t addr = 0;
    axi_dma_params_t dmaParams = { 0 };

    ADI_PERFORM_VALIDATION(adi_fpga9001_DataChain_HopTable_Transfer_Validate, fpga9001, dmaCfg);

    if (dmaCfg->trigger != ADI_FPGA9001_DMA_TRIGGER_IMMEDIATE) {
        dmaParams.triggers[(int) dmaCfg->trigger] = AXI_DMA_TOP_TRIGGER_RISING_EDGE;
    }

    dmaId = AXI_DMA_MOSI_ID;
    ADI_EXPECT(adi_fpga9001_hal_wrapper_DmaRelativeAddress_Get, fpga9001, ADI_FPGA9001_HAL_DMA_SPI, &addr);
    dmaParams.start_address = (uint64_t) addr;
    dmaParams.length = dmaCfg->length;
    dmaParams.cyclic = dmaCfg->continuous;
    dmaParams.qthreshold = 0x0;
    dmaParams.start_sync = 0x0;
    dmaParams.cyclic_sync = 0x0;

    ADI_EXPECT(axi_dma_config, fpga9001, dmaId, AXI_DMA_TOP_OFFSET, &dmaParams);
    ADI_EXPECT(axi_dma_begin, fpga9001, dmaId, AXI_DMA_TOP_OFFSET);

    ADI_API_RETURN(fpga9001);
}

int32_t adi_fpga9001_DataChain_HopTable_Transfer_Stop(adi_fpga9001_Device_t *fpga9001,
                                                      uint32_t timeout_ms)
{
    uint32_t timeout_us = 0;
    ADI_NULL_DEVICE_PTR_RETURN(fpga9001);
    timeout_us = KILO_TO_BASE_UNIT(timeout_ms);
    ADI_EXPECT(axi_dma_finish, fpga9001, AXI_DMA_MOSI_ID, AXI_DMA_TOP_OFFSET, timeout_us);
    ADI_EXPECT(axi_mspi_dma_finish, fpga9001, AXI_MSPI_ID, timeout_us);
    ADI_API_RETURN(fpga9001); 
}
static __maybe_unused int32_t adi_fpga9001_DataChain_Mspi_Dma_Configure_Validate(adi_fpga9001_Device_t *fpga9001,
                                                                  adi_fpga9001_DmaTrigger_e trigger,
                                                                  uint32_t bytesPerTable,
                                                                  uint32_t framesPerTable)
{
    ADI_RANGE_CHECK(fpga9001, trigger, ADI_FPGA9001_DMA_TRIGGER_SMA_1, ADI_FPGA9001_DMA_TRIGGER_IMMEDIATE);
    if (bytesPerTable <= 0)
    {
        ADI_ERROR_REPORT(&fpga9001->common,
            ADI_COMMON_ERRSRC_API,
            ADI_COMMON_ERR_API_FAIL,
            ADI_COMMON_ACT_ERR_CHECK_PARAM,
            bytesPerTable,
            "bytesPerTable must be a positive integer.");
        ADI_ERROR_RETURN(fpga9001->common.error.newAction);
    }
    if (framesPerTable <= 0)
    {
        ADI_ERROR_REPORT(&fpga9001->common,
            ADI_COMMON_ERRSRC_API,
            ADI_COMMON_ERR_API_FAIL,
            ADI_COMMON_ACT_ERR_CHECK_PARAM,
            framesPerTable,
            "framesPerTable must be a positive integer.");
        ADI_ERROR_RETURN(fpga9001->common.error.newAction);
    }
    ADI_API_RETURN(fpga9001);
}

int32_t adi_fpga9001_DataChain_Mspi_Dma_Configure(adi_fpga9001_Device_t *fpga9001,
                                                  adi_fpga9001_DmaTrigger_e trigger,
                                                  uint32_t bytesPerTable,
                                                  uint32_t framesPerTable)
{
    uint32_t mspiTriggerMode = 0;
    uint32_t timeout_us = 1000000 * 10;
    uint32_t status = 1;
    
    ADI_PERFORM_VALIDATION(adi_fpga9001_DataChain_Mspi_Dma_Configure_Validate, fpga9001, trigger, bytesPerTable, framesPerTable);

    axi_mspi_finish(fpga9001, AXI_MSPI_ID, timeout_us);
    axi_mspi_top_reg_triggers_set(fpga9001, AXI_MSPI_ID, 0x0);
	if (trigger == ADI_FPGA9001_DMA_TRIGGER_SYNC) {
		axi_mspi_top_trigger_mode_set(fpga9001, AXI_MSPI_ID, trigger, AXI_MSPI_TOP_TRIGGER_EDGE);
	}
	else if (trigger != ADI_FPGA9001_DMA_TRIGGER_IMMEDIATE) {
        axi_mspi_top_trigger_mode_set(fpga9001, AXI_MSPI_ID, trigger, AXI_MSPI_TOP_TRIGGER_RISING_EDGE);
    }
    axi_mspi_slave_select_set(fpga9001, AXI_MSPI_ID, 0x1);
    axi_mspi_dma_config(fpga9001, AXI_MSPI_ID, (bytesPerTable - 1), (framesPerTable - 1));

    ADI_API_RETURN(fpga9001);
}

int32_t adi_fpga9001_DataChain_Mspi_Fifo_Configure(adi_fpga9001_Device_t *fpga9001)
{
    uint32_t status = 1;
    uint32_t timeout_us = 1000000 * 10;
    ADI_NULL_DEVICE_PTR_RETURN(fpga9001);

    axi_mspi_finish(fpga9001, AXI_MSPI_ID, timeout_us);
    axi_mspi_top_reg_triggers_set(fpga9001, AXI_MSPI_ID, 0x0);
    axi_mspi_slave_select_set(fpga9001, AXI_MSPI_ID, 0x0);
    axi_mspi_fifo_config(fpga9001, AXI_MSPI_ID);
    
    ADI_API_RETURN(fpga9001);
}

static __maybe_unused int32_t adi_fpga9001_DataChain_Data_Get_22I22Q_GainChange_GainIndexValidate(adi_fpga9001_Device_t *fpga9001,
                                                                                   adi_common_Port_e port,
                                                                                   adi_common_ChannelNumber_e channel,
                                                                                   uint32_t iData[],
                                                                                   uint32_t qData[],
                                                                                   uint8_t gainChangeData[],
                                                                                   uint8_t gainIndexData[],
                                                                                   uint32_t length)
{
    ADI_RANGE_CHECK(fpga9001, port, ADI_RX, ADI_ELB);
    ADI_RANGE_CHECK(fpga9001, channel, ADI_CHANNEL_1, ADI_CHANNEL_2);

    ADI_NULL_PTR_RETURN(&fpga9001->common, iData);
    ADI_NULL_PTR_RETURN(&fpga9001->common, qData);
    ADI_NULL_PTR_RETURN(&fpga9001->common, gainChangeData);
    ADI_NULL_PTR_RETURN(&fpga9001->common, gainIndexData);

#ifndef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    ADI_RANGE_CHECK(fpga9001, length * 2, 0, ADI_FPGA9001_MAX_CAPTURE_BYTES / 4)
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY
    
    ADI_API_RETURN(fpga9001);
}

int32_t adi_fpga9001_DataChain_Data_Get_22I22Q_GainChange_GainIndex(adi_fpga9001_Device_t *fpga9001,
                                                                    adi_common_Port_e port,
                                                                    adi_common_ChannelNumber_e channel,
                                                                    uint32_t iData[],
                                                                    uint32_t qData[],
                                                                    uint8_t gainChangeData[],
                                                                    uint8_t gainIndexData[],
                                                                    uint32_t length)
{
    uint32_t i = 0;
    uint32_t ramAddress = 0;
    int32_t recoveryAction = ADI_COMMON_ACT_NO_ACTION;
#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    uint32_t *rdBuff = NULL;
#else
    uint32_t rdBuff[ADI_FPGA9001_MAX_CAPTURE_BYTES * 2] = { 0 };
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY


    ADI_PERFORM_VALIDATION(adi_fpga9001_DataChain_Data_Get_22I22Q_GainChange_GainIndexValidate, fpga9001, port, channel, iData, qData, gainChangeData, gainIndexData, length);

#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    rdBuff = calloc(length * 2, sizeof(uint32_t));
    if (rdBuff == NULL)
    {
        ADI_ERROR_REPORT(&fpga9001->common,
                         ADI_COMMON_ERRSRC_API,
                         ADI_COMMON_ERR_MEM_ALLOC_FAIL,
                         ADI_COMMON_ACT_ERR_RESET_FULL,
                         NULL,
                         "Unable to allocate memory");
        ADI_API_RETURN(fpga9001);
    }
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY

    ADI_EXPECT(adi_fpga9001_hal_wrapper_DmaRelativeAddress_Get, fpga9001, fpga9001_Dma_Get(fpga9001, port, channel), &ramAddress);
    if (ramAddress == (uint32_t) - 1)
    {
#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
        free(rdBuff);
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY
        ADI_API_RETURN(fpga9001);
    }

    recoveryAction = adi_fpga9001_hal_wrapper_Ram_Read(fpga9001, ramAddress, rdBuff, length * 2);
    if (recoveryAction != ADI_COMMON_ACT_NO_ACTION)
    {
#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
        free(rdBuff);
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY
    }
    
    for (i = 0; i < length; i++)
    {
        iData[i]          = (uint32_t)((rdBuff[2 * i + 1] & 0xFFFFFC00) >> 10); /* I =  rdBuff0[31:10] */
        qData[i]          = (uint32_t)((rdBuff[2 * i + 0] & 0xFFFFFC00) >> 10); /* Q =  rdBuff1[31:10] */
        gainChangeData[i] = (uint8_t) ((rdBuff[2 * i + 0] & 0x100)      >>  8); /* GD = rdBuff0[8] */
        gainIndexData[i]  = (uint8_t) ((rdBuff[2 * i + 0] & 0xFF)       >>  0); /* GI = rdBuff0[7:0] */
    }

#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    free(rdBuff);
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY

    ADI_API_RETURN(fpga9001);
}

static __maybe_unused int32_t adi_fpga9001_DataChain_Data_Get_22I22QSigned_GainChange_GainIndexValidate(adi_fpga9001_Device_t *fpga9001,
                                                                                         adi_common_Port_e port,
                                                                                         adi_common_ChannelNumber_e channel,
                                                                                         int32_t iData[],
                                                                                         int32_t qData[],
                                                                                         uint8_t gainChangeData[],
                                                                                         uint8_t gainIndexData[],
                                                                                         uint32_t length)
{
    ADI_RANGE_CHECK(fpga9001, port, ADI_RX, ADI_ELB);
    ADI_RANGE_CHECK(fpga9001, channel, ADI_CHANNEL_1, ADI_CHANNEL_2);

    ADI_NULL_PTR_RETURN(&fpga9001->common, iData);
    ADI_NULL_PTR_RETURN(&fpga9001->common, qData);
    ADI_NULL_PTR_RETURN(&fpga9001->common, gainChangeData);
    ADI_NULL_PTR_RETURN(&fpga9001->common, gainIndexData);

#ifndef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    ADI_RANGE_CHECK(fpga9001, length * 2, 0, ADI_FPGA9001_MAX_CAPTURE_BYTES / 4)
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY
    
    ADI_API_RETURN(fpga9001);
}

int32_t adi_fpga9001_DataChain_Data_Get_22I22QSigned_GainChange_GainIndex(adi_fpga9001_Device_t *fpga9001,
                                                                          adi_common_Port_e port,
                                                                          adi_common_ChannelNumber_e channel,
                                                                          int32_t iData[],
                                                                          int32_t qData[],
                                                                          uint8_t gainChangeData[],
                                                                          uint8_t gainIndexData[],
                                                                          uint32_t length)
{
    uint32_t i = 0;
    uint32_t ramAddress = 0;
    int32_t recoveryAction = ADI_COMMON_ACT_NO_ACTION;
    int32_t positiveFullScale = 1 << 21;
    int32_t fullScaleRange = 1 << 22;
#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    uint32_t *rdBuff = NULL;
#else
    uint32_t rdBuff[ADI_FPGA9001_MAX_CAPTURE_BYTES * 2] = { 0 };
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY


    ADI_PERFORM_VALIDATION(adi_fpga9001_DataChain_Data_Get_22I22QSigned_GainChange_GainIndexValidate, fpga9001, port, channel, iData, qData, gainChangeData, gainIndexData, length);

#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    rdBuff = calloc(length * 2, sizeof(uint32_t));
    if (rdBuff == NULL)
    {
        ADI_ERROR_REPORT(&fpga9001->common,
                         ADI_COMMON_ERRSRC_API,
                         ADI_COMMON_ERR_MEM_ALLOC_FAIL,
                         ADI_COMMON_ACT_ERR_RESET_FULL,
                         NULL,
                         "Unable to allocate memory");
        ADI_API_RETURN(fpga9001);
    }
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY

    ADI_EXPECT(adi_fpga9001_hal_wrapper_DmaRelativeAddress_Get, fpga9001, fpga9001_Dma_Get(fpga9001, port, channel), &ramAddress);
    if (ramAddress == (uint32_t) - 1)
    {
#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
        free(rdBuff);
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY
        ADI_API_RETURN(fpga9001);
    }

    recoveryAction = adi_fpga9001_hal_wrapper_Ram_Read(fpga9001, ramAddress, rdBuff, length * 2);
    if (recoveryAction != ADI_COMMON_ACT_NO_ACTION)
    {
#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
        free(rdBuff);
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY
    }
    
    for (i = 0; i < length; i++)
    {
        iData[i]          = (int32_t)((rdBuff[2 * i + 1] & 0xFFFFFC00) >> 10); /* I =  rdBuff0[31:10] */
        iData[i]          = iData[i] > positiveFullScale ? iData[i] - fullScaleRange : iData[i];
        qData[i]          = (int32_t)((rdBuff[2 * i + 0] & 0xFFFFFC00) >> 10); /* Q =  rdBuff1[31:10] */
        qData[i]          = qData[i] > positiveFullScale ? qData[i] - fullScaleRange : qData[i];
        gainChangeData[i] = (uint8_t)((rdBuff[2 * i + 0] & 0x100)      >>  8); /* GD = rdBuff0[8] */
        gainIndexData[i]  = (uint8_t)((rdBuff[2 * i + 0] & 0xFF)       >>  0); /* GI = rdBuff0[7:0] */
    }

#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    free(rdBuff);
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY

    ADI_API_RETURN(fpga9001);
}

static __maybe_unused int32_t adi_fpga9001_DataChain_Data_Get_22I22QInterleaved_GainChange_GainIndexValidate(adi_fpga9001_Device_t *fpga9001,
                                                                                              adi_common_Port_e port,
                                                                                              adi_common_ChannelNumber_e channel,
                                                                                              uint32_t iqData[],
                                                                                              uint8_t gainChangeData[],
                                                                                              uint8_t gainIndexData[],
                                                                                              uint32_t length)
{
    ADI_RANGE_CHECK(fpga9001, port, ADI_RX, ADI_ELB);
    ADI_RANGE_CHECK(fpga9001, channel, ADI_CHANNEL_1, ADI_CHANNEL_2);

    ADI_NULL_PTR_RETURN(&fpga9001->common, iqData);
    ADI_NULL_PTR_RETURN(&fpga9001->common, gainChangeData);
    ADI_NULL_PTR_RETURN(&fpga9001->common, gainIndexData);

#ifndef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    ADI_RANGE_CHECK(fpga9001, length * 2, 0, ADI_FPGA9001_MAX_CAPTURE_BYTES / 2)
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY
    
    ADI_API_RETURN(fpga9001);
    
}

int32_t adi_fpga9001_DataChain_Data_Get_22I22QInterleaved_GainChange_GainIndex(adi_fpga9001_Device_t *fpga9001,
                                                                               adi_common_Port_e port,
                                                                               adi_common_ChannelNumber_e channel,
                                                                               uint32_t iqData[],
                                                                               uint8_t gainChangeData[],
                                                                               uint8_t gainIndexData[],
                                                                               uint32_t length)
{
    uint32_t i = 0;
    uint32_t ramAddress = 0;
    int32_t recoveryAction = ADI_COMMON_ACT_NO_ACTION;
#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    uint32_t *rdBuff = NULL;
#else
    uint32_t rdBuff[ADI_FPGA9001_MAX_CAPTURE_BYTES * 2] = { 0 };
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY


    ADI_PERFORM_VALIDATION(adi_fpga9001_DataChain_Data_Get_22I22QInterleaved_GainChange_GainIndexValidate, fpga9001, port, channel, iqData, gainChangeData, gainIndexData, length);

#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    rdBuff = calloc(length * 2, sizeof(uint32_t));
    if (rdBuff == NULL)
    {
        ADI_ERROR_REPORT(&fpga9001->common,
                         ADI_COMMON_ERRSRC_API,
                         ADI_COMMON_ERR_MEM_ALLOC_FAIL,
                         ADI_COMMON_ACT_ERR_RESET_FULL,
                         NULL,
                         "Unable to allocate memory");
        ADI_API_RETURN(fpga9001);
    }
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY

    ADI_EXPECT(adi_fpga9001_hal_wrapper_DmaRelativeAddress_Get, fpga9001, fpga9001_Dma_Get(fpga9001, port, channel), &ramAddress);
    if (ramAddress == (uint32_t) - 1)
    {
#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
        free(rdBuff);
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY
        ADI_API_RETURN(fpga9001);
    }

    recoveryAction = adi_fpga9001_hal_wrapper_Ram_Read(fpga9001, ramAddress, rdBuff, length * 2);
    if (recoveryAction != ADI_COMMON_ACT_NO_ACTION)
    {
#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
        free(rdBuff);
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY
    }
    
    for (i = 0; i < length / 2; i++)
    {
        iqData[2 * i + 0]         = (uint32_t)((rdBuff[2 * i + 1] & 0xFFFFFC00) >> 10); /* I =  rdBuff0[31:10] */
        iqData[2 * i + 1]         = (uint32_t)((rdBuff[2 * i + 0] & 0xFFFFFC00) >> 10); /* Q =  rdBuff1[31:10] */
        gainChangeData[2 * i + 0] = (uint8_t) ((rdBuff[2 * i + 0] & 0x100)      >> 8); /* GD = rdBuff0[8] */
        gainIndexData[2 * i + 0]  = (uint8_t)((rdBuff[2 * i + 0] & 0xFF)        >> 0); /* GI = rdBuff0[7:0] */
        
        gainChangeData[2 * i + 1] = gainChangeData[2 * i + 0];
        gainIndexData[2 * i + 1]  = gainIndexData[2 * i + 0];
    }

#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    free(rdBuff);
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY

    ADI_API_RETURN(fpga9001);
}

static __maybe_unused int32_t adi_fpga9001_DataChain_Data_Get_22I22QInterleavedSigned_GainChange_GainIndexValidate(adi_fpga9001_Device_t *fpga9001,
                                                                                                    adi_common_Port_e port,
                                                                                                    adi_common_ChannelNumber_e channel,
                                                                                                    int32_t iqData[],
                                                                                                    uint8_t gainChangeData[],
                                                                                                    uint8_t gainIndexData[],
                                                                                                    uint32_t length)
{
    ADI_RANGE_CHECK(fpga9001, port, ADI_RX, ADI_ELB);
    ADI_RANGE_CHECK(fpga9001, channel, ADI_CHANNEL_1, ADI_CHANNEL_2);

    ADI_NULL_PTR_RETURN(&fpga9001->common, iqData);
    ADI_NULL_PTR_RETURN(&fpga9001->common, gainChangeData);
    ADI_NULL_PTR_RETURN(&fpga9001->common, gainIndexData);

#ifndef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    ADI_RANGE_CHECK(fpga9001, length * 2, 0, ADI_FPGA9001_MAX_CAPTURE_BYTES / 2)
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY
    
    ADI_API_RETURN(fpga9001);
    
}

int32_t adi_fpga9001_DataChain_Data_Get_22I22QInterleavedSigned_GainChange_GainIndex(adi_fpga9001_Device_t *fpga9001,
                                                                                     adi_common_Port_e port,
                                                                                     adi_common_ChannelNumber_e channel,
                                                                                     int32_t iqData[],
                                                                                     uint8_t gainChangeData[],
                                                                                     uint8_t gainIndexData[],
                                                                                     uint32_t length)
{
    uint32_t i = 0;
    uint32_t ramAddress = 0;
    int32_t recoveryAction = ADI_COMMON_ACT_NO_ACTION;
    int32_t positiveFullScale = 1 << 21;
    int32_t fullScaleRange = 1 << 22;
#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    uint32_t *rdBuff = NULL;
#else
    uint32_t rdBuff[ADI_FPGA9001_MAX_CAPTURE_BYTES * 2] = { 0 };
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY


    ADI_PERFORM_VALIDATION(adi_fpga9001_DataChain_Data_Get_22I22QInterleavedSigned_GainChange_GainIndexValidate, fpga9001, port, channel, iqData, gainChangeData, gainIndexData, length);

#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    rdBuff = calloc(length * 2, sizeof(uint32_t));
    if (rdBuff == NULL)
    {
        ADI_ERROR_REPORT(&fpga9001->common,
                         ADI_COMMON_ERRSRC_API,
                         ADI_COMMON_ERR_MEM_ALLOC_FAIL,
                         ADI_COMMON_ACT_ERR_RESET_FULL,
                         NULL,
                         "Unable to allocate memory");
        ADI_API_RETURN(fpga9001);
    }
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY

    ADI_EXPECT(adi_fpga9001_hal_wrapper_DmaRelativeAddress_Get, fpga9001, fpga9001_Dma_Get(fpga9001, port, channel), &ramAddress);
    if (ramAddress == (uint32_t) - 1)
    {
#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
        free(rdBuff);
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY
        ADI_API_RETURN(fpga9001);
    }

    recoveryAction = adi_fpga9001_hal_wrapper_Ram_Read(fpga9001, ramAddress, rdBuff, length * 2);
    if (recoveryAction != ADI_COMMON_ACT_NO_ACTION)
    {
#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
        free(rdBuff);
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY
    }
    
    for (i = 0; i < length / 2; i++)
    {
        iqData[2 * i + 0]         = (int32_t)((rdBuff[2 * i + 1] & 0xFFFFFC00) >> 10); /* I =  rdBuff0[31:10] */
        iqData[2 * i + 0]         = iqData[2 * i + 0] > positiveFullScale ? iqData[2 * i + 0] - fullScaleRange : iqData[2 * i + 0];
        iqData[2 * i + 1]         = (int32_t)((rdBuff[2 * i + 0] & 0xFFFFFC00) >> 10); /* Q =  rdBuff1[31:10] */
        iqData[2 * i + 1]         = iqData[2 * i + 1] > positiveFullScale ? iqData[2 * i + 1] - fullScaleRange : iqData[2 * i + 1];
        gainChangeData[2 * i + 0] = (uint8_t)((rdBuff[2 * i + 0] & 0x100)      >> 8); /* GD = rdBuff0[8] */
        gainIndexData[2 * i + 0]  = (uint8_t)((rdBuff[2 * i + 0] & 0xFF)        >> 0); /* GI = rdBuff0[7:0] */
        
        gainChangeData[2 * i + 1] = gainChangeData[2 * i + 0];
        gainIndexData[2 * i + 1]  = gainIndexData[2 * i + 0];
    }

#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    free(rdBuff);
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY

    ADI_API_RETURN(fpga9001);
}

static __maybe_unused int32_t adi_fpga9001_DataChain_Data_Get_16I16QValidate(adi_fpga9001_Device_t *fpga9001,
                                                              adi_common_Port_e port,
                                                              adi_common_ChannelNumber_e channel,
                                                              uint16_t *iData,
                                                              uint16_t *qData,
                                                              uint32_t length)
{
    ADI_RANGE_CHECK(fpga9001, port, ADI_RX, ADI_ELB);
    ADI_RANGE_CHECK(fpga9001, channel, ADI_CHANNEL_1, ADI_CHANNEL_2);

    ADI_NULL_PTR_RETURN(&fpga9001->common, iData);
    ADI_NULL_PTR_RETURN(&fpga9001->common, qData);

#ifndef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    ADI_RANGE_CHECK(fpga9001, length, 0, ADI_FPGA9001_MAX_CAPTURE_BYTES / 4)
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY
    
    ADI_API_RETURN(fpga9001);
}

int32_t adi_fpga9001_DataChain_Data_Get_16I16Q(adi_fpga9001_Device_t *fpga9001,
                                                           adi_common_Port_e port,
                                                           adi_common_ChannelNumber_e channel,
                                                           uint16_t *iData,
                                                           uint16_t *qData,
                                                           uint32_t length)
{
    uint32_t i = 0;
    uint32_t ramAddress = 0;
    int32_t recoveryAction = ADI_COMMON_ACT_NO_ACTION;
#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    uint32_t *rdBuff = NULL;
#else
    uint32_t rdBuff[ADI_FPGA9001_MAX_CAPTURE_BYTES] = { 0 };
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY


    ADI_PERFORM_VALIDATION(adi_fpga9001_DataChain_Data_Get_16I16QValidate, fpga9001, port, channel, iData, qData, length);

#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    rdBuff = calloc(length, sizeof(uint32_t));
    if (rdBuff == NULL)
    {
        ADI_ERROR_REPORT(&fpga9001->common,
                         ADI_COMMON_ERRSRC_API,
                         ADI_COMMON_ERR_MEM_ALLOC_FAIL,
                         ADI_COMMON_ACT_ERR_RESET_FULL,
                         NULL,
                         "Unable to allocate memory");
        ADI_API_RETURN(fpga9001);
    }
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY

    ADI_EXPECT(adi_fpga9001_hal_wrapper_DmaRelativeAddress_Get, fpga9001, fpga9001_Dma_Get(fpga9001, port, channel), &ramAddress);
    if (ramAddress == (uint32_t) -1)
    {
#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
        free(rdBuff);
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY
        ADI_API_RETURN(fpga9001);
    }

    recoveryAction = adi_fpga9001_hal_wrapper_Ram_Read(fpga9001, ramAddress, rdBuff, length);
    if (recoveryAction != ADI_COMMON_ACT_NO_ACTION)
    {
#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
        free(rdBuff);
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY
        ADI_API_RETURN(fpga9001);
    }

    for (i = 0; i < length; i++)
    {
        iData[i] = (uint16_t)((rdBuff[i] & 0xFFFF0000) >> 16);
        qData[i] = (uint16_t)(rdBuff[i]  & 0x0000FFFF);
    }

#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    free(rdBuff);
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY

    ADI_API_RETURN(fpga9001);
}

static __maybe_unused int32_t adi_fpga9001_DataChain_Data_Get_16I16QSignedValidate(adi_fpga9001_Device_t *fpga9001,
                                                                    adi_common_Port_e port,
                                                                    adi_common_ChannelNumber_e channel,
                                                                    int16_t *iData,
                                                                    int16_t *qData,
                                                                    uint32_t length)
{
    ADI_RANGE_CHECK(fpga9001, port, ADI_RX, ADI_ELB);
    ADI_RANGE_CHECK(fpga9001, channel, ADI_CHANNEL_1, ADI_CHANNEL_2);

    ADI_NULL_PTR_RETURN(&fpga9001->common, iData);
    ADI_NULL_PTR_RETURN(&fpga9001->common, qData);

#ifndef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    ADI_RANGE_CHECK(fpga9001, length, 0, ADI_FPGA9001_MAX_CAPTURE_BYTES / 4)
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY
    
    ADI_API_RETURN(fpga9001);
}

int32_t adi_fpga9001_DataChain_Data_Get_16I16QSigned(adi_fpga9001_Device_t *fpga9001,
                                                     adi_common_Port_e port,
                                                     adi_common_ChannelNumber_e channel,
                                                     int16_t iData[],
                                                     int16_t qData[],
                                                     uint32_t length)
{
    uint32_t i = 0;
    uint32_t ramAddress = 0;
    int32_t recoveryAction = ADI_COMMON_ACT_NO_ACTION;
    uint32_t positiveFullScale = 0x8000; // 2^15
    uint32_t fullScaleRange = 0x10000;   // 2^16
#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    uint32_t *rdBuff = NULL;
#else
    uint32_t rdBuff[ADI_FPGA9001_MAX_CAPTURE_BYTES] = { 0 };
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY


    ADI_PERFORM_VALIDATION(adi_fpga9001_DataChain_Data_Get_16I16QSignedValidate, fpga9001, port, channel, iData, qData, length);

#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    rdBuff = calloc(length, sizeof(uint32_t));
    if (rdBuff == NULL)
    {
        ADI_ERROR_REPORT(&fpga9001->common,
                         ADI_COMMON_ERRSRC_API,
                         ADI_COMMON_ERR_MEM_ALLOC_FAIL,
                         ADI_COMMON_ACT_ERR_RESET_FULL,
                         NULL,
                         "Unable to allocate memory");
        ADI_API_RETURN(fpga9001);
    }
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY

    ADI_EXPECT(adi_fpga9001_hal_wrapper_DmaRelativeAddress_Get, fpga9001, fpga9001_Dma_Get(fpga9001, port, channel), &ramAddress);
    if (ramAddress == (uint32_t) - 1)
    {
#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
        free(rdBuff);
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY
        ADI_API_RETURN(fpga9001);
    }

    recoveryAction = adi_fpga9001_hal_wrapper_Ram_Read(fpga9001, ramAddress, rdBuff, length);
    if (recoveryAction != ADI_COMMON_ACT_NO_ACTION)
    {
#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
        free(rdBuff);
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY
        ADI_API_RETURN(fpga9001);
    }

    for (i = 0; i < length; i++)
    {
        iData[i] = (int16_t)((rdBuff[i] & 0xFFFF0000) >> 16);
        iData[i] = iData[i] > positiveFullScale ? iData[i] - fullScaleRange : iData[i];
        qData[i] = (int16_t)(rdBuff[i]  & 0x0000FFFF);
        qData[i] = qData[i] > positiveFullScale ? qData[i] - fullScaleRange : qData[i];
    }

#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    free(rdBuff);
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY

    ADI_API_RETURN(fpga9001);
}

static __maybe_unused int32_t adi_fpga9001_DataChain_Data_Get_16IQInterleavedValidate(adi_fpga9001_Device_t *fpga9001,
                                                                       adi_common_Port_e port,
                                                                       adi_common_ChannelNumber_e channel,
                                                                       uint16_t *iqData,
                                                                       uint32_t length)
{
    ADI_RANGE_CHECK(fpga9001, port, ADI_RX, ADI_ORX);
    ADI_RANGE_CHECK(fpga9001, channel, ADI_CHANNEL_1, ADI_CHANNEL_2);

    ADI_NULL_PTR_RETURN(&fpga9001->common, iqData);

#ifndef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    ADI_RANGE_CHECK(fpga9001, length, 0, ADI_FPGA9001_MAX_CAPTURE_BYTES / 2)
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY

    ADI_API_RETURN(fpga9001);
}

int32_t adi_fpga9001_DataChain_Data_Get_16IQInterleaved(adi_fpga9001_Device_t *fpga9001,
                                                                    adi_common_Port_e port,
                                                                    adi_common_ChannelNumber_e channel,
                                                                    uint16_t *iqData,
                                                                    uint32_t length)
{
    uint32_t i = 0;
    uint32_t ramAddress = 0;
#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    uint32_t *rdBuff = NULL;
#else
    uint32_t rdBuff[ADI_FPGA9001_MAX_CAPTURE_BYTES] = { 0 };
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY

    ADI_PERFORM_VALIDATION(adi_fpga9001_DataChain_Data_Get_16IQInterleavedValidate, fpga9001, port, channel, iqData, length);

#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    rdBuff = calloc(length / 2, sizeof(uint32_t));
    if (rdBuff == NULL)
    {
        ADI_ERROR_REPORT(&fpga9001->common,
                         ADI_COMMON_ERRSRC_API,
                         ADI_COMMON_ERR_MEM_ALLOC_FAIL,
                         ADI_COMMON_ACT_ERR_RESET_FULL,
                         NULL,
                         "Unable to allocate memory");
        ADI_API_RETURN(fpga9001);
    }
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY

    ADI_EXPECT(adi_fpga9001_hal_wrapper_DmaRelativeAddress_Get, fpga9001, fpga9001_Dma_Get(fpga9001, port, channel), &ramAddress);
    if (ramAddress == (uint32_t) -1)
    {
#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
        free(rdBuff);
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY
        ADI_API_RETURN(fpga9001);
    }

    if (adi_fpga9001_hal_wrapper_Ram_Read(fpga9001, ramAddress, rdBuff, length / 2) != ADI_COMMON_ACT_NO_ACTION)
    {
#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
        free(rdBuff);
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY
        ADI_API_RETURN(fpga9001);
    }

    for (i = 0; i < length / 2; i++)
    {
        iqData[2 * i]     = (uint16_t)((rdBuff[i] & 0xFFFF0000) >> 16);
        iqData[2 * i + 1] = (uint16_t)(rdBuff[i]  & 0x0000FFFF);
    }

#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    free(rdBuff);
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY

    ADI_API_RETURN(fpga9001);
}

static __maybe_unused int32_t adi_fpga9001_DataChain_Data_Get_16IQInterleavedSignedValidate(adi_fpga9001_Device_t *fpga9001,
                                                                             adi_common_Port_e port,
                                                                             adi_common_ChannelNumber_e channel,
                                                                             int16_t *iqData,
                                                                             uint32_t length)
{
    ADI_RANGE_CHECK(fpga9001, port, ADI_RX, ADI_ORX);
    ADI_RANGE_CHECK(fpga9001, channel, ADI_CHANNEL_1, ADI_CHANNEL_2);

    ADI_NULL_PTR_RETURN(&fpga9001->common, iqData);

#ifndef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    ADI_RANGE_CHECK(fpga9001, length, 0, ADI_FPGA9001_MAX_CAPTURE_BYTES / 2)
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY

    ADI_API_RETURN(fpga9001);
}

int32_t adi_fpga9001_DataChain_Data_Get_16IQInterleavedSigned(adi_fpga9001_Device_t *fpga9001,
                                                              adi_common_Port_e port,
                                                              adi_common_ChannelNumber_e channel,
                                                              int16_t *iqData,
                                                              uint32_t length)
{
    uint32_t i = 0;
    uint32_t ramAddress = 0;
    uint32_t positiveFullScale = 0x8000;  // 2^15
    uint32_t fullScaleRange = 0x10000;    // 2^16
#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    uint32_t *rdBuff = NULL;
#else
    uint32_t rdBuff[ADI_FPGA9001_MAX_CAPTURE_BYTES] = { 0 };
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY

    ADI_PERFORM_VALIDATION(adi_fpga9001_DataChain_Data_Get_16IQInterleavedSignedValidate, fpga9001, port, channel, iqData, length);

#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    rdBuff = calloc(length / 2, sizeof(uint32_t));
    if (rdBuff == NULL)
    {
        ADI_ERROR_REPORT(&fpga9001->common,
                         ADI_COMMON_ERRSRC_API,
                         ADI_COMMON_ERR_MEM_ALLOC_FAIL,
                         ADI_COMMON_ACT_ERR_RESET_FULL,
                         NULL,
                         "Unable to allocate memory");
        ADI_API_RETURN(fpga9001);
    }
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY

    ADI_EXPECT(adi_fpga9001_hal_wrapper_DmaRelativeAddress_Get, fpga9001, fpga9001_Dma_Get(fpga9001, port, channel), &ramAddress);
    if (ramAddress == (uint32_t) - 1)
    {
#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
        free(rdBuff);
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY
        ADI_API_RETURN(fpga9001);
    }

    if (adi_fpga9001_hal_wrapper_Ram_Read(fpga9001, ramAddress, rdBuff, length / 2) != ADI_COMMON_ACT_NO_ACTION)
    {
#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
        free(rdBuff);
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY
        ADI_API_RETURN(fpga9001);
    }

    for (i = 0; i < length / 2; i++)
    {
        iqData[2 * i]     = (int16_t)((rdBuff[i] & 0xFFFF0000) >> 16);
        iqData[2 * i]     = iqData[2 * i] > positiveFullScale ? iqData[2 * i] - fullScaleRange : iqData[2 * i];
        iqData[2 * i + 1] = (int16_t)(rdBuff[i]  & 0x0000FFFF);
        iqData[2 * i + 1] = iqData[2 * i + 1] > positiveFullScale ? iqData[2 * i + 1] - fullScaleRange : iqData[2 * i + 1];
    }

#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    free(rdBuff);
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY

    ADI_API_RETURN(fpga9001);
}

static __maybe_unused int32_t adi_fpga9001_DataChain_Data_Get_16IValidate(adi_fpga9001_Device_t *fpga9001,
                                                           adi_common_Port_e port,
                                                           adi_common_ChannelNumber_e channel,
                                                           uint16_t *iData,
                                                           uint32_t length)
{
    ADI_RANGE_CHECK(fpga9001, port, ADI_RX, ADI_ORX);
    ADI_RANGE_CHECK(fpga9001, channel, ADI_CHANNEL_1, ADI_CHANNEL_2);

    ADI_NULL_PTR_RETURN(&fpga9001->common, iData);

#ifndef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    ADI_RANGE_CHECK(fpga9001, length, 0, ADI_FPGA9001_MAX_CAPTURE_BYTES / 2)
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY

    ADI_API_RETURN(fpga9001);
}

int32_t adi_fpga9001_DataChain_Data_Get_16I(adi_fpga9001_Device_t *fpga9001,
                                            adi_common_Port_e port,
                                            adi_common_ChannelNumber_e channel,
                                            uint16_t *iData,
                                            uint32_t length)
{
    uint32_t i = 0;
    uint32_t ramAddress = 0;
#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    uint32_t *rdBuff = NULL;
#else
    uint32_t rdBuff[ADI_FPGA9001_MAX_CAPTURE_BYTES] = { 0 };
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY
    
    ADI_PERFORM_VALIDATION(adi_fpga9001_DataChain_Data_Get_16IValidate, fpga9001, port, channel, iData, length);

#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    rdBuff = calloc(length / 2, sizeof(uint32_t));
    if (rdBuff == NULL)
    {
        ADI_ERROR_REPORT(&fpga9001->common,
                         ADI_COMMON_ERRSRC_API,
                         ADI_COMMON_ERR_MEM_ALLOC_FAIL,
                         ADI_COMMON_ACT_ERR_RESET_FULL,
                         NULL,
                         "Unable to allocate memory");
        ADI_API_RETURN(fpga9001);
    }
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY

    ADI_EXPECT(adi_fpga9001_hal_wrapper_DmaRelativeAddress_Get, fpga9001, fpga9001_Dma_Get(fpga9001, port, channel), &ramAddress);
    if (ramAddress == (uint32_t) - 1)
    {
#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
        free(rdBuff);
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY
        ADI_API_RETURN(fpga9001);
    }

    if (adi_fpga9001_hal_wrapper_Ram_Read(fpga9001, ramAddress, rdBuff, length / 2) != ADI_COMMON_ACT_NO_ACTION)
    {
#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
        free(rdBuff);
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY
        ADI_API_RETURN(fpga9001);
    }

    for (i = 0; i < length / 2; i++)
    {
        iData[2 * i]     = (uint16_t)(rdBuff[i]  & 0x0000FFFF);
        iData[2 * i + 1] = (uint16_t)((rdBuff[i] & 0xFFFF0000) >> 16);
    }

#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    free(rdBuff);
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY

    ADI_API_RETURN(fpga9001);
}

static __maybe_unused int32_t adi_fpga9001_DataChain_Data_Get_16ISignedValidate(adi_fpga9001_Device_t *fpga9001,
                                                                 adi_common_Port_e port,
                                                                 adi_common_ChannelNumber_e channel,
                                                                 int16_t *iData,
                                                                 uint32_t length)
{
    ADI_RANGE_CHECK(fpga9001, port, ADI_RX, ADI_ORX);
    ADI_RANGE_CHECK(fpga9001, channel, ADI_CHANNEL_1, ADI_CHANNEL_2);

    ADI_NULL_PTR_RETURN(&fpga9001->common, iData);

#ifndef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    ADI_RANGE_CHECK(fpga9001, length, 0, ADI_FPGA9001_MAX_CAPTURE_BYTES / 2)
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY

    ADI_API_RETURN(fpga9001);
}

int32_t adi_fpga9001_DataChain_Data_Get_16ISigned(adi_fpga9001_Device_t *fpga9001,
                                                  adi_common_Port_e port,
                                                  adi_common_ChannelNumber_e channel,
                                                  int16_t *iData,
                                                  uint32_t length)
{
    uint32_t i = 0;
    uint32_t ramAddress = 0;
    uint32_t positiveFullScale = 0x8000;  // 2^15
    uint32_t fullScaleRange = 0x10000;    // 2^16
#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    uint32_t *rdBuff = NULL;
#else
    uint32_t rdBuff[ADI_FPGA9001_MAX_CAPTURE_BYTES] = { 0 };
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY
    
    ADI_PERFORM_VALIDATION(adi_fpga9001_DataChain_Data_Get_16ISignedValidate, fpga9001, port, channel, iData, length);

#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    rdBuff = calloc(length / 2, sizeof(uint32_t));
    if (rdBuff == NULL)
    {
        ADI_ERROR_REPORT(&fpga9001->common,
                         ADI_COMMON_ERRSRC_API,
                         ADI_COMMON_ERR_MEM_ALLOC_FAIL,
                         ADI_COMMON_ACT_ERR_RESET_FULL,
                         NULL,
                         "Unable to allocate memory");
        ADI_API_RETURN(fpga9001);
    }
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY

    ADI_EXPECT(adi_fpga9001_hal_wrapper_DmaRelativeAddress_Get, fpga9001, fpga9001_Dma_Get(fpga9001, port, channel), &ramAddress);
    if (ramAddress == (uint32_t) - 1)
    {
#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
        free(rdBuff);
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY
        ADI_API_RETURN(fpga9001);
    }

    if (adi_fpga9001_hal_wrapper_Ram_Read(fpga9001, ramAddress, rdBuff, length / 2) != ADI_COMMON_ACT_NO_ACTION)
    {
#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
        free(rdBuff);
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY
        ADI_API_RETURN(fpga9001);
    }

    for (i = 0; i < length / 2; i++)
    {
        iData[2 * i]     = (int16_t)(rdBuff[i]  & 0x0000FFFF);
        iData[2 * i]     = iData[2 * i] > positiveFullScale ? iData[2 * i] - fullScaleRange : iData[2 * i];
        iData[2 * i + 1] = (int16_t)((rdBuff[i] & 0xFFFF0000) >> 16);
        iData[2 * i + 1] = iData[2 * i + 1] > positiveFullScale ? iData[2 * i + 1] - fullScaleRange : iData[2 * i + 1];
    }

#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    free(rdBuff);
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY

    ADI_API_RETURN(fpga9001);
}

static __maybe_unused int32_t adi_fpga9001_DataChain_Data_Get_15I15Q_GainChangeValidate(adi_fpga9001_Device_t *fpga9001,
                                                                         adi_common_Port_e port,
                                                                         adi_common_ChannelNumber_e channel,
                                                                         uint16_t iData[],
                                                                         uint16_t qData[],
                                                                         uint8_t gainChangeData[],
                                                                         uint32_t length)
{
    ADI_RANGE_CHECK(fpga9001, port, ADI_RX, ADI_ELB);
    ADI_RANGE_CHECK(fpga9001, channel, ADI_CHANNEL_1, ADI_CHANNEL_2);

    ADI_NULL_PTR_RETURN(&fpga9001->common, iData);
    ADI_NULL_PTR_RETURN(&fpga9001->common, qData);
    ADI_NULL_PTR_RETURN(&fpga9001->common, gainChangeData);

#ifndef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    ADI_RANGE_CHECK(fpga9001, length, 0, ADI_FPGA9001_MAX_CAPTURE_BYTES / 4)
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY
    
    ADI_API_RETURN(fpga9001);
}

int32_t adi_fpga9001_DataChain_Data_Get_15I15Q_GainChange(adi_fpga9001_Device_t *fpga9001,
                                                          adi_common_Port_e port,
                                                          adi_common_ChannelNumber_e channel,
                                                          uint16_t iData[],
                                                          uint16_t qData[],
                                                          uint8_t gainChangeData[],
                                                          uint32_t length)
{
    uint32_t i = 0;
    uint32_t ramAddress = 0;
    int32_t recoveryAction = ADI_COMMON_ACT_NO_ACTION;
#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    uint32_t *rdBuff = NULL;
#else
    uint32_t rdBuff[ADI_FPGA9001_MAX_CAPTURE_BYTES] = { 0 };
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY


    ADI_PERFORM_VALIDATION(adi_fpga9001_DataChain_Data_Get_15I15Q_GainChangeValidate, fpga9001, port, channel, iData, qData, gainChangeData, length);

#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    rdBuff = calloc(length, sizeof(uint32_t));
    if (rdBuff == NULL)
    {
        ADI_ERROR_REPORT(&fpga9001->common,
                         ADI_COMMON_ERRSRC_API,
                         ADI_COMMON_ERR_MEM_ALLOC_FAIL,
                         ADI_COMMON_ACT_ERR_RESET_FULL,
                         NULL,
                         "Unable to allocate memory");
        ADI_API_RETURN(fpga9001);
    }
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY

    ADI_EXPECT(adi_fpga9001_hal_wrapper_DmaRelativeAddress_Get, fpga9001, fpga9001_Dma_Get(fpga9001, port, channel), &ramAddress);
    if (ramAddress == (uint32_t) - 1)
    {
#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
        free(rdBuff);
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY
        ADI_API_RETURN(fpga9001);
    }

    recoveryAction = adi_fpga9001_hal_wrapper_Ram_Read(fpga9001, ramAddress, rdBuff, length);
    if (recoveryAction != ADI_COMMON_ACT_NO_ACTION)
    {
#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
        free(rdBuff);
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY
        ADI_API_RETURN(fpga9001);
    }

    for (i = 0; i < length; i++)
    {
        iData[i]          = (uint16_t)((rdBuff[i] & 0xFFFE0000) >> 16);
        qData[i]          = (uint16_t)((rdBuff[i] & 0x0000FFFE) >>  0);
        gainChangeData[i] = (uint16_t)((rdBuff[i] & 0x1)        >>  0);
    }

#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    free(rdBuff);
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY

    ADI_API_RETURN(fpga9001);
}

static __maybe_unused int32_t adi_fpga9001_DataChain_Data_Get_15I15QSigned_GainChangeValidate(adi_fpga9001_Device_t *fpga9001,
                                                                               adi_common_Port_e port,
                                                                               adi_common_ChannelNumber_e channel,
                                                                               int16_t iData[],
                                                                               int16_t qData[],
                                                                               uint8_t gainChangeData[],
                                                                               uint32_t length)
{
    ADI_RANGE_CHECK(fpga9001, port, ADI_RX, ADI_ELB);
    ADI_RANGE_CHECK(fpga9001, channel, ADI_CHANNEL_1, ADI_CHANNEL_2);

    ADI_NULL_PTR_RETURN(&fpga9001->common, iData);
    ADI_NULL_PTR_RETURN(&fpga9001->common, qData);
    ADI_NULL_PTR_RETURN(&fpga9001->common, gainChangeData);

#ifndef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    ADI_RANGE_CHECK(fpga9001, length, 0, ADI_FPGA9001_MAX_CAPTURE_BYTES / 4)
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY
    
    ADI_API_RETURN(fpga9001);
}

int32_t adi_fpga9001_DataChain_Data_Get_15I15QSigned_GainChange(adi_fpga9001_Device_t *fpga9001,
                                                                adi_common_Port_e port,
                                                                adi_common_ChannelNumber_e channel,
                                                                int16_t iData[],
                                                                int16_t qData[],
                                                                uint8_t gainChangeData[],
                                                                uint32_t length)
{
    uint32_t i = 0;
    uint32_t ramAddress = 0;
    int32_t recoveryAction = ADI_COMMON_ACT_NO_ACTION;
    uint32_t positiveFullScale = 0x8000;   // 2^15
    uint32_t fullScaleRange = 0x10000;     // 2^16
#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    uint32_t *rdBuff = NULL;
#else
    uint32_t rdBuff[ADI_FPGA9001_MAX_CAPTURE_BYTES] = { 0 };
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY


    ADI_PERFORM_VALIDATION(adi_fpga9001_DataChain_Data_Get_15I15QSigned_GainChangeValidate, fpga9001, port, channel, iData, qData, gainChangeData, length);

#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    rdBuff = calloc(length, sizeof(uint32_t));
    if (rdBuff == NULL)
    {
        ADI_ERROR_REPORT(&fpga9001->common,
                         ADI_COMMON_ERRSRC_API,
                         ADI_COMMON_ERR_MEM_ALLOC_FAIL,
                         ADI_COMMON_ACT_ERR_RESET_FULL,
                         NULL,
                         "Unable to allocate memory");
        ADI_API_RETURN(fpga9001);
    }
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY

    ADI_EXPECT(adi_fpga9001_hal_wrapper_DmaRelativeAddress_Get, fpga9001, fpga9001_Dma_Get(fpga9001, port, channel), &ramAddress);
    if (ramAddress == (uint32_t) - 1)
    {
#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
        free(rdBuff);
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY
        ADI_API_RETURN(fpga9001);
    }

    recoveryAction = adi_fpga9001_hal_wrapper_Ram_Read(fpga9001, ramAddress, rdBuff, length);
    if (recoveryAction != ADI_COMMON_ACT_NO_ACTION)
    {
#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
        free(rdBuff);
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY
        ADI_API_RETURN(fpga9001);
    }

    for (i = 0; i < length; i++)
    {
        iData[i]          = (int16_t)((rdBuff[i] & 0xFFFE0000) >> 16);
        iData[i]          = iData[i] > positiveFullScale ? iData[i] - fullScaleRange : iData[i];
        qData[i]          = (int16_t)((rdBuff[i] & 0x0000FFFE) >>  0);
        qData[i]          = qData[i] > positiveFullScale ? qData[i] - fullScaleRange : qData[i];
        gainChangeData[i] = (uint16_t)((rdBuff[i] & 0x1)        >>  0);
    }

#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    free(rdBuff);
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY

    ADI_API_RETURN(fpga9001);
}

static __maybe_unused int32_t adi_fpga9001_DataChain_Data_Get_15IQInterleaved_GainChangeValidate(adi_fpga9001_Device_t *fpga9001,
                                                                                    adi_common_Port_e port,
                                                                                    adi_common_ChannelNumber_e channel,
                                                                                    uint16_t iqData[],
                                                                                    uint8_t gainChangeData[],
                                                                                    uint32_t length)
{
    ADI_RANGE_CHECK(fpga9001, port, ADI_RX, ADI_ORX);
    ADI_RANGE_CHECK(fpga9001, channel, ADI_CHANNEL_1, ADI_CHANNEL_2);

    ADI_NULL_PTR_RETURN(&fpga9001->common, iqData);
    ADI_NULL_PTR_RETURN(&fpga9001->common, gainChangeData);

#ifndef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    ADI_RANGE_CHECK(fpga9001, length, 0, ADI_FPGA9001_MAX_CAPTURE_BYTES / 2)
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY

    ADI_API_RETURN(fpga9001);
}

int32_t adi_fpga9001_DataChain_Data_Get_15IQInterleaved_GainChange(adi_fpga9001_Device_t *fpga9001,
                                                                     adi_common_Port_e port,
                                                                     adi_common_ChannelNumber_e channel,
                                                                     uint16_t iqData[],
                                                                     uint8_t gainChangeData[],
                                                                     uint32_t length)
{
    uint32_t i = 0;
    uint32_t ramAddress = 0;
#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    uint32_t *rdBuff = NULL;
#else
    uint32_t rdBuff[ADI_FPGA9001_MAX_CAPTURE_BYTES] = { 0 };
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY

    ADI_PERFORM_VALIDATION(adi_fpga9001_DataChain_Data_Get_15IQInterleaved_GainChangeValidate, fpga9001, port, channel, iqData, gainChangeData, length);

#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    rdBuff = calloc(length / 2, sizeof(uint32_t));
    if (rdBuff == NULL)
    {
        ADI_ERROR_REPORT(&fpga9001->common,
                         ADI_COMMON_ERRSRC_API,
                         ADI_COMMON_ERR_MEM_ALLOC_FAIL,
                         ADI_COMMON_ACT_ERR_RESET_FULL,
                         NULL,
                         "Unable to allocate memory");
        ADI_API_RETURN(fpga9001);
    }
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY

    ADI_EXPECT(adi_fpga9001_hal_wrapper_DmaRelativeAddress_Get, fpga9001, fpga9001_Dma_Get(fpga9001, port, channel), &ramAddress);
    if (ramAddress == (uint32_t) - 1)
    {
#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
        free(rdBuff);
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY
        ADI_API_RETURN(fpga9001);
    }

    if (adi_fpga9001_hal_wrapper_Ram_Read(fpga9001, ramAddress, rdBuff, length / 2) != ADI_COMMON_ACT_NO_ACTION)
    {
#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
        free(rdBuff);
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY
        ADI_API_RETURN(fpga9001);
    }

    for (i = 0; i < length / 2; i++)
    {
        iqData[2 * i + 0]         = (uint16_t)((rdBuff[i] & 0xFFFE0000) >> 16);
        iqData[2 * i + 1]         = (uint16_t)((rdBuff[i] & 0x0000FFFE) >>  0);
        gainChangeData[2 * i + 0] = (uint16_t)((rdBuff[i] & 0x1)        >>  0);
        gainChangeData[2 * i + 1] = gainChangeData[2 * i + 0];
    }

#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    free(rdBuff);
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY

    ADI_API_RETURN(fpga9001);
}

static __maybe_unused int32_t adi_fpga9001_DataChain_Data_Get_15IQInterleavedSigned_GainChangeValidate(adi_fpga9001_Device_t *fpga9001,
                                                                                          adi_common_Port_e port,
                                                                                          adi_common_ChannelNumber_e channel,
                                                                                          int16_t iqData[],
                                                                                          uint8_t gainChangeData[],
                                                                                          uint32_t length)
{
    ADI_RANGE_CHECK(fpga9001, port, ADI_RX, ADI_ORX);
    ADI_RANGE_CHECK(fpga9001, channel, ADI_CHANNEL_1, ADI_CHANNEL_2);

    ADI_NULL_PTR_RETURN(&fpga9001->common, iqData);
    ADI_NULL_PTR_RETURN(&fpga9001->common, gainChangeData);

#ifndef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    ADI_RANGE_CHECK(fpga9001, length, 0, ADI_FPGA9001_MAX_CAPTURE_BYTES / 2)
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY

    ADI_API_RETURN(fpga9001);
}

static __maybe_unused int32_t adi_fpga9001_DataChain_Data_Get_12I12QValidate(adi_fpga9001_Device_t *fpga9001,
                                                              adi_common_Port_e port,
                                                              adi_common_ChannelNumber_e channel,
                                                              uint16_t *iData,
                                                              uint16_t *qData,
                                                              uint32_t length)
{
    ADI_RANGE_CHECK(fpga9001, port, ADI_RX, ADI_ELB);
    ADI_RANGE_CHECK(fpga9001, channel, ADI_CHANNEL_1, ADI_CHANNEL_2);

    ADI_NULL_PTR_RETURN(&fpga9001->common, iData);
    ADI_NULL_PTR_RETURN(&fpga9001->common, qData);

#ifndef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    ADI_RANGE_CHECK(fpga9001, length, 0, ADI_FPGA9001_MAX_CAPTURE_BYTES / 4)
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY
    
    ADI_API_RETURN(fpga9001);
}

int32_t adi_fpga9001_DataChain_Data_Get_12I12Q(adi_fpga9001_Device_t *fpga9001,
                                                           adi_common_Port_e port,
                                                           adi_common_ChannelNumber_e channel,
                                                           uint16_t *iData,
                                                           uint16_t *qData,
                                                           uint32_t length)
{
    uint32_t i = 0;
    uint32_t ramAddress = 0;
    int32_t recoveryAction = ADI_COMMON_ACT_NO_ACTION;
#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    uint32_t *rdBuff = NULL;
#else
    uint32_t rdBuff[ADI_FPGA9001_MAX_CAPTURE_BYTES] = { 0 };
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY


    ADI_PERFORM_VALIDATION(adi_fpga9001_DataChain_Data_Get_12I12QValidate, fpga9001, port, channel, iData, qData, length);

#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    rdBuff = calloc(length, sizeof(uint32_t));
    if (rdBuff == NULL)
    {
        ADI_ERROR_REPORT(&fpga9001->common,
                         ADI_COMMON_ERRSRC_API,
                         ADI_COMMON_ERR_MEM_ALLOC_FAIL,
                         ADI_COMMON_ACT_ERR_RESET_FULL,
                         NULL,
                         "Unable to allocate memory");
        ADI_API_RETURN(fpga9001);
    }
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY

    ADI_EXPECT(adi_fpga9001_hal_wrapper_DmaRelativeAddress_Get, fpga9001, fpga9001_Dma_Get(fpga9001, port, channel), &ramAddress);
    if (ramAddress == (uint32_t) -1)
    {
#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
        free(rdBuff);
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY
        ADI_API_RETURN(fpga9001);
    }

    recoveryAction = adi_fpga9001_hal_wrapper_Ram_Read(fpga9001, ramAddress, rdBuff, length);
    if (recoveryAction != ADI_COMMON_ACT_NO_ACTION)
    {
#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
        free(rdBuff);
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY
        ADI_API_RETURN(fpga9001);
    }

    for (i = 0; i < length; i++)
    {
        iData[i] = (uint16_t)((rdBuff[i] & 0x0FFF0000) >> 16);
        qData[i] = (uint16_t)(rdBuff[i]  & 0x00000FFF);
    }

#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    free(rdBuff);
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY

    ADI_API_RETURN(fpga9001);
}

int32_t adi_fpga9001_DataChain_Data_Get_12I12QSigned(adi_fpga9001_Device_t *fpga9001,
                                                           adi_common_Port_e port,
                                                           adi_common_ChannelNumber_e channel,
                                                           int16_t *iData,
                                                           int16_t *qData,
                                                           uint32_t length)
{
    uint32_t i = 0;
    uint32_t ramAddress = 0;
    int32_t recoveryAction = ADI_COMMON_ACT_NO_ACTION;
#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    uint32_t *rdBuff = NULL;
#else
    uint32_t rdBuff[ADI_FPGA9001_MAX_CAPTURE_BYTES] = { 0 };
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY


    ADI_PERFORM_VALIDATION(adi_fpga9001_DataChain_Data_Get_12I12QValidate, fpga9001, port, channel, (uint16_t*)iData, (uint16_t*)qData, length);

#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    rdBuff = calloc(length, sizeof(uint32_t));
    if (rdBuff == NULL)
    {
        ADI_ERROR_REPORT(&fpga9001->common,
                         ADI_COMMON_ERRSRC_API,
                         ADI_COMMON_ERR_MEM_ALLOC_FAIL,
                         ADI_COMMON_ACT_ERR_RESET_FULL,
                         NULL,
                         "Unable to allocate memory");
        ADI_API_RETURN(fpga9001);
    }
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY

    ADI_EXPECT(adi_fpga9001_hal_wrapper_DmaRelativeAddress_Get, fpga9001, fpga9001_Dma_Get(fpga9001, port, channel), &ramAddress);
    if (ramAddress == (uint32_t) -1)
    {
#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
        free(rdBuff);
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY
        ADI_API_RETURN(fpga9001);
    }

    recoveryAction = adi_fpga9001_hal_wrapper_Ram_Read(fpga9001, ramAddress, rdBuff, length);
    if (recoveryAction != ADI_COMMON_ACT_NO_ACTION)
    {
#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
        free(rdBuff);
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY
        ADI_API_RETURN(fpga9001);
    }

    for (i = 0; i < length; i++)
    {
        iData[i] = (int16_t)((rdBuff[i] & 0x0FFF0000) >> 12);
        iData[i] >>= 4;
        qData[i] = (int16_t)(rdBuff[i]  & 0x00000FFF);
        qData[i] <<= 4;
        qData[i] >>= 4;
    }

#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    free(rdBuff);
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY

    ADI_API_RETURN(fpga9001);
}

static __maybe_unused int32_t adi_fpga9001_DataChain_Data_Get_12IQInterleavedValidate(adi_fpga9001_Device_t *fpga9001,
                                                                       adi_common_Port_e port,
                                                                       adi_common_ChannelNumber_e channel,
                                                                       uint16_t *iqData,
                                                                       uint32_t length)
{
    ADI_RANGE_CHECK(fpga9001, port, ADI_RX, ADI_ORX);
    ADI_RANGE_CHECK(fpga9001, channel, ADI_CHANNEL_1, ADI_CHANNEL_2);

    ADI_NULL_PTR_RETURN(&fpga9001->common, iqData);

#ifndef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    ADI_RANGE_CHECK(fpga9001, length, 0, ADI_FPGA9001_MAX_CAPTURE_BYTES / 2)
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY

    ADI_API_RETURN(fpga9001);
}

int32_t adi_fpga9001_DataChain_Data_Get_12IQInterleaved(adi_fpga9001_Device_t *fpga9001,
                                                                    adi_common_Port_e port,
                                                                    adi_common_ChannelNumber_e channel,
                                                                    uint16_t *iqData,
                                                                    uint32_t length)
{
    uint32_t i = 0;
    uint32_t ramAddress = 0;
#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    uint32_t *rdBuff = NULL;
#else
    uint32_t rdBuff[ADI_FPGA9001_MAX_CAPTURE_BYTES] = { 0 };
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY

    ADI_PERFORM_VALIDATION(adi_fpga9001_DataChain_Data_Get_12IQInterleavedValidate, fpga9001, port, channel, iqData, length);

#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    rdBuff = calloc(length / 2, sizeof(uint32_t));
    if (rdBuff == NULL)
    {
        ADI_ERROR_REPORT(&fpga9001->common,
                         ADI_COMMON_ERRSRC_API,
                         ADI_COMMON_ERR_MEM_ALLOC_FAIL,
                         ADI_COMMON_ACT_ERR_RESET_FULL,
                         NULL,
                         "Unable to allocate memory");
        ADI_API_RETURN(fpga9001);
    }
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY

    ADI_EXPECT(adi_fpga9001_hal_wrapper_DmaRelativeAddress_Get, fpga9001, fpga9001_Dma_Get(fpga9001, port, channel), &ramAddress);
    if (ramAddress == (uint32_t) -1)
    {
#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
        free(rdBuff);
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY
        ADI_API_RETURN(fpga9001);
    }

    if (adi_fpga9001_hal_wrapper_Ram_Read(fpga9001, ramAddress, rdBuff, length / 2) != ADI_COMMON_ACT_NO_ACTION)
    {
#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
        free(rdBuff);
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY
        ADI_API_RETURN(fpga9001);
    }

    for (i = 0; i < length / 2; i++)
    {
        iqData[2 * i]     = (uint16_t)((rdBuff[i] & 0x0FFF0000) >> 16);
        iqData[2 * i + 1] = (uint16_t)(rdBuff[i]  & 0x00000FFF);
    }

#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    free(rdBuff);
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY

    ADI_API_RETURN(fpga9001);
}

int32_t adi_fpga9001_DataChain_Data_Get_12IQInterleavedSigned(adi_fpga9001_Device_t *fpga9001,
                                                                    adi_common_Port_e port,
                                                                    adi_common_ChannelNumber_e channel,
                                                                    int16_t *iqData,
                                                                    uint32_t length)
{
    uint32_t i = 0;
    uint32_t ramAddress = 0;
#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    uint32_t *rdBuff = NULL;
#else
    uint32_t rdBuff[ADI_FPGA9001_MAX_CAPTURE_BYTES] = { 0 };
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY

    ADI_PERFORM_VALIDATION(adi_fpga9001_DataChain_Data_Get_12IQInterleavedValidate, fpga9001, port, channel, (uint16_t*)iqData, length);

#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    rdBuff = calloc(length / 2, sizeof(uint32_t));
    if (rdBuff == NULL)
    {
        ADI_ERROR_REPORT(&fpga9001->common,
                         ADI_COMMON_ERRSRC_API,
                         ADI_COMMON_ERR_MEM_ALLOC_FAIL,
                         ADI_COMMON_ACT_ERR_RESET_FULL,
                         NULL,
                         "Unable to allocate memory");
        ADI_API_RETURN(fpga9001);
    }
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY

    ADI_EXPECT(adi_fpga9001_hal_wrapper_DmaRelativeAddress_Get, fpga9001, fpga9001_Dma_Get(fpga9001, port, channel), &ramAddress);
    if (ramAddress == (uint32_t) -1)
    {
#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
        free(rdBuff);
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY
        ADI_API_RETURN(fpga9001);
    }

    if (adi_fpga9001_hal_wrapper_Ram_Read(fpga9001, ramAddress, rdBuff, length / 2) != ADI_COMMON_ACT_NO_ACTION)
    {
#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
        free(rdBuff);
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY
        ADI_API_RETURN(fpga9001);
    }

    for (i = 0; i < length / 2; i++)
    {
        iqData[2 * i]     = (int16_t)((rdBuff[i] & 0x0FFF0000) >> 12);
        iqData[2 * i]     >>= 4;
        iqData[2 * i + 1] = (int16_t)(rdBuff[i]  & 0x00000FFF);
        iqData[2 * i]     <<= 4;
        iqData[2 * i]     >>= 4;
    }

#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    free(rdBuff);
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY

    ADI_API_RETURN(fpga9001);
}

int32_t adi_fpga9001_DataChain_Data_Get_15IQInterleavedSigned_GainChange(adi_fpga9001_Device_t *fpga9001,
                                                                           adi_common_Port_e port,
                                                                           adi_common_ChannelNumber_e channel,
                                                                           int16_t iqData[],
                                                                           uint8_t gainChangeData[],
                                                                           uint32_t length)
{
    uint32_t i = 0;
    uint32_t ramAddress = 0;
    uint32_t positiveFullScale = 0x8000;    // 2^15
    uint32_t fullScaleRange = 0x10000;      // 2^16
#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    uint32_t *rdBuff = NULL;
#else
    uint32_t rdBuff[ADI_FPGA9001_MAX_CAPTURE_BYTES] = { 0 };
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY

    ADI_PERFORM_VALIDATION(adi_fpga9001_DataChain_Data_Get_15IQInterleavedSigned_GainChangeValidate, fpga9001, port, channel, iqData, gainChangeData, length);

#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    rdBuff = calloc(length / 2, sizeof(uint32_t));
    if (rdBuff == NULL)
    {
        ADI_ERROR_REPORT(&fpga9001->common,
                         ADI_COMMON_ERRSRC_API,
                         ADI_COMMON_ERR_MEM_ALLOC_FAIL,
                         ADI_COMMON_ACT_ERR_RESET_FULL,
                         NULL,
                         "Unable to allocate memory");
        ADI_API_RETURN(fpga9001);
    }
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY

    ADI_EXPECT(adi_fpga9001_hal_wrapper_DmaRelativeAddress_Get, fpga9001, fpga9001_Dma_Get(fpga9001, port, channel), &ramAddress);
    if (ramAddress == (uint32_t) - 1)
    {
#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
        free(rdBuff);
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY
        ADI_API_RETURN(fpga9001);
    }

    if (adi_fpga9001_hal_wrapper_Ram_Read(fpga9001, ramAddress, rdBuff, length / 2) != ADI_COMMON_ACT_NO_ACTION)
    {
#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
        free(rdBuff);
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY
        ADI_API_RETURN(fpga9001);
    }

    for (i = 0; i < length / 2; i++)
    {
        iqData[2 * i + 0]         = (int16_t)((rdBuff[i] & 0xFFFE0000) >> 16);
        iqData[2 * i + 0]         = iqData[2 * i + 0] > positiveFullScale ? iqData[2 * i + 0] - fullScaleRange : iqData[2 * i + 0];
        iqData[2 * i + 1]         = (int16_t)((rdBuff[i] & 0x0000FFFE) >>  0);
        iqData[2 * i + 1]         = iqData[2 * i + 1] > positiveFullScale ? iqData[2 * i + 1] - fullScaleRange : iqData[2 * i + 1];
        gainChangeData[2 * i + 0] = (uint16_t)((rdBuff[i] & 0x1)        >>  0);
        gainChangeData[2 * i + 1] = gainChangeData[2 * i + 0];
    }

#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    free(rdBuff);
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY

    ADI_API_RETURN(fpga9001);
}

static __maybe_unused int32_t adi_fpga9001_DataGet_SValidate(adi_fpga9001_Device_t *fpga9001,
                                              adi_common_Port_e port,
                                              adi_common_ChannelNumber_e channel,
                                              uint8_t *symbols,
                                              uint32_t length)
{
    ADI_RANGE_CHECK(fpga9001, port, ADI_RX, ADI_ORX);
    ADI_RANGE_CHECK(fpga9001, channel, ADI_CHANNEL_1, ADI_CHANNEL_2);

    ADI_NULL_PTR_RETURN(&fpga9001->common, symbols);
    
#ifndef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    ADI_RANGE_CHECK(fpga9001, length, 0, ADI_FPGA9001_MAX_CAPTURE_BYTES)
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY

    ADI_API_RETURN(fpga9001);
}

static __maybe_unused int32_t adi_fpga9001_GainIndexGet_SValidate(adi_fpga9001_Device_t *fpga9001,
                                                   adi_common_Port_e port,
                                                   adi_common_ChannelNumber_e channel,
                                                   uint8_t *gainChangeBit,
                                                   uint8_t *symbols,
                                                   uint32_t length)
{
    ADI_RANGE_CHECK(fpga9001, port, ADI_RX, ADI_ORX);
    ADI_RANGE_CHECK(fpga9001, channel, ADI_CHANNEL_1, ADI_CHANNEL_2);
    ADI_NULL_PTR_RETURN(&fpga9001->common, gainChangeBit);
    ADI_NULL_PTR_RETURN(&fpga9001->common, symbols);
    
#ifndef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    ADI_RANGE_CHECK(fpga9001, length, 0, ADI_FPGA9001_MAX_CAPTURE_BYTES)
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY
    ADI_API_RETURN(fpga9001);
}

int32_t adi_fpga9001_DataChain_GainIndex_Get_8S(adi_fpga9001_Device_t *fpga9001,
                                                adi_common_ChannelNumber_e channel,
                                                uint8_t *gainChangeBit,
                                                uint8_t *symbols,
                                                uint32_t length)
{
    uint32_t i = 0;
    uint32_t ramAddress = 0;
#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    uint32_t *rdBuff = NULL;
#else
    uint32_t rdBuff[ADI_FPGA9001_MAX_CAPTURE_BYTES] = { 0 };
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY

    ADI_PERFORM_VALIDATION(adi_fpga9001_GainIndexGet_SValidate, fpga9001, ADI_RX, channel, gainChangeBit, symbols, length);

#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    rdBuff = calloc(length / 4, sizeof(uint32_t));
    if (rdBuff == NULL)
    {
        ADI_ERROR_REPORT(&fpga9001->common,
                         ADI_COMMON_ERRSRC_API,
                         ADI_COMMON_ERR_MEM_ALLOC_FAIL,
                         ADI_COMMON_ACT_ERR_RESET_FULL,
                         NULL,
                         "Unable to allocate memory");
        ADI_API_RETURN(fpga9001);
    }
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY

    ADI_EXPECT(adi_fpga9001_hal_wrapper_DmaRelativeAddress_Get, fpga9001, fpga9001_Dma_Get(fpga9001, ADI_RX, channel), &ramAddress);
    if (ramAddress == (uint32_t) - 1)
    {
#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
        free(rdBuff);
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY
        ADI_API_RETURN(fpga9001);
    }

    if (adi_fpga9001_hal_wrapper_Ram_Read(fpga9001, ramAddress, rdBuff, length / 4) != ADI_COMMON_ACT_NO_ACTION)
    {
#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
        free(rdBuff);
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY
        ADI_API_RETURN(fpga9001);
    }

    for (i = 0; i < length / 4; i++)
    {
        symbols[4 * i]       = (uint8_t)((rdBuff[i] & 0x000000FF) | 0x80);
        gainChangeBit[4 * i] = (uint8_t)((rdBuff[i] & 0x80) >> 7);

        symbols[4 * i + 1]       = (uint8_t)(((rdBuff[i] & 0x0000FF00) >> 8) | 0x80);
        gainChangeBit[4 * i + 1] = (uint8_t)((rdBuff[i] & 0x00008000) >> 15);

        symbols[4 * i + 2]       = (uint8_t)(((rdBuff[i] & 0x00FF0000) >> 16) | 0x80);
        gainChangeBit[4 * i + 2] = (uint8_t)((rdBuff[i] & 0x00800000) >> 23);

        symbols[4 * i + 3]       = (uint8_t)(((rdBuff[i] & 0xFF000000) >> 24) | 0x80);
        gainChangeBit[4 * i + 3] = (uint8_t)((rdBuff[i] & 0x80000000) >> 31);
    }

#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    free(rdBuff);
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY

    ADI_API_RETURN(fpga9001);
}

int32_t adi_fpga9001_DataChain_Data_Get_8S(adi_fpga9001_Device_t *fpga9001,
                                           adi_common_Port_e port,
                                           adi_common_ChannelNumber_e channel,
                                           uint8_t *symbols,
                                           uint32_t length)
{
    uint32_t i = 0;
    uint32_t ramAddress = 0;
#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    uint32_t *rdBuff = NULL;
#else
    uint32_t rdBuff[ADI_FPGA9001_MAX_CAPTURE_BYTES] = { 0 };
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY

    ADI_PERFORM_VALIDATION(adi_fpga9001_DataGet_SValidate, fpga9001, port, channel, symbols, length);

#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    rdBuff = calloc(length / 4, sizeof(uint32_t));
    if (rdBuff == NULL)
    {
        ADI_ERROR_REPORT(&fpga9001->common,
                         ADI_COMMON_ERRSRC_API,
                         ADI_COMMON_ERR_MEM_ALLOC_FAIL,
                         ADI_COMMON_ACT_ERR_RESET_FULL,
                         NULL,
                         "Unable to allocate memory");
        ADI_API_RETURN(fpga9001);
    }
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY

    ADI_EXPECT(adi_fpga9001_hal_wrapper_DmaRelativeAddress_Get, fpga9001, fpga9001_Dma_Get(fpga9001, port, channel), &ramAddress);
    if (ramAddress == (uint32_t) - 1)
    {
#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
        free(rdBuff);
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY
        ADI_API_RETURN(fpga9001);
    }

    if (adi_fpga9001_hal_wrapper_Ram_Read(fpga9001, ramAddress, rdBuff, length / 4) != ADI_COMMON_ACT_NO_ACTION)
    {
#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
        free(rdBuff);
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY
        ADI_API_RETURN(fpga9001);
    }

    for (i = 0; i < length / 4; i++)
    {
        symbols[4 * i]     = (uint8_t)(rdBuff[i]  & 0x000000FF);
        symbols[4 * i + 1] = (uint8_t)((rdBuff[i] & 0x0000FF00) >> 8);
        symbols[4 * i + 2] = (uint8_t)((rdBuff[i] & 0x00FF0000) >> 16);
        symbols[4 * i + 3] = (uint8_t)((rdBuff[i] & 0xFF000000) >> 24);
    }

#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    free(rdBuff);
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY

    ADI_API_RETURN(fpga9001);
}

int32_t adi_fpga9001_DataChain_Data_Get_2S(adi_fpga9001_Device_t *fpga9001,
                                           adi_common_Port_e port,
                                           adi_common_ChannelNumber_e channel,
                                           uint8_t *symbols,
                                           uint32_t length)
{
    uint32_t i = 0;
    uint32_t ramAddress = 0;
#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    uint32_t *rdBuff = NULL;
#else
    uint32_t rdBuff[ADI_FPGA9001_MAX_CAPTURE_BYTES] = { 0 };
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY

    ADI_PERFORM_VALIDATION(adi_fpga9001_DataGet_SValidate, fpga9001, port, channel, symbols, length);

#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    rdBuff = calloc(length / 4, sizeof(uint32_t));
    if (rdBuff == NULL)
    {
        ADI_ERROR_REPORT(&fpga9001->common,
                         ADI_COMMON_ERRSRC_API,
                         ADI_COMMON_ERR_MEM_ALLOC_FAIL,
                         ADI_COMMON_ACT_ERR_RESET_FULL,
                         NULL,
                         "Unable to allocate memory");
        ADI_API_RETURN(fpga9001);
    }
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY

    ADI_EXPECT(adi_fpga9001_hal_wrapper_DmaRelativeAddress_Get, fpga9001, fpga9001_Dma_Get(fpga9001, port, channel), &ramAddress);
    if (ramAddress == (uint32_t) - 1)
    {
#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
        free(rdBuff);
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY
        ADI_API_RETURN(fpga9001);
    }

    if (adi_fpga9001_hal_wrapper_Ram_Read(fpga9001, ramAddress, rdBuff, length / 4) != ADI_COMMON_ACT_NO_ACTION)
    {
#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
        free(rdBuff);
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY
        ADI_API_RETURN(fpga9001);
    }

    for (i = 0; i < length / 4; i++)
    {
        symbols[4 * i]     = (uint8_t)(rdBuff[i]  & 0x00000003);
        symbols[4 * i + 1] = (uint8_t)((rdBuff[i] & 0x00000300) >> 8);
        symbols[4 * i + 2] = (uint8_t)((rdBuff[i] & 0x00030000) >> 16);
        symbols[4 * i + 3] = (uint8_t)((rdBuff[i] & 0x03000000) >> 24);
    }

#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    free(rdBuff);
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY

    ADI_API_RETURN(fpga9001);
}

static __maybe_unused int32_t adi_fpga9001_DataChain_DmaLength_Set(adi_fpga9001_Device_t *fpga9001,
                                                    adi_common_Port_e port,
                                                    adi_common_ChannelNumber_e channel,
                                                    uint32_t numBytes)
{
    uint32_t dmaId;

    ADI_NULL_DEVICE_PTR_RETURN(fpga9001);
    ADI_RANGE_CHECK(fpga9001, port, ADI_RX, ADI_ORX);
    ADI_RANGE_CHECK(fpga9001, channel, ADI_CHANNEL_1, ADI_CHANNEL_2);

    if (numBytes % 8 != 0)
    {
        ADI_ERROR_REPORT(&fpga9001->common, ADI_COMMON_ERRSRC_API, ADI_COMMON_ERR_INV_PARAM,
            ADI_COMMON_ACT_ERR_CHECK_PARAM, numBytes,
            "Invalid parameter value, numBytes must be a multiple of 8.");
        ADI_ERROR_RETURN(fpga9001->common.error.newAction);
    }

    dmaId = fpga9001_DMAIdGet(fpga9001, port, channel);
    axi_dma_top_length_set((void *)fpga9001, dmaId, AXI_DMA_TOP_OFFSET, numBytes);
    ADI_API_RETURN(fpga9001);
}

int32_t adi_fpga9001_DataChain_Data_Set_16I16Q(adi_fpga9001_Device_t *fpga9001,
                                               adi_common_Port_e port,
                                               adi_common_ChannelNumber_e channel,
                                               uint16_t *iData,
                                               uint16_t *qData,
                                               uint32_t length)
{
    uint32_t i = 0;
    uint32_t ramAddress = 0;
#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    uint32_t *wrBuff = NULL;
#else
    uint32_t wrBuff[ADI_FPGA9001_MAX_CAPTURE_BYTES] = { 0 };
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY

    ADI_PERFORM_VALIDATION(adi_fpga9001_DataChain_Data_Get_16I16QValidate, fpga9001, port, channel, iData, qData, length);

    ADI_EXPECT(adi_fpga9001_DataChain_DmaLength_Set, fpga9001, port, channel, length * 4);

#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    wrBuff = calloc(length, sizeof(uint32_t));
    if (wrBuff == NULL)
    {
        ADI_ERROR_REPORT(&fpga9001->common,
                         ADI_COMMON_ERRSRC_API,
                         ADI_COMMON_ERR_MEM_ALLOC_FAIL,
                         ADI_COMMON_ACT_ERR_RESET_FULL,
                         NULL,
                         "Unable to allocate memory");
        ADI_API_RETURN(fpga9001);
    }
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY

    ADI_EXPECT(adi_fpga9001_hal_wrapper_DmaRelativeAddress_Get, fpga9001, fpga9001_Dma_Get(fpga9001, port, channel), &ramAddress);
    if (ramAddress == (uint32_t) - 1)
    {
#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
        free(wrBuff);
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY
        ADI_API_RETURN(fpga9001);
    }

    for (i = 0; i < length; i++)
    {
        wrBuff[i] = (uint32_t)(iData[i] << 16) | (uint32_t)qData[i];
    }

    adi_fpga9001_hal_wrapper_Ram_Write(fpga9001, ramAddress, wrBuff, length);

#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    free(wrBuff);
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY

    ADI_API_RETURN(fpga9001);
}

int32_t adi_fpga9001_DataChain_Data_Set_16IQInterleaved(adi_fpga9001_Device_t *fpga9001,
                                                        adi_common_Port_e port,
                                                        adi_common_ChannelNumber_e channel,
                                                        uint16_t *iqData,
                                                        uint32_t length)
{
    uint32_t i = 0;
    uint32_t ramAddress = 0;
#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    uint32_t *wrBuff = NULL;
#else
    uint32_t wrBuff[ADI_FPGA9001_MAX_CAPTURE_BYTES] = { 0 };
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY

    ADI_PERFORM_VALIDATION(adi_fpga9001_DataChain_Data_Get_16IQInterleavedValidate, fpga9001, port, channel, iqData, length);

    ADI_EXPECT(adi_fpga9001_DataChain_DmaLength_Set, fpga9001, port, channel, length * 2);

#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    wrBuff = calloc(length / 2, sizeof(uint32_t));
    if (wrBuff == NULL)
    {
        ADI_ERROR_REPORT(&fpga9001->common,
                         ADI_COMMON_ERRSRC_API,
                         ADI_COMMON_ERR_MEM_ALLOC_FAIL,
                         ADI_COMMON_ACT_ERR_RESET_FULL,
                         NULL,
                         "Unable to allocate memory");
        ADI_API_RETURN(fpga9001);
    }
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY

    ADI_EXPECT(adi_fpga9001_hal_wrapper_DmaRelativeAddress_Get, fpga9001, fpga9001_Dma_Get(fpga9001, port, channel), &ramAddress);
    if (ramAddress == (uint32_t) - 1)
    {
#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
        free(wrBuff);
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY
        ADI_API_RETURN(fpga9001);
    }

    for (i = 0; i < length / 2; i++)
    {
        wrBuff[i] = (uint32_t)(iqData[2 * i] << 16) | (uint32_t)iqData[2 * i + 1];
        iqData[2 * i]     = (uint16_t)((wrBuff[i] & 0xFFFF0000) >> 16);
        iqData[2 * i + 1] = (uint16_t)(wrBuff[i]  & 0x0000FFFF);
    }

    adi_fpga9001_hal_wrapper_Ram_Write(fpga9001, ramAddress, wrBuff, length / 2);

#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    free(wrBuff);
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY

    ADI_API_RETURN(fpga9001);
}

int32_t adi_fpga9001_DataChain_Data_Set_16I(adi_fpga9001_Device_t *fpga9001,
                                            adi_common_Port_e port,
                                            adi_common_ChannelNumber_e channel,
                                            uint16_t *iData,
                                            uint32_t length)
{
    uint32_t i = 0;
    uint32_t ramAddress = 0;
#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    uint32_t *wrBuff = NULL;
#else
    uint32_t wrBuff[ADI_FPGA9001_MAX_CAPTURE_BYTES] = { 0 };
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY

    ADI_PERFORM_VALIDATION(adi_fpga9001_DataChain_Data_Get_16IValidate, fpga9001, port, channel, iData, length);

    ADI_EXPECT(adi_fpga9001_DataChain_DmaLength_Set, fpga9001, port, channel, length * 2);

#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    wrBuff = calloc(length / 2, sizeof(uint32_t));
    if (wrBuff == NULL)
    {
        ADI_ERROR_REPORT(&fpga9001->common,
                         ADI_COMMON_ERRSRC_API,
                         ADI_COMMON_ERR_MEM_ALLOC_FAIL,
                         ADI_COMMON_ACT_ERR_RESET_FULL,
                         NULL,
                         "Unable to allocate memory");
        ADI_API_RETURN(fpga9001);
    }
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY

    ADI_EXPECT(adi_fpga9001_hal_wrapper_DmaRelativeAddress_Get, fpga9001, fpga9001_Dma_Get(fpga9001, port, channel), &ramAddress);
    if (ramAddress == (uint32_t) - 1)
    {
#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
        free(wrBuff);
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY
        ADI_API_RETURN(fpga9001);
    }

    for (i = 0; i < length / 2; i++)
    {
        wrBuff[i] = (uint32_t)(iData[2 * i + 1] << 16) | (uint32_t)iData[2 * i];
    }

    adi_fpga9001_hal_wrapper_Ram_Write(fpga9001, ramAddress, wrBuff, length / 2);

#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    free(wrBuff);
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY

    ADI_API_RETURN(fpga9001);
}

int32_t adi_fpga9001_DataChain_Data_Set_12I12Q(adi_fpga9001_Device_t *fpga9001,
                                               adi_common_Port_e port,
                                               adi_common_ChannelNumber_e channel,
                                               uint16_t *iData,
                                               uint16_t *qData,
                                               uint32_t length)
{
    uint32_t i = 0;
    uint32_t ramAddress = 0;
#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    uint32_t *wrBuff = NULL;
#else
    uint32_t wrBuff[ADI_FPGA9001_MAX_CAPTURE_BYTES] = { 0 };
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY

    ADI_PERFORM_VALIDATION(adi_fpga9001_DataChain_Data_Get_12I12QValidate, fpga9001, port, channel, iData, qData, length);

    ADI_EXPECT(adi_fpga9001_DataChain_DmaLength_Set, fpga9001, port, channel, length * 4);

#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    wrBuff = calloc(length, sizeof(uint32_t));
    if (wrBuff == NULL)
    {
        ADI_ERROR_REPORT(&fpga9001->common,
                         ADI_COMMON_ERRSRC_API,
                         ADI_COMMON_ERR_MEM_ALLOC_FAIL,
                         ADI_COMMON_ACT_ERR_RESET_FULL,
                         NULL,
                         "Unable to allocate memory");
        ADI_API_RETURN(fpga9001);
    }
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY

    ADI_EXPECT(adi_fpga9001_hal_wrapper_DmaRelativeAddress_Get, fpga9001, fpga9001_Dma_Get(fpga9001, port, channel), &ramAddress);
    if (ramAddress == (uint32_t) - 1)
    {
#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
        free(wrBuff);
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY
        ADI_API_RETURN(fpga9001);
    }

    for (i = 0; i < length; i++)
    {
        wrBuff[i] = (uint32_t)(iData[i] << 16) | (uint32_t)qData[i];
    }

    adi_fpga9001_hal_wrapper_Ram_Write(fpga9001, ramAddress, wrBuff, length);

#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    free(wrBuff);
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY

    ADI_API_RETURN(fpga9001);
}

int32_t adi_fpga9001_DataChain_Data_Set_12IQInterleaved(adi_fpga9001_Device_t *fpga9001,
                                                        adi_common_Port_e port,
                                                        adi_common_ChannelNumber_e channel,
                                                        uint16_t *iqData,
                                                        uint32_t length)
{
    uint32_t i = 0;
    uint32_t ramAddress = 0;
#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    uint32_t *wrBuff = NULL;
#else
    uint32_t wrBuff[ADI_FPGA9001_MAX_CAPTURE_BYTES] = { 0 };
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY

    ADI_PERFORM_VALIDATION(adi_fpga9001_DataChain_Data_Get_12IQInterleavedValidate, fpga9001, port, channel, iqData, length);

    ADI_EXPECT(adi_fpga9001_DataChain_DmaLength_Set, fpga9001, port, channel, length * 2);

#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    wrBuff = calloc(length / 2, sizeof(uint32_t));
    if (wrBuff == NULL)
    {
        ADI_ERROR_REPORT(&fpga9001->common,
                         ADI_COMMON_ERRSRC_API,
                         ADI_COMMON_ERR_MEM_ALLOC_FAIL,
                         ADI_COMMON_ACT_ERR_RESET_FULL,
                         NULL,
                         "Unable to allocate memory");
        ADI_API_RETURN(fpga9001);
    }
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY

    ADI_EXPECT(adi_fpga9001_hal_wrapper_DmaRelativeAddress_Get, fpga9001, fpga9001_Dma_Get(fpga9001, port, channel), &ramAddress);
    if (ramAddress == (uint32_t) - 1)
    {
#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
        free(wrBuff);
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY
        ADI_API_RETURN(fpga9001);
    }

    for (i = 0; i < length / 2; i++)
    {
        wrBuff[i] = (uint32_t)(iqData[2 * i] << 16) | (uint32_t)iqData[2 * i + 1];
    }

    adi_fpga9001_hal_wrapper_Ram_Write(fpga9001, ramAddress, wrBuff, length / 2);

#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    free(wrBuff);
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY

    ADI_API_RETURN(fpga9001);
}

int32_t adi_fpga9001_DataChain_Data_Set_8S(adi_fpga9001_Device_t *fpga9001,
                                           adi_common_Port_e port,
                                           adi_common_ChannelNumber_e channel,
                                           uint8_t *symbols,
                                           uint32_t length)
{
    uint32_t i = 0;
    uint32_t ramAddress = 0;
#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    uint32_t *wrBuff = NULL;
#else
    uint32_t wrBuff[ADI_FPGA9001_MAX_CAPTURE_BYTES] = { 0 };
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY

    ADI_PERFORM_VALIDATION(adi_fpga9001_DataGet_SValidate, fpga9001, port, channel, symbols, length);

    ADI_EXPECT(adi_fpga9001_DataChain_DmaLength_Set, fpga9001, port, channel, length);

#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    wrBuff = calloc(length / 4, sizeof(uint32_t));
    if (wrBuff == NULL)
    {
        ADI_ERROR_REPORT(&fpga9001->common,
                         ADI_COMMON_ERRSRC_API,
                         ADI_COMMON_ERR_MEM_ALLOC_FAIL,
                         ADI_COMMON_ACT_ERR_RESET_FULL,
                         NULL,
                         "Unable to allocate memory");
        ADI_API_RETURN(fpga9001);
    }
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY

    ADI_EXPECT(adi_fpga9001_hal_wrapper_DmaRelativeAddress_Get, fpga9001, fpga9001_Dma_Get(fpga9001, port, channel), &ramAddress);
    if (ramAddress == (uint32_t) - 1)
    {
#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
        free(wrBuff);
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY
        ADI_API_RETURN(fpga9001);
    }

    for (i = 0; i < length / 4; i++)
    {
        wrBuff[i] = (uint32_t)(symbols[4 * i + 3] << 24) |
                    (uint32_t)(symbols[4 * i + 2] << 16) |
                    (uint32_t)(symbols[4 * i + 1] << 8)  |
                    (uint32_t)symbols[4 * i];
    }

    adi_fpga9001_hal_wrapper_Ram_Write(fpga9001, ramAddress, wrBuff, length / 4);

#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    free(wrBuff);
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY

    ADI_API_RETURN(fpga9001);
}

int32_t adi_fpga9001_DataChain_Data_Set_2S(adi_fpga9001_Device_t *fpga9001,
                                           adi_common_Port_e port,
                                           adi_common_ChannelNumber_e channel,
                                           uint8_t *symbols,
                                           uint32_t length)
{
    uint32_t i = 0;
    uint32_t ramAddress = 0;
#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    uint32_t *wrBuff = NULL;
#else
    uint32_t wrBuff[ADI_FPGA9001_MAX_CAPTURE_BYTES] = { 0 };
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY

    ADI_PERFORM_VALIDATION(adi_fpga9001_DataGet_SValidate, fpga9001, port, channel, symbols, length);

    ADI_EXPECT(adi_fpga9001_DataChain_DmaLength_Set, fpga9001, port, channel, length);

#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    wrBuff = calloc(length / 4, sizeof(uint32_t));
    if (wrBuff == NULL)
    {
        ADI_ERROR_REPORT(&fpga9001->common,
                         ADI_COMMON_ERRSRC_API,
                         ADI_COMMON_ERR_MEM_ALLOC_FAIL,
                         ADI_COMMON_ACT_ERR_RESET_FULL,
                         NULL,
                         "Unable to allocate memory");
        ADI_API_RETURN(fpga9001);
    }
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY

    ADI_EXPECT(adi_fpga9001_hal_wrapper_DmaRelativeAddress_Get, fpga9001, fpga9001_Dma_Get(fpga9001, port, channel), &ramAddress);
    if (ramAddress == (uint32_t) - 1)
    {
#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
        free(wrBuff);
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY
        ADI_API_RETURN(fpga9001);
    }

    for (i = 0; i < length / 4; i++)
    {
        wrBuff[i] = (uint32_t)((symbols[4 * i + 3] & 0x03) << 24) |
                    (uint32_t)((symbols[4 * i + 2] & 0x03) << 16) |
                    (uint32_t)((symbols[4 * i + 1] & 0x03) << 8) |
                    (uint32_t)(symbols[4 * i] & 0x03);
    }

    adi_fpga9001_hal_wrapper_Ram_Write(fpga9001, ramAddress, wrBuff, length / 4);

#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    free(wrBuff);
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY

    ADI_API_RETURN(fpga9001);
}

static __maybe_unused int32_t adi_fpga9001_DataChain_HopTable_Set_8S_Validate(adi_fpga9001_Device_t *fpga9001,
                                                               uint8_t spiPackedFhTable[],
                                                               uint32_t byteCount)
{
    ADI_NULL_PTR_RETURN(&fpga9001->common, spiPackedFhTable);
    ADI_ENTRY_PTR_ARRAY_EXPECT(fpga9001, spiPackedFhTable, byteCount);
    ADI_API_RETURN(fpga9001);
}

int32_t adi_fpga9001_DataChain_HopTable_Set_8S(adi_fpga9001_Device_t *fpga9001,
                                               uint8_t spiPackedFhTable[],
                                               uint32_t byteCount)
{
    uint32_t i = 0;
    uint32_t ramAddress = 0;
    uint32_t length_32 = 0;
    //uint32_t data_32 = 0;

#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    uint32_t *wrBuff = NULL;
#else
    uint32_t wrBuff[ADI_FPGA9001_MAX_CAPTURE_BYTES] = { 0 };
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY

    ADI_PERFORM_VALIDATION(adi_fpga9001_DataChain_HopTable_Set_8S_Validate, fpga9001, spiPackedFhTable, byteCount);

#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    
    length_32 = (byteCount + 3) / 4;

    wrBuff = calloc(length_32, sizeof(uint32_t));
    if (wrBuff == NULL)
    {
        ADI_ERROR_REPORT(&fpga9001->common,
                         ADI_COMMON_ERRSRC_API,
                         ADI_COMMON_ERR_MEM_ALLOC_FAIL,
                         ADI_COMMON_ACT_ERR_RESET_FULL,
                         NULL,
                         "Unable to allocate memory");
        ADI_API_RETURN(fpga9001);
    }
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY

    ADI_EXPECT(adi_fpga9001_hal_wrapper_DmaRelativeAddress_Get, fpga9001, ADI_FPGA9001_HAL_DMA_SPI, &ramAddress);
    
    for (i = 0; i < length_32; i++)
    {
        wrBuff[i] = (uint32_t)(spiPackedFhTable[4 * i + 3] << 24) |
                    (uint32_t)(spiPackedFhTable[4 * i + 2] << 16) |
                    (uint32_t)(spiPackedFhTable[4 * i + 1] << 8)  |
                    (uint32_t)spiPackedFhTable[4 * i];
    }

    adi_fpga9001_hal_wrapper_Ram_Write(fpga9001, ramAddress, wrBuff, length_32);
    
#ifdef ADI_FPGA9001_USE_DYNAMIC_MEMORY
    free(wrBuff);
#endif // ADI_FPGA9001_USE_DYNAMIC_MEMORY

    ADI_API_RETURN(fpga9001);
}
