/**
 * \file
 * \brief Contains FPGA9001 API configuration and run-time type definitions
 *
 * Copyright 2018-2025 Analog Devices Inc.
 * Released under the FPGA9001 API license, for more information
 * see the "LICENSE.txt" file in this zip file.
 *
 */

#include "adi_common_error.h"

#include "adi_fpga9001.h"
#include "adi_fpga9001_error.h"
#include "adi_fpga9001_hal.h"
#include "adi_fpga9001_gpio.h"
#include "adi_fpga9001_hal_wrapper.h"
#include "fpga9001_utilities.h"
#include "adi_fpga9001_ssi.h"
#include "adi_adrv9001_types.h"
#include "adrv9001_reg_addr_macros.h"

int32_t adi_fpga9001_VersionGet(adi_fpga9001_Device_t *fpga9001,
                                adi_fpga9001_Version_t *version)
{
    uint32_t regVal = 0;

    axi_sysid_top_sys_version_get((void *)fpga9001, AXI_SYSID_ID, &regVal);
    version->major = (uint8_t)((regVal >> 16) & 0xff);
    version->minor = (uint8_t)((regVal >>  8) & 0xff);
    version->patch = (uint8_t)((regVal >>  0) & 0xff);
    ADI_API_RETURN(fpga9001);
}

int32_t adi_fpga9001_Initialize(adi_fpga9001_Device_t *fpga9001Device,
                                adi_adrv9001_Init_t *init,
                                adi_fpga9001_SsiCalibrationCfg_t *ssiCalibration,
                                adi_fpga9001_Mmcm_ClockOutput_Divisor_e adrv9001DeviceClockDivisor)
{
    uint8_t i = 0;
    uint32_t version = 0;
    uint32_t locked = 0;
    axi_adrv9001_clk_params_t clk_params = {0};

    adi_fpga9001_GpioSmaCfg_t smaConfig = { 0 };
    
    uint32_t RX_CHANNELS[] = {
        ADI_ADRV9001_RX1,
        ADI_ADRV9001_RX2,
        ADI_ADRV9001_ORX1,
        ADI_ADRV9001_ORX2,
        ADI_ADRV9001_ILB1,
        ADI_ADRV9001_ILB2,
        ADI_ADRV9001_ELB1,
        ADI_ADRV9001_ELB2};

    bool ssiDelayConfigured = false;
    bool rx1SsiConfigured = false;
    bool rx2SsiConfigured = false;    
    
    uint32_t BITM_RX1 = ADI_ADRV9001_RX1 | ADI_ADRV9001_ORX1 | ADI_ADRV9001_ILB1 | ADI_ADRV9001_ELB1;
    uint32_t BITM_RX2 = ADI_ADRV9001_RX2 | ADI_ADRV9001_ORX2 | ADI_ADRV9001_ILB2 | ADI_ADRV9001_ELB2;

    /* Check fpga9001 info is valid */
    ADI_ENTRY_EXPECT(fpga9001Device);

    for (i = 0; ((i < ADI_ADRV9001_MAX_RXCHANNELS) && (ssiDelayConfigured == false)); i++)
    {
        if (ADRV9001_BF_EQUAL(init->rx.rxInitChannelMask, RX_CHANNELS[i]))
        {
            ADI_EXPECT(adi_fpga9001_ssi_Delay_Configure,
                       fpga9001Device,
                       (adi_fpga9001_SsiType_e)(init->rx.rxChannelCfg[i].profile.rxSsiConfig.ssiType),
                       ssiCalibration);

            ssiDelayConfigured = true;
        }
    }

    /* ideally you want application to report versions, this is meant to be a */
    /* developer friendly api, and users are free to use any version at their own risk */
    /* do not block, simply inform -- better to change it to a NOTE. */

    axi_sysid_top_sys_version_get((void *)fpga9001Device, AXI_SYSID_ID, &version);
    ADI_ERROR_REPORT(&fpga9001Device->common, ADI_COMMON_ERRSRC_API,
                     0, 0, version, "INFO: Platform FPGA version.");

    /* deassert reset to the fpga9001 */
    axi_adrv9001_top_resetb_set((void *)fpga9001Device, AXI_ADRV9001_ID, AXI_ADRV9001_TOP_OFFSET, 0x1);

    smaConfig.smaPin1 = true;
    smaConfig.smaPin2 = true;
    ADI_EXPECT(adi_fpga9001_gpio_smapin_Configure, fpga9001Device, &smaConfig);
    
    /* Configure the RX ssi port using the first found valid port */
    for (i = 0; i < ADI_ADRV9001_MAX_RXCHANNELS; i++)
    {
        if (ADRV9001_BF_EQUAL(init->rx.rxInitChannelMask, RX_CHANNELS[i]))
        {
            if ((BITM_RX1 & RX_CHANNELS[i]) && (!rx1SsiConfigured))
            {
                ADI_EXPECT(adi_fpga9001_ssi_Configure, fpga9001Device, ADI_RX, ADI_CHANNEL_1, init->rx.rxChannelCfg[i].profile.rxInterfaceSampleRate_Hz, init->rx.rxChannelCfg[i].profile.rxOutputRate_Hz, (adi_fpga9001_SsiConfig_t *)(&init->rx.rxChannelCfg[i].profile.rxSsiConfig));
                rx1SsiConfigured = true;
            }
            if ((BITM_RX2 & RX_CHANNELS[i]) && (!rx2SsiConfigured))
            {
                ADI_EXPECT(adi_fpga9001_ssi_Configure, fpga9001Device, ADI_RX, ADI_CHANNEL_2, init->rx.rxChannelCfg[i].profile.rxInterfaceSampleRate_Hz, init->rx.rxChannelCfg[i].profile.rxOutputRate_Hz, (adi_fpga9001_SsiConfig_t *)(&init->rx.rxChannelCfg[i].profile.rxSsiConfig));
                rx2SsiConfigured = true;
            }
        }
    }

    ADI_EXPECT(adi_fpga9001_ssi_Configure, fpga9001Device, ADI_TX, ADI_CHANNEL_1, init->tx.txProfile[0].txInterfaceSampleRate_Hz, init->tx.txProfile[0].txInputRate_Hz, (adi_fpga9001_SsiConfig_t *)(&init->tx.txProfile[0].txSsiConfig));
    ADI_EXPECT(adi_fpga9001_ssi_Configure, fpga9001Device, ADI_TX, ADI_CHANNEL_2, init->tx.txProfile[0].txInterfaceSampleRate_Hz, init->tx.txProfile[0].txInputRate_Hz, (adi_fpga9001_SsiConfig_t *)(&init->tx.txProfile[1].txSsiConfig));

    axi_adrv9001_top_mcs_swreset_set(fpga9001Device, AXI_ADRV9001_ID, AXI_ADRV9001_TOP_OFFSET, 0x0);
    axi_adrv9001_top_reg_mcs_rx_resetn_set(fpga9001Device, AXI_ADRV9001_ID, AXI_ADRV9001_TOP_OFFSET, (uint32_t) -1);
    axi_adrv9001_top_reg_mcs_tx_resetn_set(fpga9001Device, AXI_ADRV9001_ID, AXI_ADRV9001_TOP_OFFSET, (uint32_t) -1);

    clk_params.ref0_clk_freq_hz = (1000*init->clocks.deviceClock_kHz);

    if (ADI_FPGA9001_MMCM_CLKDIV_DISABLED == adrv9001DeviceClockDivisor)
    {
        clk_params.ref1_clk_freq_hz = -1;
    }
    else
    {
        clk_params.ref1_clk_freq_hz = (1000*init->clocks.deviceClock_kHz) >> adrv9001DeviceClockDivisor;
    }

    clk_params.gen_dev_clk_freq_hz = (1000*init->clocks.deviceClock_kHz);

    if (axi_adrv9001_clk_config((void *)fpga9001Device, AXI_ADRV9001_ID, &clk_params, &locked) != 0)
    {
        ADI_ERROR_REPORT(&fpga9001Device->common,
                         ADI_COMMON_ERRSRC_API,
                         ADI_COMMON_ERR_API_FAIL,
                         ADI_COMMON_ACT_ERR_API_NOT_IMPLEMENTED,
                         0,
                         "function axi_adrv9001_clk_config has failed");
        ADI_API_RETURN(fpga9001Device);
    }

    if (locked != 1)
    {
        ADI_ERROR_REPORT(&fpga9001Device->common,
                         ADI_COMMON_ERRSRC_API,
                         ADI_COMMON_ERR_API_FAIL,
                         ADI_COMMON_ACT_ERR_API_NOT_IMPLEMENTED,
                         0,
                         "platform VCO clock NOT locked");
        ADI_API_RETURN(fpga9001Device);
    }

    axi_adrv9001_top_trig0_ext_oe_set((void *)fpga9001Device, AXI_ADRV9001_ID, AXI_ADRV9001_TOP_OFFSET, 0x1);
    axi_adrv9001_top_trig1_ext_oe_set((void *)fpga9001Device, AXI_ADRV9001_ID, AXI_ADRV9001_TOP_OFFSET, 0x1);

    ADI_API_RETURN(fpga9001Device);
}
