/*!
* \file
* \brief TDD types
*
* FPGA9001 API Version: $ADI_FPGA9001_API_VERSION$
*/

/**
 * Copyright 2019-2025 Analog Devices Inc.
 * Released under the FPGA9001 API license, for more information
 * see the "LICENSE.txt" file in this zip file.
 */

#ifndef _ADI_FPGA9001_TDD_TYPES_H_
#define _ADI_FPGA9001_TDD_TYPES_H_

#ifdef __cplusplus
extern "C" {
#endif

#include <stdint.h>
#include <stdbool.h>
   
#include "axi_adrv9001.h"
#include "axi_adrv9001_top_regs.h"
#include "axi_adrv9001_ssi_regs.h"
#include "axi_adrv9001_tdd_regs.h"
#include "axi_adrv9001_gpio_regs.h"
#include "adi_adrv9001_types.h"

/**
* The TDD timing, sequence, and framing parameters are illustrated in the following timing diagram.
* 
* \image html "FPGA TDD timing diagram.png" "FPGA TDD timing diagram"
*/

#define ADI_FPGA9001_TDDSELECT_MAX_COUNT 28

/**
* \brief TDD enables supported
*/
typedef enum adi_fpga9001_TddSelect
{
    ADI_FPGA9001_TDDSELECT_RX1                 = 0,
    ADI_FPGA9001_TDDSELECT_TX1                 = 1,
    ADI_FPGA9001_TDDSELECT_RX2                 = 2,
    ADI_FPGA9001_TDDSELECT_TX2                 = 3,
    ADI_FPGA9001_TDDSELECT_ORX1                = 4,
    ADI_FPGA9001_TDDSELECT_ORX2                = 5,
    ADI_FPGA9001_TDDSELECT_DMA_RX1             = 6,
    ADI_FPGA9001_TDDSELECT_DMA_TX1             = 7,
    ADI_FPGA9001_TDDSELECT_DMA_RX2             = 8,
    ADI_FPGA9001_TDDSELECT_DMA_TX2             = 9,
    ADI_FPGA9001_TDDSELECT_DMA_ORX1            = 10,
    ADI_FPGA9001_TDDSELECT_DMA_ORX2            = 11,
    ADI_FPGA9001_TDDSELECT_GENERAL_PURPOSE_1    = 12,
    ADI_FPGA9001_TDDSELECT_GENERAL_PURPOSE_2    = 13,
    ADI_FPGA9001_TDDSELECT_GENERAL_PURPOSE_3    = 14,
    ADI_FPGA9001_TDDSELECT_GENERAL_PURPOSE_4    = 15,
    ADI_FPGA9001_TDDSELECT_GENERAL_PURPOSE_5    = 16,
    ADI_FPGA9001_TDDSELECT_GENERAL_PURPOSE_6    = 17,
    ADI_FPGA9001_TDDSELECT_SMA1_TRIG           = 18,
    ADI_FPGA9001_TDDSELECT_SMA2_TRIG           = 19,
    ADI_FPGA9001_TDDSELECT_RX1_DMA_TRIG        = 20,
    ADI_FPGA9001_TDDSELECT_TX1_DMA_TRIG        = 21,
    ADI_FPGA9001_TDDSELECT_RX2_DMA_TRIG        = 22,
    ADI_FPGA9001_TDDSELECT_TX2_DMA_TRIG        = 23,
    ADI_FPGA9001_TDDSELECT_ORX1_DMA_TRIG       = 24,
    ADI_FPGA9001_TDDSELECT_ORX2_DMA_TRIG       = 25,
    ADI_FPGA9001_TDDSELECT_SPI_TRIG            = 26,
    ADI_FPGA9001_TDDSELECT_GPIO_TRIG           = 27,
} adi_fpga9001_TddSelect_e;

/**
* \brief Enumerated list of FPGA  TDD trigger sources
*/
typedef enum adi_fpga9001_TddTrigger
{
    ADI_FPGA9001_TDD_TRIGGER_SMA_1      = 0,
    ADI_FPGA9001_TDD_TRIGGER_SMA_2      = 1,
    ADI_FPGA9001_TDD_TRIGGER_MCS        = 2,
    ADI_FPGA9001_TDD_TRIGGER_GPIO       = 3,
    ADI_FPGA9001_TDD_TRIGGER_IMMEDIATE  = 4
} adi_fpga9001_TddTrigger_e;

/**
* \brief TDD modes supported
*/
typedef enum adi_fpga9001_TddMode
{
    ADI_FPGA9001_TDD_ENABLE_LOW   = 0,
    ADI_FPGA9001_TDD_ENABLE_HIGH  = 1,
    ADI_FPGA9001_TDD_ENABLE_AUTO  = 2
} adi_fpga9001_TddMode_e;

/**
* \brief TDD frame timing info
* 
* The TDD counter starts at 0, ends at framePeriod, it can optionally be repeated,
* it can also be prematurely terminated.
*/
typedef struct adi_fpga9001_TddFrameTiming
{
    uint32_t framePeriod;                       /*!< Duration of TDD frame in tdd_clk cycles */
    uint32_t numberFrames;                      /*!< Desired number of TDD frames in sequence */
    bool repeatFrameSequence;                   /*!< Set this value to 1 to repeat the TDD frame sequence (if not, tdd frame comes to an end). */
    uint32_t frameSwitchTime;                   /*!< When current frame == (frameSwitchNum, frameSwitchTime), that frame is prematurely terminated. */
    uint32_t frameSwitchNum;                    /*!< When current frame == (frameSwitchNum, frameSwitchTime), that frame is prematurely terminated. */
    bool frameSwitchEnable;                     /*!< Set this value to 1 to enable frame premature switching. Otherwise, set to 0 */
    adi_fpga9001_TddTrigger_e frameStartTrig;   /*!< Set this to the desired frame synchronization trigger source. */
    bool inMicroSeconds;                        /*!< Set this to true if all parameters are in micro seconds otherwise counts. */
} adi_fpga9001_TddFrameTiming_t;

/**
* \brief Timing of enable assert/deassert within a TDD frame
* 
* All parameters can be between 0 and framePeriod (inclusive). A pair can be disabled by
* setting the assert and deassert parameters to be the same
*/
typedef struct adi_fpga9001_TddEnableTiming
{
    bool enable;                      /*!< Enable TDD auto or keep TDD enable asserted low. */
    uint32_t primaryAssert;           /*!< Number of tdd_clk cycles after start of TDD frame to perform primary assert */
    uint32_t primaryFrameAssert;      /*!< First frame in primary sequence in which enable is asserted */
    uint32_t primaryDeassert;         /*!< Number of tdd_clk cycles after start of TDD frame to perform primary deassert */
    uint32_t primaryFrameDeassert;    /*!< Final frame in primary sequence in which enable is deasserted */
    uint32_t secondaryAssert;         /*!< Number of tdd_clk cycles after start of TDD frame to perform secondary assert */
    uint32_t secondaryFrameAssert;    /*!< First frame in secondary sequence in which enable is asserted */
    uint32_t secondaryDeassert;       /*!< Number of tdd_clk cycles after start of TDD frame to perform secondary deassert */
    uint32_t secondaryFrameDeassert;  /*!< Final frame in secondary sequence in which enable is deasserted */
    bool inMicroSeconds;              /*!< Set this to true if all parameters are in micro seconds otherwise counts. */
    /** 
     * If true, enables are synchronized with the datapath clocks, enable-on-time is rounded to the nearest sample count. 
     * If false, enable-on-time is as specified
     */
    bool enableSyncToSample;
} adi_fpga9001_TddEnableTiming_t;

/**
* \brief structure to hold TDD configuration
*/
typedef struct adi_fpga9001_TddConfig
{
    adi_fpga9001_TddFrameTiming_t frameTiming;
    adi_fpga9001_TddEnableTiming_t enables[ADI_FPGA9001_TDDSELECT_MAX_COUNT];
} adi_fpga9001_TddConfig_t;


#ifdef __cplusplus
}
#endif

#endif  /* _ADI_FPGA9001_TDD_TYPES_H_ */