/**
 * \file
 * \brief Contains top level fpga9001 related function prototypes for TDD specific features
 *
 * FPGA9001 API Version: $ADI_FPGA9001_API_VERSION$
 */
/**
 * Copyright 2015 - 2019 Analog Devices Inc.
 * Released under the FPGA9001 API license, for more information
 * see the "LICENSE.pdf" file in this zip file.
 */

#ifndef _ADI_FPGA9001_TDD_H_
#define _ADI_FPGA9001_TDD_H_

#include "adi_fpga9001_types.h"
#include "adi_fpga9001_tdd_types.h"

#ifdef __cplusplus
extern "C" {
#endif

/**
 * \brief Control TDD enables and assertion modes
 *
 * This function provides software control of TDD (asserted low, high, or hardware modes).
 *
 * \pre The ADRV9001 must be configured for pin enable mode by calling adi_adrv9001_Radio_ChannelEnableMode_Set()
 *
 * \param[in] fpga9001          Context variable - Pointer to FPGA9001 device data structure
 * \param[in] tddSelect         TDD enable to configure.
 * \param[in] mode              The desired mode of the pin
 *
 * \returns A code indicating success (ADI_COMMON_ACT_NO_ACTION) or the required action to recover
 */
int32_t adi_fpga9001_Tdd_Enable_Set(adi_fpga9001_Device_t *fpga9001,
                                    adi_fpga9001_TddSelect_e tddSelect,
                                    adi_fpga9001_TddMode_e mode);
/**
 * \brief Read back the TDD assertion modes
 *
 * \param[in]  fpga9001         Context variable - Pointer to FPGA9001 device data structure
 * \param[in]  tddSelect        TDD enable for which to read assertion mode
 * \param[out] mode             The desired mode of the pin
 *
 * \returns A code indicating success (ADI_COMMON_ACT_NO_ACTION) or the required action to recover
 */
int32_t adi_fpga9001_Tdd_Enable_Get(adi_fpga9001_Device_t *fpga9001,
                                    adi_fpga9001_TddSelect_e tddSelect,
                                    adi_fpga9001_TddMode_e *mode);

/**
 * \brief Configure the FPGA TDD for programmed mode
 *
 * \param[in] fpga9001          Context variable - Pointer to FPGA9001 device data structure
 * \param[in] tddConfig         The desired TDD programmed mode configuration
 *
 * \returns A code indicating success (ADI_COMMON_ACT_NO_ACTION) or the required action to recover
 */
int32_t adi_fpga9001_Tdd_Configure(adi_fpga9001_Device_t *fpga9001,
                                   adi_fpga9001_TddConfig_t *tddConfig);

/**
 * \brief Inspect the FPGA TDD programmed mode configuration
 *
 * \param[in]  fpga9001         Context variable - Pointer to FPGA9001 device data structure
 * \param[out] tddConfig        The current TDD programmed mode configuration
 *
 * \returns A code indicating success (ADI_COMMON_ACT_NO_ACTION) or the required action to recover
 */
int32_t adi_fpga9001_Tdd_Inspect(adi_fpga9001_Device_t *fpga9001,
                                 adi_fpga9001_TddConfig_t *tddConfig);

/**
 * \brief Start the TDD state machine based on the trigger specified in the configuration
 * 
 * \param[in]  fpga9001         Context variable - Pointer to FPGA9001 device data structure
 * 
 * \returns A code indicating success (ADI_COMMON_ACT_NO_ACTION) or the required action to recover
 */
int32_t adi_fpga9001_Tdd_Start(adi_fpga9001_Device_t *fpga9001);

/**
 * \brief Stop the TDD state machine, bringing all enable pins low at the end of the current frame
 * 
 * \param[in]  fpga9001         Context variable - Pointer to FPGA9001 device data structure
 * 
 * \returns A code indicating success (ADI_COMMON_ACT_NO_ACTION) or the required action to recover
 */
int32_t adi_fpga9001_Tdd_Stop(adi_fpga9001_Device_t *fpga9001);

#ifdef __cplusplus
}
#endif

#endif /* _ADI_FPGA9001_TDD_H_ */