/**
 * \file
 * \brief Types for FPGA9001 Synchronous Serial Interface (SSI) configuration
 *
 * ADRV9001 API Version: $ADI_FPGA9001_API_VERSION$
 */

/**
 * Copyright 2018 Analog Devices Inc.
 * Released under the FPGA9001 API license, for more information
 * see the "LICENSE.txt" file in this zip file.
 */

#ifndef _ADI_FPGA9001_SSI_TYPES_H_
#define _ADI_FPGA9001_SSI_TYPES_H_

#ifdef __KERNEL__
#include <linux/types.h>
#else
#include <stdint.h>
#include <stdbool.h>
#endif

#ifdef __cplusplus
extern "C" {
#endif

#define ADI_FPGA9001_NUM_CHANNELS 2

/**
 * \brief Enumeration of SSI type
 */
typedef enum adi_fpga9001_SsiType
{
    ADI_FPGA9001_SSI_TYPE_DISABLE = 0,           /*!< Disable SSI Type */
    ADI_FPGA9001_SSI_TYPE_CMOS = 1,              /*!< CMOS SSI Type for both Channel 1 and Channel 2. In 1T1R case, CMOS for Channel1/Channel2 */
    ADI_FPGA9001_SSI_TYPE_LVDS = 2,              /*!< LVDS SSI Type for both Channel 1 and Channel 2. In 1T1R case, LVDS for Channel1/Channel2 */
	ADI_FPGA9001_SSI_TYPE_CH1CMOS_CH2LVDS = 3,   /*!< CMOS for Channel1 and LVDS for Channel 2 */
	ADI_FPGA9001_SSI_TYPE_CH1LVDS_CH2CMOS = 4,   /*!< LVDS for Channel1 and CMOS for Channel 2 */
    ADI_FPGA9001_SSI_TYPE_CMOS_COMMON_CLOCK = 5, /*!< CMOS SSI Type for both Channels. txRefClockPins disabled/repurposed for GPIO. Transmit and receive share sameclock */
    ADI_FPGA9001_SSI_TYPE_LVDS_COMMON_CLOCK = 6  /*!< LVDS SSI Type for both Channels. txRefClockPins disabled/repurposed for GPIO. Transmit and receive share sameclock */
} adi_fpga9001_SsiType_e;

/**
 * \brief SSI formats
 * \note Copy of adi_adrv9001_SsiDataFormat_e
 */
typedef enum adi_fpga9001_SsiFormat
{
    ADI_FPGA9001_SSI_FORMAT_2_BIT_SYMBOL_DATA,  /*!< 2 bit symbol data (CMOS) */
    ADI_FPGA9001_SSI_FORMAT_8_BIT_SYMBOL_DATA,  /*!< 8 bit symbol data (CMOS) */
    ADI_FPGA9001_SSI_FORMAT_16_BIT_SYMBOL_DATA, /*!< 16 bit symbol data (CMOS) */
    ADI_FPGA9001_SSI_FORMAT_12_BIT_I_Q_DATA,    /*!< 12 bit I/Q data (LVDS) */
    ADI_FPGA9001_SSI_FORMAT_16_BIT_I_Q_DATA,    /*!< 16 bit I/Q data (CMOS/LVDS) */

    /*!< 15 bit I/Q data, 1 bit Rx Gain Change (CMOS/LVDS) */
    ADI_FPGA9001_SSI_FORMAT_15_BIT_I_Q_DATA_1_BIT_GAIN_CHANGE = 5,

    /*!< 22 bit I/Q data, 1 bit Rx Gain Change, 8 bit Rx Gain Index (CMOS/LVDS) */
    ADI_FPGA9001_SSI_FORMAT_22_BIT_I_Q_DATA_1_BIT_GAIN_CHANGE_8_BIT_GAIN_INDEX = 6
} adi_fpga9001_SsiFormat_e;

/**
 * \brief Type of data to transmit over SSI
 */
typedef enum adi_fpga9001_SsiTestModeData
{
    ADI_FPGA9001_SSI_TESTMODE_DATA_NORMAL = 0,      /*!< No check available for this data type */
    ADI_FPGA9001_SSI_TESTMODE_DATA_FIXED_PATTERN,
    ADI_FPGA9001_SSI_TESTMODE_DATA_RAMP_NIBBLE,     /*!< CSSI-Only */
    ADI_FPGA9001_SSI_TESTMODE_DATA_RAMP_16_BIT,     /*!< CSSI and LSSI */
    ADI_FPGA9001_SSI_TESTMODE_DATA_PRBS15,          /*!< LSSI-Only */
    ADI_FPGA9001_SSI_TESTMODE_DATA_PRBS7,           /*!< LSSI-Only */
} adi_fpga9001_SsiTestModeData_e;

/**
 * \brief Enumeration of the SSI Strobe
 */
typedef enum adi_fpga9001_SsiStrobeType
{
    ADI_FPGA9001_SSI_SHORT_STROBE = 0, /*!< Short SSI Strobe */
    ADI_FPGA9001_SSI_LONG_STROBE  = 1  /*!< Long SSI Strobe */
} adi_fpga9001_SsiStrobeType_e;

/**
 * \brief Enumeration of the SSI number of lane
 */
typedef enum adi_fpga9001_SsiNumLane
{
    ADI_FPGA9001_SSI_1_LANE = 0, /*!< 1 lane (CMOS/LVDS) */
    ADI_FPGA9001_SSI_2_LANE = 1, /*!< 2 lane (LVDS) */
    ADI_FPGA9001_SSI_4_LANE = 2  /*!< 4 lane (CMOS) */
} adi_fpga9001_SsiNumLane_e;

/**
 * \brief Tx SSI reference clock pin option enumeration
 */
typedef enum adi_fpga9001_SsiTxRefClockPin
{
    ADI_FPGA9001_SSI_TX_REF_CLOCK_PIN_DISABLED = 0u,                 /*!< Tx reference clock out pin disabled */
    ADI_FPGA9001_SSI_TX_REF_CLOCK_PIN_LVDS_DCLK_OUT_ENABLED = 1u,    /*!< Tx reference clock out pin TX_DCLK_OUT+/- enabled in LVDS mode */
    ADI_FPGA9001_SSI_TX_REF_CLOCK_PIN_CMOS_DCLK_OUT_P_ENABLED = 1u,  /*!< Tx reference clock out pin TX_DCLK_OUT+   enabled in CMOS mode */
    ADI_FPGA9001_SSI_TX_REF_CLOCK_PIN_CMOS_DCLK_OUT_N_ENABLED = 2u,  /*!< Tx reference clock out pin TX_DCLK_OUT-   enabled in CMOS mode */
    ADI_FPGA9001_SSI_TX_REF_CLOCK_PIN_CMOS_STROBE_IN_N_ENABLED = 3u, /*!< Tx reference clock out pin TX_STROBE_IN-  enabled in CMOS mode */
} adi_fpga9001_SsiTxRefClockPin_e;
    
/**
 * \brief Data structure to hold FPGA9001 SSI configuration.
 */
typedef struct adi_fpga9001_SsiConfig
{
    adi_fpga9001_SsiType_e       ssiType;                   /*!< SSI type */
    adi_fpga9001_SsiFormat_e     ssiDataFormatSel;          /*!< SSI data format */
    adi_fpga9001_SsiNumLane_e    numLaneSel;                /*!< SSI number of lanes */
    adi_fpga9001_SsiStrobeType_e strobeType;				/*!< SSI strobe type */
    uint8_t						 lsbFirst;					/*!< SSI LSB first */
    uint8_t						 qFirst;					/*!< SSI Q data first */
    adi_fpga9001_SsiTxRefClockPin_e	 txRefClockPin;         /*!< Reference clock pin select  */
    bool                         lvdsIBitInversion;         /*!< LVDS SSI I bit inversion */
    bool                         lvdsQBitInversion;         /*!< LVDS SSI Q bit inversion */
    bool                         lvdsStrobeBitInversion;    /*!< LVDS SSI Strobe bit inversion */
    uint8_t						 lvdsUseLsbIn12bitMode;		/*!< LVDS use LSB in 12 bit mode */
    bool						 lvdsRxClkInversionEn;      /*!< LVDS Rx clock inversion enable */
    bool                         cmosDdrPosClkEn;           /*!< CMOS DDR positive clock enable */
    bool                         cmosClkInversionEn;        /*!< CMOS DDR clock inversion enable */
    bool                         ddrEn;                     /*!< Double Data Rate (DDR) enable */
    bool                         rxMaskStrobeEn;            /*!< Rx enable for strobe mask */
} adi_fpga9001_SsiConfig_t;

/**
 * \brief Data structure to transmit FPGA9001 SSI test mode configuration
 */
typedef struct adi_fpga9001_TxSsiTestModeCfg
{
    adi_fpga9001_SsiTestModeData_e testData;   /*!< Type of data to transmit over SSI */
    uint32_t fixedDataPatternToTransmit;       /*!< Value of Fixed pattern to transmit over interface */
} adi_fpga9001_TxSsiTestModeCfg_t;

/**
 * \brief Data structure to receive FPGA9001 SSI test mode configuration
 */
typedef struct adi_fpga9001_RxSsiTestModeCfg
{
    adi_fpga9001_SsiTestModeData_e testData;   /*!< Type of data to receive over SSI and check */
    uint32_t fixedDataPatternToCheck;          /*!< Value of Fixed pattern to check against pattern recevied over interface */
} adi_fpga9001_RxSsiTestModeCfg_t;

/**
 * \brief Data structure to hold FPGA9001 CMOS delay calibration configuration for each channel.
 */
typedef struct adi_fpga9001_SsiCalibrationCfg
{
    uint8_t rxStrobeDelay[ADI_FPGA9001_NUM_CHANNELS];   /*!< Strobe delay for Rx1/Rx2 channels */
    uint8_t rxIDataDelay[ADI_FPGA9001_NUM_CHANNELS];    /*!< I data delay for Rx1/Rx2 channels */
    uint8_t rxQDataDelay[ADI_FPGA9001_NUM_CHANNELS];    /*!< Q data delay for Rx1/Rx2 channels */

    uint8_t txClkDelay[ADI_FPGA9001_NUM_CHANNELS];      /*!< CLK delay for Tx1/Tx2 channels */
    uint8_t txStrobeDelay[ADI_FPGA9001_NUM_CHANNELS];   /*!< Strobe delay for Tx1/Tx2 channels */
    uint8_t txIDataDelay[ADI_FPGA9001_NUM_CHANNELS];    /*!< I data delay for Tx1/Tx2 channels */
    uint8_t txQDataDelay[ADI_FPGA9001_NUM_CHANNELS];    /*!< Q data delay for Tx1/Tx2 channels */
    bool txClkInverted[ADI_FPGA9001_NUM_CHANNELS];      /*!< Whether to invert the Tx1/Tx2 SSI clock polarity */
} adi_fpga9001_SsiCalibrationCfg_t;

#ifdef __cplusplus
}
#endif

#endif // !_ADI_FPGA9001_SSI_TYPES_H_